/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.window;

import org.apache.flink.api.java.functions.KeySelector;
import org.apache.flink.dropwizard.metrics.DropwizardMeterWrapper;
import org.apache.flink.metrics.Counter;
import org.apache.flink.metrics.Meter;
import org.apache.flink.runtime.state.StateSnapshotContext;
import org.apache.flink.streaming.runtime.streamrecord.StreamRecord;
import org.apache.flink.table.api.window.TimeWindow;
import org.apache.flink.table.dataformat.BaseRow;
import org.apache.flink.table.runtime.window.aligned.AlignedWindowAggregator;
import org.apache.flink.table.runtime.window.aligned.AlignedWindowTrigger;
import org.apache.flink.table.runtime.window.assigners.SlidingWindowAssigner;
import org.apache.flink.table.runtime.window.assigners.WindowAssigner;

import java.util.Collection;
import java.util.Collections;

/**
 * An operator that used as local window aggregate operator of two stage window optimization.
 * It accumulates input data into local window buffer, and flush the buffer both when the size
 * of buffer exceeds the threshold and when watermark comes.
 */
public class LocalAlignedWindowOperator extends AbstractAlignedWindowOperator {

	private static final long serialVersionUID = 1L;

	/** KeySelector is used to extract key for aligned window aggregator buffer. */
	private final KeySelector<BaseRow, BaseRow> keySelector;

	private final boolean finishBundleBeforeSnapshot;

	private transient Counter numLateRecordsDropped;
	private transient Meter lateRecordsDroppedRate;

	public LocalAlignedWindowOperator(
		AlignedWindowAggregator<BaseRow, TimeWindow, BaseRow> windowRunner,
		WindowAssigner<TimeWindow> windowAssigner,
		AlignedWindowTrigger windowTrigger,
		int rowtimeIndex,
		boolean finishBundleBeforeSnapshot,
		KeySelector<BaseRow, BaseRow> keySelector) {
		super(windowRunner, windowAssigner, windowTrigger, rowtimeIndex);

		this.finishBundleBeforeSnapshot = finishBundleBeforeSnapshot;
		this.keySelector = keySelector;
	}

	@Override
	public void open() throws Exception {
		super.open();
		this.numLateRecordsDropped = metrics.counter(LATE_ELEMENTS_DROPPED_METRIC_NAME);
		this.lateRecordsDroppedRate = metrics.meter(
			LATE_ELEMENTS_DROPPED_RATE_METRIC_NAME,
			new DropwizardMeterWrapper(new com.codahale.metrics.Meter()));
	}

	@Override
	public void processElement(StreamRecord<BaseRow> record) throws Exception {
		// prepare inputRow and timestamp
		BaseRow inputRow = record.getValue();
		// TODO: support processing time in the future
		long timestamp = inputRow.getLong(this.rowtimeIndex);

		Collection<TimeWindow> windows;
		if (windowAssigner instanceof SlidingWindowAssigner) {
			// using pane optimization to reduce network shuffle.
			TimeWindow pane = ((SlidingWindowAssigner) windowAssigner)
				.assignPane(inputRow, timestamp);
			windows = Collections.singletonList(pane);
		} else {
			windows = windowAssigner.assignWindows(inputRow, timestamp);
		}

		boolean isElementDropped = true;
		for (TimeWindow window : windows) {
			if (!isWindowLate(window, getCurrentWatermark())) {
				isElementDropped = false;
				windowRunner.addElement(getKey(inputRow), window, inputRow);
			}
			// TODO: support late arrival trigger
		}

		if (isElementDropped) {
			numLateRecordsDropped.inc();
			lateRecordsDroppedRate.markEvent();
		}
	}

	@Override
	protected void advanceWatermark(long watermark) throws Exception {
		// flush all window buffer acc to downstream directly no matter
		// whether current watermark passes the window end.
		finishBundle();
	}

	@Override
	public void prepareSnapshotPreBarrier(long checkpointId) throws Exception {
		if (finishBundleBeforeSnapshot) {
			// flush all window buffer acc to downstream directly
			finishBundle();
		}
	}

	private void finishBundle() throws Exception {
		for (TimeWindow window: windowRunner.windows()) {
			windowRunner.fireWindow(window);
		}
		windowRunner.expireAllWindows();
	}

	@Override
	public void snapshotState(StateSnapshotContext context) throws Exception {
		super.snapshotState(context);
		if (!finishBundleBeforeSnapshot) {
			windowRunner.snapshot();
		}
	}

	@Override
	protected BaseRow getKey(BaseRow input) throws Exception {
		return this.keySelector.getKey(input);
	}

	@Override
	public boolean requireState() {
		// if finishBundleBeforeSnapshot, then no state requirement
		return !finishBundleBeforeSnapshot;
	}
}
