/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.window.aligned;

import org.apache.flink.api.common.typeinfo.TypeInformation;
import org.apache.flink.table.api.window.TimeWindow;
import org.apache.flink.table.codegen.GeneratedSubKeyedAggsHandleFunction;
import org.apache.flink.table.dataformat.BaseRow;
import org.apache.flink.table.runtime.functions.ExecutionContext;
import org.apache.flink.table.runtime.functions.SubKeyedAggsHandleFunction;

/**
 * An implementation of {@link BufferedAlignedWindowAggregator} for global phase of
 * two phase window aggregate.
 */
public final class GlobalAlignedWindowAggregator extends BufferedAlignedWindowAggregator {

	private static final long serialVersionUID = 8587640048900077451L;

	private final GeneratedSubKeyedAggsHandleFunction<TimeWindow> generatedLocalWindowAgg;
	private transient SubKeyedAggsHandleFunction<TimeWindow> localWindowAggregator;

	public GlobalAlignedWindowAggregator(
		TypeInformation<BaseRow> accTypeInfo,
		TypeInformation<BaseRow> aggResultTypeInfo,
		GeneratedSubKeyedAggsHandleFunction<TimeWindow> generatedLocalWindowAgg,
		GeneratedSubKeyedAggsHandleFunction<TimeWindow> generatedGlobalWindowAgg,
		long minibatchSize,
		boolean sendRetraction) {

		super(accTypeInfo, aggResultTypeInfo,
			generatedGlobalWindowAgg, minibatchSize, sendRetraction);
		this.generatedLocalWindowAgg = generatedLocalWindowAgg;
	}

	@Override
	public void open(Context context) throws Exception {
		super.open(context);

		ExecutionContext ctx = context.getExecutionContext();
		// the type cast is needed here, otherwise compile will complain
		this.localWindowAggregator = (SubKeyedAggsHandleFunction<TimeWindow>) generatedLocalWindowAgg.newInstance(
			ctx.getRuntimeContext().getUserCodeClassLoader());
		this.localWindowAggregator.open(ctx);
	}

	@Override
	protected BaseRow processInput(TimeWindow window, BaseRow input, BaseRow acc) throws Exception {
		// null namespace means using heap data views
		localWindowAggregator.setAccumulators(null, acc);
		localWindowAggregator.merge(window, input);
		return localWindowAggregator.getAccumulators();
	}
}
