/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.window.aligned;

import org.apache.flink.api.common.typeinfo.TypeInformation;
import org.apache.flink.table.api.window.TimeWindow;
import org.apache.flink.table.codegen.GeneratedSubKeyedAggsHandleFunction;
import org.apache.flink.table.dataformat.BaseRow;
import org.apache.flink.table.dataformat.util.BaseRowUtil;

/**
 * An implementation of {@link BufferedAlignedWindowAggregator} for one phase window aggregate.
 */
public class OnePhaseAlignedWindowAggregator extends BufferedAlignedWindowAggregator {

	private static final long serialVersionUID = 2229485674427835761L;

	public OnePhaseAlignedWindowAggregator(
		TypeInformation<BaseRow> accTypeInfo,
		TypeInformation<BaseRow> aggResultTypeInfo,
		GeneratedSubKeyedAggsHandleFunction<TimeWindow> generatedWindowAggregator,
		long minibatchSize,
		boolean sendRetraction) {

		super(accTypeInfo, aggResultTypeInfo, generatedWindowAggregator, minibatchSize, sendRetraction);
	}

	@Override
	protected BaseRow processInput(TimeWindow window, BaseRow input, BaseRow acc) throws Exception {
		// null namespace means using heap data views
		windowAggregator.setAccumulators(null, acc);
		if (BaseRowUtil.isAccumulateMsg(input)) {
			windowAggregator.accumulate(input);
		} else {
			windowAggregator.retract(input);
		}
		return windowAggregator.getAccumulators();
	}
}
