/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.sinks

import org.apache.flink.table.connector.DefinedDistribution

import java.util

/**
  * An abstract class with trait about partitionable table sink. This is mainly used for
  * static partitions. For sql statement:
  * <pre>
  * <code>
  * INSERT INTO A PARTITION(a='ab', b='cd') select c, d from B
  * </code>
  * </pre>
  * We Assume the A has partition columns as &lt;a&gt;, &lt;b&gt;, &lt;c&gt;.
  * The columns &lt;a&gt; and &lt;b&gt; are called static partition columns, while c is called
  * dynamic partition column.
  * Note:
  *   1. current class implementation don't support partition pruning which means constant
  * partition columns will still be kept in result row
  *   2. Map<String, String> returned by getStaticPartitions method in which key is the partition
  *   column name and the value is the literal value. There is an special case that string literal
  *   will be converted to 'value'. Take the above sql as an example, the value will be 'ab' instead
  *   of ab.
  */
abstract class PartitionableTableSink extends DefinedDistribution {
  private[this] var staticPartitions: util.LinkedHashMap[String, String] = _

  /**
    * With static partition info from sql statement.
    *
    * @param partitions mapping from static partition column names to string literal values.
    */
  def withStaticPartitions(partitions: util.LinkedHashMap[String, String]): Unit = {
    staticPartitions = partitions
  }

  /**
    * @return static partitions from sql statement, never returns null.
    */
  final def getStaticPartitions: util.LinkedHashMap[String, String] = {
    if (staticPartitions == null) {
      new util.LinkedHashMap[String, String]()
    } else {
      staticPartitions
    }
  }
}
