/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.util;

import org.apache.flink.table.api.BatchTableEnvironment;
import org.apache.flink.table.plan.nodes.exec.BatchExecNode;
import org.apache.flink.table.plan.nodes.exec.ExecNode;
import org.apache.flink.table.plan.nodes.exec.batch.BatchExecNodeVisitorImpl;
import org.apache.flink.table.plan.nodes.physical.batch.BatchExecCalc;
import org.apache.flink.table.plan.nodes.physical.batch.BatchExecExchange;
import org.apache.flink.table.plan.nodes.physical.batch.BatchExecHashJoinBase;
import org.apache.flink.table.plan.nodes.physical.batch.BatchExecSort;
import org.apache.flink.table.plan.nodes.physical.batch.BatchExecSortMergeJoinBase;
import org.apache.flink.table.plan.nodes.process.DAGProcessContext;
import org.apache.flink.table.plan.nodes.process.DAGProcessor;

import java.util.List;

/**
 * Let calc output BinaryRow directly, avoiding the serializer converting BaseRow to BinaryRow
 * in Exchange.
 */
public class CalcOutputBinaryRowProcessor extends BatchExecNodeVisitorImpl implements DAGProcessor {

	@Override
	public List<ExecNode<?, ?>> process(List<ExecNode<?, ?>> sinkNodes, DAGProcessContext context) {
		List<BatchExecNode<?>> batchExecNodes = DAGProcessor.getNonSinkBatchExecNodes(sinkNodes);
		batchExecNodes.forEach(b -> b.accept(this));
		return sinkNodes;
	}

	@Override
	public void visit(BatchExecExchange exchange) {
		super.visit(exchange);
		enableCalcOutputBinaryRow(exchange.getInputNodes().get(0));
	}

	@Override
	public void visit(BatchExecSort sort) {
		super.visit(sort);
		enableCalcOutputBinaryRow(sort.getInputNodes().get(0));
	}

	@Override
	public void visit(BatchExecSortMergeJoinBase join) {
		super.visit(join);
		if (!join.leftSorted()) {
			enableCalcOutputBinaryRow(join.getInputNodes().get(0));
		}
		if (!join.rightSorted()) {
			enableCalcOutputBinaryRow(join.getInputNodes().get(1));
		}
	}

	@Override
	public void visit(BatchExecHashJoinBase join) {
		super.visit(join);
		if (join.leftIsBuild()) {
			enableCalcOutputBinaryRow(join.getInputNodes().get(0));
		} else {
			enableCalcOutputBinaryRow(join.getInputNodes().get(1));
		}
	}

	private void enableCalcOutputBinaryRow(ExecNode<BatchTableEnvironment, ?> input) {
		if (input instanceof BatchExecCalc) {
			((BatchExecCalc) input).enableOutputBinaryRow();
		}
	}
}
