/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.cloud.context.statistics;

import com.alibaba.cloud.context.AliCloudConfiguration;
import com.alibaba.cloud.context.AlicloudSdkProduct;
import com.alibaba.cloud.context.edas.*;
import com.alibaba.cloud.context.utils.NetUtils;
import com.aliyuncs.edas.model.v20170801.GetSecureTokenResponse;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author xiaolongzuo
 */
public class StatisticsTask {

    private static final Long DEFAULT_INTERVAL = 1000L * 60L * 10L;

    private static final String KEYS_UUID = "vmUuid";

    private static final String KEYS_IP = "ip";

    private static final String KEYS_COMPONENTS = "components";

    private static final String KEYS_TENANT_ID = "tenantId";

    private static final String KEYS_EDAS_ID = "edasId";

    private static final String KEYS_USER_ID = "userId";

    private static final String KEYS_VERSION = "version";

    private static final String KEYS_SPRING_BOOT_VERSION = "springBootVersion";

    private static final AtomicBoolean started = new AtomicBoolean(false);

    private long interval;

    private AliCloudEdasSdk aliCloudEdasSdk;

    private EdasConfiguration edasConfiguration;

    private List<String> components;

    private EdasChangeOrderConfiguration edasChangeOrderConfiguration = EdasChangeOrderConfigurationFactory
        .getEdasChangeOrderConfiguration();

    public StatisticsTask(AliCloudConfiguration aliCloudConfiguration, EdasConfiguration edasConfiguration,
                          List<String> components) {
        this(DEFAULT_INTERVAL, null, edasConfiguration, components);
        try {
            this.aliCloudEdasSdk = AliCloudEdasSdkFactory.getDefaultAliCloudEdasSdk(aliCloudConfiguration,
                edasConfiguration.getRegionId());
        } catch (Exception e) {
            //ignored
        }
    }

    public StatisticsTask(AliCloudEdasSdk aliCloudEdasSdk, EdasConfiguration edasConfiguration,
                          List<String> components) {
        this(DEFAULT_INTERVAL, aliCloudEdasSdk, edasConfiguration, components);
    }

    public StatisticsTask(long interval, AliCloudEdasSdk aliCloudEdasSdk, EdasConfiguration edasConfiguration,
                          List<String> components) {
        this.interval = interval;
        this.aliCloudEdasSdk = aliCloudEdasSdk;
        this.edasConfiguration = edasConfiguration;
        this.components = components == null ? new ArrayList<String>() : components;
    }

    public void start() {
        if (components == null || components.size() == 0) {
            return;
        }
        if (started.compareAndSet(false, true)) {
            if (edasChangeOrderConfiguration.isEdasManaged()) {
                Map<String, String> params = new HashMap<>(10);
                params.put(KEYS_UUID, getUuid());
                params.put(KEYS_IP, getIp());
                params.put(KEYS_COMPONENTS, getComponents());
                params.putAll(getIds());
                params.put(KEYS_VERSION, getVersion());
                params.put(KEYS_SPRING_BOOT_VERSION, getSpringBootVersion());
                Thread thread = new Thread(new ManagedReportTask(interval, params));
                thread.setDaemon(true);
                thread.setName("Alicloud Managed Report Thread");
                thread.start();
            } else {
                Map<String, String> params = new HashMap<>(10);
                params.put("scaPing", "true");
                params.put("params", getParamsString());
                Thread thread = new Thread(new UnmanagedReportTask(interval, params));
                thread.setDaemon(true);
                thread.setName("Alicloud Unmanaged Report Thread");
                thread.start();
            }
        }
    }

    private boolean isSpringCloud() {
        return components != null && components.size() > 0 && components.get(0).startsWith("SC-");
    }

    private boolean isDubbo() {
        return components != null && components.size() > 0 && components.get(0).startsWith("DUBBO-");
    }

    private String getParamsString() {
        List<String> paramsList = new ArrayList<>();
        paramsList.add(getUuid());
        paramsList.add(getIp());
        paramsList.add(getIdsString());
        paramsList.add(getComponents());
        paramsList.add(getVersions());

        StringBuffer paramsString = new StringBuffer();
        for (String param : paramsList) {
            paramsString.append("|").append(param);
        }
        return paramsString.substring(1);
    }

    private String getUuid() {
        String uuid = "";
        try {
            String fileSeparator = System.getProperty("file.separator");
            String userHome = System.getProperty("user.home");
            File file = new File(
                userHome + fileSeparator + ".sca" + fileSeparator + "sca_temp.uid");
            if (file.exists()) {
                BufferedReader bufferedReader = new BufferedReader(new FileReader(file));
                uuid = bufferedReader.readLine().trim();
            } else {
                uuid = UUID.randomUUID().toString();
                File dir = new File(userHome + fileSeparator + ".sca");
                if (!dir.exists()) {
                    dir.mkdir();
                }
                FileWriter fileWriter = new FileWriter(file);
                fileWriter.write(uuid);
                fileWriter.flush();
                fileWriter.close();
            }
        } catch (Exception e) {
            // ignored
        }
        return uuid;
    }

    private String getIp() {
        return NetUtils.findFirstNonLoopbackAddress().getHostAddress();
    }

    private String getIdsString() {
        String tenantId = "";
        String edasId = "";
        String userId = "";
        Map<String, String> idsMap = getIds();
        if (idsMap.get(KEYS_TENANT_ID) != null) {
            tenantId = idsMap.get(KEYS_TENANT_ID);
        }
        if (idsMap.get(KEYS_EDAS_ID) != null) {
            tenantId = idsMap.get(KEYS_EDAS_ID);
        }
        if (idsMap.get(KEYS_USER_ID) != null) {
            tenantId = idsMap.get(KEYS_USER_ID);
        }
        return edasId + "|" + userId + "|" + tenantId;
    }

    private Map<String, String> getIds() {
        Map<String, String> idsMap = new HashMap<>(3);
        if (edasChangeOrderConfiguration.isEdasManaged()) {
            idsMap.put(KEYS_TENANT_ID, edasChangeOrderConfiguration.getTenantId());
            return idsMap;
        }
        if (aliCloudEdasSdk == null || edasConfiguration == null
            || edasConfiguration.getNamespace() == null) {
            return idsMap;
        }
        if (!aliCloudEdasSdk.getAliCloudSdk().contains(AlicloudSdkProduct.EDAS,
            edasConfiguration.getRegionId())) {
            return idsMap;
        }
        try {
            GetSecureTokenResponse.SecureToken secureToken = aliCloudEdasSdk
                .getSecureToken(edasConfiguration.getNamespace());
            if (secureToken != null) {
                if (secureToken.getTenantId() != null) {
                    idsMap.put(KEYS_TENANT_ID, secureToken.getTenantId());
                }
                if (secureToken.getEdasId() != null) {
                    idsMap.put(KEYS_EDAS_ID, secureToken.getEdasId());
                }
                if (secureToken.getUserId() != null) {
                    idsMap.put(KEYS_USER_ID, secureToken.getUserId());
                }
            }
        } catch (Exception e) {
            // ignored
        }
        return idsMap;
    }

    private String getComponents() {
        StringBuffer componentsString = new StringBuffer();
        for (String component : components) {
            componentsString.append(",").append(component);
        }
        return componentsString.length() > 0 ? componentsString.substring(1) : "";
    }

    private String getVersions() {
        return getVersion() + "|" + getSpringBootVersion();
    }

    private String getVersion() {
        String version = "";
        if (isSpringCloud()) {
            try {
                Class<?> clazz = Class.forName(
                    "org.springframework.cloud.alicloud.context.statistics.StatisticsTaskStarter");
                version = clazz.getPackage().getImplementationVersion();
            } catch (ClassNotFoundException e) {
                //ignored
            }
        } else if (isDubbo()) {
            try {
                Class<?> clazz = Class.forName(
                    "com.alibaba.edas.dubbo.EdasRegistry");
                version = clazz.getPackage().getImplementationVersion();
            } catch (ClassNotFoundException e) {
                //ignored
            }
        }
        return version;
    }

    private String getSpringBootVersion() {
        String springBootVersion = "";
        try {
            Class<?> clazz = Class.forName("org.springframework.boot.ApplicationRunner");
            springBootVersion = clazz.getPackage().getImplementationVersion();
        } catch (ClassNotFoundException e) {
            //ignored
        }
        return springBootVersion;
    }

}
