/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.hbase.client;

import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Queue;

import com.alibaba.hbase.exception.BatchExceptions;
import com.alibaba.lindorm.client.WideColumnService;
import com.alibaba.lindorm.client.core.meta.ExternalIndexConfig;
import com.alibaba.lindorm.client.core.meta.ExternalIndexField;
import com.alibaba.lindorm.client.core.widecolumnservice.WAppend;
import com.alibaba.lindorm.client.core.widecolumnservice.WDelete;
import com.alibaba.lindorm.client.core.widecolumnservice.WGet;
import com.alibaba.lindorm.client.core.widecolumnservice.WIncrement;
import com.alibaba.lindorm.client.core.widecolumnservice.WPut;
import com.alibaba.lindorm.client.core.widecolumnservice.WResult;
import com.alibaba.lindorm.client.core.widecolumnservice.WRowMutations;
import com.alibaba.lindorm.client.core.widecolumnservice.WScan;
import com.alibaba.lindorm.client.core.widecolumnservice.WScanner;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WCompareFilter;
import com.alibaba.lindorm.client.schema.IndexState;
import com.alibaba.lindorm.client.schema.LindormFamilyAttributes;
import com.alibaba.lindorm.client.schema.LindormIndexDescriptor;
import com.alibaba.lindorm.client.schema.LindormTableDescriptor;
import com.google.protobuf.Service;
import com.google.protobuf.ServiceException;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.NamespaceDescriptor;
import org.apache.hadoop.hbase.NamespaceNotFoundException;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Append;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Increment;
import org.apache.hadoop.hbase.client.Mutation;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Row;
import org.apache.hadoop.hbase.client.RowMutations;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.coprocessor.Batch;
import org.apache.hadoop.hbase.client.index.AliHBaseIndexDescriptor;
import org.apache.hadoop.hbase.client.metrics.ScanMetrics;
import org.apache.hadoop.hbase.filter.CompareFilter;
import org.apache.hadoop.hbase.protobuf.generated.AggregateProtos;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;

public class AliHBaseAPIProxyDirectImpl implements AliHBaseAPIProxy {


  private AliHBaseDirectImplFactory factory;
  private TableName myTableName;
  private int defaultScannerCaching;
  private WideColumnService wideColumnService;
  private String tableNameWithoutNamespace;

  public AliHBaseAPIProxyDirectImpl(AliHBaseDirectImplFactory factory, TableName tableName) throws IOException {
    this.factory = factory;
    this.myTableName = tableName;
    this.defaultScannerCaching = factory.getConf().getInt(HConstants.HBASE_CLIENT_SCANNER_CACHING,
        AliHBaseConstants.ALIHBASE_CLIENT_SCANNER_CACHING_MAX);
    if (defaultScannerCaching > AliHBaseConstants.ALIHBASE_CLIENT_SCANNER_CACHING_MAX) {
      this.defaultScannerCaching = AliHBaseConstants.ALIHBASE_CLIENT_SCANNER_CACHING_MAX;
    }
    if (myTableName != null) {
      this.wideColumnService = factory.getWideColumnService(myTableName.getNamespaceAsString());
      this.tableNameWithoutNamespace = myTableName.getQualifierAsString();
    }
  }

  @Override
  public boolean tableExists(TableName tableName) throws IOException {
    return factory.getWideColumnService(tableName.getNamespaceAsString()).listTables()
        .contains(tableName.getQualifierAsString());  }

  @Override
  public List<HTableDescriptor> listTableDescriptors() throws IOException {
    TableName[] tableNames = listTableNames();
    List<HTableDescriptor> tableDescriptors = new ArrayList<>();
    for (TableName tableName : tableNames) {
      HTableDescriptor tableDescriptor = getDescriptor(tableName);
      tableDescriptors.add(tableDescriptor);
    }
    return tableDescriptors;
  }

  @Override
  public TableName[] listTableNames() throws IOException {
    List<String> tables = factory.getAdminService().listTableNamesByPrefix("");
    TableName[] tableNames = new TableName[tables.size()];
    for (int i = 0; i < tables.size(); i++) {
      tableNames[i] = ElementConvertor.toHBaseTableName(tables.get(i));
    }
    return tableNames;
  }

  @Override
  public HTableDescriptor getDescriptor(TableName tableName) throws IOException {
    LindormTableDescriptor lindormTableDescriptor = factory
        .getWideColumnService(tableName.getNamespaceAsString())
        .describeTable(tableName.getQualifierAsString());
    return ElementConvertor
        .toHbaseTableDescriptor(tableName.getNamespaceAsString(), lindormTableDescriptor);
  }

  @Override
  public TableName[] listTableNamesByNamespace(String name) throws IOException {
    List<String> tables = factory.getWideColumnService(name).listTables();
    TableName[] tableNames = new TableName[tables.size()];
    for (int i = 0; i < tables.size(); i++) {
      tableNames[i] = TableName.valueOf(name, tables.get(i));
    }
    return tableNames;
  }

  @Override
  public List<HTableDescriptor> listTableDescriptorsByNamespace(String name) throws IOException {
    TableName[] tableNames = listTableNamesByNamespace(name);
    List<HTableDescriptor> descriptors = new ArrayList<>();
    for (TableName tableName : tableNames) {
      descriptors.add(getDescriptor(tableName));
    }
    return descriptors;
  }

  @Override
  public void createTable(HTableDescriptor desc, byte[][] splitKeys) throws IOException {
    LindormTableDescriptor lindormTableDescriptor = ElementConvertor.toLindormTableDescripter(desc, false);
    TableName tableName = desc.getTableName();
    factory.getWideColumnService(tableName.getNamespaceAsString()).createTable(lindormTableDescriptor, splitKeys);
  }

  @Override
  public void deleteTable(TableName tableName) throws IOException {
    factory.getWideColumnService(tableName.getNamespaceAsString())
        .deleteTable(tableName.getQualifierAsString());
  }

  @Override
  public void truncateTable(TableName tableName, boolean preserveSplits) throws IOException {
    factory.getWideColumnService(tableName.getNamespaceAsString())
        .truncateTable(tableName.getQualifierAsString());
  }

  @Override
  public void enableTable(TableName tableName) throws IOException {
    factory.getWideColumnService(tableName.getNamespaceAsString())
        .onlineTable(tableName.getQualifierAsString());
  }

  @Override
  public void disableTable(TableName tableName) throws IOException {
    factory.getWideColumnService(tableName.getNamespaceAsString())
        .offlineTable(tableName.getQualifierAsString());
  }

  @Override
  public boolean isTableEnabled(TableName tableName) throws IOException {
    return factory.getWideColumnService(tableName.getNamespaceAsString())
        .isTableOnline(tableName.getQualifierAsString());
  }

  @Override
  public boolean isTableDisabled(TableName tableName) throws IOException {
    return factory.getWideColumnService(tableName.getNamespaceAsString())
        .isTableOffline(tableName.getQualifierAsString());
  }

  @Override
  public boolean isTableAvailable(TableName tableName) throws IOException {
    return isTableEnabled(tableName);
  }

  @Override
  public void addColumnFamily(TableName tableName, HColumnDescriptor columnFamily)
      throws IOException {
    factory.getWideColumnService(tableName.getNamespaceAsString())
        .addFamily(tableName.getQualifierAsString(),
            ElementConvertor.toLindormcolumnFamilyDescriptor(columnFamily, false));
  }

  @Override
  public void deleteColumnFamily(TableName tableName, byte[] columnFamily) throws IOException {
    factory.getWideColumnService(tableName.getNamespaceAsString())
        .deleteFamily(tableName.getQualifierAsString(), Bytes.toString(columnFamily));
  }

  @Override
  public void modifyColumnFamily(TableName tableName, HColumnDescriptor columnFamily)
      throws IOException {
    List<LindormFamilyAttributes> lindormFamilyAttributes = ElementConvertor
        .toLindormFamilyAttributes(columnFamily, false);
    factory.getWideColumnService(tableName.getNamespaceAsString())
        .modifyFamilyAttributes(tableName.getQualifierAsString(), lindormFamilyAttributes);
  }

  @Override
  public void modifyTable(HTableDescriptor td) throws IOException {
    LindormTableDescriptor lindormTableDescriptor = ElementConvertor
        .toLindormTableDescripter(td, false);
    factory.getWideColumnService(td.getTableName().getNamespaceAsString())
        .modifyTableAttributes(lindormTableDescriptor);
  }

  @Override
  public void deleteNamespace(String name) throws IOException {
    factory.getAdminService().removeNamespace(name);
  }

  @Override
  public NamespaceDescriptor getNamespaceDescriptor(String name)
      throws NamespaceNotFoundException, IOException {
    if (factory.getAdminService().listNamespaces().contains(name)) {
      return NamespaceDescriptor.create(name).build();
    } else {
      throw new NamespaceNotFoundException(name + " is not found");
    }
  }

  @Override
  public NamespaceDescriptor[] listNamespaceDescriptors() throws IOException {
    List<String> namespaces = factory.getAdminService().listNamespaces();
    NamespaceDescriptor[] namespaceDescriptors = new NamespaceDescriptor[namespaces.size()];
    for (int i = 0; i < namespaces.size(); i++) {
      namespaceDescriptors[i] = NamespaceDescriptor.create(namespaces.get(i)).build();
    }
    return namespaceDescriptors;
  }

  @Override
  public void createNamespace(NamespaceDescriptor descriptor) throws IOException {
    factory.getAdminService().createNamespace(descriptor.getName());
  }

  @Override
  public void flushRegion(String regionName) {
    try {
      for (String idc : factory.getSystemService().getLConnection().getAllIDC()) {
        factory.getSystemService().flush(regionName, idc);
      }
    } catch (Throwable t) {
      throw new IllegalArgumentException(t);
    }
  }

  @Override
  public void compactRegion(String regionName) {
    try {
      for (String idc : factory.getSystemService().getLConnection().getAllIDC()) {
        factory.getSystemService().compact(regionName, idc);
      }
    } catch (Throwable t) {
      throw new IllegalArgumentException(t);
    }
  }

  @Override
  public void majorCompactRegion(String regionName) {
    try {
      for (String idc : factory.getSystemService().getLConnection().getAllIDC()) {
        factory.getSystemService().majorCompact(regionName, idc);
      }
    } catch (Throwable t) {
      throw new IllegalArgumentException(t);
    }
  }

  @Override
  public void flushTable(TableName tableName) {
    String table = ElementConvertor.toLindormTableFullName(tableName);
    try {
      for (String idc : factory.getSystemService().getLConnection().getAllIDC()) {
        factory.getSystemService().flush(table, idc);
      }
    } catch (Throwable t) {
      throw new IllegalArgumentException(t);
    }
  }

  @Override
  public void compactTable(TableName tableName) {
    String table = ElementConvertor.toLindormTableFullName(tableName);
    try {
      for (String idc : factory.getSystemService().getLConnection().getAllIDC()) {
        factory.getSystemService().compact(table, idc);
      }
    } catch (Throwable t) {
      throw new IllegalArgumentException(t);
    }
  }

  @Override
  public void majorCompactTable(TableName tableName) {
    String table = ElementConvertor.toLindormTableFullName(tableName);
    try {
      for (String idc : factory.getSystemService().getLConnection().getAllIDC()) {
        factory.getSystemService().majorCompact(table, idc);
      }
    } catch (Throwable t) {
      throw new IllegalArgumentException(t);
    }
  }

  private HRegionLocation createFakeRegionLocation(byte[] startKey, byte[] endKey) {
    HRegionInfo regionInfo = new HRegionInfo(myTableName, startKey, endKey);
    return new HRegionLocation(regionInfo, ServerName.valueOf("localhost", 0, 0));
  }

  private Pair<byte[], byte[]> getStartEndKey(byte[] row) throws IOException {
    byte[][] endKeys = getStartEndKeys().getSecond();
    if (endKeys.length == 1) {
      return new Pair<>(HConstants.EMPTY_START_ROW, HConstants.EMPTY_START_ROW);
    }

    byte[] preStartKey = HConstants.EMPTY_START_ROW;
    for (byte[] endKey : endKeys) {
      if (Bytes.compareTo(row, endKey) < 0) {
        return new Pair<>(preStartKey, endKey);
      }
      preStartKey = endKey;
    }

    //last region
    return new Pair<>(endKeys[endKeys.length - 2], HConstants.EMPTY_START_ROW);
  }

  @Override
  public HRegionLocation getRegionLocation(byte[] row) throws IOException {
    Pair<byte[], byte[]> startEndKey = getStartEndKey(row);
    return createFakeRegionLocation(startEndKey.getFirst(), startEndKey.getSecond());
  }

  @Override
  public List<HRegionLocation> getAllRegionLocations() throws IOException {
    List<HRegionLocation> fakeRegionLocations = new ArrayList<>();
    byte[] preStartKey = HConstants.EMPTY_START_ROW;
    byte[][] endKeys = getStartEndKeys().getSecond();
    for (int i = 0; i < endKeys.length; i++) {
      byte[] endKey = endKeys[i];
      fakeRegionLocations.add(createFakeRegionLocation(preStartKey, endKey));
      preStartKey = endKey;
    }
    return fakeRegionLocations;
  }

  @Override
  public Pair<byte[][], byte[][]> getStartEndKeys() throws IOException {
    com.alibaba.lindorm.client.core.utils.Pair<byte[][], byte[][]> pair = wideColumnService
        .getStartEndKeys(tableNameWithoutNamespace);
    return new Pair<byte[][], byte[][]>(pair.getFirst(), pair.getSecond());
  }

  @Override
  public boolean exists(Get get) throws IOException {
    WGet wGet = ElementConvertor.toLindormGet(get);
    return wideColumnService
        .exists(tableNameWithoutNamespace, wGet);
  }

  @Override
  public boolean[] exists(List<Get> gets) throws IOException {
    Result[] results = get(gets);
    boolean[] existResult = new boolean[results.length];
    for (int i = 0; i < results.length; i++) {
      existResult[i] = (results[i].getRow() != null);
    }
    return existResult;
  }

  @Override
  public void batch(List<? extends Row> actions, Object[] results) throws IOException {
    List<Object> tactions = new ArrayList<Object>(actions.size());
    for (Row action : actions) {
      if (action instanceof Get) {
        tactions.add(ElementConvertor.toLindormGet((Get) action));
      } else if (action instanceof Put) {
        tactions.add(ElementConvertor.toLindormPut((Put) action));
      } else if (action instanceof Delete) {
        tactions.add(ElementConvertor.toLindormDelete((Delete) action));
      } else if (action instanceof Append) {
        tactions.add(ElementConvertor.toLindormAppend((Append) action));
      } else if (action instanceof Increment) {
        tactions.add(ElementConvertor.toLindormIncrement((Increment) action));
      } else {
        throw new UnsupportedOperationException(
            "Unsupported type " + action.getClass().getName() + " in batch operation.");
      }
    }
    Object[] hbaseResults = null;
    BatchExceptions batchExceptions = new BatchExceptions();
    try {
      hbaseResults = batch(tactions);
    } finally {
      for (int i = 0; i < hbaseResults.length; i++) {
        if (results != null) {
          if (hbaseResults[i] == null) {
            results[i] = null;
          } else if (hbaseResults[i] instanceof WResult) {
            results[i] = ElementConvertor.toHBaseResult((WResult) hbaseResults[i]);
          } else if (hbaseResults[i] instanceof Result) {
            results[i] = hbaseResults[i];
          } else if (hbaseResults[i] instanceof Throwable) {
            results[i] = hbaseResults[i];
            batchExceptions.add((Throwable) hbaseResults[i], actions.get(i), null);
          } else {
            throw new IOException("Not unsupported result type " + hbaseResults[i]);
          }
        } else {
          if (hbaseResults[i] != null && hbaseResults[i] instanceof Throwable) {
            batchExceptions.add((Throwable) hbaseResults[i], actions.get(i), null);
          }
        }
      }
      if (batchExceptions.hasErrors()) {
        throw batchExceptions.makeException();
      }
    }
  }

  private Object[] batch(List<Object> actions) throws IOException {
    Object[] finalResults = new Object[actions.size()];

    List<WGet> getActions = new ArrayList<>();
    List<Integer> getActionsIndex = new ArrayList<>();

    List<WPut> putActions = new ArrayList<>();
    List<Integer> putActionsIndex = new ArrayList<>();

    List<WDelete> delActions = new ArrayList<>();
    List<Integer> delActionsIndex = new ArrayList<>();

    List<WAppend> appendActions = new ArrayList<>();
    List<Integer> appendActionsIndex = new ArrayList<>();

    List<WIncrement> incActions = new ArrayList<>();
    List<Integer> incActionsIndex = new ArrayList<>();

    for (int i = 0; i < actions.size(); i++) {
      Object a = actions.get(i);
      if (a instanceof WGet) {
        getActions.add((WGet) a);
        getActionsIndex.add(i);
      } else if (a instanceof WPut) {
        putActions.add((WPut) a);
        putActionsIndex.add(i);
      } else if (a instanceof WDelete) {
        delActions.add((WDelete) a);
        delActionsIndex.add(i);
      } else if (a instanceof WAppend) {
        appendActions.add((WAppend) a);
        appendActionsIndex.add(i);
      } else if (a instanceof WIncrement) {
        incActions.add((WIncrement) a);
        incActionsIndex.add(i);
      } else {
        throw new UnsupportedOperationException("Not supported action " + a.getClass().getName());
      }
    }

    if (!getActions.isEmpty()) {
      try {
        WResult[] getResult = wideColumnService.batchGet(tableNameWithoutNamespace, getActions);
        for (int i = 0; i < getActions.size(); i++) {
          int index = getActionsIndex.get(i);
          finalResults[index] = getResult[i];
        }
      } catch (Throwable e) {
        for (Integer index : getActionsIndex) {
          finalResults[index] = e;
        }
      }
    }
    if (!putActions.isEmpty()) {
      try {
        wideColumnService.batchPut(tableNameWithoutNamespace, putActions);
        for (Integer index : putActionsIndex) {
          finalResults[index] = new Result();
        }
      } catch (Throwable e) {
        for (Integer index : putActionsIndex) {
          finalResults[index] = e;
        }
      }
    }
    if (!delActions.isEmpty()) {
      try {
        wideColumnService.batchDelete(tableNameWithoutNamespace, delActions);
        for (Integer index : delActionsIndex) {
          finalResults[index] = new Result();
        }
      } catch (Throwable e) {
        for (Integer index : delActionsIndex) {
          finalResults[index] = e;
        }
      }
    }
    if (!appendActions.isEmpty()) {
      try {
        WResult[] appendResults = wideColumnService.batchAppend(tableNameWithoutNamespace, appendActions);
        for (int i = 0; i < appendActions.size(); i++) {
          int index = appendActionsIndex.get(i);
          finalResults[index] = appendResults[i];
        }
      } catch (Throwable e) {
        for (Integer index : appendActionsIndex) {
          finalResults[index] = e;
        }
      }
    }
    if (!incActions.isEmpty()) {
      try {
        WResult[] incResults = wideColumnService.batchIncrement(tableNameWithoutNamespace, incActions);
        for (int i = 0; i < incActions.size(); i++) {
          int index = incActionsIndex.get(i);
          finalResults[index] = incResults[i];
        }
      } catch (Throwable e) {
        for (Integer index : appendActionsIndex) {
          finalResults[index] = e;
        }
      }
    }
    return finalResults;
  }

  @Override
  public Result get(Get get) throws IOException {
    WResult result = wideColumnService
        .get(tableNameWithoutNamespace, ElementConvertor.toLindormGet(get));
    return ElementConvertor.toHBaseResult(result);
  }

  @Override
  public Result[] get(List<Get> gets) throws IOException {
    WResult[] results = wideColumnService
        .batchGet(tableNameWithoutNamespace, ElementConvertor.toLindormGets(gets));
    return ElementConvertor.toHBaseResults(results);
  }

  @Override
  public ResultScanner getScanner(Scan scan) throws IOException {
    return new Scanner(scan);
  }

  @Override
  public void put(Put put) throws IOException {
    WPut wPut = ElementConvertor.toLindormPut(put);
    wideColumnService
        .put(tableNameWithoutNamespace, wPut);
  }

  @Override
  public void put(List<Put> puts) throws IOException {
    List<WPut> wPuts = ElementConvertor.toLindormPuts(puts);
    wideColumnService
        .batchPut(tableNameWithoutNamespace, wPuts);
  }

  @Override
  public void delete(Delete delete) throws IOException {
    WDelete wDelete = ElementConvertor.toLindormDelete(delete);
    wideColumnService
        .delete(tableNameWithoutNamespace, wDelete);
  }

  @Override
  public void delete(List<Delete> deletes) throws IOException {
    List<WDelete> wDeletes = ElementConvertor.toLindormDeleteList(deletes);
    wideColumnService
        .batchDelete(tableNameWithoutNamespace, wDeletes);
  }

  @Override
  public boolean checkAndPut(byte[] row, byte[] family, byte[] qualifier,
      CompareFilter.CompareOp compareOp, byte[] value, Put put) throws IOException {
    WPut wPut = ElementConvertor.toLindormPut(put);
    WCompareFilter.CompareOp wOp = ElementConvertor.toLindormCompareOp(compareOp);
    return wideColumnService
        .checkAndPut(tableNameWithoutNamespace, row, family, qualifier, wOp, value, wPut);
  }

  @Override
  public boolean checkAndDelete(byte[] row, byte[] family, byte[] qualifier,
      CompareFilter.CompareOp compareOp, byte[] value, Delete delete) throws IOException {
    WDelete wDelete = ElementConvertor.toLindormDelete(delete);
    WCompareFilter.CompareOp wOp = ElementConvertor.toLindormCompareOp(compareOp);
    return wideColumnService
        .checkAndDelete(tableNameWithoutNamespace, row, family, qualifier, wOp, value,
            wDelete);
  }

  @Override
  public boolean checkAndMutate(byte[] row, byte[] family, byte[] qualifier,
      CompareFilter.CompareOp op, byte[] value, RowMutations rowMutations) throws IOException {
    WRowMutations wRowMutations = ElementConvertor.toLindormRowMutation(rowMutations);
    WCompareFilter.CompareOp wOp = ElementConvertor.toLindormCompareOp(op);
    return wideColumnService
        .checkAndMutate(tableNameWithoutNamespace, row, family, qualifier, wOp, value,
            wRowMutations);
  }

  @Override
  public void mutateRow(RowMutations rm) throws IOException {
    List<Put> puts = new ArrayList<>();
    List<Delete> deletes = new ArrayList<>();

    for (Mutation mutation : rm.getMutations()) {
      if (mutation instanceof Put) {
        puts.add((Put) mutation);
      } else if (mutation instanceof Delete) {
        deletes.add((Delete) mutation);
      } else {
        throw new IOException("mutateRow not supported, use put(List<Put> puts), "
            + "get(List<Get> gets) or delete(List<Delete> deletes) respectively");
      }
    }

    if (!deletes.isEmpty()) {
      delete(deletes);
    }

    if (!puts.isEmpty()) {
      put(puts);
    }
  }

  @Override
  public Result append(Append append) throws IOException {
    WAppend wAppend = ElementConvertor.toLindormAppend(append);
    WResult wResult = wideColumnService
        .append(tableNameWithoutNamespace, wAppend);
    return ElementConvertor.toHBaseResult(wResult);
  }

  @Override
  public Result increment(Increment increment) throws IOException {
    WIncrement wIncrement = ElementConvertor.toLindormIncrement(increment);
    WResult wResult = wideColumnService
        .increment(tableNameWithoutNamespace, wIncrement);
    return ElementConvertor.toHBaseResult(wResult);
  }

  @Override
  public <T extends Service, R> void coprocessorService(Class<T> service, byte[] startKey,
      byte[] endKey, Batch.Call<T, R> callable, Batch.Callback<R> callback)
      throws ServiceException, Throwable {
    if (service != AggregateProtos.AggregateService.class) {
      throw new UnsupportedOperationException("coprocessorService " + service + " not supported");
    }
    T instance = (T) new AliHBaseUEAggregateService(
        factory.getTableServiceService(myTableName.getNamespaceAsString()),
        tableNameWithoutNamespace);
    R result = callable.call(instance);
    if (callback != null) {
      callback.update(null, null, result);
    }
  }

  @Override
  public void addExternalIndex(ExternalIndexConfig config, List<ExternalIndexField> fields)
      throws IOException {
      wideColumnService.addExternalIndex(tableNameWithoutNamespace, config, fields);
    }

  @Override
  public void removeExternalIndex(List<String> fields) throws IOException {
    wideColumnService.removeExternalIndex(tableNameWithoutNamespace, fields);
  }

  @Override
  public void buildExternalIndex() throws IOException {
    wideColumnService.buildExternalIndex(tableNameWithoutNamespace);
  }

  @Override
  public void cancelBuildExternalIndex() throws IOException {
    wideColumnService.cancelBuildExternalIndex(tableNameWithoutNamespace);
  }

  @Override
  public void close() throws IOException {

  }

  /**
   * A scanner to perform scan from thrift server
   * getScannerResults is used in this scanner
   */
  private class Scanner implements ResultScanner {
    private final WScanner wScanner;
    protected Result lastResult = null;
    protected final Queue<Result> cache = new ArrayDeque<>();;


    public Scanner(Scan scan) throws IOException {
      if (scan.getCaching() <= 0) {
        scan.setCaching(defaultScannerCaching);
      } else if (scan.getCaching() == 1 && scan.isReversed()){
        // for reverse scan, we need to pass the last row to the next scanner
        // we need caching number bigger than 1
        scan.setCaching(scan.getCaching() + 1);
      }
      WScan wScan = ElementConvertor.toLindormScan(scan);
      this.wScanner = wideColumnService.getScanner(tableNameWithoutNamespace, wScan);
    }


    public boolean renewLease() {
      return false;
    }


    public ScanMetrics getScanMetrics() {
      return null;
    }

    @Override
    public Iterator<Result> iterator() {
      return new Iterator<Result>() {
        private Iterator<WResult> it = wScanner.iterator();
        public boolean hasNext() {
          return it.hasNext();
        }

        public Result next() {
          WResult wResult = it.next();
          if (wResult == null) {
            return null;
          } else {
            return ElementConvertor.toHBaseResult(wResult);
          }
        }

        public void remove(){

        }
      };
    }


    @Override
    public Result next() throws IOException {
      WResult r = wScanner.next();
      if (r == null) {
        return null;
      }
      return ElementConvertor.toHBaseResult(r);
    }

    @Override
    public void close() {
    }

    @Override
    public Result[] next(int nbRows) throws IOException {
      // Collect values to be returned here
      ArrayList<Result> resultSets = new ArrayList<Result>(nbRows);
      for(int i = 0; i < nbRows; i++) {
        Result next = next();
        if (next != null) {
          resultSets.add(next);
        } else {
          break;
        }
      }
      return resultSets.toArray(new Result[resultSets.size()]);
    }
  }

  @Override
  public List<AliHBaseIndexDescriptor> describeIndex(TableName dataTableName) throws IOException {
    LindormTableDescriptor desc = factory
            .getWideColumnService(dataTableName.getNamespaceAsString())
            .describeTable(dataTableName.getQualifierAsString());

    List<LindormIndexDescriptor> indexDescriptors = desc.getIndexes();
    List<AliHBaseIndexDescriptor> ret = Collections.emptyList();
    if (indexDescriptors != null && !indexDescriptors.isEmpty()) {
      ret = new ArrayList<>(indexDescriptors.size());
      for (LindormIndexDescriptor lid : indexDescriptors) {
        ret.add(ElementConvertor.toAliHBaseIndexDescriptor(dataTableName.getNamespaceAsString(), lid));
      }
    }
    return ret;
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor indexDescriptor)
      throws IOException {
    LindormIndexDescriptor lid = ElementConvertor.toLindormIndexDescriptor(indexDescriptor);
    TableName tableName = indexDescriptor.getDataTable();
    factory.getTableServiceService(tableName.getNamespaceAsString()).createIndex(lid);
    buildIndexSync(indexDescriptor.getIndexName(), indexDescriptor.getDataTable());
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor indexDescriptor,
      byte[][] splitKeys) throws IOException {
    LindormIndexDescriptor lid = ElementConvertor.toLindormIndexDescriptor(indexDescriptor);
    TableName tableName = indexDescriptor.getDataTable();
    factory.getTableServiceService(tableName.getNamespaceAsString()).createIndex(lid, splitKeys);
    buildIndexSync(indexDescriptor.getIndexName(), indexDescriptor.getDataTable());
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor indexDescriptor,
      byte[] startKey, byte[] endKey, int numRegions) throws IOException {
    LindormIndexDescriptor lid = ElementConvertor.toLindormIndexDescriptor(indexDescriptor);
    TableName tableName = indexDescriptor.getDataTable();
    factory.getTableServiceService(tableName.getNamespaceAsString()).createIndex(lid, startKey, endKey, numRegions);
    buildIndexSync(indexDescriptor.getIndexName(), indexDescriptor.getDataTable());
  }

  @Override
  public void deleteIndex(String indexName, TableName dataTable)
      throws IOException {
    factory.getTableServiceService(dataTable.getNamespaceAsString()).deleteIndex(indexName, dataTable.getQualifierAsString());
  }

  @Override
  public void offlineIndex(String indexName, TableName dataTable)
      throws IOException {
    factory.getTableServiceService(dataTable.getNamespaceAsString()).alterIndexState(indexName, dataTable.getQualifierAsString(), IndexState.DISABLED);
  }

  //TODO build index synchronously
  private void buildIndexSync(String indexName, TableName dataTable) throws IOException {
    factory.getTableServiceService(dataTable.getNamespaceAsString())
        .buildIndex(dataTable.getNamespaceAsString(), dataTable.getQualifierAsString(), indexName);
    blockingAndWaitIndexState(indexName, dataTable, IndexState.ACTIVE, Integer.MAX_VALUE);
  }

  private void blockingAndWaitIndexState(String indexName, TableName dataTable,
      IndexState indexState, int blockTimeInSecond) throws IOException {
    long startTime = System.currentTimeMillis();
    long blockTimeInMs = Integer.MAX_VALUE;
    if (blockTimeInMs != Integer.MAX_VALUE) {
      blockTimeInMs = Integer.MAX_VALUE * 1000; // to ms
    }
    long deadline = startTime + blockTimeInMs;
    long remaining = deadline - System.currentTimeMillis();
    long pauseTime = 1000; // 10s
    int loop = 0;
    IndexState currIndexState = null;
    while (remaining >= 0) {
      currIndexState = factory.getTableServiceService(dataTable.getNamespaceAsString()).getIndexState(indexName,
          dataTable.getQualifierAsString());
      if (currIndexState == indexState) {
        return;
      }

      loop++;
      long sleepTime = pauseTime;
      if (loop > 60) {
        sleepTime = pauseTime * 10;
      }
      try {
        Thread.sleep(sleepTime);
      } catch (InterruptedException e) {
        throw new IOException("Failed to wait index state of " + indexName
            + " to:" + indexState + ", current index state is:" + currIndexState, e);
      }
    }
  }
}
