/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.hbase.client;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import com.alibaba.hbase.thrift2.generated.THBaseService;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.CookieStore;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.BasicCookieStore;
import org.apache.http.params.CoreConnectionPNames;
import org.apache.thrift.protocol.TBinaryProtocol;
import org.apache.thrift.protocol.TProtocol;
import org.apache.thrift.transport.TTransport;
import org.apache.thrift.transport.TTransportException;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLEncoder;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;

public class AliHBaseThriftClientBuilder extends ThriftClientBuilder {


  //Used internally
  public static final String ACCESSKEYHEADER = "ACCESSKEYID";
  public static final String ACCESSTIMESTAMPHEADER = "ACCESSTIMESTAMP";
  public static final String ACCESSSIGNATUREHEADER = "ACCESSSIGNATURE";
  public static final String INSTANCEIDHEADER = "INSTANCEID";
  public static final String INSTANCEHOST = "INSTANCEHOST";
  public static final String ACCESSUUID = "ACCESSUUID";

  public static final String TIMEOUTHEADER = "TIMEOUT";

  Map<String,String> customHeader = new HashMap<>();


  public AliHBaseThriftClientBuilder(AliHBaseThriftImplFactory factory) {
    super(factory);
  }

  public void addCostumHeader(String key, String value) {
    customHeader.put(key, value);
  }

  @Override
  public Pair<THBaseService.Client, TTransport> getClient() throws IOException {
    String url = factory.getHost();
    try {
      AliHBaseClient httpClient = new AliHBaseClient(url, factory.getHttpClient(),
          factory.getConf());
      for (Map.Entry<String, String> header : customHeader.entrySet()) {
        httpClient.setCustomHeader(header.getKey(), header.getValue());
      }
      httpClient.open();
      TProtocol prot = new TBinaryProtocol(httpClient);
      THBaseService.Client client = new THBaseService.Client(prot);
      return new Pair<THBaseService.Client, TTransport>(client, httpClient);
    } catch (TTransportException | NoSuchAlgorithmException e) {
      throw new IOException(e);
    }
  }

  public static byte[] toSHA1(byte[] key) throws NoSuchAlgorithmException {
    MessageDigest md = MessageDigest.getInstance("SHA-1");
    return md.digest(key);
  }

  public static byte[] toReversed(String key) {
    return Bytes.toBytes(new StringBuilder(key).reverse().toString());
  }

  /**
   * Calculate the xor between the given two bytes.
   * @param left
   * @param right
   * @return the byte array of xor result
   */
  public static byte[] xor(byte[] left, byte[] right) {
    byte[] longbytes;
    byte[] shortbytes;
    if (left.length >= right.length) {
      longbytes = left;
      shortbytes = right;
    } else {
      longbytes = right;
      shortbytes = left;
    }
    byte[] xorBytes = new byte[longbytes.length];
    int i = 0;
    for (; i < shortbytes.length; i++) {
      xorBytes[i] = (byte) (shortbytes[i] ^ longbytes[i]);
    }
    for (; i < longbytes.length; i++) {
      xorBytes[i] = longbytes[i];
    }
    return xorBytes;
  }

  public static String hmacSHA1Signature(String accessKeyID, byte[] hashedPassword, byte[] encodedNameAndPass, long
      timestamp, String UUID) throws IOException{
    try {
      byte[] token = xor(toSHA1(Bytes.add(Bytes.toBytes(timestamp),
          Bytes.toBytes(UUID), encodedNameAndPass)),
          hashedPassword);
      return Bytes.toStringBinary(token);
    } catch (Throwable t) {
      throw new IOException("Failed creating signature for '"
          + accessKeyID + "'", t);
    }
  }

  /**
   * HTTP implementation of the TTransport interface. Used for working with a
   * Thrift web services implementation (using for example TServlet).
   *
   * This class offers two implementations of the HTTP transport.
   * One uses HttpURLConnection instances, the other HttpClient from Apache
   * Http Components.
   * The chosen implementation depends on the constructor used to
   * create the THttpClient instance.
   * Using the THttpClient(String url) constructor or passing null as the
   * HttpClient to THttpClient(String url, HttpClient client) will create an
   * instance which will use HttpURLConnection.
   *
   * When using HttpClient, the following configuration leads to 5-15%
   * better performance than the HttpURLConnection implementation:
   *
   * http.protocol.version=HttpVersion.HTTP_1_1
   * http.protocol.content-charset=UTF-8
   * http.protocol.expect-continue=false
   * http.connection.stalecheck=false
   *
   * Also note that under high load, the HttpURLConnection implementation
   * may exhaust the open file descriptor limit.
   *
   * @see <a href="https://issues.apache.org/jira/browse/THRIFT-970">THRIFT-970</a>
   */

  public class AliHBaseClient extends TTransport {

    private URL url_ = null;

    private final ByteArrayOutputStream requestBuffer_ = new ByteArrayOutputStream();

    private InputStream inputStream_ = null;

    private int connectTimeout_ = 0;

    private int readTimeout_ = 0;

    private Map<String,String> customHeaders_ = null;

    private final HttpHost host;

    private final HttpClient client;
    private HttpClientContext httpContext;
    private CookieStore cookieStore;

    private Configuration conf;

    final private String userName;

    final private String instanceID;

    final private String UUID = java.util.UUID.randomUUID().toString();

    final private byte[] hashedPassword;

    final private byte[] encodedNameAndPass;

    private int operationTimeout;

    public AliHBaseClient(String url, HttpClient client, Configuration conf)
        throws TTransportException, NoSuchAlgorithmException {
      try {
        url_ = new URL(url);
        this.client = client;
        this.httpContext = HttpClientContext.create();
        this.cookieStore = new BasicCookieStore();
        httpContext.setCookieStore(this.cookieStore);

        this.host = new HttpHost(url_.getHost(),
            -1 == url_.getPort() ? url_.getDefaultPort() : url_.getPort(), url_.getProtocol());
        this.conf = conf;
        this.userName = conf.get(AliHBaseConstants.USERNAME);
        if (this.userName == null) {
          throw new RuntimeException(AliHBaseConstants.USERNAME + " is not set");
        }
        String password = conf.get(AliHBaseConstants.PASSWORD);
        if (password == null) {
          throw new RuntimeException(AliHBaseConstants.PASSWORD + " is not set");
        }
        hashedPassword = toSHA1(Bytes.toBytes(password));
        encodedNameAndPass = toSHA1(Bytes.add(hashedPassword, toReversed(this.userName)));
        this.instanceID = conf.get(AliHBaseConstants.INSTANCEID);
        if (this.instanceID == null) {
          throw new RuntimeException(AliHBaseConstants.INSTANCEID + " is not set");

        }
        this.operationTimeout = conf.getInt(HConstants.HBASE_CLIENT_OPERATION_TIMEOUT,
            HConstants.DEFAULT_HBASE_CLIENT_OPERATION_TIMEOUT);
      } catch (IOException iox) {
        throw new TTransportException(iox);
      }
    }



    public void setConnectTimeout(int timeout) {
      connectTimeout_ = timeout;
      if (null != this.client) {
        // WARNING, this modifies the HttpClient params, this might have an impact elsewhere if the
        // same HttpClient is used for something else.
        client.getParams().setParameter(CoreConnectionPNames.CONNECTION_TIMEOUT, connectTimeout_);
      }
    }

    public void setReadTimeout(int timeout) {
      readTimeout_ = timeout;
      if (null != this.client) {
        // WARNING, this modifies the HttpClient params, this might have an impact elsewhere if the
        // same HttpClient is used for something else.
        client.getParams().setParameter(CoreConnectionPNames.SO_TIMEOUT, readTimeout_);
      }
    }

    public void setCustomHeaders(Map<String,String> headers) {
      customHeaders_ = headers;
    }

    public void setCustomHeader(String key, String value) {
      if (customHeaders_ == null) {
        customHeaders_ = new HashMap<String, String>();
      }
      customHeaders_.put(key, value);
    }

    public void open() {}

    public void close() {
      if (null != inputStream_) {
        try {
          inputStream_.close();
        } catch (IOException ioe) {
          ;
        }
        inputStream_ = null;
      }
    }

    public boolean isOpen() {
      return true;
    }

    public int read(byte[] buf, int off, int len) throws TTransportException {
      if (inputStream_ == null) {
        throw new TTransportException("Response buffer is empty, no request.");
      }
      try {
        int ret = inputStream_.read(buf, off, len);
        if (ret == -1) {
          throw new TTransportException("No more data available.");
        }
        return ret;
      } catch (IOException iox) {
        throw new TTransportException(iox);
      }
    }

    public void write(byte[] buf, int off, int len) {
      requestBuffer_.write(buf, off, len);
    }

    /**
     * copy from org.apache.http.util.EntityUtils#consume. Android has it's own httpcore
     * that doesn't have a consume.
     */
    private void consume(final HttpEntity entity) throws IOException {
      if (entity == null) {
        return;
      }
      if (entity.isStreaming()) {
        InputStream instream = entity.getContent();
        if (instream != null) {
          instream.close();
        }
      }
    }

    private void addAKAuthHeader(HttpPost post) throws IOException {
      long timestamp = System.currentTimeMillis();
      String signature = hmacSHA1Signature(userName, hashedPassword, encodedNameAndPass, timestamp,
          UUID);
      post.addHeader(ACCESSKEYHEADER, userName);
      post.addHeader(ACCESSTIMESTAMPHEADER, Long.toString(timestamp));
      post.addHeader(ACCESSSIGNATUREHEADER, URLEncoder.encode(signature, "UTF-8"));
      post.addHeader(INSTANCEIDHEADER, instanceID);
      post.addHeader(INSTANCEHOST, this.url_.getHost());
      post.addHeader(ACCESSUUID, UUID);
    }

    private void flushUsingHttpClient() throws TTransportException {

      if (null == this.client) {
        throw new TTransportException("Null HttpClient, aborting.");
      }

      // Extract request and reset buffer
      byte[] data = requestBuffer_.toByteArray();
      requestBuffer_.reset();

      HttpPost post = null;

      InputStream is = null;

      try {
        // Set request to path + query string
        post = new HttpPost(this.url_.getFile());

        //
        // Headers are added to the HttpPost instance, not
        // to HttpClient.
        //

        post.setHeader("Content-Type", "application/x-thrift");
        post.setHeader("Accept", "application/x-thrift");
        post.setHeader("User-Agent", "Java/THttpClient/HC");

        if (null != customHeaders_) {
          for (Map.Entry<String, String> header : customHeaders_.entrySet()) {
            post.setHeader(header.getKey(), header.getValue());
          }
        }
        addAKAuthHeader(post);

        post.setHeader(TIMEOUTHEADER, String.valueOf(operationTimeout));

        post.setEntity(new ByteArrayEntity(data));

        HttpResponse response = this.client.execute(this.host, post, this.httpContext);
        int responseCode = response.getStatusLine().getStatusCode();
        //
        // Retrieve the inputstream BEFORE checking the status code so
        // resources get freed in the finally clause.
        //

        is = response.getEntity().getContent();

        // Read the responses into a byte array so we can release the connection
        // early. This implies that the whole content will have to be read in
        // memory, and that momentarily we might use up twice the memory (while the
        // thrift struct is being read up the chain).
        // Proceeding differently might lead to exhaustion of connections and thus
        // to app failure.

        byte[] buf = new byte[1024];
        ByteArrayOutputStream baos = new ByteArrayOutputStream();

        int len = 0;
        do {
          len = is.read(buf);
          if (len > 0) {
            baos.write(buf, 0, len);
          }
        } while (-1 != len);

        try {
          // Indicate we're done with the content.
          consume(response.getEntity());
        } catch (IOException ioe) {
          // We ignore this exception, it might only mean the server has no
          // keep-alive capability.
        }
        if (responseCode != HttpStatus.SC_OK) {
          String error = Bytes.toString(baos.toByteArray());
          throw new TTransportException(error);
        }


        inputStream_ = new ByteArrayInputStream(baos.toByteArray());
      } catch (IOException ioe) {
        // Abort method so the connection gets released back to the connection manager
        if (null != post) {
          post.abort();
        }
        throw new TTransportException(ioe);
      } finally {
        if (null != is) {
          // Close the entity's input stream, this will release the underlying connection
          try {
            is.close();
          } catch (IOException ioe) {
            throw new TTransportException(ioe);
          }
        }
      }
    }

    public void flush() throws TTransportException {
      flushUsingHttpClient();
    }
  }
}