/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.hbase.client;


import com.alibaba.lindorm.client.core.LindormWideColumnService;
import com.alibaba.lindorm.client.core.utils.Pair;
import com.alibaba.lindorm.client.core.utils.SchemaUtils;
import com.alibaba.lindorm.client.core.utils.StringUtils;
import com.alibaba.lindorm.client.core.widecolumnservice.WAppend;
import com.alibaba.lindorm.client.core.widecolumnservice.WColumn;
import com.alibaba.lindorm.client.core.widecolumnservice.WDelete;
import com.alibaba.lindorm.client.core.widecolumnservice.WGet;
import com.alibaba.lindorm.client.core.widecolumnservice.WIncrement;
import com.alibaba.lindorm.client.core.widecolumnservice.WPut;
import com.alibaba.lindorm.client.core.widecolumnservice.WResult;
import com.alibaba.lindorm.client.core.widecolumnservice.WRowMutations;
import com.alibaba.lindorm.client.core.widecolumnservice.WScan;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WBinaryComparator;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WBinaryPrefixComparator;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WBitComparator;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WByteArrayComparable;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WColumnCountGetFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WColumnPrefixFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WColumnRangeFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WCompareFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WDependentColumnFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WFamilyFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WFilterList;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WFirstKeyOnlyFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WFuzzyRowFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WInclusiveStopFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WKeyOnlyFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WMultipleColumnPrefixFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WNullComparator;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WPrefixFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WQualifierFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WRandomRowFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WRegexStringComparator;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WRowFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WSingleColumnValueExcludeFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WSingleColumnValueFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WSkipFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WSubstringComparator;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WTimestampsFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WValueFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WWhileMatchFilter;
import com.alibaba.lindorm.client.dml.ColumnKey;
import com.alibaba.lindorm.client.dml.ColumnValue;
import com.alibaba.lindorm.client.schema.ColumnFamilyDescriptor;
import com.alibaba.lindorm.client.schema.DataType;
import com.alibaba.lindorm.client.schema.IndexState;
import com.alibaba.lindorm.client.schema.IndexedColumnSchema;
import com.alibaba.lindorm.client.schema.LindormAttribute;
import com.alibaba.lindorm.client.schema.LindormFamilyAttributeConstants;
import com.alibaba.lindorm.client.schema.LindormFamilyAttributes;
import com.alibaba.lindorm.client.schema.LindormIndexDescriptor;
import com.alibaba.lindorm.client.schema.LindormTableDescriptor;
import com.alibaba.lindorm.client.schema.SortOrder;
import com.google.protobuf.InvalidProtocolBufferException;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Append;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Durability;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Increment;
import org.apache.hadoop.hbase.client.IsolationLevel;
import org.apache.hadoop.hbase.client.Mutation;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Query;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.RowMutations;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.coprocessor.BigDecimalColumnInterpreter;
import org.apache.hadoop.hbase.client.coprocessor.DoubleColumnInterpreter;
import org.apache.hadoop.hbase.client.coprocessor.LongColumnInterpreter;
import org.apache.hadoop.hbase.client.index.AliHBaseColumn;
import org.apache.hadoop.hbase.client.index.AliHBaseIndexDescriptor;
import org.apache.hadoop.hbase.coprocessor.ColumnInterpreter;
import org.apache.hadoop.hbase.filter.BinaryComparator;
import org.apache.hadoop.hbase.filter.BinaryPrefixComparator;
import org.apache.hadoop.hbase.filter.BitComparator;
import org.apache.hadoop.hbase.filter.ByteArrayComparable;
import org.apache.hadoop.hbase.filter.ColumnCountGetFilter;
import org.apache.hadoop.hbase.filter.ColumnPrefixFilter;
import org.apache.hadoop.hbase.filter.ColumnRangeFilter;
import org.apache.hadoop.hbase.filter.DependentColumnFilter;
import org.apache.hadoop.hbase.filter.FamilyFilter;
import org.apache.hadoop.hbase.filter.Filter;
import org.apache.hadoop.hbase.filter.FilterList;
import org.apache.hadoop.hbase.filter.FirstKeyOnlyFilter;
import org.apache.hadoop.hbase.filter.FirstKeyValueMatchingQualifiersFilter;
import org.apache.hadoop.hbase.filter.FuzzyRowFilter;
import org.apache.hadoop.hbase.filter.InclusiveStopFilter;
import org.apache.hadoop.hbase.filter.KeyOnlyFilter;
import org.apache.hadoop.hbase.filter.MultipleColumnPrefixFilter;
import org.apache.hadoop.hbase.filter.NullComparator;
import org.apache.hadoop.hbase.filter.PrefixFilter;
import org.apache.hadoop.hbase.filter.QualifierFilter;
import org.apache.hadoop.hbase.filter.RandomRowFilter;
import org.apache.hadoop.hbase.filter.RegexStringComparator;
import org.apache.hadoop.hbase.filter.RowFilter;
import org.apache.hadoop.hbase.filter.SingleColumnValueExcludeFilter;
import org.apache.hadoop.hbase.filter.SingleColumnValueFilter;
import org.apache.hadoop.hbase.filter.SkipFilter;
import org.apache.hadoop.hbase.filter.SubstringComparator;
import org.apache.hadoop.hbase.filter.TimestampsFilter;
import org.apache.hadoop.hbase.filter.ValueFilter;
import org.apache.hadoop.hbase.filter.WhileMatchFilter;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.io.TimeRange;
import org.apache.hadoop.hbase.protobuf.generated.ComparatorProtos;
import org.apache.hadoop.hbase.protobuf.generated.FilterProtos;
import org.apache.hadoop.hbase.protobuf.generated.HBaseProtos;
import org.apache.hadoop.hbase.regionserver.BloomType;
import org.apache.hadoop.hbase.util.Bytes;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static com.alibaba.hbase.client.AliHBaseConstants.CHS_L;
import static com.alibaba.hbase.client.AliHBaseConstants.CHS_MAX_LAYER;
import static com.alibaba.hbase.client.AliHBaseConstants.CHS_PROMOTE_ON_MAJOR;
import static com.alibaba.hbase.client.AliHBaseConstants.COLD_BOUNDARY;
import static com.alibaba.hbase.client.AliHBaseConstants.STORAGETYPE_COLD;
import static com.alibaba.hbase.client.AliHBaseConstants.STORAGETYPE_DEFAULT;
import static com.alibaba.hbase.client.AliHBaseConstants.STORAGE_POLICY;
import static com.alibaba.lindorm.client.schema.LindormFamilyAttributeConstants.DATA_BLOCK_ENCODING;
import static org.apache.hadoop.hbase.HColumnDescriptor.COMPRESSION;

public class ElementConvertor {

  private static final long DEFAULT_MAX_RESULT_SIZE = -1L;

  private static final String COLD_HOT_COLD_STORAGETYPE_CONFIG = "storagetype=" + STORAGETYPE_COLD;
  private static final byte[] ROWKEY_NAME = Bytes.toBytes(LindormWideColumnService.UNIFIED_PK_COLUMN_NAME);

  private static final Map<String, DataType> INTERPRETER_DATATYPE_MAP = new HashMap<>();

  static {
    INTERPRETER_DATATYPE_MAP.put(LongColumnInterpreter.class.getName(), DataType.LONG);
    INTERPRETER_DATATYPE_MAP.put(DoubleColumnInterpreter.class.getName(), DataType.DOUBLE);
    INTERPRETER_DATATYPE_MAP.put(BigDecimalColumnInterpreter.class.getName(), DataType.DECIMAL);
  }


  private static final String TABLEMETAVERSION = "TABLEMETAVERSION";
  private static final byte[] TABLEMETAVERSIONBYTES = Bytes.toBytes(TABLEMETAVERSION);

  private static WColumn.Type convertTypeToWType(byte type) {
    switch (KeyValue.Type.codeToType(type)) {
      case Minimum:
        return WColumn.Type.Minimum;
      case Put:
        return WColumn.Type.Put;
      case Delete:
        return WColumn.Type.Delete;
      case DeleteColumn:
        return WColumn.Type.DeleteColumn;
      case DeleteFamily:
        return WColumn.Type.DeleteFamily;
      case Maximum:
        return WColumn.Type.Maximum;
    }
    throw new UnsupportedOperationException("Unsupport op type " + KeyValue.Type.codeToType(type));
  }

  private static KeyValue.Type convertWTypeToHType(WColumn.Type type) {
    switch (type) {
      case Minimum:
        return KeyValue.Type.Minimum;
      case Put:
        return KeyValue.Type.Put;
      case Delete:
        return KeyValue.Type.Delete;
      case DeleteColumn:
        return KeyValue.Type.DeleteColumn;
      case DeleteFamily:
        return KeyValue.Type.DeleteFamily;
      case Maximum:
        return KeyValue.Type.Maximum;
    }
    throw new UnsupportedOperationException("Unsupport op type " + type);
  }

  /**
   * Creates a {@link WDelete} (Tablestore) from a {@link Delete} (HBase).
   *
   * @param delete the <code>Delete</code> to convert
   * @return converted <code>WDelete</code>
   */
  public static WDelete toLindormDelete(Delete delete) throws IOException {
    checkDeleteSupport(delete);
    // TODO: 06/09/2018 lock id ?
    WDelete wdel = new WDelete(delete.getRow(), delete.getTimeStamp());

//    if (Durability.SKIP_WAL.equals(delete.getDurability())) {
//      wdel.setWriteToWAL(false);
//    }

    for (Map.Entry<byte[], List<Cell>> entry : delete.getFamilyCellMap().entrySet()) {
      for (Cell kv : entry.getValue()) {
        wdel.addDeleteMarker(new WColumn(CellUtil.cloneFamily(kv), CellUtil.cloneQualifier(kv),
            CellUtil.cloneValue(kv), kv.getTimestamp(),
            convertTypeToWType(kv.getTypeByte())));
      }
    }
    return wdel;
  }

  public static List<Delete> toHBaseDeletes(List<WDelete> in) throws IOException {
    List<Delete> out = new ArrayList<Delete>(in.size());
    for (WDelete delete : in) {
      out.add(toHBaseDelete(delete));
    }
    return out;
  }

  public static Delete toHBaseDelete(WDelete in) {
    Delete out = new Delete(in.getRowKey());

    Map<byte[], List<WColumn>> familyMap = in.getFamilyMap();
    for (byte[] family : familyMap.keySet()) {
      for (WColumn columnValue : familyMap.get(family)) {
        byte[] qualifier = columnValue.getQualifier();
        long timestamp = columnValue.getTs();
        if (columnValue.getType() == WColumn.Type.Delete) {
          out.addColumn(family, qualifier, timestamp);
        } else if (columnValue.getType() == WColumn.Type.DeleteColumn) {
          out.addColumns(family, qualifier, timestamp);
        } else if (columnValue.getType() == WColumn.Type.DeleteFamily) {
          out.addFamily(family, timestamp);
        }
      }
    }
    return out;
  }

  private static void checkPutSupport(Put in) throws IOException {
    if (in.getACL() != null) {
      throw new IOException("Put#setACL() is not supported");
    }
    if (in.getTTL() != Long.MAX_VALUE) {
      throw new IOException("Put#setTTL() is not supported");
    }
    if (!in.getAttributesMap().isEmpty()) {
      throw new IOException("Put#setAttribute() is not supported");
    }
  }

  private static void checkIncrementSupport(Increment in) throws IOException {
    if (in.getACL() != null) {
      throw new IOException("Increment#setACL() is not supported");
    }
    if (in.getTTL() != Long.MAX_VALUE) {
      throw new IOException("Increment#setTTL() is not supported");
    }
    if (!in.getAttributesMap().isEmpty()) {
      if(in.getAttributesMap().size() > 1 || !in.getAttributesMap().containsKey("_rr_")) {
        throw new IOException("Increment#setAttribute() is not supported");
      }
      if(!Bytes.toBoolean(in.getAttributesMap().get("_rr_"))) {
        throw new IOException("Increment#setAttribute() is not supported");
      }
    }
    if (in.getTimeStamp() != HConstants.LATEST_TIMESTAMP) {
      throw new IOException("Timestamp is not supported");
    }
  }

  private static void checkAppendSupport(Append in) throws IOException {
    if (in.getACL() != null) {
      throw new IOException("Append#setACL() is not supported");
    }
    if (in.getTTL() != Long.MAX_VALUE) {
      throw new IOException("Append#setTTL() is not supported");
    }
    if (!in.getAttributesMap().isEmpty()) {
      if(in.getAttributesMap().size() > 1 || !in.getAttributesMap().containsKey("_rr_")) {
        throw new IOException("Append#setAttribute() is not supported");
      }
      if(!Bytes.toBoolean(in.getAttributesMap().get("_rr_"))) {
        throw new IOException("Append#setAttribute() is not supported");
      }
    }
    if (in.getTimeStamp() != HConstants.LATEST_TIMESTAMP) {
      throw new IOException("Timestamp is not supported");
    }
  }

  private static void checkDeleteSupport(Delete in) throws IOException {
    if (in.getACL() != null) {
      throw new IOException("Delete#setACL() is not supported");
    }
    if (in.getTTL() != Long.MAX_VALUE) {
      throw new IOException("Delete#setTTL() is not supported");
    }
    if (!in.getAttributesMap().isEmpty()) {
      throw new IOException("Delete#setAttribute() is not supported");
    }
  }

  /**
   * Converts multiple {@link Delete}s (HBase) into a list of {@link WDelete}s
   * (Lindorm).
   *
   * @param in list of <code>Delete</code>s to convert
   * @return list of converted <code>WDelete</code>s
   */
  public static List<WDelete> toLindormDeleteList(List<Delete> in) throws IOException {
    List<WDelete> out = new ArrayList<WDelete>(in.size());
    for (Delete delete : in) {
      if (delete != null) {
        out.add(toLindormDelete(delete));
      }
    }
    return out;
  }

  private static boolean isQueryHotOnly(Query query) {
    byte[] value = query.getAttribute(AliHBaseConstants.HOT_ONLY);
    return value != null && Bytes.toBoolean(value);
  }

  private static boolean isHotColdAutoMerge(Query query) {
    byte[] value = query.getAttribute(AliHBaseConstants.COLD_HOT_MERGE);
    return value != null && Bytes.toBoolean(value);
  }

  /**
   * Creates a {@link WGet} (Lindorm) from a {@link Get} (HBase).
   *
   * @param get the <code>Get</code> to convert
   * @return <code>WGet</code> object
   * @throws IOException
   */
  public static WGet toLindormGet(Get get) throws IOException {
    checkGetRowSupport(get);
    WGet wget = new WGet(get.getRow());
    wget.setCacheBlocks(get.getCacheBlocks());
    wget.setMaxVersions(get.getMaxVersions());
    TimeRange tr = get.getTimeRange();
    if (!tr.isAllTime()) {
      wget.setTimeRange(tr.getMin(), tr.getMax());
    }
    wget.setFamilyMap(get.getFamilyMap());
    wget.setFilter(toLindormFilter(get.getFilter()));
    if (isQueryHotOnly(get)) {
      wget.setQueryHotOnly(true);
    } else if (isHotColdAutoMerge(get)) {
      wget.setHotColdAutoMerge(true);
    }
    return wget;
  }

  private static void checkGetRowSupport(Get in) throws IOException{
    if (in.isClosestRowBefore()) {
      throw new IOException("Get#setClosestRowBefore(true) is not supported");
    }
    /**if (in.getACL() != null) {
      throw new IOException("Get#setACL() is not supported");
    }
    if (in.getMaxResultsPerColumnFamily() != -1) {
      throw new IOException("Get#setMaxResultsPerColumnFamily() is not supported");
    }**/
    if (in.getRowOffsetPerColumnFamily() != 0) {
      throw new IOException("Get#setRowOffsetPerColumnFamily() is not supported");
    }
    /**if (in.getConsistency() == Consistency.TIMELINE) {
      throw new IOException(
          "Get#setConsistency(Consistency.TIMELINE) is not supported");
    }
    if (!in.getAttributesMap().isEmpty()) {
      throw new IOException("Get#setAttribute() is not supported");
    }
    if (in.getIsolationLevel() == IsolationLevel.READ_UNCOMMITTED) {
      throw new IOException(
          "Get#setIsolationLevel(READ_UNCOMMITTED) is not supported");
    }
    if (in.getReplicaId() > 0) {
      throw new IOException("Get#setReplicaId() is not supported");
    }
    if (in.isCheckExistenceOnly()) {
      throw new IOException("Get#setCheckExistenceOnly() is not supported");
    }*/
  }

  private static WFilter toLindormFilter(Filter filter) throws IOException {
    if (filter == null) {
      return null;
    }
    if (filter instanceof FilterList) {
      FilterList filterList = ((FilterList) filter);
      WFilterList wlist = null;
      if (filterList.getOperator() == FilterList.Operator.MUST_PASS_ALL) {
        wlist = new WFilterList(WFilterList.Operator.MUST_PASS_ALL);
      } else {
        wlist = new WFilterList(WFilterList.Operator.MUST_PASS_ONE);
      }
      for (Filter childFilter : filterList.getFilters()) {
        wlist.addFilter(toLindormFilter(childFilter));
      }
      return wlist;
    } else if (filter instanceof SingleColumnValueExcludeFilter) {
      SingleColumnValueExcludeFilter sFilter = (SingleColumnValueExcludeFilter) filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(sFilter.getOperator());
      WByteArrayComparable wComparator = toLindormComparator(sFilter.getComparator());
      return new WSingleColumnValueExcludeFilter(sFilter.getFamily(), sFilter.getQualifier(), op,
          wComparator);
    } else if (filter instanceof SingleColumnValueFilter) {
      SingleColumnValueFilter sFilter = (SingleColumnValueFilter) filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(sFilter.getOperator());
      WByteArrayComparable wComparator = toLindormComparator(sFilter.getComparator());
      WSingleColumnValueFilter wsFilter = new WSingleColumnValueFilter(sFilter.getFamily(),
          sFilter.getQualifier(), op, wComparator);
      wsFilter.setLatestVersionOnly(sFilter.getLatestVersionOnly());
      wsFilter.setFilterIfMissing(sFilter.getFilterIfMissing());
      return wsFilter;
    } else if (filter instanceof ColumnCountGetFilter) {
      ColumnCountGetFilter sFilter = (ColumnCountGetFilter) filter;
      return new WColumnCountGetFilter(sFilter.getLimit());
    } else if (filter instanceof ColumnPrefixFilter) {
      ColumnPrefixFilter sFilter = (ColumnPrefixFilter) filter;
      return new WColumnPrefixFilter(sFilter.getPrefix());
    } else if (filter instanceof ColumnRangeFilter) {
      ColumnRangeFilter sFilter = (ColumnRangeFilter) filter;
      return new WColumnRangeFilter(sFilter.getMinColumn(), sFilter.getMinColumnInclusive(),
          sFilter.getMaxColumn(), sFilter.getMaxColumnInclusive());
    } else if (filter instanceof DependentColumnFilter) {
      DependentColumnFilter sFilter = (DependentColumnFilter) filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(sFilter.getOperator());
      WByteArrayComparable wComparator = null;
      if(null != sFilter.getComparator()){
        wComparator = toLindormComparator(sFilter.getComparator());
      }
      return new WDependentColumnFilter(sFilter.getFamily(), sFilter.getQualifier(),
          sFilter.getDropDependentColumn(), op, wComparator);
    } else if (filter instanceof FamilyFilter) {
      FamilyFilter sFilter = (FamilyFilter) filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(sFilter.getOperator());
      WByteArrayComparable wComparator = toLindormComparator(sFilter.getComparator());
      return new WFamilyFilter(op, wComparator);
    } else if (filter instanceof QualifierFilter) {
      QualifierFilter sFilter = (QualifierFilter) filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(sFilter.getOperator());
      WByteArrayComparable wComparator = toLindormComparator(sFilter.getComparator());
      return new WQualifierFilter(op, wComparator);
    } else if (filter instanceof RowFilter) {
      RowFilter sFilter = (RowFilter) filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(sFilter.getOperator());
      WByteArrayComparable wComparator = toLindormComparator(sFilter.getComparator());
      return new WRowFilter(op, wComparator);
    } else if (filter instanceof ValueFilter) {
      ValueFilter sFilter = (ValueFilter) filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(sFilter.getOperator());
      WByteArrayComparable wComparator = toLindormComparator(sFilter.getComparator());
      return new WValueFilter(op, wComparator);
    } else if(filter instanceof FirstKeyValueMatchingQualifiersFilter){
      throw new IOException("Unsupport filter " + filter.getClass().getName());
    } else if (filter instanceof FirstKeyOnlyFilter) {
      return new WFirstKeyOnlyFilter();
    } else if (filter instanceof InclusiveStopFilter) {
      InclusiveStopFilter sFilter = (InclusiveStopFilter) filter;
      return new WInclusiveStopFilter(sFilter.getStopRowKey());
    } else if (filter instanceof MultipleColumnPrefixFilter) {
      MultipleColumnPrefixFilter sFilter = (MultipleColumnPrefixFilter) filter;
      return new WMultipleColumnPrefixFilter(sFilter.getPrefix());
      //    } else if (filter instanceof PageFilter) {
      //      PageFilter sFilter = (PageFilter) filter;
      //      return new WPageFilter(sFilter.getPageSize());
    } else if (filter instanceof PrefixFilter) {
      PrefixFilter sFilter = (PrefixFilter) filter;
      return new WPrefixFilter(sFilter.getPrefix());
    } else if (filter instanceof RandomRowFilter) {
      RandomRowFilter sFilter = (RandomRowFilter) filter;
      return new WRandomRowFilter(sFilter.getChance());
    } else if (filter instanceof SkipFilter) {
      SkipFilter sFilter = (SkipFilter) filter;
      return new WSkipFilter(toLindormFilter(sFilter.getFilter()));
    } else if (filter instanceof TimestampsFilter) {
      TimestampsFilter sFilter = (TimestampsFilter) filter;
      return new WTimestampsFilter(sFilter.getTimestamps());
    } else if (filter instanceof WhileMatchFilter) {
      WhileMatchFilter sFilter = (WhileMatchFilter) filter;
      return new WWhileMatchFilter(toLindormFilter(sFilter.getFilter()));
    } else if (filter instanceof FuzzyRowFilter) {
      FuzzyRowFilter fuzzyRowFilter = (FuzzyRowFilter)filter;
      byte[] bytes = fuzzyRowFilter.toByteArray();
      FilterProtos.FuzzyRowFilter proto;
      try {
        proto = FilterProtos.FuzzyRowFilter.parseFrom(bytes);
      } catch (InvalidProtocolBufferException e) {
        throw new IOException(e);
      }
      int count = proto.getFuzzyKeysDataCount();
      ArrayList<Pair<byte[], byte[]>> fuzzyKeysData =
          new ArrayList<>(count);
      for (int i = 0; i < count; ++i) {
        HBaseProtos.BytesBytesPair current = proto.getFuzzyKeysData(i);
        byte[] keyBytes = current.getFirst().toByteArray();
        byte[] keyMeta = current.getSecond().toByteArray();
        // mask adapter ( -1(0xff) -> 0, 2 -> 1), FuzzyRowFilter is fast version in HBase 1.x+, add adapter
        keyMeta = maskAdapter(keyMeta);
        fuzzyKeysData.add(new Pair<>(keyBytes, keyMeta));
      }
      return new WFuzzyRowFilter(fuzzyKeysData);
    } else if (filter instanceof KeyOnlyFilter) {
      KeyOnlyFilter keyOnlyFilter = (KeyOnlyFilter)filter;
      byte[] bytes = keyOnlyFilter.toByteArray();
      FilterProtos.KeyOnlyFilter proto;
      try {
        proto = FilterProtos.KeyOnlyFilter.parseFrom(bytes);
      } catch (InvalidProtocolBufferException e) {
        throw new IOException(e);
      }
      return new WKeyOnlyFilter(proto.getLenAsVal());
    }
    else {
      throw new IOException("Unsupport filter " + filter.getClass().getName());
    }
  }

  private static byte[] maskAdapter(byte[] mask) {
    if (!isPreprocessedMask(mask)) return mask;
    for (int i = 0; i < mask.length; i++) {
      if (mask[i] == -1) {
        mask[i] = 0; // -1 -> 0
      } else if (mask[i] == 2) {
        mask[i] = 1;// 2 -> 1
      }
    }
    return mask;
  }

  private static boolean isPreprocessedMask(byte[] mask) {
    for (int i = 0; i < mask.length; i++) {
      if (mask[i] != -1 && mask[i] != 2) {
        return false;
      }
    }
    return true;
  }


  public static WCompareFilter.CompareOp toLindormCompareOp(
      org.apache.hadoop.hbase.filter.CompareFilter.CompareOp operator) throws IOException {
    switch (operator) {
      case LESS:
        return WCompareFilter.CompareOp.LESS;
      case LESS_OR_EQUAL:
        return WCompareFilter.CompareOp.LESS_OR_EQUAL;
      case EQUAL:
        return WCompareFilter.CompareOp.EQUAL;
      case NOT_EQUAL:
        return WCompareFilter.CompareOp.NOT_EQUAL;
      case GREATER_OR_EQUAL:
        return WCompareFilter.CompareOp.GREATER_OR_EQUAL;
      case GREATER:
        return WCompareFilter.CompareOp.GREATER;
      case NO_OP:
        return WCompareFilter.CompareOp.NO_OP;
    }
    throw new IOException("Unkown operator " + operator);
  }

  private static WBitComparator.BitwiseOp toLindormBitwiseOp(
      BitComparator.BitwiseOp bitwiseOp) throws IOException {
    switch(bitwiseOp){
      case AND:
        return WBitComparator.BitwiseOp.AND;
      case OR:
        return WBitComparator.BitwiseOp.OR;
      case XOR:
        return WBitComparator.BitwiseOp.XOR;
    }
    throw new IOException("Unkown BitwiseOp " + bitwiseOp);
  }

  private static WByteArrayComparable toLindormComparator(
      ByteArrayComparable comparator) throws IOException {
    if(comparator instanceof BinaryComparator){
      return new WBinaryComparator(comparator.getValue());
    }else if(comparator instanceof BinaryPrefixComparator){
      return new WBinaryPrefixComparator(comparator.getValue());
    }else if(comparator instanceof BitComparator){
      WBitComparator.BitwiseOp wbitWistOp = toLindormBitwiseOp(((BitComparator) comparator).getOperator());
      return new WBitComparator(comparator.getValue(), wbitWistOp);
    }else if(comparator instanceof NullComparator){
      return new WNullComparator();
    }else if(comparator instanceof RegexStringComparator){
      ComparatorProtos.RegexStringComparator proto;
      try {
        proto = ComparatorProtos.RegexStringComparator.parseFrom(comparator.toByteArray());
      } catch (InvalidProtocolBufferException e) {
        throw new IOException(e.getMessage());
      }
      return new WRegexStringComparator(proto.getPattern());
    }else if(comparator instanceof SubstringComparator){
      return new WSubstringComparator(Bytes.toString(comparator.getValue()));
    }
    throw new IOException("Unknow comparator " + comparator);
  }

  /**
   * Converts multiple {@link WPut}s (Lindorm) into a list of {@link Put}s (HBase).
   *
   * @param in list of <code>WPut</code>s to convert
   * @return list of converted <code>Put</code>s
   */
  public static List<Put> toHBasePuts(List<WPut> in) {
    List<Put> out = new ArrayList<Put>(in.size());
    for (WPut put : in) {
      out.add(toHBasePut(put));
    }
    return out;
  }

  /**
   * Creates a {@link Put} (HBase) from a {@link WPut} (Lindorm)
   *
   * @param in the <code>WPut</code> to convert
   * @return converted <code>Put</code>
   */
  public static Put toHBasePut(WPut in) {
    if (in.isEmpty()) {
      throw new IllegalArgumentException("No columns to insert");
    }
    Put out = new Put(in.getRowKey());
    Map<byte[], List<WColumn>> familyMap = in.getFamilyMap();
    for (byte[] family : familyMap.keySet()) {
      for (WColumn col : familyMap.get(family)) {
        out.addColumn(col.getFamily(), col.getQualifier(), col.getTs(), col.getValue());
      }
    }
    return out;
  }

  /**
   * Converts multiple {@link Put}s (HBase) into a list of {@link WPut}s (Lindorm).
   *
   * @param in list of <code>Put</code>s to convert
   * @return list of converted <code>WPut</code>s
   */
  public static List<WPut> toLindormPuts(List<Put> in) throws IOException {
    List<WPut> out = new ArrayList<WPut>(in.size());
    for (Put put : in) {
      out.add(toLindormPut(put));
    }
    return out;
  }

  /**
   * Creates a {@link WPut} (Lindorm) from a {@link Put} (HBase)
   *
   * @param put the <code>Put</code> to convert
   * @return converted <code>WPut</code>
   */
  public static WPut toLindormPut(Put put)throws IOException {
    if (put.isEmpty()) {
      throw new IllegalArgumentException("No columns to insert");
    }
    checkPutSupport(put);
    WPut wput = new WPut(put.getRow());
//    wput.setWriteToWAL(! (put.getDurability() == Durability.SKIP_WAL));
    for (List<Cell> list : put.getFamilyCellMap().values()) {
      for (Cell kv : list) {
        wput.add(new WColumn(CellUtil.cloneFamily(kv), CellUtil.cloneQualifier(kv), CellUtil.cloneValue(kv),
            kv.getTimestamp(), convertTypeToWType(kv.getTypeByte())));
      }
    }
    return wput;
  }

  public static WIncrement toLindormIncrement(Increment inc) throws IOException {
    if (inc.isEmpty()) {
      throw new IllegalArgumentException("No incremented values found.");
    }
    checkIncrementSupport(inc);
    WIncrement winc = new WIncrement(inc.getRow());
    for (List<Cell> list : inc.getFamilyCellMap().values()) {
      for (Cell kv : list) {
        winc.addColumn(CellUtil.cloneFamily(kv), CellUtil.cloneQualifier(kv),
            Bytes.toLong(CellUtil.cloneValue(kv)));
      }
    }
    winc.setTimeRange(inc.getTimeRange().getMin(), inc.getTimeRange().getMax());

    return winc;
  }

  public static WAppend toLindormAppend(Append append) throws IOException {
    if (append.isEmpty()) {
      throw new IllegalArgumentException("No append values found.");
    }
    checkAppendSupport(append);
    WAppend wAppend = new WAppend(append.getRow());
    for (List<Cell> list : append.getFamilyCellMap().values()) {
      for (Cell kv : list) {
        wAppend.add(CellUtil.cloneFamily(kv), CellUtil.cloneQualifier(kv),
            CellUtil.cloneValue(kv));
      }
    }
    wAppend.setReturnResults(append.isReturnResults());

    return wAppend;
  }

  public static WScan toLindormScan(Scan scan) throws IOException {
    checkScanSupport(scan);
    WScan wScan = new WScan(scan.getStartRow(), scan.getStopRow());
    wScan.setFamilyMap(scan.getFamilyMap());
    wScan.setCacheBlocks(scan.getCacheBlocks());
    wScan.setCaching(scan.getCaching());
    if (scan.getMaxResultSize() > 0) {
      wScan.setMaxResultSize(scan.getMaxResultSize());
    }
    wScan.setOffset(scan.getRowOffsetPerColumnFamily());
    wScan.setReversed(scan.isReversed());
    if (scan.getMaxVersions() > 1) {
      wScan.setMaxVersions(scan.getMaxVersions());
    }
    TimeRange timeRange = scan.getTimeRange();
    if (!timeRange.isAllTime()) {
      wScan.setTimeRange(timeRange.getMin(), timeRange.getMax());
    }
    /**if (scan.getBatch() != -1) {
      throw new IllegalArgumentException("Batch is not supported in scan");
    }*/
    wScan.setRaw(scan.isRaw());
    wScan.setIsolationLevel(toLindormIsolationLevel(scan.getIsolationLevel()));
    if (scan.getFilter() != null) {
      wScan.setFilter(toLindormFilter(scan.getFilter()));
    }
    if (isQueryHotOnly(scan)) {
      wScan.setQueryHotOnly(true);
    } else if (isHotColdAutoMerge(scan)) {
      wScan.setHotColdAutoMerge(true);
    }
    return wScan;
  }

  private static WScan.IsolationLevel toLindormIsolationLevel(IsolationLevel isolationLevel)
      throws IOException {
    switch (isolationLevel) {
      case READ_COMMITTED:
        return WScan.IsolationLevel.READ_COMMITTED;
      case READ_UNCOMMITTED:
        return WScan.IsolationLevel.READ_UNCOMMITTED;
    }
    throw new IOException("Unkown isolationLevel " + isolationLevel);
  }

  private static void checkScanSupport(Scan in) throws IOException {
    /**
     if (in.getBatch() != -1) {
      throw new IOException("Scan#SetBatch() is not supported");
    }
    if (in.getMaxResultSize() != -1) {
      throw new IOException("Scan#SetMaxResultSize() is not supported");
    }
    if (!in.getIsolationLevel().equals(IsolationLevel.READ_COMMITTED)) {
      throw new IOException(
          "Scan#setIsolationLevel(READ_UNCOMMITTED) is not supported");
    }
    if (in.getReplicaId() != -1) {
      throw new IOException("Scan#setReplicaId() is not supported");
    }
    if (in.getACL() != null) {
      throw new IOException("Scan#setACL() is not supported");
    }
    if (in.getConsistency().equals(Consistency.TIMELINE)) {
      throw new IOException("Scan#setConsistency(TIMELINE) is not supported");
    }
    if (!in.getAttributesMap().isEmpty()) {
      if(in.getAttributesMap().size() > 1 || !in.getAttributesMap().containsKey("_raw_")) {
        throw new IOException("Scan#setAttribute() is not supported");
      }
    }
    if (in.getAllowPartialResults()) {
      throw new IOException("Scan#setAllowPartialResults() is not supported");
    }
    if (in.getLoadColumnFamiliesOnDemandValue() != null) {
      throw new IOException(
          "Scan#setLoadColumnFamiliesOnDemandValue() is not supported");
    }
    if (in.getRowOffsetPerColumnFamily() > 0) {
      throw new IOException("Scan#setRowOffsetPerColumnFamily() is not supported");
    }
    if (in.getMaxResultsPerColumnFamily() > 0) {
      throw new IOException("Scan#setMaxResultsPerColumnFamily() is not supported");
    }
     */
    if (in.getRowOffsetPerColumnFamily() != 0) {
      throw new IOException("Scan#SetRowOffsetPerColumnFamily() is not supported");
    }
  }

  /**
   * Converts multiple {@link Get}s (HBase) into a list of {@link WGet}s (Lindorm).
   *
   * @param in list of <code>Get</code>s to convert
   * @return list of <code>WGet</code> objects
   * @throws IOException
   */
  public static List<WGet> toLindormGets(List<Get> in) throws IOException {
    List<WGet> out = new ArrayList<WGet>(in.size());
    for (Get get : in) {
      out.add(toLindormGet(get));
    }
    return out;
  }

  /**
   * Converts multiple {@link WResult}s (Lindorm) into a list of {@link Result}s
   * (HBase).
   *
   * @param in array of <code>WResult</code>s to convert
   * @return array of converted <code>Result</code>s
   */
  public static Result[] toHBaseResults(List<WResult> in) {
    List<Result> out = new ArrayList<Result>(in.size());
    for (WResult result : in) {
      out.add(toHBaseResult(result));
    }
    return out.toArray(new Result[0]);
  }

  public static Result[] toHBaseResults(WResult[] in) {
    List<Result> out = new ArrayList<Result>(in.length);
    for (WResult result : in) {
      out.add(toHBaseResult(result));
    }
    return out.toArray(new Result[0]);
  }

  /**
   * Create a {@link Result} (HBase) from a {@link WResult} (Lindorm)
   *
   * @param in
   * @return converted result
   */
  public static Result toHBaseResult(WResult in) {
    Cell[] cells = new Cell[in.size()];
    int i = 0;
    for (WColumn c : in.raw()) {
      Cell cell = CellUtil.createCell(in.getRowKey(), c.getFamily(), c.getQualifier(), c.getTs(),
                      convertWTypeToHType(c.getType()).getCode(), c.getValue());
      cells[i] = cell;
      i++;
    }
    // sort in case lindorm may return unordered KVs.
    // this is necessary because Result#getColumnLatestCell use binary search to get specific column.
    Arrays.sort(cells, KeyValue.COMPARATOR);

    return Result.create(cells);
  }

  public static HColumnDescriptor toHbasecolumnFamilyDescriptor(
      LindormTableDescriptor tableDescriptor, ColumnFamilyDescriptor in) {
    HColumnDescriptor builder = new HColumnDescriptor(in.getName());
    //Do not pass compress, data block encoding, dfsreplication
    builder.setValue(COMPRESSION, tableDescriptor.getCompression());
    builder.setValue(HColumnDescriptor.DATA_BLOCK_ENCODING, tableDescriptor.getDataBlockEncoding());
    //builder.setDFSReplication(tableDescriptor.getDfsReplication());
    builder.setMaxVersions(tableDescriptor.getMaxVersions());
    builder.setMinVersions(tableDescriptor.getMinVersions());
    builder.setTimeToLive(tableDescriptor.getTimeToLive());
    builder.setValue(COLD_BOUNDARY, tableDescriptor.getColdHotSeparateBoundary());
    for (int i = 1; i <= CHS_MAX_LAYER; i++) {
      String key = CHS_L + i;
      String value = tableDescriptor.getColdHotLayerConfig(i);
      builder.setValue(key, value);
    }
    builder.setValue(CHS_PROMOTE_ON_MAJOR,
        Boolean.toString(tableDescriptor.isColdHotPromoteOnMajor()));
    //column family attributes will override table level attributes
    builder.setBloomFilterType(BloomType.valueOf(in.getBloomFilterType()));
    builder.setValue(COMPRESSION, in.getCompression());
    builder.setValue(HColumnDescriptor.DATA_BLOCK_ENCODING, in.getDataBlockEncoding());
    builder.setMaxVersions(in.getMaxVersions());
    builder.setMinVersions(in.getMinVersions());
    builder.setTimeToLive(in.getTimeToLive());
    builder.setValue(COLD_BOUNDARY, in.getColdHotSeparateBoundary());
    builder.setBlockCacheEnabled(in.isBlockCacheEnabled());
    // set major peirod;
    try {
      Long period = in.getCompactionMajorPeriodNoDefault();
      if (period != null) {
        builder.setConfiguration("hbase.hregion.majorcompaction", String.valueOf(period));
      }
    } catch (Throwable t) {
      // error setting major compaction period
    }
    for (int i = 1; i <= CHS_MAX_LAYER; i++) {
      String key = CHS_L + i;
      String value = in.getColdHotLayerConfig(i);
      builder.setValue(key, value);
    }
    builder.setValue(CHS_PROMOTE_ON_MAJOR,
        Boolean.toString(in.isColdHotPromoteOnMajor()));
    return builder;
  }

  public static ColumnFamilyDescriptor toLindormcolumnFamilyDescriptor(
      HColumnDescriptor in, boolean limitedDescriptorEnable) {
    ColumnFamilyDescriptor out = new ColumnFamilyDescriptor(in.getName());
    if(limitedDescriptorEnable){
      out.setTimeToLive(in.getTimeToLive(), TimeUnit.SECONDS);
      out.setMaxVersions(in.getMaxVersions());
      out.setMinVersions(in.getMinVersions());
    }else{
      // set to index encoding if it is diff
      out.setCompression(Bytes.toString(in.getValue(Bytes.toBytes(COMPRESSION))));
      out.setDataBlockEncoding(Bytes.toString(in.getValue(Bytes.toBytes(HColumnDescriptor.DATA_BLOCK_ENCODING))));
      out.setBlockCacheEnabled(in.isBlockCacheEnabled());
      // set major peirod;
      try {
        String majorPeriod = in.getConfigurationValue("hbase.hregion.majorcompaction");
        if (majorPeriod != null && majorPeriod.length() > 0) {
          long period = Long.valueOf(majorPeriod);
          out.setCompactionMajorPeriod(period);
        }
      } catch (Throwable t) {
        // error setting major compaction period
      }
      //Don't pass dfs replication in
      //out.setDfsReplication(in.getDFSReplication());
      out.setMaxVersions(in.getMaxVersions());
      out.setMinVersions(in.getMinVersions());
      out.setTimeToLive(in.getTimeToLive(), TimeUnit.SECONDS);
      if (in.getValue(STORAGE_POLICY) != null) {
        String storageType = in.getValue(STORAGE_POLICY).toUpperCase();
        if (storageType.equals(STORAGETYPE_DEFAULT) || storageType.equals(STORAGETYPE_COLD)) {
          out.setStorageType(storageType);
        } else {
          throw new IllegalArgumentException("only accept storage type:" +
              STORAGETYPE_DEFAULT + " or " + STORAGETYPE_COLD);
        }
      }
      byte[] key = Bytes.toBytes(COLD_BOUNDARY);
      String boundary = null;
      if (in.getValue(key) != null) {
        if (in.getValue(STORAGE_POLICY) != null &&
            in.getValue(STORAGE_POLICY).toUpperCase().equals(STORAGETYPE_COLD)) {
          throw new IllegalArgumentException("can not both set cold hot separate and cold storage type on family:"
              + in.getNameAsString());
        }
        boundary = Bytes.toString(in.getValue(key));
        if (StringUtils.isNullOrEmpty(boundary)) {
          boundary = null;
        } else {
          out.setColdHotSeparateBoundary(boundary);
        }
      }
      for (int i = 1; i <= CHS_MAX_LAYER && !StringUtils.isNullOrEmpty(boundary); i++) {
        String layer = CHS_L + i;
        key = Bytes.toBytes(layer);
        if (in.getValue(key) != null) {
          out.setColdHotLayerConfig(i, Bytes.toString(in.getValue(key)));
        }
      }
      // set default cold storage type to OSS
      if (!StringUtils.isNullOrEmpty(boundary)) {
        int oldestIndex = boundary.split(",").length + 1;
        if (oldestIndex <= CHS_MAX_LAYER) {
          key = Bytes.toBytes(CHS_L + oldestIndex);
          if (in.getValue(key) == null) {
            out.setColdHotLayerConfig(oldestIndex, COLD_HOT_COLD_STORAGETYPE_CONFIG);
          }
        }
      }
      key = Bytes.toBytes(CHS_PROMOTE_ON_MAJOR);
      if (in.getValue(key) != null) {
        String value = Bytes.toString(in.getValue(key));
        out.setColdHotPromoteOnMajor(Boolean.valueOf(value));
      }
    }
    return out;
  }

  public static List<LindormFamilyAttributes> toLindormFamilyAttributes (
      HColumnDescriptor in, boolean limitedDescriptorEnable) throws IOException {
    List<LindormAttribute> lindormAttributes = new ArrayList<>();
    if(limitedDescriptorEnable){
      if (in.getTimeToLive() > 0) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.TTL,
            Integer.toString(in.getTimeToLive())));
      }
      if (in.getMaxVersions() > 0) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.MAX_VERSIONS,
            Integer.toString(in.getMaxVersions())));
      }
      if (in.getMinVersions() > 0) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.MIN_VERSIONS,
            Integer.toString(in.getMinVersions())));
      }
    }else {
      if (in.getValue(Bytes.toBytes(COMPRESSION)) != null) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.COMPRESSION,
            Bytes.toString(in.getValue(Bytes.toBytes(COMPRESSION)))));
      }
      if (in.getValue(Bytes.toBytes(HColumnDescriptor.DATA_BLOCK_ENCODING)) != null) {
        lindormAttributes.add(new LindormAttribute(DATA_BLOCK_ENCODING,
            Bytes.toString(in.getValue(Bytes.toBytes(HColumnDescriptor.DATA_BLOCK_ENCODING)))));
      }
      if (in.getMaxVersions() > 0) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.MAX_VERSIONS,
            Integer.toString(in.getMaxVersions())));
      }
      if (in.getMinVersions() > 0) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.MIN_VERSIONS,
            Integer.toString(in.getMinVersions())));
      }
      if (in.getTimeToLive() > 0) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.TTL,
            Integer.toString(in.getTimeToLive())));
      }
    }
    LindormFamilyAttributes lindormFamilyAttributes = new LindormFamilyAttributes(Bytes.toString(in.getName()),
        lindormAttributes);
    return Arrays.asList(lindormFamilyAttributes);
  }

  public static HTableDescriptor toHbaseTableDescriptor(String namespace,
      LindormTableDescriptor in) {
    HTableDescriptor builder = new HTableDescriptor(TableName.valueOf(namespace, in.getName()));
    for (ColumnFamilyDescriptor column : in.getFamilies()) {
      builder.addFamily(toHbasecolumnFamilyDescriptor(in, column));
    }
    if(in.getTableAttributes() != null) {
      for (Map.Entry<String, byte[]> entry : in.getTableAttributes().getAttributesMap().entrySet()) {
        builder.setValue(Bytes.toBytes(entry.getKey()), entry.getValue());
      }
    }
    builder.setValue(TABLEMETAVERSIONBYTES, Bytes.toBytes(in.getMetaVersion()));
    if (in.isDeferredLogFlush()) {
      builder.setDurability(Durability.ASYNC_WAL);
    } else {
      builder.setDurability(Durability.USE_DEFAULT);
    }
    return builder;
  }

  public static LindormTableDescriptor toLindormTableDescripter(HTableDescriptor htd, boolean limitedDescriptorEnable) {
    LindormTableDescriptor lindormTableDesc = new LindormTableDescriptor(
        htd.getTableName().getQualifierAsString());
    for (HColumnDescriptor hbaseFamily : htd
        .getColumnFamilies()) {
      ColumnFamilyDescriptor column = toLindormcolumnFamilyDescriptor(hbaseFamily, limitedDescriptorEnable);
      lindormTableDesc.addFamily(column);
    }
    if(htd.getValues() != null) {
      if(!limitedDescriptorEnable) {
        for (Map.Entry<ImmutableBytesWritable, ImmutableBytesWritable> entry : htd.getValues().entrySet()) {
          if (Bytes.equals(entry.getKey().get(), TABLEMETAVERSIONBYTES)) {
            continue;
          }
          lindormTableDesc.setTableAttributes(Bytes.toString(entry.getKey().get()), entry.getValue().get());
        }
      }
    }
    byte[] metaVersion = htd.getValue(TABLEMETAVERSIONBYTES);
    if (metaVersion != null && metaVersion.length != 0) {
      lindormTableDesc.setMetaVersion(Bytes.toInt(metaVersion));
    }
    lindormTableDesc.setDynamicColumnsEnabled(true);
    if (htd.getDurability().equals(Durability.ASYNC_WAL)) {
      lindormTableDesc.setDeferredLogFlush(true);
    } else {
      lindormTableDesc.setDeferredLogFlush(false);
    }
    return lindormTableDesc;
  }

  /**
   * There are 3 types of table names in lindorm
   * a) tt
   * b) ns.tt
   * c) ns.tt.idx
   * We should support all of them.
   * @param table lindorm table name
   * @return hbase table name
   */
  public static TableName toHBaseTableName(String table) {
    int pos = table.indexOf(".");
    if (pos < 0) {
      // no '.' found, scenario a
      return TableName.valueOf(table);
    } else {
      // scenario b or c
      String namespace = table.substring(0, pos);
      String name = table.substring(pos + 1);
      return TableName.valueOf(namespace, name);
    }
  }

  public static WRowMutations toLindormRowMutation(RowMutations rowMutations) throws IOException {
    WRowMutations wRowMutations = new WRowMutations(rowMutations.getRow());
    for (Mutation mutation : rowMutations.getMutations()) {
      if (mutation instanceof Put) {
        WPut wPut = toLindormPut((Put)mutation);
        wRowMutations.add(wPut);
      } else if (mutation instanceof Delete) {
        WDelete wDelete = toLindormDelete((Delete)mutation);
        wRowMutations.add(wDelete);
      } else  {
        throw new IOException("Only Put or Delete is supported in RowMutation: " + mutation);
      }
    }
    return wRowMutations;
  }

  public static DataType toInterpreterDataType(String interpreterClassName) {
    DataType dataType = INTERPRETER_DATATYPE_MAP.get(interpreterClassName);
    if (dataType == null) {
      throw new UnsupportedOperationException("Unsupported interpreterClass " + interpreterClassName);
    }
    return dataType;
  }

  public static byte[] toValueBytes(ColumnValue cv, ColumnInterpreter interpreterClassName) {
    BigDecimal bigDecimal = cv.getDecimal();
    if (interpreterClassName instanceof LongColumnInterpreter) {
      return Bytes.toBytes(bigDecimal.longValue());
    }
    if (interpreterClassName instanceof DoubleColumnInterpreter) {
      return Bytes.toBytes(bigDecimal.doubleValue());
    }
    if (interpreterClassName instanceof BigDecimalColumnInterpreter) {
      return Bytes.toBytes(bigDecimal);
    }
    throw new UnsupportedOperationException("Unsupported interpreterClass " + interpreterClassName.getClass());
  }

  public static String toLindormTableFullName(TableName tableName) {
    return tableName.getNamespaceAsString() + "." + tableName.getQualifierAsString();
  }

  private static void initLindormIndexFamilyDescriptor(LindormIndexDescriptor out, HColumnDescriptor in) {
    out.setCompression(in.getCompressionType().getName());
    out.setDataBlockEncoding(in.getDataBlockEncoding().toString());
    out.setBlockCacheEnabled(in.isBlockCacheEnabled());
    if (in.getValue(STORAGE_POLICY) != null) {
      String storageType = in.getValue(STORAGE_POLICY).toUpperCase();
      if (storageType.equals(STORAGETYPE_DEFAULT) || storageType.equals(STORAGETYPE_COLD)) {
        out.setStorageType(storageType);
      } else {
        throw new IllegalArgumentException("only accept storage type:" +
            STORAGETYPE_DEFAULT + " or " + STORAGETYPE_COLD);
      }
    }
    byte[] key = Bytes.toBytes(COLD_BOUNDARY);
    String boundary = null;
    if (in.getValue(key) != null) {
      if (in.getValue(STORAGE_POLICY) != null &&
          in.getValue(STORAGE_POLICY).toUpperCase().equals(STORAGETYPE_COLD)) {
        throw new IllegalArgumentException("can not both set cold hot separate and cold storage type on index:"
            + out.getIndexName());
      }
      boundary = Bytes.toString(in.getValue(key));
      if (StringUtils.isNullOrEmpty(boundary)) {
        boundary = null;
      } else {
        out.setColdHotSeparateBoundary(boundary);
      }
    }
    for (int i = 1; i <= CHS_MAX_LAYER && !StringUtils.isNullOrEmpty(boundary); i++) {
      String layer = CHS_L + i;
      key = Bytes.toBytes(layer);
      if (in.getValue(key) != null) {
        out.setColdHotLayerConfig(i, Bytes.toString(in.getValue(key)));
      }
    }
    // set default cold storage type to OSS
    if (!StringUtils.isNullOrEmpty(boundary)) {
      int oldestIndex = boundary.split(",").length + 1;
      if (oldestIndex <= CHS_MAX_LAYER) {
        key = Bytes.toBytes(CHS_L + oldestIndex);
        if (in.getValue(key) == null) {
          out.setColdHotLayerConfig(oldestIndex, COLD_HOT_COLD_STORAGETYPE_CONFIG);
        }
      }
    }
    key = Bytes.toBytes(CHS_PROMOTE_ON_MAJOR);
    if (in.getValue(key) != null) {
      String value = Bytes.toString(in.getValue(key));
      out.setColdHotPromoteOnMajor(Boolean.getBoolean(value));
    }
  }

  public static LindormIndexDescriptor toLindormIndexDescriptor(AliHBaseIndexDescriptor idxDesc) {
    LindormIndexDescriptor lid = new LindormIndexDescriptor(
            idxDesc.getIndexName(), idxDesc.getDataTable().getQualifierAsString());
    initLindormIndexFamilyDescriptor(lid, idxDesc.getFamilyDescriptor());

    for (AliHBaseColumn ic: idxDesc.getIndexedColumns()) {
      SortOrder order = SortOrder.getDefault();
      if (ic.getSortOrder() != null) {
        order = ic.getSortOrder() == AliHBaseColumn.SortOrder.ASC ? SortOrder.ASC : SortOrder.DESC;
      }
      IndexedColumnSchema ics = new IndexedColumnSchema(ic.getFamily(), ic.getQualifier(), order);
      lid.addIndexedColumns(ics);
    }
    for (AliHBaseColumn cc: idxDesc.getCoveredColumns()) {
      if (cc == AliHBaseColumn.COVER_ALL) {
        lid.addCoveredColumn(LindormIndexDescriptor.COVERED_DYNAMIC_COLUMNS);
        // ignore other covered columns, because it is covering all columns now.
        break;
      }
      lid.addCoveredColumn(new ColumnKey(cc.getFamily(), cc.getQualifier()));
    }

    // set index state
    lid.setIndexState(IndexState.BUILDING);
    return lid;
  }


  public static AliHBaseIndexDescriptor toAliHBaseIndexDescriptor(String namespace, LindormIndexDescriptor desc) {
    String indexName = desc.getIndexName();
    TableName dataTableName = TableName.valueOf(namespace, desc.getDataTableName());
    AliHBaseIndexDescriptor ret = new AliHBaseIndexDescriptor(indexName, dataTableName);

    for (IndexedColumnSchema col : desc.getIndexedColumns()) {
      if (isRowkeyField(col.getColumnKey())) {
        continue;
      }

      ret.addIndexedColumn(getFamilyName(col.getColumnKey()), col.getColumnKey().getQualifier(),
              AliHBaseColumn.SortOrder.fromImplSortOrder(col.getSortOrder()));
    }

    for (ColumnKey ck : desc.getCoveredColumns()) {
      if (ck == LindormIndexDescriptor.COVERED_DYNAMIC_COLUMNS) {
        ret.setCoveredAllColumns();
        break;
      } else {
        ret.addCoveredColumn(getFamilyName(ck), ck.getQualifier());
      }
    }

    return ret;
  }

  private static boolean isRowkeyField(ColumnKey ck) {
    byte[] f = ck.getFamily();
    return (f == null || f.length == 0) && Arrays.equals(ck.getQualifier(), ROWKEY_NAME);
  }

  private static byte[] getFamilyName(ColumnKey ck) {
    byte[] family = ck.getFamily();
    if (family == null || family.length == 0) {
      family = SchemaUtils.DEFAULT_FAMILY_NAME_BYTES;
    }
    return family;
  }

  public static String indexSchemaToString(List<AliHBaseIndexDescriptor> indexes, String dataTableName) {
    if (indexes == null || indexes.isEmpty()) {
      return "No index found for table " + dataTableName;
    } else {
      StringBuilder str = new StringBuilder();
      for (AliHBaseIndexDescriptor idx : indexes) {
        str.append(idx.getSchema());
        str.append("\n");
      }
      str.append("Total " + indexes.size() + " indexes found for table " + dataTableName);
      return str.toString();
    }
  }
}

