package com.alibaba.hbase.haclient;

import static com.alibaba.hbase.haclient.ClusterSwitchUtil.ZOOKEEPER_CREATE_NODE;
import static com.alibaba.hbase.haclient.ClusterSwitchUtil.ZOOKEEPER_CREATE_NODE_DEFAULT;
import static com.alibaba.hbase.haclient.ClusterSwitchUtil.ZOOKEEPER_MODIFY_NODE;
import static com.alibaba.hbase.haclient.ClusterSwitchUtil.ZOOKEEPER_MODIFY_NODE_DEFAULT;

import java.io.IOException;
import java.util.List;

import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.protobuf.generated.ClusterSwitchProto;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.util.Bytes;

import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;
import org.apache.hadoop.hbase.zookeeper.ZooKeeperWatcher;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;

public class AdminUtil {

  private static final String NODE_PREFIX = "1x";

  public static Pair<HTableDescriptor, byte[][]> toCreateInfo(byte[] data) throws IOException {
    if (data == null) {
      return  null;
    }
    Pair<HTableDescriptor, byte[][]> result = new Pair<>();
    try {
      Pair<String, List<String>> createInfoStringPair = toCreateInfoStringPair(data);
      String tableDescriptorString = createInfoStringPair.getFirst();
      List<String> splitKeyList = createInfoStringPair.getSecond();
      HTableDescriptor tableDescriptor = getTableDescriptorFromString(tableDescriptorString);
      byte[][] splitKeys = getSplitKeysFromStringList(splitKeyList);
      result.setFirst(tableDescriptor);
      result.setSecond(splitKeys);
      return result;
    } catch (DeserializationException e) {
      throw new IOException(e);
    }
  }

  public static byte[] toCreateInfoBytes(Pair<HTableDescriptor, byte[][]> createInfo) {
    ClusterSwitchProto.CreateInfo.Builder builder = ClusterSwitchProto.CreateInfo.newBuilder();
    byte[] tableDescriptorBytes = createInfo.getFirst().toByteArray();
    builder.setTableDescriptor(Bytes.toStringBinary(tableDescriptorBytes));
    byte[][] splitKeys = createInfo.getSecond();
    if(splitKeys != null) {
      for (int i = 0; i < splitKeys.length; i++) {
        builder.addSplitKeys(Bytes.toStringBinary(splitKeys[i]));
      }
    }

    return ProtobufUtil.prependPBMagic(builder.build().toByteArray());
  }

  public static HTableDescriptor toModifyInfo(byte[] data) throws IOException {
    if (data == null) {
      return  null;
    }
    try {
      String tableDescriptorString = toModifyInfoString(data);
      HTableDescriptor tableDescriptor = getTableDescriptorFromString(tableDescriptorString);
      return tableDescriptor;
    } catch (DeserializationException e) {
      throw new IOException(e);
    }
  }

  public static byte[] toModifyInfoBytes(HTableDescriptor tableDescriptor) {
    ClusterSwitchProto.ModifyInfo.Builder builder = ClusterSwitchProto.ModifyInfo.newBuilder();
    byte[] tableDescriptorBytes = tableDescriptor.toByteArray();
    builder.setTableDescriptor(Bytes.toStringBinary(tableDescriptorBytes));

    return ProtobufUtil.prependPBMagic(builder.build().toByteArray());
  }

  /**
   * Get CreateInfo from zk
   * @param zkCluster
   * @param conf
   * @param node
   * @return Pair<TableDescriptor, byte[][]>
   * @throws IOException
   * @throws KeeperException
   */
  public static Pair<HTableDescriptor, byte[][]> getCreateInfoFromZk(String zkCluster,
                                                                     Configuration conf,
                                                                     String node) throws IOException, KeeperException {
    Configuration createConf = ClusterSwitchUtil.createConfWithConnectKey(zkCluster, conf);
    ZooKeeperWatcher zk = new ZooKeeperWatcher(createConf, "GetCreateInfo", null, false);
    try {

      if (ZKUtil.checkExists(zk, node) == -1) {
        return null;
      } else {
        byte[] data = ZKUtil.getDataNoWatch(zk, node, null);
        Pair<HTableDescriptor, byte[][]> createInfo = toCreateInfo(data);
        return createInfo;
      }
    } finally {
      if (zk != null) {
        zk.close();
      }
    }
  }

  /**
   * Get ModifyInfo from zk
   * @param zkCluster
   * @param conf
   * @param node
   * @return TableDescriptor
   * @throws IOException
   * @throws KeeperException
   */
  public static HTableDescriptor getModifyInfoFromZk(String zkCluster, Configuration conf,
                                                     String node) throws IOException, KeeperException {
    Configuration modifyConf = ClusterSwitchUtil.createConfWithConnectKey(zkCluster, conf);
    ZooKeeperWatcher zk = new ZooKeeperWatcher(modifyConf, "GetModifyInfo", null, false);
    try {

      if (ZKUtil.checkExists(zk, node) == -1) {
        return null;
      } else {
        byte[] data = ZKUtil.getDataNoWatch(zk, node, null);
        HTableDescriptor modifyInfo = toModifyInfo(data);
        return modifyInfo;
      }
    } finally {
      if (zk != null) {
        zk.close();
      }
    }
  }

  public static void setCreateInfoNode(Configuration clientConf, HTableDescriptor tableDescriptor,
                                       byte[][] splitKey)throws IOException, KeeperException{
    byte[] data = toCreateInfoBytes(new Pair<>(tableDescriptor, splitKey));
    String parentNode = clientConf.get(ZOOKEEPER_CREATE_NODE, ZOOKEEPER_CREATE_NODE_DEFAULT);
    String node = NODE_PREFIX + StrUtil.generateRandomString(5);
    setDataToZK(clientConf, parentNode, node, data);
  }

  public static void setModifyInfoNode(Configuration clientConf, HTableDescriptor tableDescriptor)throws IOException, KeeperException{
    byte[] data = toModifyInfoBytes(tableDescriptor);
    String parentNode = clientConf.get(ZOOKEEPER_MODIFY_NODE, ZOOKEEPER_MODIFY_NODE_DEFAULT);
    String node = NODE_PREFIX + StrUtil.generateRandomString(5);
    setDataToZK(clientConf, parentNode, node, data);
  }

  public static void setDataToZK(Configuration clientConf, String parentNode, String node,
                                 byte[] data)throws IOException, KeeperException {

    ZooKeeperWatcher zk = getZkWatcher(clientConf);
    try {
      String baseNode = ClusterSwitchUtil.getBaseNode(clientConf.get(AliHBaseConstants.HACLIENT_BASE_NODE,
        AliHBaseConstants.HACLIENT_BASE_NODE_DEFAULT), clientConf.get(AliHBaseConstants.HACLIENT_CLUSTER_ID));

      String createNode = ZKUtil.joinZNode(baseNode, parentNode);
      //create if basenode not exist
      ZKUtil.createNodeIfNotExistsNoWatch(zk, baseNode, null,
        CreateMode.PERSISTENT);

      //create if createNode not exist
      ZKUtil.createNodeIfNotExistsNoWatch(zk, createNode, null,
        CreateMode.PERSISTENT);

      String dataNode = ZKUtil.joinZNode(createNode, node);
      if (ZKUtil.checkExists(zk, dataNode) == -1) {
        ZKUtil.createNodeIfNotExistsNoWatch(zk, dataNode, data,
          CreateMode.PERSISTENT);
      } else {
        ZKUtil.setData(zk, dataNode, data);
      }
    } finally {
      if (zk != null) {
        zk.close();
      }
    }
  }

  public static ZooKeeperWatcher getZkWatcher(Configuration clientConf) throws IOException{
    String endpoint = clientConf.get(AliHBaseConstants.ALIHBASE_SERVER_NAME);
    Configuration conf = new Configuration();
    if (!ClusterSwitchUtil.isValidEndpoint(endpoint)) {
      throw new IOException("Endpoint " + endpoint + " has invalid format, vaild format is " +
        "hostname:port");
    }
    conf.set(HConstants.ZOOKEEPER_QUORUM, endpoint.split(":")[0]);
    conf.set(HConstants.ZOOKEEPER_CLIENT_PORT, endpoint.split(":")[1]);
    ZooKeeperWatcher zk = new ZooKeeperWatcher(conf, "GetData", null, false);
    return zk;
  }

  public static HTableDescriptor getTableDescriptorFromString(String tableDescriptorString) throws DeserializationException, IOException{
    HTableDescriptor tableDescriptor =
      HTableDescriptor.parseFrom(Bytes.toBytesBinary(tableDescriptorString));
    return tableDescriptor;
  }

  public static byte[][] getSplitKeysFromStringList(List<String> splitKeyList){
    byte[][] splitKeys = null;
    if(!splitKeyList.isEmpty()){
      splitKeys = new byte[splitKeyList.size()][];
      for (int i = 0; i < splitKeyList.size(); i++) {
        splitKeys[i] = Bytes.toBytesBinary(splitKeyList.get(i));
      }
    }
    return splitKeys;
  }

  public static Pair<String, List<String>> toCreateInfoStringPair(byte[] data) throws DeserializationException, IOException{
    ProtobufUtil.expectPBMagicPrefix(data);
    int prefixLen = ProtobufUtil.lengthOfPBMagic();
    ClusterSwitchProto.CreateInfo.Builder builder =
      ClusterSwitchProto.CreateInfo.newBuilder();
    ProtobufUtil.mergeFrom(builder, data, prefixLen, data.length - prefixLen);
    ClusterSwitchProto.CreateInfo createInfo = builder.build();
    String tableDescriptorString = createInfo.getTableDescriptor();
    List<String> splitKeyList = createInfo.getSplitKeysList();
    return new Pair<>(tableDescriptorString, splitKeyList);
  }

  public static String toModifyInfoString(byte[] data) throws DeserializationException, IOException{
    ProtobufUtil.expectPBMagicPrefix(data);
    int prefixLen = ProtobufUtil.lengthOfPBMagic();
    ClusterSwitchProto.ModifyInfo.Builder builder =
      ClusterSwitchProto.ModifyInfo.newBuilder();
    ProtobufUtil.mergeFrom(builder, data, prefixLen, data.length - prefixLen);
    ClusterSwitchProto.ModifyInfo modifyInfo = builder.build();
    String tableDescriptorString = modifyInfo.getTableDescriptor();
    return tableDescriptorString;
  }
}