package com.alibaba.hbase.haclient.dualservice;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.haclient.ClusterSwitchUtil;
import com.alibaba.hbase.protobuf.generated.ClusterSwitchProto;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;
import org.apache.hadoop.hbase.zookeeper.ZooKeeperWatcher;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.protobuf.ProtobufUtil;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;

public class DualUtil {
  private static Log LOG = LogFactory.getLog(DualUtil.class);

  public static final String DUALTABLE_DIR = "hbase.haclient.dualtable.dir";
  public static final String DUALTABLE_DIR_DEFATULT = "/alihbase/dualtable/";
  public static final String DUALTABLE_FILE = "hbase.haclient.dualtable.filename";
  public static final String DUALTABLE_FILE_DEFAULT = "dualtable.xml";
  public static final String DUAL_TABLE = "hbase.haclient.dual.table";
  public static final String DUAL_CONFIG_SEP = "#";
  public static final String DUAL_CONFIG_GLITCH = "glitchtimeout";


  public static List<String> toDualTables(byte[] data) throws IOException {
    if (data == null) {
      return  null;
    }
    try {
      ProtobufUtil.expectPBMagicPrefix(data);
      int prefixLen = ProtobufUtil.lengthOfPBMagic();
      ClusterSwitchProto.DualTable.Builder builder =
        ClusterSwitchProto.DualTable.newBuilder();
      ProtobufUtil.mergeFrom(builder, data, prefixLen, data.length - prefixLen);
      ClusterSwitchProto.DualTable dualTable = builder.build();
      List<String> dualTables = dualTable.getTableListList();
      return dualTables;
    } catch (DeserializationException e) {
      throw new IOException(e);
    }
  }


  /**
   * Get DualTables from zk
   * @param endpoint
   * @return List<String>
   * @throws IOException
   * @throws KeeperException
   */
  public static List<String> getDualTablesFromZK(String endpoint, Configuration clientConf)
    throws IOException, KeeperException {
    Configuration conf = new Configuration();
    if (!ClusterSwitchUtil.isValidEndpoint(endpoint)) {
      throw new IOException("Endpoint " + endpoint + " has invalid format, vaild format is " +
        "hostname:port");
    }
    conf.set(HConstants.ZOOKEEPER_QUORUM, endpoint.split(":")[0]);
    conf.set(HConstants.ZOOKEEPER_CLIENT_PORT, endpoint.split(":")[1]);
    ZooKeeperWatcher zk = new ZooKeeperWatcher(conf, "GetCommand", null, false);
    try {
      String baseNode = ClusterSwitchUtil.getBaseNode(clientConf.get(AliHBaseConstants.HACLIENT_BASE_NODE,
        AliHBaseConstants.HACLIENT_BASE_NODE_DEFAULT), clientConf.get(AliHBaseConstants.HACLIENT_CLUSTER_ID));
      if (ZKUtil.checkExists(zk, baseNode) == -1) {
        return new ArrayList<String>();
      }
      String dualTableNode = ZKUtil.joinZNode(baseNode,
        clientConf.get(ClusterSwitchUtil.ZOOKEEPER_DUAL_TABLE_NODE,
          ClusterSwitchUtil.ZOOKEEPER_DUAL_TABLE_NODE_DEFAULT));
      if (ZKUtil.checkExists(zk, dualTableNode) == -1) {
        return new ArrayList<String>();
      } else {
        byte[] data = ZKUtil.getDataNoWatch(zk, dualTableNode, null);
        List<String> dualTables = toDualTables(data);
        return dualTables;
      }
    } finally {
      if (zk != null) {
        zk.close();
      }
    }
  }

  /**
   * if can not get dualtables from endpoint get dualtables from xml
   * @return List<String>
   */
  public static List<String> getDualTablesFromXML(Configuration conf){
    return getDualTablesFromXML(System.getProperty("user.home") + conf.get(DUALTABLE_DIR,
      DUALTABLE_DIR_DEFATULT) + conf.get(DUALTABLE_FILE, DUALTABLE_FILE_DEFAULT));
  }


  /**
   * get dualtables from xml
   * @param file
   * @return List<String>
   */
  public static List<String> getDualTablesFromXML(String file){
    try{
      Configuration conf  = new Configuration();
      conf.addResource(new Path(file));
      String dualTablesString = conf.get(DUAL_TABLE, null);
      if(dualTablesString == null){
        return new ArrayList<String>();
      }
      List<String> dualTables = new ArrayList<>(Arrays.asList(StringUtils.split(dualTablesString,
        ",")));

      return dualTables;
    } catch(Exception e){
      LOG.error("Get dual tables from xml failed, " + e);
      return null;
    }
  }

  /**
   * flush dual tables to user home dir, if can not dual tables from endpoint
   * get dual tables from disk
   * @param dualTables
   */
  public static void flushDualTables(List<String> dualTables, Configuration clientConf){
    if(dualTables == null || dualTables.isEmpty()){
      return;
    }
    Configuration conf  = new Configuration();
    conf.clear();
    conf.set(DUAL_TABLE, StringUtils.join(dualTables, ","));
    String dirName = System.getProperty("user.home") + clientConf.get(DUALTABLE_DIR, DUALTABLE_DIR_DEFATULT);
    String fileName = dirName + clientConf.get(DUALTABLE_FILE, DUALTABLE_FILE_DEFAULT);
    File dir = new File(dirName);
    if(!dir.exists()){
      dir.mkdirs();
    }

    FileOutputStream out = null;
    try{
      File file = new File(fileName);
      if(!file.exists()){
        file.createNewFile();
      }
      out =
        new FileOutputStream(file);
      conf.writeXml(out);
      out.flush();
    } catch(Exception e){
      LOG.warn("flush dual tables failed : " + e);
    } finally {
      try {
        if (out != null) {
          out.close();
        }
      }catch(IOException ioe){
        LOG.debug("flush dual tables FileOutputStream close failed");
      }
    }
  }


  public static byte[] toDualTablesBytes(List<String> dualTables) {
    ClusterSwitchProto.DualTable.Builder builder = ClusterSwitchProto.DualTable.newBuilder();
    for(String dualTable : dualTables){
      builder.addTableList(dualTable);
    }
    return ProtobufUtil.prependPBMagic(builder.build().toByteArray());
  }

  /**
   * set dual trace to zk
   * @param endpoint
   * @return String
   * @throws IOException
   * @throws KeeperException
   */
  public static void setDualTraceToZK(ZooKeeperWatcher zk ,String endpoint,
                                      Configuration clientConf, String nodeName, String metrics) throws IOException, KeeperException {
      String baseNode = ClusterSwitchUtil.getBaseNode(clientConf.get(AliHBaseConstants.HACLIENT_BASE_NODE,
        AliHBaseConstants.HACLIENT_BASE_NODE_DEFAULT), clientConf.get(AliHBaseConstants.HACLIENT_CLUSTER_ID));
      ZKUtil.createNodeIfNotExistsNoWatch(zk, baseNode, null, CreateMode.PERSISTENT);
      String dualTraceBaseNode = ZKUtil.joinZNode(baseNode,
        clientConf.get(ClusterSwitchUtil.ZOOKEEPER_DUAL_TRACE_NODE,
          ClusterSwitchUtil.ZOOKEEPER_DUAL_TRACE_NODE_DEFAULT));
      ZKUtil.createNodeIfNotExistsNoWatch(zk, dualTraceBaseNode, null, CreateMode.PERSISTENT);
      String dualTraceNode = ZKUtil.joinZNode(dualTraceBaseNode, nodeName);
      if (ZKUtil.checkExists(zk, dualTraceNode) == -1) {
        ZKUtil.createNodeIfNotExistsNoWatch(zk, dualTraceNode, toDualTraceBytes(metrics),
          CreateMode.EPHEMERAL);
      } else {
        ZKUtil.setData(zk, dualTraceNode, toDualTraceBytes(metrics));
      }
  }


  public static ZooKeeperWatcher getZookeeperWatcher(String endpoint) throws IOException{
    Configuration conf = new Configuration();
    if (!ClusterSwitchUtil.isValidEndpoint(endpoint)) {
      throw new IOException("Endpoint " + endpoint + " has invalid format, vaild format is " +
        "hostname:port");
    }
    conf.set(HConstants.ZOOKEEPER_QUORUM, endpoint.split(":")[0]);
    conf.set(HConstants.ZOOKEEPER_CLIENT_PORT, endpoint.split(":")[1]);
    return new ZooKeeperWatcher(conf, "GetCommand", null, false);
  }

  public static byte[] toDualTraceBytes(String metrics) {
    ClusterSwitchProto.DualTrace.Builder builder = ClusterSwitchProto.DualTrace.newBuilder();
    builder.setMetrics(metrics);
    return ProtobufUtil.prependPBMagic(builder.build().toByteArray());
  }

  public static String getDualTraceMetrics(byte[] data) throws IOException {
    if (data == null) {
      return  null;
    }
    try {
      ProtobufUtil.expectPBMagicPrefix(data);
      int prefixLen = ProtobufUtil.lengthOfPBMagic();
      ClusterSwitchProto.DualTrace.Builder builder =
        ClusterSwitchProto.DualTrace.newBuilder();
      ProtobufUtil.mergeFrom(builder, data, prefixLen, data.length - prefixLen);
      ClusterSwitchProto.DualTrace dualTrace = builder.build();
      if(dualTrace == null){
        return null;
      }
      return dualTrace.getMetrics();
    } catch (DeserializationException e) {
      throw new IOException(e);
    }
  }

}