/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.client;

import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.index.AliHBaseIndexDescriptor;

import java.io.IOException;
import java.util.List;

/**
 * Provide some new feature interfaces
 */
public interface AliHBaseAdminInterface extends Admin {

  /**
   * Get index schema for specified data table. Return empty list if no index found, never return null.
   * @param dataTableName data table name
   * @return all index descriptors of the specified data table; if no index found for this data table, then return an empty list.
   */
  List<AliHBaseIndexDescriptor> describeIndex(TableName dataTableName) throws IOException;

  /**
   * Create a global secondary index table against a specified data table, block forever until operation success.
   * This call needs to wait for the index data to be built, may consume relatively long time, and thus may block current thread.
   * @param indexDescriptor descriptor of index table
   * @throws IOException
   */
  void createIndex(AliHBaseIndexDescriptor indexDescriptor) throws IOException ;

  /**
   * Create a global secondary index table against a specified data table with split keys, block forever until operation success.
   * This call needs to wait for the index data to be built, may consume relatively long time, and thus may block current thread.
   * @param indexDescriptor descriptor of index table
   * @param splitKeys pre-split range of index table
   * @throws IOException
   */
  void createIndex(AliHBaseIndexDescriptor indexDescriptor, byte[][] splitKeys) throws IOException;

  /**
   * Create a global secondary index table against a specified data table, block forever until operation success.
   * Index table will be split into numRegions initial regions, with startKey, endKey specified.
   * This call needs to wait for the index data to be built, may consume relatively long time, and thus may block current thread.
   * @param indexDescriptor descriptor of index table
   * @param startKey start range key of index table
   * @param endKey end range key of index table
   * @param numRegions number of initial regions
   * @throws IOException
   */
  void createIndex(AliHBaseIndexDescriptor indexDescriptor, byte[] startKey, byte[] endKey, int numRegions) throws IOException;

  /**
   * Drop index and clean all index data synchronously, block forever until operation done.
   * The #{@link AliHBaseAdminInterface#offlineIndex(String, TableName)} operation must be performed first.
   * @param indexName index name
   * @param dataTable name of main table
   * @throws IOException
   */
  void deleteIndex(String indexName, TableName dataTable) throws IOException;

  /**
   * Modify index status to DISABLED, block forever until operation done.
   * Re-online is not supported, index must be deleted and re-created.
   * @param indexName index name
   * @param dataTable name of main table
   * @throws IOException
   */
  void offlineIndex(String indexName, TableName dataTable) throws IOException;
}
