package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.haclient.dualservice.DualExecutor;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;
import com.google.protobuf.Service;
import com.google.protobuf.ServiceException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.coprocessor.Batch;
import org.apache.hadoop.hbase.filter.CompareFilter;
import org.apache.hadoop.hbase.ipc.CoprocessorRpcChannel;
import org.apache.hadoop.hbase.util.Pair;

public class AliHBaseMultiTable extends HTable implements HTableInterface,  RegionLocator{
  private static final Log LOG = LogFactory.getLog(AliHBaseMultiTable.class);

  private volatile HTable activeHTable = null;
  private volatile HTable standbyHTable = null;
  //switch count decide switch or not
  private long switchCount = 0;
  private AliHBaseMultiClusterConnection connection;
  private Configuration conf;
  private TableName tableName;
  private AliHBaseConstants.ClusterType currentType;
  private boolean tableDualEnabled = false;
  private int glitchTimeout;
  private int scannerTimeout;

  public AliHBaseMultiTable(TableName tableName, AliHBaseMultiClusterConnection connection,
                            HTable table) throws IOException{
    super(tableName, connection, new ConnectionConfiguration(connection.getConfiguration()), null
      , null, new NoopExecutorService());
    this.activeHTable = table;
    this.connection = connection;
    this.tableName = tableName;
    this.currentType =
      AliHBaseConstants.ClusterType.valueOf(connection.getConfiguration().get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
    this.switchCount = connection.getSwitchCount();
    this.conf = connection.getConfiguration();
    if(this.conf.getBoolean(AliHBaseConstants.DUALSERVICE_TABLE_ENABLE,
      AliHBaseConstants.DEFAULT_DUALSERVICE_TABLE_ENABLE)){
      tableDualEnabled = true;
    }else{
      tableDualEnabled = this.conf.getBoolean(DualExecutor.createTableConfKey(tableName.getNameAsString(),
        AliHBaseConstants.DUALSERVICE_ENABLE), false);
    }
    this.glitchTimeout = this.conf.getInt(
      DualExecutor.createTableConfKey(tableName.getNameAsString(),
        AliHBaseConstants.DUALSERVICE_GLITCHTIMEOUT), this.conf.getInt(
        AliHBaseConstants.DUALSERVICE_GLITCHTIMEOUT, AliHBaseConstants.DEFAULT_DUALSERVICE_GLITCHTIMEOUT));
    this.scannerTimeout = this.conf.getInt(HConstants.HBASE_CLIENT_SCANNER_TIMEOUT_PERIOD,
      HConstants.DEFAULT_HBASE_CLIENT_SCANNER_TIMEOUT_PERIOD);
    if(tableDualEnabled){
      this.standbyHTable = this.connection.getHTableWithStandbyConnection(tableName);
    }
  }

  private HTable getCurrentHTable() throws IOException{
    if(this.switchCount < connection.getSwitchCount()){
      Configuration currentConf = this.connection.getConfiguration();
      AliHBaseConstants.ClusterType clusterType =
        AliHBaseConstants.ClusterType.valueOf(currentConf.get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
      //only HBase to HBase no need update HTable
      if(!(this.currentType == AliHBaseConstants.ClusterType.HBASE && clusterType == AliHBaseConstants.ClusterType.HBASE)) {
        LOG.debug("Update HTable, from " + this.currentType + ", to " + clusterType);
        HTable newTable = connection.getHTableByType(tableName);
        HTable lastHTable = this.activeHTable;
        this.activeHTable = newTable;
        if (lastHTable != null) {
          try {
            lastHTable.close();
          } catch (IOException e) {
            LOG.warn("last htable close failed" + e);
          }
        }
      }
      this.currentType = clusterType;
      this.switchCount = connection.getSwitchCount();
      this.conf = currentConf;
      if(tableDualEnabled){
        this.standbyHTable = this.connection.getHTableWithStandbyConnection(tableName);
      }
    }
    return this.activeHTable;
  }

  //visiable for test
  public boolean getTableDualEnable(){
    return this.tableDualEnabled;
  }

  private static class NoopExecutorService implements ExecutorService {
    @Override
    public void shutdown() {

    }

    @Override
    public List<Runnable> shutdownNow() {
      return null;
    }

    @Override
    public boolean isShutdown() {
      return false;
    }

    @Override
    public boolean isTerminated() {
      return false;
    }

    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
      return false;
    }

    @Override
    public <T> Future<T> submit(Callable<T> task) {
      return null;
    }

    @Override
    public <T> Future<T> submit(Runnable task, T result) {
      return null;
    }

    @Override
    public Future<?> submit(Runnable task) {
      return null;
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks)
      throws InterruptedException {
      return null;
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks, long timeout,
                                         TimeUnit unit) throws InterruptedException {
      return null;
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks)
      throws InterruptedException, ExecutionException {
      return null;
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
      throws InterruptedException, ExecutionException, TimeoutException {
      return null;
    }

    @Override
    public void execute(Runnable command) {

    }
  }

  @Override
  public Configuration getConfiguration() {
    try {
      return this.getCurrentHTable().getConfiguration();
    }catch(IOException e){
      LOG.error("getConfiguration failed : " + e);
      return null;
    }
  }

  @Override
  public HRegionLocation getRegionLocation(String row) throws IOException {
    return this.getCurrentHTable().getRegionLocation(row);
  }

  @Override
  public HRegionLocation getRegionLocation(byte[] row) throws IOException {
    return this.getCurrentHTable().getRegionLocation(row);
  }

  @Override
  public HRegionLocation getRegionLocation(byte[] row, boolean reload) throws IOException {
    return this.getCurrentHTable().getRegionLocation(row, reload);
  }

  @Override
  public byte[] getTableName() {
    return this.tableName.getName();
  }

  @Override
  public TableName getName() {
    return this.tableName;
  }

  @Override
  public int getScannerCaching() {
    try{
      return this.getCurrentHTable().getScannerCaching();
    }catch(IOException e){
      LOG.error("getScannerCaching failed : " + e);
      return 0;
    }
  }

  @Override
  public List<Row> getWriteBuffer() {
    try{
      return this.getCurrentHTable().getWriteBuffer();
    }catch(IOException e){
      LOG.error("getWriteBuffer failed : " + e);
      return null;
    }
  }

  @Override
  public void setScannerCaching(int scannerCaching) {
    try {
      this.getCurrentHTable().setScannerCaching(scannerCaching);
    }catch(IOException e){
      LOG.error("setScannerCaching failed : " + e);
    }
  }

  @Override
  public byte[][] getStartKeys() throws IOException {
    return this.getCurrentHTable().getStartKeys();
  }

  @Override
  public byte[][] getEndKeys() throws IOException {
    return this.getCurrentHTable().getEndKeys();
  }

  @Override
  public Pair<byte[][], byte[][]> getStartEndKeys() throws IOException {
    return this.getCurrentHTable().getStartEndKeys();
  }

  @Override
  public NavigableMap<HRegionInfo, ServerName> getRegionLocations() throws IOException {
    return this.getCurrentHTable().getRegionLocations();
  }

  @Override
  public List<HRegionLocation> getAllRegionLocations() throws IOException {
    return this.getCurrentHTable().getAllRegionLocations();
  }

  @Override
  public List<HRegionLocation> getRegionsInRange(byte[] startKey, byte[] endKey) throws IOException {
    return this.getCurrentHTable().getRegionsInRange(startKey, endKey);
  }

  @Override
  public List<HRegionLocation> getRegionsInRange(byte[] startKey, byte[] endKey, boolean reload) throws IOException {
    return this.getCurrentHTable().getRegionsInRange(startKey, endKey, reload);
  }

  @Override
  public Result getRowOrBefore(byte[] row, final byte[] family) throws IOException {
    return this.getCurrentHTable().getRowOrBefore(row, family);
  }

  @Override
  public HConnection getConnection() {
    return this.connection;
  }

  @Override
  public HTableDescriptor getTableDescriptor() throws IOException {
    return this.getCurrentHTable().getTableDescriptor();
  }

  @Override
  public ResultScanner getScanner(Scan scan) throws IOException {
    if(tableDualEnabled){
      DualExecutor executor = this.connection.getDualExecutor();
      if(executor != null && scan.isSmall()){
        HTable activeTable = this.getCurrentHTable();
        return new AliHBaseResultScanner(activeTable, this.standbyHTable, executor, scan,
          glitchTimeout, scannerTimeout);
      }
    }
    return this.getCurrentHTable().getScanner(scan);
  }

  @Override
  public ResultScanner getScanner(byte[] family) throws IOException {
    Scan scan = new Scan();
    scan.addFamily(family);
    return this.getScanner(scan);
  }

  @Override
  public ResultScanner getScanner(byte[] family, byte[] qualifier) throws IOException {
    Scan scan = new Scan();
    scan.addColumn(family, qualifier);
    return this.getScanner(scan);
  }

  @Override
  public Result get(Get get) throws IOException {
    if(tableDualEnabled){
      DualExecutor executor = this.connection.getDualExecutor();
      if(executor != null) {
        HTable activeTable = this.getCurrentHTable();
        return executor.dualGet(activeTable, this.standbyHTable, tableName.getName(), get,
          this.glitchTimeout, this.getOperationTimeout());
      }
    }
    return this.getCurrentHTable().get(get);
  }

  @Override
  public Result[] get(List<Get> gets) throws IOException {
    if(tableDualEnabled){
      DualExecutor executor = this.connection.getDualExecutor();
      if(executor != null) {
        HTable activeTable = this.getCurrentHTable();
        return executor.dualBatchGet(activeTable, this.standbyHTable, tableName.getName(), gets,
          this.glitchTimeout, this.getOperationTimeout());
      }
    }
    return this.getCurrentHTable().get(gets);
  }

  @Override
  public void batch(List<? extends Row> actions, Object[] results) throws InterruptedException, IOException {
    this.getCurrentHTable().batch(actions, results);
  }

  @Override
  public Object[] batch(List<? extends Row> actions) throws InterruptedException, IOException {
    return this.getCurrentHTable().batch(actions);
  }

  @Override
  public <R> void batchCallback(List<? extends Row> actions, Object[] results, Batch.Callback<R> callback) throws IOException, InterruptedException {
    this.getCurrentHTable().batchCallback(actions, results, callback);
  }

  @Override
  public <R> Object[] batchCallback(List<? extends Row> actions, Batch.Callback<R> callback) throws IOException, InterruptedException {
    return this.getCurrentHTable().batchCallback(actions, callback);
  }

  @Override
  public void delete(final Delete delete) throws IOException {
    if(tableDualEnabled){
      DualExecutor executor = this.connection.getDualExecutor();
      if(executor != null) {
        HTable activeTable = this.getCurrentHTable();
        executor.dualDelete(activeTable, this.standbyHTable, tableName.getName(), delete,
          this.glitchTimeout, this.getOperationTimeout());
        return;
      }
    }
    this.getCurrentHTable().delete(delete);
  }

  @Override
  public void delete(List<Delete> deletes) throws IOException {
    if(tableDualEnabled){
      DualExecutor executor = this.connection.getDualExecutor();
      if(executor != null) {
        HTable activeTable = this.getCurrentHTable();
        executor.dualBatchDelete(activeTable, this.standbyHTable, tableName.getName(), deletes,
          this.glitchTimeout, this.getOperationTimeout());
        return;
      }
    }
    this.getCurrentHTable().delete(deletes);
  }

  @Override
  public void put(final Put put) throws IOException {
    if(tableDualEnabled){
      DualExecutor executor = this.connection.getDualExecutor();
      if(executor != null) {
        HTable activeTable = this.getCurrentHTable();
        executor.dualPut(activeTable, this.standbyHTable, tableName.getName(), put,
          this.glitchTimeout, this.getOperationTimeout());
        return;
      }
    }
    this.getCurrentHTable().put(put);
  }

  @Override
  public void put(List<Put> puts) throws IOException {
    if(tableDualEnabled){
      DualExecutor executor = this.connection.getDualExecutor();
      if(executor != null) {
        HTable activeTable = this.getCurrentHTable();
        executor.dualBatchPut(activeTable, this.standbyHTable, tableName.getName(), puts,
          this.glitchTimeout, this.getOperationTimeout());
        return;
      }
    }
    this.getCurrentHTable().put(puts);
  }

  @Override
  public void mutateRow(final RowMutations rm) throws IOException {
    this.getCurrentHTable().mutateRow(rm);
  }

  @Override
  public Result append(final Append append) throws IOException {
    return this.getCurrentHTable().append(append);
  }

  @Override
  public Result increment(final Increment increment) throws IOException {
    return this.getCurrentHTable().increment(increment);
  }

  @Override
  public long incrementColumnValue(byte[] row, byte[] family, byte[] qualifier, long amount) throws IOException {
    return this.getCurrentHTable().incrementColumnValue(row, family, qualifier, amount);
  }

  @Override
  public long incrementColumnValue(byte[] row, byte[] family, byte[] qualifier, long amount, boolean writeToWAL) throws IOException {
    return this.getCurrentHTable().incrementColumnValue(row, family, qualifier, amount, writeToWAL);
  }

  @Override
  public long incrementColumnValue(final byte[] row, final byte[] family, final byte[] qualifier, final long amount, final Durability durability) throws IOException {
    return this.getCurrentHTable().incrementColumnValue(row, family, qualifier, amount, durability);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public boolean checkAndPut(byte[] row, byte[] family, byte[] qualifier, byte[] value, Put put) throws IOException {
    return this.getCurrentHTable().checkAndPut(row, family, qualifier, value, put);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public boolean checkAndPut(byte[] row, byte[] family, byte[] qualifier, CompareFilter.CompareOp compareOp, byte[] value, Put put) throws IOException {
    return this.getCurrentHTable().checkAndPut(row, family, qualifier, compareOp, value, put);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public boolean checkAndDelete(byte[] row, byte[] family, byte[] qualifier, byte[] value, Delete delete) throws IOException {
    return this.getCurrentHTable().checkAndDelete(row, family, qualifier, value, delete);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public boolean checkAndDelete(byte[] row, byte[] family, byte[] qualifier, CompareFilter.CompareOp compareOp, byte[] value, Delete delete) throws IOException {
    return this.getCurrentHTable().checkAndDelete(row, family, qualifier, compareOp, value, delete);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public boolean checkAndMutate(byte[] row, byte[] family, byte[] qualifier, CompareFilter.CompareOp compareOp, byte[] value, RowMutations rm) throws IOException {
    return this.getCurrentHTable().checkAndMutate(row, family, qualifier, compareOp, value, rm);
  }

  @Override
  public boolean exists(Get get) throws IOException {
    return this.getCurrentHTable().exists(get);
  }

  @Override
  public boolean[] existsAll(List<Get> gets) throws IOException {
   return this.getCurrentHTable().existsAll(gets);
  }

  @Override
  public Boolean[] exists(List<Get> gets) throws IOException {
    return this.getCurrentHTable().exists(gets);
  }

  @Override
  public void flushCommits() throws IOException {
    this.getCurrentHTable().flushCommits();
  }

  @Override
  public <R> void processBatchCallback(List<? extends Row> list, Object[] results, Batch.Callback<R> callback) throws IOException, InterruptedException {
    this.getCurrentHTable().processBatchCallback(list, results, callback);
  }

  public void processBatch(List<? extends Row> list, Object[] results) throws IOException, InterruptedException {
    this.getCurrentHTable().batch(list, results);
  }

  @Override
  public void close() throws IOException {
    super.close();
    if(this.activeHTable != null){
      this.activeHTable.close();
    }
    if(this.standbyHTable != null){
      this.standbyHTable.close();
    }
  }

  @Override
  public void validatePut(Put put) throws IllegalArgumentException {
    try{
      this.getCurrentHTable().validatePut(put);
    }catch(IOException e){
      LOG.error("validatePut failed : " + e);
    }
  }

  @Override
  public void clearRegionCache() {
    try{
      this.getCurrentHTable().clearRegionCache();
    }catch(IOException e){
      LOG.error("clearRegionCache failed : " + e);
    }
  }

  @Override
  public CoprocessorRpcChannel coprocessorService(byte[] row) {
    try {
      return this.getCurrentHTable().coprocessorService(row);
    }catch(IOException e){
      LOG.error("coprocessorService failed : " + e);
      return null;
    }
  }

  @Override
  public <T extends Service, R> Map<byte[], R> coprocessorService(Class<T> service, byte[] startKey, byte[] endKey, Batch.Call<T, R> callable) throws ServiceException, Throwable {
    return this.getCurrentHTable().coprocessorService(service, startKey, endKey, callable);
  }

  @Override
  public <T extends Service, R> void coprocessorService(final Class<T> service, byte[] startKey, byte[] endKey, final Batch.Call<T, R> callable, final Batch.Callback<R> callback) throws ServiceException, Throwable {
    this.getCurrentHTable().coprocessorService(service, startKey, endKey, callable, callback);
  }

  @Override
  public boolean isAutoFlush() {
    try {
      return this.getCurrentHTable().isAutoFlush();
    }catch(IOException e){
      LOG.error("isAutoFlush failed : " + e);
      return false;
    }
  }

  @Override
  public void setAutoFlush(boolean autoFlush) {
    try{
      this.getCurrentHTable().setAutoFlush(autoFlush);
    }catch(IOException e){
      LOG.error("setAutoFlush failed : " + e);
    }
  }

  @Override
  public void setAutoFlushTo(boolean autoFlush) {
    try{
      this.getCurrentHTable().setAutoFlushTo(autoFlush);
    }catch(IOException e){
      LOG.error("setAutoFlushTo failed : " + e);
    }
  }

  @Override
  public void setAutoFlush(boolean autoFlush, boolean clearBufferOnFail) {
    try{
      this.getCurrentHTable().setAutoFlush(autoFlush, clearBufferOnFail);
    }catch(IOException e){
      LOG.error("setAutoFlush failed : " + e);
    }
  }

  @Override
  public long getWriteBufferSize() {
    try{
      return this.getCurrentHTable().getWriteBufferSize();
    }catch(IOException e){
      LOG.error("getWriteBufferSize failed : " + e);
      return 0;
    }
  }

  @Override
  public void setWriteBufferSize(long writeBufferSize) throws IOException {
    this.getCurrentHTable().setWriteBufferSize(writeBufferSize);
  }

  @Deprecated
  @Override
  public int getRpcTimeout() {
    try {
      return this.getCurrentHTable().getRpcTimeout();
    }catch(IOException e){
      LOG.error("getRpcTimeout failed : " + e);
      return 0;
    }
  }

  /** @deprecated */
  @Deprecated
  @Override
  public void setRpcTimeout(int rpcTimeout) {
    try{
      this.getCurrentHTable().setRpcTimeout(rpcTimeout);
    }catch(IOException e){
      LOG.error("setRpcTimeout failed : " + e);
    }
  }

  @Override
  public int getOperationTimeout() {
    try{
      return this.getCurrentHTable().getOperationTimeout();
    }catch(IOException e){
      LOG.error("getOperationTimeout failed : " + e);
      return 0;
    }
  }

  @Override
  public void setOperationTimeout(int operationTimeout) {
    try{
      this.getCurrentHTable().setOperationTimeout(operationTimeout);
    }catch(IOException e){
      LOG.error("setOperationTimeout failed : " + e);
    }
  }

  @Override
  public String toString() {
    try{
      return this.getCurrentHTable().toString();
    }catch(IOException e){
      LOG.error("toString failed : " + e);
      return null;
    }
  }

  @Override
  public <R extends Message> Map<byte[], R> batchCoprocessorService(Descriptors.MethodDescriptor methodDescriptor, Message request, byte[] startKey, byte[] endKey, R responsePrototype) throws ServiceException, Throwable {
    return this.getCurrentHTable().batchCoprocessorService(methodDescriptor, request, startKey, endKey
      , responsePrototype);
  }

  @Override
  public <R extends Message> void batchCoprocessorService(Descriptors.MethodDescriptor methodDescriptor, Message request, byte[] startKey, byte[] endKey, R responsePrototype, Batch.Callback<R> callback) throws ServiceException, Throwable {
    this.getCurrentHTable().batchCoprocessorService(methodDescriptor, request, startKey, endKey,
      responsePrototype, callback);
  }

  @Override
  public RegionLocator getRegionLocator() {
    try{
      return this.getCurrentHTable().getRegionLocator();
    }catch(IOException e){
      LOG.error("getRegionLocator failed : " + e);
      return null;
    }
  }

}