/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Queue;
import java.util.TreeMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import com.alibaba.hbase.client.AliHBaseAPIProxy;
import com.alibaba.hbase.client.AliHBaseTableInterface;
import com.alibaba.hbase.client.AliHBaseUEAggregateService;
import com.alibaba.hbase.client.AliHBaseUEBufferedMutator;
import com.alibaba.hbase.client.AliHBaseUEConnection;
import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.client.ElementConvertor;
import com.alibaba.hbase.exception.BatchExceptions;
import com.alibaba.lindorm.client.TableService;
import com.alibaba.lindorm.client.WideColumnService;
import com.alibaba.lindorm.client.core.meta.ExternalIndexConfig;
import com.alibaba.lindorm.client.core.meta.ExternalIndexField;
import com.alibaba.lindorm.client.core.meta.ExternalIndexRowFormatterType;
import com.alibaba.lindorm.client.core.meta.ExternalIndexType;
import com.alibaba.lindorm.client.core.widecolumnservice.WAppend;
import com.alibaba.lindorm.client.core.widecolumnservice.WDelete;
import com.alibaba.lindorm.client.core.widecolumnservice.WGet;
import com.alibaba.lindorm.client.core.widecolumnservice.WIncrement;
import com.alibaba.lindorm.client.core.widecolumnservice.WPut;
import com.alibaba.lindorm.client.core.widecolumnservice.WResult;
import com.alibaba.lindorm.client.core.widecolumnservice.WRowMutations;
import com.alibaba.lindorm.client.core.widecolumnservice.WScan;
import com.alibaba.lindorm.client.core.widecolumnservice.WScanner;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WCompareFilter;
import com.alibaba.lindorm.client.schema.LindormTableDescriptor;

import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;
import com.google.protobuf.Service;
import com.google.protobuf.ServiceException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.coprocessor.Batch;
import org.apache.hadoop.hbase.filter.CompareFilter;
import org.apache.hadoop.hbase.ipc.CoprocessorRpcChannel;
import org.apache.hadoop.hbase.protobuf.generated.AggregateProtos;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;

public class AliHBaseUETable extends HTable implements AliHBaseTableInterface, RegionLocator {


  private TableName myTableName;
  private AliHBaseUEConnection connection;
  private Configuration conf;
  private AliHBaseAPIProxy proxy;
  private boolean isClosed;
  private boolean autoFlush = true;

  private int defaultScannerCaching;

  private AliHBaseUEBufferedMutator mutator;


  public AliHBaseUETable(TableName tableName, AliHBaseUEConnection connection) throws IOException {
    super(tableName, new AliHBaseUEClusterConnection(connection), new NoopExecutorService());
    this.myTableName = tableName;
    this.connection = connection;
    this.conf = connection.getConfiguration();
    this.defaultScannerCaching = conf.getInt(HConstants.HBASE_CLIENT_SCANNER_CACHING,
        AliHBaseConstants.ALIHBASE_CLIENT_SCANNER_CACHING_DEFAULT);
    if (defaultScannerCaching > AliHBaseConstants.ALIHBASE_CLIENT_SCANNER_CACHING_DEFAULT) {
      this.defaultScannerCaching = AliHBaseConstants.ALIHBASE_CLIENT_SCANNER_CACHING_DEFAULT;
    }
    this.proxy = connection.getAPIProxy(tableName);


  }

  private static class NoopExecutorService implements ExecutorService {
    @Override
    public void shutdown() {

    }

    @Override
    public List<Runnable> shutdownNow() {
      return null;
    }

    @Override
    public boolean isShutdown() {
      return false;
    }

    @Override
    public boolean isTerminated() {
      return false;
    }

    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
      return false;
    }

    @Override
    public <T> Future<T> submit(Callable<T> task) {
      return null;
    }

    @Override
    public <T> Future<T> submit(Runnable task, T result) {
      return null;
    }

    @Override
    public Future<?> submit(Runnable task) {
      return null;
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks)
        throws InterruptedException {
      return null;
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks, long timeout,
        TimeUnit unit) throws InterruptedException {
      return null;
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks)
        throws InterruptedException, ExecutionException {
      return null;
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
        throws InterruptedException, ExecutionException, TimeoutException {
      return null;
    }

    @Override
    public void execute(Runnable command) {

    }
  }

  @Override
  public HRegionLocation getRegionLocation(byte[] row) throws IOException {
    return getRegionLocation(row, false);
  }

  @Override
  public HRegionLocation getRegionLocation(byte[] row, boolean b) throws IOException {
    return proxy.getRegionLocation(row);
  }


  @Override
  public List<HRegionLocation> getAllRegionLocations() throws IOException {
    return proxy.getAllRegionLocations();
  }

  @Override
  public byte[][] getStartKeys() throws IOException {
    return getStartEndKeys().getFirst();
  }

  @Override
  public byte[][] getEndKeys() throws IOException {
    return getStartEndKeys().getSecond();
  }

  @Override
  public Pair<byte[][], byte[][]> getStartEndKeys() throws IOException {
    return proxy.getStartEndKeys();
  }

  @Override
  public TableName getName() {
    return myTableName;
  }

  @Override
  public Configuration getConfiguration() {
    return conf;
  }

  @Override
  public HTableDescriptor getTableDescriptor() throws IOException {
    return proxy.getDescriptor(myTableName);
  }

  @Override
  public boolean exists(Get get) throws IOException {
    return proxy.exists(get);
  }

  @Override
  public boolean[] existsAll(List<Get> gets) throws IOException {
    return proxy.exists(gets);
  }


  @Override
  public <R> void batchCallback(List<? extends Row> actions, Object[] results,
      Batch.Callback<R> callback) throws IOException {
    throw new IOException("BatchCallback not supported, use put(List<Put> puts), "
        + "get(List<Get> gets) or delete(List<Delete> deletes) respectively");
  }

  @Override
  public Result get(Get get) throws IOException {
    return proxy.get(get);
  }

  @Override
  public Result[] get(List<Get> gets) throws IOException {
    return proxy.get(gets);
  }



  @Override
  public ResultScanner getScanner(Scan scan) throws IOException {
    return proxy.getScanner(scan);
  }

  @Override
  public ResultScanner getScanner(byte[] family) throws IOException {
    Scan scan = new Scan();
    scan.addFamily(family);
    return getScanner(scan);
  }

  @Override
  public ResultScanner getScanner(byte[] family, byte[] qualifier) throws IOException {
    Scan scan = new Scan();
    scan.addColumn(family, qualifier);
    return getScanner(scan);
  }

  @Override
  public void put(Put put) throws IOException {
    if (autoFlush) {
      proxy.put(put);
    } else {
      getBufferedMutator().mutate(put);
    }
  }

  @Override
  public void put(List<Put> puts) throws IOException {
    if (autoFlush) {
      proxy.put(puts);
    } else {
      getBufferedMutator().mutate(puts);
    }
  }

  @Override
  public void delete(Delete delete) throws IOException {
    proxy.delete(delete);
  }

  @Override
  public void delete(List<Delete> deletes) throws IOException {
    proxy.delete(deletes);
  }

  @Override
  public boolean checkAndMutate(byte[] row, byte[] family, byte[] qualifier,  CompareFilter.CompareOp op,
      byte[] value, RowMutations rowMutations) throws IOException {
    return proxy.checkAndMutate(row, family, qualifier, op, value, rowMutations);
  }



  @Override
  public boolean checkAndPut(final byte [] row,
      final byte [] family, final byte [] qualifier, final byte [] value,
      final Put put)
      throws IOException {
    return checkAndPut(row, family, qualifier, CompareFilter.CompareOp.EQUAL, value, put);
  }

  @Override
  public boolean checkAndDelete(final byte [] row,
      final byte [] family, final byte [] qualifier, final byte [] value,
      final Delete delete) throws IOException {
    return this.checkAndDelete(row, family, qualifier, CompareFilter.CompareOp.EQUAL, value, delete);
  }

  @Override
  public boolean checkAndPut(final byte [] row,
      final byte [] family, final byte [] qualifier,
      CompareFilter.CompareOp compareOp, final byte [] value,
      final Put put) throws IOException {
    return proxy.checkAndPut(row, family, qualifier, compareOp, value, put);
  }

  @Override
  public boolean checkAndDelete(final byte [] row,
      final byte [] family, final byte [] qualifier,
      CompareFilter.CompareOp compareOp, final byte [] value, Delete delete) throws IOException {
    return proxy.checkAndDelete(row, family, qualifier, compareOp, value, delete);
  }

  @Override
  public long incrementColumnValue(final byte [] row, final byte [] family,
      final byte [] qualifier, final long amount)
      throws IOException {
    return incrementColumnValue(row, family, qualifier, amount, Durability.SYNC_WAL);
  }

  @Override
  public long incrementColumnValue(final byte [] row, final byte [] family,
      final byte [] qualifier, final long amount, final Durability durability) throws IOException {
      Increment increment = new Increment(row);
      increment.addColumn(family, qualifier, amount);
      Result result =  increment(increment);
      return Long.valueOf(Bytes.toLong(result.getValue(family, qualifier)));
  }

  @Override
  public void mutateRow(RowMutations rm) throws IOException {
    proxy.mutateRow(rm);
  }

  @Override
  public Result append(Append append) throws IOException {
    return proxy.append(append);
  }

  @Override
  public Result increment(Increment increment) throws IOException {
    return proxy.increment(increment);
  }

  @Override
  public void close() throws IOException {
    if (isClosed) {
      return;
    }
    flushCommits();
    if (mutator != null) {
      mutator.close();
    }
    proxy.close();
    isClosed = true;
  }

  @Override
  public CoprocessorRpcChannel coprocessorService(byte[] row) {
    throw new UnsupportedOperationException("coprocessorService not supported");
  }

  @Override
  public void batch(List<? extends Row> actions, Object[] results)
      throws IOException {
    proxy.batch(actions, results);
  }

  @Override
  public Object[] batch(List<? extends Row> actions) throws IOException {
    Object[] results = new Object[actions.size()];
    batch(actions, results);
    return results;
  }

  @Override
  public <R> Object[] batchCallback(List<? extends Row> list, Batch.Callback<R> callback)
      throws IOException, InterruptedException {
    throw new UnsupportedOperationException("batchCallback not supported");
  }

  @Override
  public <T extends Service, R> Map<byte[], R> coprocessorService(Class<T> service, byte[] startKey, byte[] endKey,
      Batch.Call<T, R> call) throws ServiceException, Throwable {
    throw new UnsupportedOperationException("coprocessorService not supported");
  }

  @Override
  public <T extends Service, R> void coprocessorService(Class<T> service, byte[] startKey, byte[] endKey,
      Batch.Call<T, R> callable, Batch.Callback<R> callback) throws ServiceException, Throwable {
    proxy.coprocessorService(service, startKey, endKey, callable, callback);
  }

  @Override
  public long getWriteBufferSize() {
    if(mutator != null) {
      return mutator.getWriteBufferSize();
    } else {
      return 0;
    }
  }

  @Override
  public void setWriteBufferSize(long l) throws IOException {
    getBufferedMutator();
    mutator.setWriteBufferSize(l);
  }

  @Override
  public <R extends Message> Map<byte[], R> batchCoprocessorService(
      Descriptors.MethodDescriptor methodDescriptor, Message message, byte[] startKey, byte[] endKey,
      R responsePrototype) throws ServiceException, Throwable {
    throw new UnsupportedOperationException("batchCoprocessorService not supported");
  }

  @Override
  public <R extends Message> void batchCoprocessorService(
      Descriptors.MethodDescriptor methodDescriptor, Message message, byte[] startKey, byte[] endKey,
      R responsePrototype, Batch.Callback<R> callback) throws ServiceException, Throwable {
    throw new UnsupportedOperationException("batchCoprocessorService not supported");
  }

  public void setOperationTimeout(int i) {
  }

  public int getOperationTimeout() {
    return connection.getOperationTimeout();
  }

  public int getRpcTimeout() {
    return connection.getOperationTimeout();
  }

  public void setRpcTimeout(int i) {
  }

  public int getReadRpcTimeout() {
    return connection.getOperationTimeout();
  }

  public void setReadRpcTimeout(int i) {
  }

  public int getWriteRpcTimeout() {
    return connection.getOperationTimeout();
  }

  public void setWriteRpcTimeout(int i) {
  }

  @Override
  public byte[] getTableName() {
    return myTableName.toBytes();
  }

  @Override
  public long incrementColumnValue(byte[] row, byte[] family, byte[] qualifier, long amount,
      boolean writeToWAL) throws IOException {
    return incrementColumnValue(row, family, qualifier, amount);
  }

  @Override
  public Boolean[] exists(List<Get> gets) throws IOException {
    Result[] results = get(gets);
    Boolean[] existResult = new Boolean[results.length];
    for (int i = 0; i < results.length; i++) {
      existResult[i] = (results[i].getRow() != null);
    }
    return existResult;
  }

  @Override
  public void setAutoFlush(boolean autoFlush) {
    this.autoFlush = autoFlush;
  }

  @Override
  public void setAutoFlush(boolean autoFlush, boolean clearBufferOnFail) {
    this.autoFlush = autoFlush;
    if (mutator != null) {
      mutator.setClearBufferOnFail(clearBufferOnFail);
    }
  }

  @Override
  public void setAutoFlushTo(boolean autoFlush) {
    this.autoFlush = autoFlush;
  }

  @Override
  public boolean isAutoFlush() {
    return autoFlush;
  }

  @Override
  public void flushCommits() throws IOException {
    if (mutator == null) {
      // nothing to flush if there's no mutator; don't bother creating one.
      return;
    }
    getBufferedMutator().flush();
  }

  @Override
  public Result getRowOrBefore(byte[] row, byte[] family) throws IOException {
    Scan scan = new Scan(row);
    if (family != null) {
      scan.addFamily(family);
    }
    scan.setReversed(true);
    scan.setCaching(2);
    ResultScanner scanner = getScanner(scan);
    try {
      return scanner.next();
    } finally {
      scanner.close();
    }
  }

  @Override
  public HRegionLocation getRegionLocation(String row) throws IOException {
    return getRegionLocation(Bytes.toBytes(row));
  }

  @Override
  public HConnection getConnection() {
    return super.getConnection();
  }

  @Override
  public int getScannerCaching() {
    return defaultScannerCaching;
  }

  @Override
  public List<Row> getWriteBuffer() {
    return null;
  }

  @Override
  public void setScannerCaching(int scannerCaching) {

  }

  @Override
  public NavigableMap<HRegionInfo, ServerName> getRegionLocations() throws IOException {
    return new TreeMap<>();
  }

  @Override
  public List<HRegionLocation> getRegionsInRange(byte[] startKey, byte[] endKey)
      throws IOException {
    return getRegionsInRange(startKey, endKey, false);
  }

  @Override
  public List<HRegionLocation> getRegionsInRange(byte[] startKey, byte[] endKey, boolean reload)
      throws IOException {
    return getKeysAndRegionsInRange(startKey, endKey, false, false).getSecond();

  }

  private Pair<List<byte[]>, List<HRegionLocation>> getKeysAndRegionsInRange(
      final byte[] startKey, final byte[] endKey, final boolean includeEndKey,
      final boolean reload) throws IOException {
    final boolean endKeyIsEndOfTable = Bytes.equals(endKey,HConstants.EMPTY_END_ROW);
    if ((Bytes.compareTo(startKey, endKey) > 0) && !endKeyIsEndOfTable) {
      throw new IllegalArgumentException(
          "Invalid range: " + Bytes.toStringBinary(startKey) +
              " > " + Bytes.toStringBinary(endKey));
    }
    List<byte[]> keysInRange = new ArrayList<byte[]>();
    List<HRegionLocation> regionsInRange = new ArrayList<HRegionLocation>();
    byte[] currentKey = startKey;
    do {
      HRegionLocation regionLocation = getRegionLocation(currentKey, reload);
      keysInRange.add(currentKey);
      regionsInRange.add(regionLocation);
      currentKey = regionLocation.getRegionInfo().getEndKey();
    } while (!Bytes.equals(currentKey, HConstants.EMPTY_END_ROW)
        && (endKeyIsEndOfTable || Bytes.compareTo(currentKey, endKey) < 0
        || (includeEndKey && Bytes.compareTo(currentKey, endKey) == 0)));
    return new Pair<List<byte[]>, List<HRegionLocation>>(keysInRange,
        regionsInRange);
  }



  @Override
  public <R> void processBatchCallback(List<? extends Row> list, Object[] results,
      Batch.Callback<R> callback) throws IOException, InterruptedException {
    throw new UnsupportedOperationException("batchCallback not supported");
  }


  public void processBatch(List<? extends Row> list, Object[] results)
      throws IOException, InterruptedException {
    batch(list, results);
  }

  @Override
  public void validatePut(Put put) throws IllegalArgumentException {
    super.validatePut(put);
  }

  @Override
  ExecutorService getPool() {
    return super.getPool();
  }

  @Override
  public void clearRegionCache() {
  }

  @Override
  public String toString() {
    return super.toString();
  }

  @Override
  public RegionLocator getRegionLocator() {
    return this;
  }

  @Override
  BufferedMutator getBufferedMutator() throws IOException {
    if (mutator == null) {
      mutator = (AliHBaseUEBufferedMutator)connection.getBufferedMutator(myTableName);
    }
    return mutator;
  }

  @Override
  public void addExternalIndex(String targetIndexName, ExternalIndexType type,
      ExternalIndexRowFormatterType rowFormatterType, List<ExternalIndexField> fields)
      throws IOException {
    ExternalIndexConfig config = new ExternalIndexConfig(targetIndexName, type, rowFormatterType);
    addExternalIndex(config, fields);
  }

  @Override
  public void addExternalIndex(String targetIndexName, ExternalIndexType type,
      ExternalIndexRowFormatterType rowFormatterType, ExternalIndexField... fields)
      throws IOException {
    List<ExternalIndexField> list = new ArrayList<ExternalIndexField>();
    for (ExternalIndexField field : fields) {
      list.add(field);
    }
    addExternalIndex(targetIndexName, type, rowFormatterType, list);

  }

  @Override
  public void addExternalIndex(ExternalIndexConfig config, List<ExternalIndexField> fields)
      throws IOException {
    proxy.addExternalIndex(config, fields);

  }

  @Override
  public void addExternalIndex(ExternalIndexConfig config, ExternalIndexField... fields)
      throws IOException {
    List<ExternalIndexField> list = new ArrayList<ExternalIndexField>();
    for (ExternalIndexField field : fields) {
      list.add(field);
    }
    addExternalIndex(config, fields);
  }

  @Override
  public void addExternalIndex(List<ExternalIndexField> fields) throws IOException {
    addExternalIndex(null, fields);
  }

  @Override
  public void addExternalIndex(ExternalIndexField... fields) throws IOException {
    addExternalIndex(null, fields);
  }

  @Override
  public void removeExternalIndex(List<String> fields) throws IOException {
    proxy.removeExternalIndex(fields);
  }

  @Override
  public void removeExternalIndex(String... fields) throws IOException {
    List<String> list = new ArrayList<String>();
    for (String field : fields) {
      list.add(field);
    }
    removeExternalIndex(list);
  }

  @Override
  public void buildExternalIndex() throws IOException {
    proxy.buildExternalIndex();
  }

  @Override
  public void cancelBuildExternalIndex() throws IOException {
    proxy.cancelBuildExternalIndex();
  }
}
