/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.client.index;

import com.alibaba.lindorm.client.core.utils.SchemaUtils;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.util.Bytes;

import java.util.ArrayList;
import java.util.List;

/**
 * Similar to {@link org.apache.hadoop.hbase.HTableDescriptor}, describes an index table.
 * Clients should use this descriptor to create an index.
 * TODO: index descriptor may not extends all attributes from table descriptor
 * @see org.apache.hadoop.hbase.client.AliHBaseAdminInterface#createIndex(AliHBaseIndexDescriptor)
 */
public class AliHBaseIndexDescriptor {
  // custom index name
  private String indexName;
  // table name that need to add indexes
  private TableName dataTable;

  // indexed columns of main table
  private List<AliHBaseColumn> indexedColumns = new ArrayList<>();
  // covered columns of main table
  private List<AliHBaseColumn> coveredColumns = new ArrayList<>();

  // descriptor of index table, index table has only one column family
  private HColumnDescriptor familyDescriptor;

  public AliHBaseIndexDescriptor(String indexName, TableName dataTable) {
    this.indexName = indexName;
    this.dataTable = dataTable;
    // note that family name is not used here.
    this.familyDescriptor = new HColumnDescriptor(SchemaUtils.DEFAULT_FAMILY_NAME_BYTES);
  }

  public AliHBaseIndexDescriptor(String indexName, TableName dataTable, HColumnDescriptor familyDesc) {
    this.indexName = indexName;
    this.dataTable = dataTable;
    if (familyDesc == null) {
      throw new IllegalArgumentException("Index column family must not be null");
    }
    this.familyDescriptor = familyDesc;
  }

  public String getIndexName() {
    return indexName;
  }

  public TableName getDataTable() {
    return dataTable;
  }

  public List<AliHBaseColumn> getIndexedColumns() {
    return indexedColumns;
  }

  public List<AliHBaseColumn> getCoveredColumns() {
    return coveredColumns;
  }

  public HColumnDescriptor getFamilyDescriptor() {
    return familyDescriptor;
  }

  public void setFamilyDescriptor(HColumnDescriptor familyDescriptor) {
    this.familyDescriptor = familyDescriptor;
  }

  /**
   * Add an indexed column to this index, with default sort order ASC.
   * @param family family name bytes
   * @param qualifier qualifier name bytes
   */
  public void addIndexedColumn(byte[] family, byte[] qualifier) {
    addIndexedColumn(family, qualifier, AliHBaseColumn.SortOrder.getDefault());
  }

  public void addIndexedColumn(String family, String qualifier) {
    addIndexedColumn(Bytes.toBytes(family), Bytes.toBytes(qualifier));
  }

  /**
   * Add an indexed column to this index, with an explicit sort order.
   * @param family family name bytes
   * @param qualifier qualifier name bytes
   * @param sortOrder sort order
   */
  public void addIndexedColumn(byte[] family, byte[] qualifier, AliHBaseColumn.SortOrder sortOrder) {
    if (indexedColumns == null) {
      indexedColumns = new ArrayList<>();
    }
    indexedColumns.add(AliHBaseColumn.createIndexedColumn(family, qualifier, sortOrder));
  }

  public void addIndexedColumn(String family, String qualifier, AliHBaseColumn.SortOrder sortOrder) {
    addIndexedColumn(Bytes.toBytes(family), Bytes.toBytes(qualifier), sortOrder);
  }

  /**
   * Add a covered column
   * @param family family name bytes
   * @param qualifier qualifier name bytes
   */
  public void addCoveredColumn(byte[] family, byte[] qualifier) {
    if (coveredColumns == null) {
      coveredColumns = new ArrayList<>();
    }
    coveredColumns.add(AliHBaseColumn.createCoveredColumn(family, qualifier));
  }

  public void addCoveredColumn(String family, String qualifier) {
    addCoveredColumn(Bytes.toBytes(family), Bytes.toBytes(qualifier));
  }

  /**
   * Mark this index will cover all columns in data table.
   */
  public void setCoveredAllColumns() {
    if (coveredColumns == null) {
      coveredColumns = new ArrayList<>();
    }

    coveredColumns.clear();
    coveredColumns.add(AliHBaseColumn.COVER_ALL);
  }

  public String getSchema() {
    StringBuilder str = new StringBuilder();
    str.append("Index [");
    str.append(indexName);
    str.append("] on [");
    str.append(dataTable);
    str.append("] (");

    if (indexedColumns != null && indexedColumns.size() > 0) {
      for (AliHBaseColumn ic : indexedColumns) {
        str.append(ic.toString());
        str.append(",");
      }
      str.setLength(str.length() - 1);
    }
    str.append(")");

    if (coveredColumns != null && coveredColumns.size() > 0) {
      str.append(" includes (");
      for (AliHBaseColumn key : coveredColumns) {
        str.append(key);
        str.append(",");
      }
      str.setLength(str.length() - 1);
      str.append(")");
    }

    return str.toString();
  }

  @Override
  public String toString() {
    StringBuilder str = new StringBuilder();
    str.append(getSchema());
    str.append(", INDEX_FAMILY_ATTR: ");
    str.append(getFamilyDescriptor().toString());
    return str.toString();
  }
}
