/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.hbase.client;

import com.alibaba.lindorm.client.WideColumnService;
import com.alibaba.lindorm.client.core.widecolumnservice.WDelete;
import com.alibaba.lindorm.client.core.widecolumnservice.WPut;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.BufferedMutator;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Mutation;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.RetriesExhaustedWithDetailsException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;

public class AliHBaseUEBufferedMutator implements BufferedMutator {

  private static final Logger LOG = LoggerFactory.getLogger(AliHBaseUEBufferedMutator.class);

  private final TableName tableName;

  private AliHBaseUEConnection connection;

  private volatile long writeBufferSize;

  private long currentWriteBufferSize;

  private volatile boolean clearBufferOnFail;

  private final ConcurrentLinkedQueue<Mutation> writeBuffer;

  private WideColumnService wideColumnService;

  public AliHBaseUEBufferedMutator(AliHBaseUEConnection connection, TableName tableName) throws IOException {
    this.tableName = tableName;
    this.connection = connection;
    this.writeBuffer = new ConcurrentLinkedQueue<Mutation>();
    this.writeBufferSize = this.connection.getConfiguration().getLong("hbase.client.write.buffer", 2097152);
    this.currentWriteBufferSize = 0;
    this.clearBufferOnFail = this.connection.getConfiguration()
        .getBoolean("hbase.client.bufferedmutator.clearbufferonfail", true);
    this.wideColumnService = connection.getWideColumnService(tableName.getNamespaceAsString());
  }

  @Override
  public TableName getName() {
    return tableName;
  }

  @Override
  public Configuration getConfiguration() {
    return connection.getConfiguration();
  }

  @Override
  public void mutate(Mutation mutation) throws InterruptedIOException, RetriesExhaustedWithDetailsException {
    mutate(Collections.singletonList(mutation));
  }

  @Override
  public void mutate(List<? extends Mutation> list)
      throws InterruptedIOException, RetriesExhaustedWithDetailsException {
    List<WPut> flushPuts = new ArrayList<WPut>();
    List<WDelete> flushDeletes = new ArrayList<WDelete>();

    for (Mutation mutation : list) {
      writeBuffer.add(mutation);
      currentWriteBufferSize += mutation.heapSize();
    }
    try {
      if (currentWriteBufferSize >= writeBufferSize) {
        extractOMutation(flushPuts, flushDeletes);
      }
      flush(flushPuts, flushDeletes);
    } catch (IOException e) {
      // TODO: 06/09/2018
      LOG.error("Mutate exception ", e);
      throw new InterruptedIOException(e.getMessage());
    }
  }

  private void extractOMutation(List<WPut> flushPuts, List<WDelete> flushDeletes) throws IOException {
    for (Mutation mutation : writeBuffer) {
      if (mutation instanceof Put) {
        flushPuts.add(ElementConvertor.toLindormPut((Put)mutation));
      } else if (mutation instanceof Delete) {
        flushDeletes.add(ElementConvertor.toLindormDelete((Delete)mutation));
      } else {
        throw new UnsupportedOperationException("Unsupported mutation type " + mutation.getClass());
      }
    }
    writeBuffer.clear();
    currentWriteBufferSize = 0;
  }

  private void flush(List<WPut> flushPuts, List<WDelete> flushDeletes) throws IOException {
    if (!flushPuts.isEmpty()) {
      commitPuts(flushPuts);
    }
    if (!flushDeletes.isEmpty()) {
      commitDeletes(flushDeletes);
    }
  }

  @Override
  public void close() throws IOException {
    flush();
  }

  @Override
  public void flush() throws InterruptedIOException, RetriesExhaustedWithDetailsException{
    List<WPut> flushPuts = new ArrayList<WPut>();
    List<WDelete> flushDeletes = new ArrayList<WDelete>();
    try {
      synchronized (writeBuffer) {
        extractOMutation(flushPuts, flushDeletes);
      }
      flush(flushPuts, flushDeletes);
    } catch (IOException e) {
      // TODO: 06/09/2018
      LOG.error("Flush exception ", e);
      throw new InterruptedIOException(e.getMessage());
    }
  }

  @Override
  public long getWriteBufferSize() {
    return writeBufferSize;
  }

  public void setClearBufferOnFail(boolean clearBufferOnFail) {
    this.clearBufferOnFail = clearBufferOnFail;
  }

  private void commitPuts(final List<WPut> puts) throws IOException {
    boolean flushSuccessfully = false;
    try {
      this.wideColumnService.batchPut(tableName.getQualifierAsString(), puts);
      flushSuccessfully = true;
    } finally {
      if (!flushSuccessfully && !clearBufferOnFail) {
        List<Put> hputs = ElementConvertor.toHBasePuts(puts);
        synchronized (writeBuffer) {
          for (Put put : hputs) {
            writeBuffer.add(put);
            currentWriteBufferSize += put.heapSize();
          }
        }
      }
    }
  }

  private void commitDeletes(final List<WDelete> deletes) throws IOException {
    boolean flushSuccessfully = false;
    try {
      this.wideColumnService.batchDelete(tableName.getQualifierAsString(), deletes);
      flushSuccessfully = true;
    } finally {
      if (!flushSuccessfully && !clearBufferOnFail) {
        List<Delete> hDeletes = ElementConvertor.toHBaseDeletes(deletes);
        synchronized (writeBuffer) {
          for (Delete delete : hDeletes) {
            writeBuffer.add(delete);
            currentWriteBufferSize += delete.heapSize();
          }
        }
      }
    }
  }

}