/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.hbase.client;

import java.io.IOException;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import com.alibaba.lindorm.client.AdminService;
import com.alibaba.lindorm.client.LindormClientConfig;
import com.alibaba.lindorm.client.LindormClientConstants;
import com.alibaba.lindorm.client.LindormServiceProvider;
import com.alibaba.lindorm.client.SystemService;
import com.alibaba.lindorm.client.TableService;
import com.alibaba.lindorm.client.WideColumnService;
import com.alibaba.lindorm.client.core.utils.ClientEnvLogUtil;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.TableName;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AliHBaseDirectImplFactory extends AliHBaseImplFactory {

  private static final Logger LOG = LoggerFactory.getLogger(AliHBaseDirectImplFactory.class);


  private Configuration conf;

  private String seedServer;
  private String userName;
  private String password;

  private int operationTimeout;

  private boolean accessFromInternet;
  private ConcurrentHashMap<String, WideColumnService> wideColumnMap = new ConcurrentHashMap<>();
  private ConcurrentHashMap<String, TableService> tableServiceMap = new ConcurrentHashMap<>();
  private AdminService adminService;
  private SystemService systemService;

  private LindormClientConfig lindormClientConfig;

  static {
    ClientEnvLogUtil.setEnabled(false);
  }

  public AliHBaseDirectImplFactory(Configuration conf) {
    this.conf = conf;
    String host = conf.get(AliHBaseConstants.ALIHBASE_SERVER_NAME);
    if (host == null || host.trim().isEmpty()) {
      this.seedServer = conf.get(LindormClientConstants.SEED_SERVERS);
    } else {
      if (host.contains(":")) {
        // the port is set with the host name
        this.seedServer = host;
      } else {
        int port = AliHBaseConstants.DEFAULT_ALIHBASEUE_PORT;
        this.seedServer = host + ":" + port;
      }
    }

    this.operationTimeout = conf.getInt(HConstants.HBASE_CLIENT_OPERATION_TIMEOUT,
        HConstants.DEFAULT_HBASE_CLIENT_OPERATION_TIMEOUT);

    this.userName = conf.get(AliHBaseConstants.USERNAME);
    this.password = conf.get(AliHBaseConstants.PASSWORD);

    String publicIdentifier = conf.get(AliHBaseConstants.PUBLIC_HOSTNAME_IDENTIFIER,
        AliHBaseConstants.DEFAULT_PUBLIC_HOSTNAME_IDENTIFIER);

    if (seedServer.contains(publicIdentifier)) {
      this.accessFromInternet = true;
    }
    if (conf.get(AliHBaseConstants.ALIHBASE_ACCESS_FROM_INTERNET) != null) {
      this.accessFromInternet = conf.getBoolean(AliHBaseConstants.ALIHBASE_ACCESS_FROM_INTERNET,
          AliHBaseConstants.ALIHBASE_ACCESS_FROM_INTERNET_DEFAULT);
    }
  }

  @Override
  public Configuration getConf() {
    return conf;
  }

  AdminService getAdminService() throws IOException {
    if (adminService == null) {
      synchronized (this) {
        if (adminService == null) {
          adminService = LindormServiceProvider.AdminServiceProvider.create(getLindormConfig());
        }
      }
    }
    return adminService;
  }

  public SystemService getSystemService() throws IOException {
    if (systemService == null) {
      synchronized (this) {
        if (systemService == null) {
          systemService = LindormServiceProvider.SystemServiceProvider.create(getLindormConfig());
        }
      }
    }
    return systemService;
  }

  WideColumnService getWideColumnService(String namespace) throws IOException {
    if (namespace == null) {
      throw new IOException("namespace can't be null!");
    }
    WideColumnService wideColumnService = wideColumnMap.get(namespace);
    if (wideColumnService == null) {
      synchronized (this) {
        LindormClientConfig config = getLindormConfig();
        config.setNamespace(namespace);
        WideColumnService service = LindormServiceProvider.WideColumnServiceProvider.create(config);
        WideColumnService old = wideColumnMap.putIfAbsent(namespace, service);
        if (old != null) {
          service.close();
          return old;
        } else {
          return service;
        }
      }

    } else {
      return wideColumnService;
    }
  }



  LindormClientConfig createLindormConfig() throws IOException {
    LindormClientConfig lindormClientConfig = LindormClientConfig.create();
    Iterator<Map.Entry<String, String>> it = conf.iterator();
    while (it.hasNext()) {
      Map.Entry<String, String> entry = it.next();
      if (entry.getKey() != null && entry.getValue() != null) {
        lindormClientConfig.set(entry.getKey(), entry.getValue());
      }
    }
    if (userName != null && !userName.isEmpty() && password != null && !password.isEmpty()) {
      lindormClientConfig.setUser(userName, password);
    }
    lindormClientConfig.setSeedServer(seedServer.trim());
    lindormClientConfig.setTimeout(operationTimeout);
    if (accessFromInternet) {
      lindormClientConfig.setBoolean(LindormClientConstants.LINDOM_RPC_ONLY_USE_SEEDSERVER, true);
    }
    if (conf.get(LindormClientConstants.CLIENT_SORTER_TYPE) != null) {
      lindormClientConfig.set(LindormClientConstants.CLIENT_SORTER_TYPE,
          conf.get(LindormClientConstants.CLIENT_SORTER_TYPE));
    } else {
      lindormClientConfig
          .set(LindormClientConstants.CLIENT_SORTER_TYPE, LindormClientConstants.CLIENT_NAME_SORTER);
    }
    return lindormClientConfig;
  }

  LindormClientConfig getLindormConfig() throws IOException {
    if (lindormClientConfig == null) {
      synchronized (this) {
        if (lindormClientConfig == null) {
          lindormClientConfig = createLindormConfig();
        }
      }
    }
    return new LindormClientConfig(lindormClientConfig);
  }

  TableService getTableServiceService(String namespace) throws IOException {
    if (namespace == null) {
      throw new IOException("namespace can't be null!");
    }
    TableService tableService = tableServiceMap.get(namespace);
    if (tableService == null) {
      synchronized (this) {
        LindormClientConfig config = getLindormConfig();
        config.setNamespace(namespace);
        TableService service = LindormServiceProvider.TableServiceProvider.create(config);
        TableService old = tableServiceMap.putIfAbsent(namespace, service);
        if (old != null) {
          service.close();
          return old;
        } else {
          return service;
        }
      }
    } else {
      return tableService;
    }
  }

  @Override
  public AliHBaseAPIProxy getHBaseAPIProxyImpl(TableName tableName) throws IOException {
    return new AliHBaseAPIProxyDirectImpl(this, tableName);
  }

  @Override
  public void close() throws IOException {
    try {
      if (adminService != null) {
        adminService.close();
      }
    } catch (IOException ioE) {
      LOG.error("Error happened when closing adminservice", ioE);
    }
    try {
      if (systemService != null) {
        systemService.close();
      }
    } catch (Throwable e) {
      LOG.error("Error happened when closing systemService", e);
    }
    for (Map.Entry<String, WideColumnService> entry : wideColumnMap.entrySet()) {
      if (entry.getValue() != null) {
        try {
          entry.getValue().close();
        } catch (IOException ioE) {
          LOG.error("Error happened when closing WideColumnService for namespace " + entry.getKey(), ioE);
        }
      }
    }

    for (Map.Entry<String, TableService> entry : tableServiceMap.entrySet()) {
      if (entry.getValue() != null) {
        try {
          entry.getValue().close();
        } catch (Exception e) {
          LOG.error("Error happened when closing TableService for namespace " + entry.getKey(), e);
        }
      }
    }
  }
}
