package com.alibaba.hbase.haclient.dualservice;

import static com.alibaba.hbase.client.AliHBaseConstants.DEFAULT_DUALSERVICE_ACTIVE_EXECUTOR_QUEUE;
import static com.alibaba.hbase.client.AliHBaseConstants.DEFAULT_DUALSERVICE_ACTIVE_EXECUTOR_THREAD;
import static com.alibaba.hbase.client.AliHBaseConstants.DEFAULT_DUALSERVICE_STANDBY_EXECUTOR_QUEUE;
import static com.alibaba.hbase.client.AliHBaseConstants.DEFAULT_DUALSERVICE_STANDBY_EXECUTOR_THREAD;
import static com.alibaba.hbase.client.AliHBaseConstants.DUALSERVICE_ACTIVE_EXECUTOR_QUEUE;
import static com.alibaba.hbase.client.AliHBaseConstants.DUALSERVICE_ACTIVE_EXECUTOR_THREAD;
import static com.alibaba.hbase.client.AliHBaseConstants.DUALSERVICE_STANDBY_EXECUTOR_QUEUE;
import static com.alibaba.hbase.client.AliHBaseConstants.DUALSERVICE_STANDBY_EXECUTOR_THREAD;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;


import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.util.Threads;

public class DualExecutor {

  private Configuration conf;

  private ThreadPoolExecutor dualActiveThreadPool = null;

  private static ThreadPoolExecutor dualStandbyThreadPool = null;

  public DualExecutor(Configuration conf){
    this.conf = conf;
    int activeDualThreads = conf.getInt(DUALSERVICE_ACTIVE_EXECUTOR_THREAD,
        DEFAULT_DUALSERVICE_ACTIVE_EXECUTOR_THREAD);
    int activeQueueSize = conf.getInt(DUALSERVICE_ACTIVE_EXECUTOR_QUEUE,
        DEFAULT_DUALSERVICE_ACTIVE_EXECUTOR_QUEUE);
    int standbyDualThreads = conf.getInt(DUALSERVICE_STANDBY_EXECUTOR_THREAD,
        DEFAULT_DUALSERVICE_STANDBY_EXECUTOR_THREAD);
    int standbyQueueSize = conf.getInt(DUALSERVICE_STANDBY_EXECUTOR_QUEUE,
        DEFAULT_DUALSERVICE_STANDBY_EXECUTOR_QUEUE);
    //init active thread pool
    dualActiveThreadPool = new ThreadPoolExecutor(activeDualThreads, activeDualThreads, 60,
      TimeUnit.SECONDS, new ArrayBlockingQueue<Runnable>(activeQueueSize), Threads.newDaemonThreadFactory("DualService-Active-Executor"),
      new ThreadPoolExecutor.AbortPolicy());
    if(dualStandbyThreadPool == null) {
      synchronized (DualExecutor.class) {
        if (dualStandbyThreadPool == null) {
          //init standby thread pool
          dualStandbyThreadPool = new ThreadPoolExecutor(standbyDualThreads, standbyDualThreads, 60,
            TimeUnit.SECONDS, new ArrayBlockingQueue<Runnable>(standbyQueueSize),
            Threads.newDaemonThreadFactory("DualService-Standby-Executor"),
            new ThreadPoolExecutor.AbortPolicy());
        }
      }
    }
  }

  public static String createTableConfKey(String tableName, String confKey){
    return tableName + "." + confKey;
  }

  public Result dualGet(HTable activeTable, HTable standbyTable, final byte[] tableName,
                      final Get get, final int glitchTimeout, final int operationTimeout) throws IOException{
    DualContext<Result> dualContext = new DualContext<Result>(tableName, glitchTimeout, operationTimeout);
    Callable<Result> activeCallable = new DualCallable<Result, Get>(dualContext, activeTable, get
      , ActionType.GET, Role.ACTIVE);
    Callable<Result> standbyCallable = new DualCallable<Result, Get>(dualContext, standbyTable,
      get, ActionType.GET, Role.STANDBY);
    return doDualOperation(activeCallable, standbyCallable, dualContext);
  }

  public Void dualPut(HTable activeTable, HTable standbyTable, final byte[] tableName,
                    final Put put, final int glitchTimeout, final int operationTimeout) throws IOException{
    DualContext<Void> dualContext = new DualContext<Void>(tableName, glitchTimeout,
      operationTimeout);
    Callable<Void> activeCallable = new DualCallable<Void, Put>(dualContext, activeTable, put
      , ActionType.PUT, Role.ACTIVE);
    Callable<Void> standbyCallable = new DualCallable<Void, Put>(dualContext, standbyTable,
      put, ActionType.PUT, Role.STANDBY);
    return doDualOperation(activeCallable, standbyCallable, dualContext);
  }

  public Void dualDelete(HTable activeTable, HTable standbyTable, final byte[] tableName,
                       final Delete delete, final int glitchTimeout, final int operationTimeout) throws IOException{
    DualContext<Void> dualContext = new DualContext<Void>(tableName, glitchTimeout,
      operationTimeout);
    Callable<Void> activeCallable = new DualCallable<Void, Delete>(dualContext, activeTable, delete
      , ActionType.DELETE, Role.ACTIVE);
    Callable<Void> standbyCallable = new DualCallable<Void, Delete>(dualContext, standbyTable,
      delete, ActionType.DELETE, Role.STANDBY);
    return doDualOperation(activeCallable, standbyCallable, dualContext);
  }

  public Result[] dualBatchGet(HTable activeTable, HTable standbyTable, final byte[] tableName,
                           final List<Get> gets, final int glitchTimeout, final int operationTimeout) throws IOException{
    DualContext<Result[]> dualContext = new DualContext<Result[]>(tableName, glitchTimeout,
      operationTimeout);
    Callable<Result[]> activeCallable = new DualCallable<Result[], List<Get> >(dualContext,
      activeTable, gets, ActionType.BATCHGET, Role.ACTIVE);
    Callable<Result[]> standbyCallable = new DualCallable<Result[], List<Get> >(dualContext,
      standbyTable, gets, ActionType.BATCHGET, Role.STANDBY);
    return doDualOperation(activeCallable, standbyCallable, dualContext);
  }

  public Void dualBatchPut(HTable activeTable, HTable standbyTable, final byte[] tableName,
                             final List<Put> puts, final int glitchTimeout,
                             final int operationTimeout) throws IOException{
    DualContext<Void> dualContext = new DualContext<Void>(tableName, glitchTimeout,
      operationTimeout);
    Callable<Void> activeCallable = new DualCallable<Void, List<Put> >(dualContext,
      activeTable, puts, ActionType.BATCHPUT, Role.ACTIVE);
    Callable<Void> standbyCallable = new DualCallable<Void, List<Put> >(dualContext,
      standbyTable, puts, ActionType.BATCHPUT, Role.STANDBY);
    return doDualOperation(activeCallable, standbyCallable, dualContext);
  }

  public Void dualBatchDelete(HTable activeTable, HTable standbyTable, final byte[] tableName,
                                 final List<Delete> deletes, final int glitchTimeout,
                                 final int operationTimeout) throws IOException{
    DualContext<Void> dualContext = new DualContext<Void>(tableName, glitchTimeout,
      operationTimeout);
    Callable<Void> activeCallable = new DualCallable<Void, List<Delete> >(dualContext,
      activeTable, deletes, ActionType.BATCHDELETE, Role.ACTIVE);
    Callable<Void> standbyCallable = new DualCallable<Void, List<Delete> >(dualContext,
      standbyTable, deletes, ActionType.BATCHDELETE, Role.STANDBY);
    return doDualOperation(activeCallable, standbyCallable, dualContext);
  }

  public DualScannerResult dualNext(ResultScanner activeScanner, ResultScanner standbyScanner,
                                    final byte[] tableName, final int glitchTimeout,
                                    final int operationTimeout) throws IOException{
    DualContext<DualScannerResult> dualContext = new DualContext<DualScannerResult>(tableName, glitchTimeout,
      operationTimeout);
    Callable<DualScannerResult> activeCallable = new DualScannerCallable<DualScannerResult>(dualContext,
      activeScanner, Role.ACTIVE);
    Callable<DualScannerResult> standbyCallable = new DualScannerCallable<DualScannerResult>(dualContext,
      standbyScanner, Role.STANDBY);
    return doDualOperation(activeCallable, standbyCallable, dualContext);
  }

  private<T> T doDualOperation(Callable<T> activeCallable, Callable<T> standbyCallable,
                               DualContext<T> dualContext)throws IOException{
    Future<T> activeFuture = null;
    Future<T> standbyFuture = null;
    try{
      if(dualActiveThreadPool != null && dualActiveThreadPool.getQueue().size() < 10){
        dualContext.start();
        activeFuture = dualActiveThreadPool.submit(activeCallable);
        T result = dualContext.getResultInGlitchTimeout();
        // result is comed back from active
        if (dualContext.usePrimaryAsResult() != null) {
          return result;
        }
        standbyFuture = dualStandbyThreadPool.submit(standbyCallable);
        dualContext.waitOperationTimeout();
        return dualContext.getResult();
      }else{
        //dual active thread pool is full,disable dual service use active call
        try {
          return activeCallable.call();
        }catch(Exception e){
          throw new IOException(e);
        }
      }
    }finally {
      if(activeFuture != null && !activeFuture.isDone()){
        activeFuture.cancel(true);
      }
      if(standbyFuture != null && !standbyFuture.isDone()){
        standbyFuture.cancel(true);
      }
    }
  }

  public void close(){
    if(this.dualActiveThreadPool != null){
      this.dualActiveThreadPool.shutdown();
    }
  }

  public enum ActionType{
    GET,
    PUT,
    DELETE,
    BATCHGET,
    BATCHPUT,
    BATCHDELETE;
  }

  public enum Role{
    ACTIVE,
    STANDBY;
  }
}

