package org.apache.hadoop.hbase.client;

import java.io.Closeable;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.haclient.ClusterSwitchUtil;
import com.alibaba.hbase.haclient.dualservice.DualExecutor;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.MasterNotRunningException;
import org.apache.hadoop.hbase.RegionLocations;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.ZooKeeperConnectionException;
import org.apache.hadoop.hbase.client.backoff.ClientBackoffPolicy;
import org.apache.hadoop.hbase.client.coprocessor.Batch;
import org.apache.hadoop.hbase.ipc.RpcControllerFactory;
import org.apache.hadoop.hbase.protobuf.generated.AdminProtos;
import org.apache.hadoop.hbase.protobuf.generated.ClientProtos;
import org.apache.hadoop.hbase.protobuf.generated.MasterProtos;
import org.apache.hadoop.hbase.security.User;
import org.apache.hadoop.hbase.util.Threads;

public class AliHBaseMultiClusterConnection extends
  ConnectionManager.HConnectionImplementation implements Closeable {
  private static final Log LOG = LogFactory
    .getLog(AliHBaseMultiClusterConnection.class);

  private volatile ClusterConnection activeConnectionImpl;
  //standby connection use for dual service
  private volatile ClusterConnection standbyConnectionImpl;

  protected volatile long switchCount = 0;
  private volatile MetaCache metaCache;
  private volatile RpcRetryingCallerFactory rpcCallerFactory;

  private volatile RpcControllerFactory rpcControllerFactory;

  private volatile ConnectionConfiguration connectionConfiguration;

  private volatile TableConfiguration tableConfiguration;

  //HBASE-15295 change TableConfiguration to ConnectionConfiguration, so when create HTable
  // we need to try different constructor with different parameter
  private Constructor<?> constructorWithTableConfiguration = null;
  private Constructor<?> constructorWithConnectionConfiguration = null;
  private volatile boolean hasCellBlockSupport = false;

  protected volatile Configuration conf;
  protected volatile Configuration standbyConf;
  protected User user;
  protected volatile boolean closed = false;
  protected volatile boolean aborted = false;
  protected volatile boolean cleanupPool = false;
  protected volatile boolean cleanupStandbyPool = false;
  // thread executor shared by all HTableInterface instances created
  // by this connection
  private ExecutorService batchPool = null;
  private ExecutorService standbyBatchPool = null;
  private AsyncProcess asyncProcess = null;

  private AliHBaseConstants.ClusterType activeClusterType = null;
  private AliHBaseConstants.ClusterType standbyClusterType = null;

  protected boolean dualServiceEnable;
  protected DualExecutor dualExecutor;

  protected AliHBaseMultiClusterConnection(Configuration conf,
                                           ExecutorService pool, User user) throws IOException {
    this(conf, false, pool, user);
  }

  protected AliHBaseMultiClusterConnection(Configuration conf, boolean managed) throws IOException {
    this(conf, managed, null, null);
  }

  protected AliHBaseMultiClusterConnection(Configuration conf, boolean managed,
                                   ExecutorService pool, User user) throws IOException {
    super(conf, false);
    if (managed == true) {
      throw new IOException("Managed connection is not supported for AliHBaseMultiClusterConnection");
    }
    this.conf = conf;
    this.user = user;
    this.batchPool = pool;
    //get dual service enable or not
    this.dualServiceEnable = conf.getBoolean(AliHBaseConstants.DUALSERVICE_ENABLE,
      AliHBaseConstants.DUALSERVICE_ENABLE_DEFAULT);
    if(dualServiceEnable){
      dualExecutor = new DualExecutor(this.conf);
    }
  }

  // For tests to override.
  protected AsyncProcess createAsyncProcess() {
    return null;
  }

  protected ExecutorService getThreadPool(int maxThreads, int coreThreads, String nameHint,
                                          BlockingQueue<Runnable> passedWorkQueue) {
    // shared HTable thread executor not yet initialized
    if (maxThreads == 0) {
      maxThreads = Runtime.getRuntime().availableProcessors() * 8;
    }
    if (coreThreads == 0) {
      coreThreads = Runtime.getRuntime().availableProcessors() * 8;
    }
    long keepAliveTime = conf.getLong("hbase.hconnection.threads.keepalivetime", 60);
    BlockingQueue<Runnable> workQueue = passedWorkQueue;
    if (workQueue == null) {
      workQueue =
        new LinkedBlockingQueue<Runnable>(maxThreads *
          conf.getInt(HConstants.HBASE_CLIENT_MAX_TOTAL_TASKS,
            HConstants.DEFAULT_HBASE_CLIENT_MAX_TOTAL_TASKS));
    }
    ThreadPoolExecutor tpe = new ThreadPoolExecutor(
      coreThreads,
      maxThreads,
      keepAliveTime,
      TimeUnit.SECONDS,
      workQueue,
      Threads.newDaemonThreadFactory(toString() + nameHint));
    tpe.allowCoreThreadTimeOut(true);
    return tpe;
  }

  private ExecutorService getBatchPool() {
    if (batchPool == null) {
      synchronized (this) {
        if (batchPool == null) {
          this.batchPool = getThreadPool(conf.getInt("hbase.hconnection.threads.max", 256),
            conf.getInt("hbase.hconnection.threads.core", 256), "-shared-", null);
          cleanupPool = true;
        }
      }
    }
    return this.batchPool;
  }

  private ExecutorService getStandbyBatchPool() {
    if (standbyBatchPool == null) {
      synchronized (this) {
        if (standbyBatchPool == null) {
          this.standbyBatchPool = getThreadPool(standbyConf.getInt("hbase.hconnection.threads.max", 256),
            standbyConf.getInt("hbase.hconnection.threads.core", 256), "-shared-", null);
          cleanupStandbyPool = true;
        }
      }
    }
    return this.standbyBatchPool;
  }

  /**
   * For the reason of compatibility, we need to get some private field from the connection
   */
  private void getPrivatesFromCurrentConnection() {
    if (activeConnectionImpl != null) {
      //Get the private filed metaCache from the current connection
      try {
        Field field = activeConnectionImpl.getClass().getDeclaredField("metaCache");
        field.setAccessible(true);
        this.metaCache = (MetaCache) field.get(activeConnectionImpl);
      } catch (Throwable ex) {
        LOG.error("Error when get metaCache field", ex);
      }
      try {
        Field field = activeConnectionImpl.getClass().getDeclaredField("rpcCallerFactory");
        field.setAccessible(true);
        this.rpcCallerFactory = (RpcRetryingCallerFactory) field.get(activeConnectionImpl);
      } catch (Throwable ex) {
        LOG.error("Error when get rpcCallerFactory field", ex);
      }
      try {
        Field field = activeConnectionImpl.getClass().getDeclaredField("rpcControllerFactory");
        field.setAccessible(true);
        this.rpcControllerFactory = (RpcControllerFactory) field.get(activeConnectionImpl);
      } catch (Throwable ex) {
        LOG.error("Error when get rpcControllerFactory field", ex);
      }
      this.connectionConfiguration = new ConnectionConfiguration(
        activeConnectionImpl.getConfiguration());
      try {
        Method method = activeConnectionImpl.getClass().getMethod("hasCellBlockSupport");
        hasCellBlockSupport = (Boolean) method.invoke(activeConnectionImpl);
      } catch (Throwable ex) {
        hasCellBlockSupport = false;
      }
      tableConfiguration = new TableConfiguration(activeConnectionImpl.getConfiguration());
      try {
        constructorWithTableConfiguration =
          HTable.class.getDeclaredConstructor(TableName.class, ClusterConnection.class,
            TableConfiguration.class,
            RpcRetryingCallerFactory.class,
            RpcControllerFactory.class,
            ExecutorService.class);
        constructorWithTableConfiguration.setAccessible(true);
      } catch (NoSuchMethodException ex) {

      }
      if (constructorWithTableConfiguration == null) {
        try {
          constructorWithConnectionConfiguration =
            HTable.class.getDeclaredConstructor(TableName.class, ClusterConnection.class,
              ConnectionConfiguration.class,
              RpcRetryingCallerFactory.class,
              RpcControllerFactory.class,
              ExecutorService.class);
          constructorWithConnectionConfiguration.setAccessible(true);
        } catch (NoSuchMethodException ex) {

        }
      }

    }
  }

  protected ClusterConnection createConnection(Configuration conf, User user, ExecutorService pool)
    throws IOException {
    ClusterConnection newClusterConnection = null;
    AliHBaseConstants.ClusterType clusterType = getClusterTypeFromConf(conf);
    if(clusterType == AliHBaseConstants.ClusterType.HBASE) {
      newClusterConnection = new ConnectionManager.HConnectionImplementation(
        conf, false, pool, user);
    }else if(clusterType == AliHBaseConstants.ClusterType.HBASEUE){
      newClusterConnection = new AliHBaseUEClusterConnection(
        conf, pool, user);
    }else{
      throw new IOException("Cluster is null, can not create connection");
    }
    return newClusterConnection;
  }

  protected synchronized void onChangeCluster(Configuration activeConf, Configuration standbyConf)
    throws IOException {

    // Copy the configs in case change outside
    this.conf = new Configuration(activeConf);
    String activeConnectKey = ClusterSwitchUtil.getConnectKey(activeConf);
    AliHBaseConstants.ClusterType localActiveType = getClusterTypeFromConf(this.conf);

    if(standbyConf == null) {
      AliHBaseConstants.ClusterType lastClusterType = this.activeClusterType;
      if(lastClusterType == null){
        lastClusterType = localActiveType;
      }
      Connection lastConnection = activeConnectionImpl;
      LOG.info("Creating new connection to " + activeConnectKey);
      this.activeConnectionImpl = createConnection(activeConf, user, getBatchPool());
      this.activeClusterType = localActiveType;
      if (localActiveType == AliHBaseConstants.ClusterType.HBASE) {
        getPrivatesFromCurrentConnection();
      }
      if (lastConnection != null && !lastConnection.isClosed() && lastClusterType == AliHBaseConstants.ClusterType.HBASE) {
        LOG.info("Closing old connection on "
          + ClusterSwitchUtil.getConnectKey(lastConnection.getConfiguration())
          + " since destination cluster is changed");
        lastConnection.close();
      }
    }else{
      // Copy the configs in case change outside
      this.standbyConf = new Configuration(standbyConf);
      String standbyConnectKey = ClusterSwitchUtil.getConnectKey(standbyConf);
      AliHBaseConstants.ClusterType localStandbyType = getClusterTypeFromConf(standbyConf);

      //switch active and standby
      ClusterConnection tempConnection = this.activeConnectionImpl;
      this.activeConnectionImpl = this.standbyConnectionImpl;
      this.standbyConnectionImpl = tempConnection;

      //if active connection not exist or closed, create new connection to active
      if(activeConnectionImpl == null || activeConnectionImpl.isClosed()){
        LOG.info("Creating new active connection to " + activeConnectKey);
        this.activeConnectionImpl = createConnection(activeConf, user, getBatchPool());
      }
      //if standby connection not exist or closed, create new connect to standby
      if(standbyConnectionImpl == null || standbyConnectionImpl.isClosed()){
        LOG.info("Creating new standby connection to " + standbyConnectKey);
        try {
          this.standbyConnectionImpl = createConnection(standbyConf, user, getStandbyBatchPool());
        }catch(Throwable t){
          LOG.warn("Create standby connection failed with error : " + t);
        }
      }

      if (localActiveType == AliHBaseConstants.ClusterType.HBASE) {
        getPrivatesFromCurrentConnection();
      }

      this.activeClusterType = localActiveType;
      this.standbyClusterType = localStandbyType;
    }
    this.switchCount++;
  }

  private AliHBaseConstants.ClusterType getClusterTypeFromConf(Configuration conf){
    return AliHBaseConstants.ClusterType.valueOf(conf.get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
  }


  public long getSwitchCount() {
    return switchCount;
  }

  /** @return - true if the master server is running
   * @deprecated this has been deprecated without a replacement */
  @Deprecated
  @Override
  public boolean isMasterRunning()
    throws MasterNotRunningException, ZooKeeperConnectionException {
    return activeConnectionImpl.isMasterRunning();
  }

  /**
   * Use this api to check if the table has been created with the specified number of
   * splitkeys which was used while creating the given table.
   * Note : If this api is used after a table's region gets splitted, the api may return
   * false.
   * @param tableName
   *          tableName
   * @param splitKeys
   *          splitKeys used while creating table
   * @throws IOException
   *           if a remote or network exception occurs
   */
  @Override
  public boolean isTableAvailable(TableName tableName, byte[][] splitKeys)
    throws IOException {
    return activeConnectionImpl.isTableAvailable(tableName, splitKeys);
  }

  /**
   * Find the location of the region of <i>tableName</i> that <i>row</i>
   * lives in.
   * @param tableName name of the table <i>row</i> is in
   * @param row row key you're trying to find the region of
   * @return HRegionLocation that describes where to find the region in
   * question
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public HRegionLocation locateRegion(TableName tableName, byte[] row)
    throws IOException {
    return activeConnectionImpl.locateRegion(tableName, row);
  }

  /**
   * Allows flushing the region cache.
   */
  @Override
  public void clearRegionCache() {
    activeConnectionImpl.clearRegionCache();
  }

  public void cacheLocation(TableName tableName,
                            RegionLocations regionLocations) {
    metaCache.cacheLocation(tableName, regionLocations);
  }

  /**
   * Allows flushing the region cache of all locations that pertain to
   * <code>tableName</code>
   * @param tableName Name of the table whose regions we are to remove from
   * cache.
   */
  @Override
  public void clearRegionCache(TableName tableName) {
    activeConnectionImpl.clearRegionCache(tableName);
  }

  /**
   * Deletes cached locations for the specific region.
   * @param location The location object for the region, to be purged from cache.
   */
  @Override
  public void deleteCachedRegionLocation(HRegionLocation location) {
    activeConnectionImpl.deleteCachedRegionLocation(location);
  }

  /**
   * Find the location of the region of <i>tableName</i> that <i>row</i>
   * lives in, ignoring any value that might be in the cache.
   * @param tableName name of the table <i>row</i> is in
   * @param row row key you're trying to find the region of
   * @return HRegionLocation that describes where to find the region in
   * question
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public HRegionLocation relocateRegion(TableName tableName, byte[] row)
    throws IOException {
    return activeConnectionImpl.relocateRegion(tableName, row);
  }

  /**
   * Find the location of the region of <i>tableName</i> that <i>row</i>
   * lives in, ignoring any value that might be in the cache.
   * @param tableName name of the table <i>row</i> is in
   * @param row row key you're trying to find the region of
   * @param replicaId the replicaId of the region
   * @return RegionLocations that describe where to find the region in
   * question
   * @throws IOException if a remote or network exception occurs
   */
  public RegionLocations relocateRegion(TableName tableName, byte[] row,
                                        int replicaId) throws IOException {
    return activeConnectionImpl.relocateRegion(tableName, row, replicaId);
  }

  /**
   * Update the location cache. This is used internally by HBase, in most cases it should not be
   *  used by the client application.
   * @param tableName the table name
   * @param regionName the region name
   * @param rowkey the row
   * @param exception the exception if any. Can be null.
   * @param source the previous location
   */
  @Override
  public void updateCachedLocations(TableName tableName, byte[] regionName,
                                    byte[] rowkey, Object exception, ServerName source) {
    activeConnectionImpl.updateCachedLocations(tableName, regionName, rowkey,
      exception, source);
  }

  /**
   * Gets the location of the region of <i>regionName</i>.
   * @param regionName name of the region to locate
   * @return HRegionLocation that describes where to find the region in
   * question
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public HRegionLocation locateRegion(byte[] regionName) throws IOException {
    return activeConnectionImpl.locateRegion(regionName);
  }

  /**
   * Gets the locations of all regions in the specified table, <i>tableName</i>.
   * @param tableName table to get regions of
   * @return list of region locations for all regions of table
   * @throws IOException
   */
  @Override
  public List<HRegionLocation> locateRegions(TableName tableName)
    throws IOException {
    return activeConnectionImpl.locateRegions(tableName);
  }

  /**
   * Gets the locations of all regions in the specified table, <i>tableName</i>.
   * @param tableName table to get regions of
   * @param useCache Should we use the cache to retrieve the region information.
   * @param offlined True if we are to include offlined regions, false and we'll leave out offlined
   *          regions from returned list.
   * @return list of region locations for all regions of table
   * @throws IOException
   */
  @Override
  public List<HRegionLocation> locateRegions(final TableName tableName,
                                             final boolean useCache, final boolean offlined) throws IOException {
    return activeConnectionImpl.locateRegions(tableName, useCache, offlined);
  }

  /**
   *
   * @param tableName table to get regions of
   * @param row the row
   * @param useCache Should we use the cache to retrieve the region information.
   * @param retry do we retry
   * @return region locations for this row.
   * @throws IOException
   */
  public RegionLocations locateRegion(TableName tableName,
                                      byte[] row, boolean useCache, boolean retry) throws IOException {
    return locateRegion(tableName, row, useCache, retry, RegionReplicaUtil.DEFAULT_REPLICA_ID);
  }

  /**
   *
   * @param tableName table to get regions of
   * @param row the row
   * @param useCache Should we use the cache to retrieve the region information.
   * @param retry do we retry
   * @param replicaId the replicaId for the region
   * @return region locations for this row.
   * @throws IOException
   */
  public RegionLocations locateRegion(TableName tableName,
                                      byte[] row, boolean useCache, boolean retry, int replicaId) throws IOException {
    return activeConnectionImpl.locateRegion(tableName, row, useCache, retry, replicaId);
  }

  /**
   * Returns a {@link MasterKeepAliveConnection} to the active master
   */
  @Override
  public MasterProtos.MasterService.BlockingInterface getMaster() {
    try {
      return activeConnectionImpl.getMaster();
    }catch(IOException e){
      return null;
    }
  }

  /**
   * Establishes a connection to the region server at the specified address.
   * @param serverName
   * @return proxy for HRegionServer
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public AdminProtos.AdminService.BlockingInterface getAdmin(
    ServerName serverName) throws IOException {
    return activeConnectionImpl.getAdmin(serverName);
  }

  /**
   * Establishes a connection to the region server at the specified address, and returns
   * a region client protocol.
   *
   * @param serverName
   * @return ClientProtocol proxy for RegionServer
   * @throws IOException if a remote or network exception occurs
   *
   */
  @Override
  public ClientProtos.ClientService.BlockingInterface getClient(
    ServerName serverName) throws IOException {
    return activeConnectionImpl.getClient(serverName);
  }

  /**
   * Find region location hosting passed row
   * @param tableName table name
   * @param row Row to find.
   * @param reload If true do not use cache, otherwise bypass.
   * @return Location of row.
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public HRegionLocation getRegionLocation(TableName tableName, byte [] row,
                                           boolean reload) throws IOException {
    return activeConnectionImpl.getRegionLocation(tableName, row, reload);
  }

  /**
   * Clear any caches that pertain to server name <code>sn</code>.
   * @param sn A server name
   */
  @Override
  public void clearCaches(ServerName sn) {
    activeConnectionImpl.clearCaches(sn);
  }

  /**
   * This function allows HBaseAdmin and potentially others to get a shared MasterService
   * connection.
   * @return The shared instance. Never returns null.
   * @throws MasterNotRunningException
   */
  @Override
  @Deprecated
  public MasterKeepAliveConnection getKeepAliveMasterService()
    throws MasterNotRunningException {
    return activeConnectionImpl.getKeepAliveMasterService();
  }

  /**
   * @param serverName
   * @return true if the server is known as dead, false otherwise.
   * @deprecated internal method, do not use thru HConnection */
  @Override
  @Deprecated
  public boolean isDeadServer(ServerName serverName) {
    return activeConnectionImpl.isDeadServer(serverName);
  }

  /**
   * @return Nonce generator for this HConnection; may be null if disabled in configuration.
   */
  @Override
  public NonceGenerator getNonceGenerator() {
    return activeConnectionImpl.getNonceGenerator();
  }

  /**
   * @return Default AsyncProcess associated with this connection.
   */
  @Override
  public AsyncProcess getAsyncProcess() {
    // return the asyncProcess created. Not the one in the currentConnectionImpl
    // Since when a switch happens, the pool in the currentConnectionImpl will be close
    // The user will always receive a RejectedExecutionException unless they recreate
    // the Table instance from the connection
    if (asyncProcess != null) {
      return asyncProcess;
    }
    synchronized (this) {
      if (asyncProcess != null) {
        return asyncProcess;
      }
      asyncProcess = createAsyncProcess();
      return asyncProcess;
    }
  }

  /**
   * Returns a new RpcRetryingCallerFactory from the given {@link Configuration}.
   * This RpcRetryingCallerFactory lets the users create {@link RpcRetryingCaller}s which can be
   * intercepted with the configured {@link RetryingCallerInterceptor}
   * @param conf
   * @return RpcRetryingCallerFactory
   */
  @Override
  public RpcRetryingCallerFactory getNewRpcRetryingCallerFactory(
    Configuration conf) {
    return activeConnectionImpl.getNewRpcRetryingCallerFactory(conf);
  }

  /**
   * @return Connection's RpcRetryingCallerFactory instance
   */
  public RpcRetryingCallerFactory getRpcRetryingCallerFactory() {
    return rpcCallerFactory;
  }

  /**
   * @return Connection's RpcControllerFactory instance
   */
  public RpcControllerFactory getRpcControllerFactory() {
    return rpcControllerFactory;
  }

  /**
   * @return a ConnectionConfiguration object holding parsed configuration values
   */
  public ConnectionConfiguration getConnectionConfiguration() {
    return connectionConfiguration;
  }



  /**
   *
   * @return true if this is a managed connection.
   */
  @Override
  public boolean isManaged() {
    return activeConnectionImpl.isManaged();
  }


  /**
   * @return the current statistics tracker associated with this connection
   */
  @Override
  public ServerStatisticTracker getStatisticsTracker() {
    if(activeConnectionImpl == null){
      return super.getStatisticsTracker();
    }
    return activeConnectionImpl.getStatisticsTracker();
  }

  /**
   * @return the configured client backoff policy
   */
  @Override
  public ClientBackoffPolicy getBackoffPolicy() {
    return activeConnectionImpl.getBackoffPolicy();
  }


  /**
   * @return true when this connection uses a {@link org.apache.hadoop.hbase.codec.Codec} and so
   *         supports cell blocks.
   */
  public boolean hasCellBlockSupport() {
    return hasCellBlockSupport;
  }

  @Override
  public Configuration getConfiguration() {
    if(this.conf == null){
      return super.getConfiguration();
    }
    return conf;
  }

  /**
   * Retrieve an HTableInterface implementation for access to a table.
   * The returned HTableInterface is not thread safe, a new instance should
   * be created for each using thread.
   * This is a lightweight operation, pooling or caching of the returned HTableInterface
   * is neither required nor desired.
   * Note that the HConnection needs to be unmanaged
   * (created with {@link HConnectionManager#createConnection(Configuration)}).
   * @param tableName
   * @return an HTable to use for interactions with this table
   */
  @Override
  public HTableInterface getTable(String tableName) throws IOException {
    return getTable(TableName.valueOf(tableName));
  }

  /**
   * Retrieve an HTableInterface implementation for access to a table.
   * The returned HTableInterface is not thread safe, a new instance should
   * be created for each using thread.
   * This is a lightweight operation, pooling or caching of the returned HTableInterface
   * is neither required nor desired.
   * Note that the HConnection needs to be unmanaged
   * (created with {@link HConnectionManager#createConnection(Configuration)}).
   * @param tableName
   * @return an HTable to use for interactions with this table
   */
  @Override
  public HTableInterface getTable(byte[] tableName) throws IOException {
    return getTable(TableName.valueOf(tableName));
  }

  /**
   * Retrieve an HTableInterface implementation for access to a table.
   * The returned HTableInterface is not thread safe, a new instance should
   * be created for each using thread.
   * This is a lightweight operation, pooling or caching of the returned HTableInterface
   * is neither required nor desired.
   * Note that the HConnection needs to be unmanaged
   * (created with {@link HConnectionManager#createConnection(Configuration)}).
   * @param tableName
   * @return an HTable to use for interactions with this table
   */
  @Override
  public HTableInterface getTable(TableName tableName) throws IOException {
    return getTable(tableName, getBatchPool());
  }

  /**
   * Retrieve an HTableInterface implementation for access to a table.
   * The returned HTableInterface is not thread safe, a new instance should
   * be created for each using thread.
   * This is a lightweight operation, pooling or caching of the returned HTableInterface
   * is neither required nor desired.
   * Note that the HConnection needs to be unmanaged
   * (created with {@link HConnectionManager#createConnection(Configuration)}).
   * @param tableName
   * @param pool The thread pool to use for batch operations, null to use a default pool.
   * @return an HTable to use for interactions with this table
   */
  @Override
  public HTableInterface getTable(String tableName, ExecutorService pool)
    throws IOException {
    return getTable(TableName.valueOf(tableName), pool);
  }

  /**
   * Retrieve an HTableInterface implementation for access to a table.
   * The returned HTableInterface is not thread safe, a new instance should
   * be created for each using thread.
   * This is a lightweight operation, pooling or caching of the returned HTableInterface
   * is neither required nor desired.
   * Note that the HConnection needs to be unmanaged
   * (created with {@link HConnectionManager#createConnection(Configuration)}).
   * @param tableName
   * @param pool The thread pool to use for batch operations, null to use a default pool.
   * @return an HTable to use for interactions with this table
   */
  @Override
  public HTableInterface getTable(byte[] tableName, ExecutorService pool)
    throws IOException {
    return getTable(TableName.valueOf(tableName), pool);
  }

  /**
   * Retrieve an HTableInterface implementation for access to a table.
   * The returned HTableInterface is not thread safe, a new instance should
   * be created for each using thread.
   * This is a lightweight operation, pooling or caching of the returned HTableInterface
   * is neither required nor desired.
   * Note that the HConnection needs to be unmanaged
   * (created with {@link HConnectionManager#createConnection(Configuration)}).
   * @param tableName
   * @param pool The thread pool to use for batch operations, null to use a default pool.
   * @return an HTable to use for interactions with this table
   */
  @Override
  public HTableInterface getTable(TableName tableName,
                         ExecutorService pool) throws IOException {
    HTable htable = getHTableByType(tableName, pool, this.activeClusterType, this, false);
    return new AliHBaseMultiTable(tableName, this, htable);
  }

  public HTable getHTableByType(TableName tableName) throws IOException {
    return getHTableByType(tableName, getBatchPool(), this.activeClusterType, this, false);
  }

  public HTable getHTableWithStandbyConnection(TableName tableName) throws IOException{
    return getHTableByType(tableName, getStandbyBatchPool(), this.standbyClusterType,
      this.standbyConnectionImpl, true);
  }

  public HTable getHTableByType(TableName tableName, ExecutorService pool,
                                AliHBaseConstants.ClusterType clusterType,
                                ClusterConnection connection, boolean isStandby) throws IOException {
    if(clusterType == AliHBaseConstants.ClusterType.HBASE) {
      // Need pass in the MultiClusterConnection itself
      try {
        if (constructorWithTableConfiguration != null) {
          return (HTable)constructorWithTableConfiguration.newInstance(tableName, connection, tableConfiguration,
            getRpcRetryingCallerFactory(),
            getRpcControllerFactory(), pool);
        } else if (constructorWithConnectionConfiguration != null) {
          return (HTable)constructorWithConnectionConfiguration
            .newInstance(tableName, connection, getConnectionConfiguration(),
              getRpcRetryingCallerFactory(), getRpcControllerFactory(), pool);
        } else {
          throw new IOException("No constructor found");
        }
      } catch (Throwable t) {
        if (t instanceof IOException) {
          throw (IOException)t;
        } else {
          throw new IOException(t);
        }
      }
    }else if(clusterType == AliHBaseConstants.ClusterType.HBASEUE){
      if(!isStandby) {
        return (HTable) this.activeConnectionImpl.getTable(tableName, pool);
      }else{
        if(this.standbyConnectionImpl != null && !this.standbyConnectionImpl.isClosed()){
          return (HTable) this.standbyConnectionImpl.getTable(tableName, pool);
        }else{
          return null;
        }
      }
    }else{
      LOG.error("getTable ClusterType is null , return null");
      return null;
    }
  }

  /**
   * Retrieve a RegionLocator implementation to inspect region information on a table. The returned
   * RegionLocator is not thread-safe, so a new instance should be created for each using thread.
   *
   * This is a lightweight operation.  Pooling or caching of the returned RegionLocator is neither
   * required nor desired.
   *
   * RegionLocator needs to be unmanaged
   * (created with {@link HConnectionManager#createConnection(Configuration)}).
   *
   * @param tableName Name of the table who's region is to be examined
   * @return A RegionLocator instance
   */
  @Override
  public RegionLocator getRegionLocator(TableName tableName)
    throws IOException {
    AliHBaseConstants.ClusterType clusterType =
      AliHBaseConstants.ClusterType.valueOf(this.conf.get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
    if(clusterType == AliHBaseConstants.ClusterType.HBASE) {
      // Need pass in the MultiClusterConnection itself
      return new HRegionLocator(tableName, this);
    }else if(clusterType == AliHBaseConstants.ClusterType.HBASEUE){
      return this.activeConnectionImpl.getRegionLocator(tableName);
    }else{
      LOG.error("getRegionLocator ClusterType is null , return null");
      return null;
    }
  }

  /**
   * Retrieve an Admin implementation to administer an HBase cluster.
   * The returned Admin is not guaranteed to be thread-safe.  A new instance should be created for
   * each using thread.  This is a lightweight operation.  Pooling or caching of the returned
   * Admin is not recommended.  Note that HConnection needs to be unmanaged
   *
   * @return an Admin instance for cluster administration
   */
  @Override
  public Admin getAdmin() throws IOException {
    HBaseAdmin hbaseAdmin = getHBaseAdmin();
    return new AliHBaseMultiAdmin(this, hbaseAdmin);
  }

  public HBaseAdmin getHBaseAdmin() throws IOException {
    AliHBaseConstants.ClusterType clusterType = getClusterTypeFromConf(this.conf);
    if(clusterType == AliHBaseConstants.ClusterType.HBASE) {
      // Need pass in the MultiClusterConnection itself
      return new HBaseAdmin(this);
    }else if(clusterType == AliHBaseConstants.ClusterType.HBASEUE){
      return (AliHBaseUEAdmin)this.activeConnectionImpl.getAdmin();
    }else{
      LOG.error("getHBaseAdmin ClusterType is null , return null");
      return null;
    }
  }

  /**
   * A table that isTableEnabled == false and isTableDisabled == false
   * is possible. This happens when a table has a lot of regions
   * that must be processed.
   * @param tableName table name
   * @return true if the table is enabled, false otherwise
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public boolean isTableEnabled(TableName tableName) throws IOException {
    return activeConnectionImpl.isTableEnabled(tableName);
  }

  @Deprecated
  @Override
  public boolean isTableEnabled(byte[] tableName) throws IOException {
    return activeConnectionImpl.isTableEnabled(tableName);
  }

  /**
   * @param tableName table name
   * @return true if the table is disabled, false otherwise
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public boolean isTableDisabled(TableName tableName) throws IOException {
    return activeConnectionImpl.isTableDisabled(tableName);
  }

  @Deprecated
  @Override
  public boolean isTableDisabled(byte[] tableName) throws IOException {
    return activeConnectionImpl.isTableDisabled(tableName);
  }

  /**
   * @param tableName table name
   * @return true if all regions of the table are available, false otherwise
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public boolean isTableAvailable(TableName tableName) throws IOException {
    return activeConnectionImpl.isTableAvailable(tableName);
  }

  @Deprecated
  @Override
  public boolean isTableAvailable(byte[] tableName) throws IOException {
    return activeConnectionImpl.isTableAvailable(tableName);
  }

  @Deprecated
  @Override
  public boolean isTableAvailable(byte[] tableName, byte[][] splitKeys)
    throws IOException {
    return activeConnectionImpl.isTableAvailable(tableName, splitKeys);
  }

  /**
   * List all the userspace tables.  In other words, scan the hbase:meta table.
   *
   * @return - returns an array of HTableDescriptors
   * @throws IOException if a remote or network exception occurs
   * @deprecated Use {@link Admin#listTables()} instead.
   */
  @Deprecated
  @Override
  public HTableDescriptor[] listTables() throws IOException {
    return activeConnectionImpl.listTables();
  }

  /**
   * @deprecated Use {@link Admin#listTableNames()} instead.
   */
  @Deprecated
  @Override
  public String[] getTableNames() throws IOException {
    return activeConnectionImpl.getTableNames();
  }

  /**
   * @deprecated Use {@link Admin#listTables()} instead.
   */
  @Deprecated
  @Override
  public TableName[] listTableNames() throws IOException {
    return activeConnectionImpl.listTableNames();
  }

  /**
   * @param tableName table name
   * @return table metadata
   * @throws IOException if a remote or network exception occurs
   */
  @Deprecated
  @Override
  public HTableDescriptor getHTableDescriptor(TableName tableName)
    throws IOException {
    return activeConnectionImpl.getHTableDescriptor(tableName);
  }

  @Deprecated
  @Override
  public HTableDescriptor getHTableDescriptor(byte[] tableName) throws IOException {
    return activeConnectionImpl.getHTableDescriptor(tableName);
  }

  @Deprecated
  @Override
  public HRegionLocation locateRegion(final byte[] tableName,
                                      final byte [] row)
    throws IOException {
    return activeConnectionImpl.locateRegion(tableName, row);
  }

  @Deprecated
  @Override
  public void clearRegionCache(byte[] tableName) {
    activeConnectionImpl.clearRegionCache(tableName);
  }

  @Deprecated
  @Override
  public HRegionLocation relocateRegion(final byte[] tableName,
                                        final byte [] row)
    throws IOException {
    return activeConnectionImpl.relocateRegion(tableName, row);
  }

  @Deprecated
  @Override
  public void updateCachedLocations(TableName tableName, byte[] rowkey,
                                    Object exception, HRegionLocation source) {
    activeConnectionImpl.updateCachedLocations(tableName, rowkey, exception, source);
  }

  @Deprecated
  @Override
  public void updateCachedLocations(byte[] tableName, byte[] rowkey,
                                    Object exception, HRegionLocation source) {
    activeConnectionImpl.updateCachedLocations(tableName, rowkey, exception, source);
  }

  @Deprecated
  @Override
  public List<HRegionLocation> locateRegions(byte[] tableName) throws IOException {
    return activeConnectionImpl.locateRegions(tableName);
  }

  @Deprecated
  @Override
  public List<HRegionLocation> locateRegions(final byte[] tableName,
                                             final boolean useCache,
                                             final boolean offlined) throws IOException {
    return activeConnectionImpl.locateRegions(tableName, useCache, offlined);
  }

  /**
   * Establishes a connection to the region server at the specified address.
   * @param serverName
   * @param getMaster do we check if master is alive
   * @return proxy for HRegionServer
   * @throws IOException if a remote or network exception occurs
   * @deprecated You can pass master flag but nothing special is done.
   */
  @Deprecated
  @Override
  public AdminProtos.AdminService.BlockingInterface getAdmin(
    ServerName serverName, boolean getMaster) throws IOException {
    return activeConnectionImpl.getAdmin(serverName, getMaster);
  }

  @Deprecated
  @Override
  public HRegionLocation getRegionLocation(byte[] tableName, byte [] row,
                                           boolean reload) throws IOException {
    return activeConnectionImpl.getRegionLocation(tableName, row, reload);
  }

  /**
   * Process a mixed batch of Get, Put and Delete actions. All actions for a
   * RegionServer are forwarded in one RPC call.
   *
   *
   * @param actions The collection of actions.
   * @param tableName Name of the hbase table
   * @param pool thread pool for parallel execution
   * @param results An empty array, same size as list. If an exception is thrown,
   * you can test here for partial results, and to determine which actions
   * processed successfully.
   * @throws IOException if there are problems talking to META. Per-item
   * exceptions are stored in the results array.
   * @deprecated since 0.96 - Use {@link HTableInterface#batch} instead
   */
  @Deprecated
  @Override
  public void processBatch(List<? extends Row> actions, final TableName tableName,
                           ExecutorService pool, Object[] results)
    throws IOException, InterruptedException {
    activeConnectionImpl.processBatch(actions, tableName, pool, results);

  }

  @Deprecated
  @Override
  public void processBatch(List<? extends Row> actions, final byte[] tableName,
                           ExecutorService pool, Object[] results)
    throws IOException, InterruptedException {
    activeConnectionImpl.processBatch(actions, tableName, pool, results);
  }

  /**
   * Parameterized batch processing, allowing varying return types for different
   * {@link Row} implementations.
   * @deprecated since 0.96 - Use {@link HTableInterface#batchCallback} instead
   */
  @Deprecated
  @Override
  public <R> void processBatchCallback(List<? extends Row> list,
                                       final TableName tableName,
                                       ExecutorService pool,
                                       Object[] results,
                                       Batch.Callback<R> callback) throws IOException, InterruptedException {
    activeConnectionImpl.processBatchCallback(list, tableName, pool, results, callback);
  }

  @Deprecated
  @Override
  public <R> void processBatchCallback(List<? extends Row> list,
                                       final byte[] tableName,
                                       ExecutorService pool,
                                       Object[] results,
                                       Batch.Callback<R> callback) throws IOException, InterruptedException {
    activeConnectionImpl.processBatchCallback(list, tableName, pool, results, callback);
  }

  /**
   * @deprecated does nothing since since 0.99
   **/
  @Deprecated
  @Override
  public void setRegionCachePrefetch(final TableName tableName,
                                     final boolean enable) {
    activeConnectionImpl.setRegionCachePrefetch(tableName, enable);
  }

  /**
   * @deprecated does nothing since 0.99
   **/
  @Deprecated
  @Override
  public void setRegionCachePrefetch(final byte[] tableName,
                                     final boolean enable) {
    activeConnectionImpl.setRegionCachePrefetch(tableName, enable);
  }

  /**
   * @deprecated always return false since 0.99
   **/
  @Deprecated
  @Override
  public boolean getRegionCachePrefetch(TableName tableName) {
    return activeConnectionImpl.getRegionCachePrefetch(tableName);
  }

  /**
   * @deprecated always return false since 0.99
   **/
  @Deprecated
  @Override
  public boolean getRegionCachePrefetch(byte[] tableName) {
    return activeConnectionImpl.getRegionCachePrefetch(tableName);
  }

  /**
   * @return the number of region servers that are currently running
   * @throws IOException if a remote or network exception occurs
   * @deprecated This method will be changed from public to package protected.
   */
  @Deprecated
  @Override
  public int getCurrentNrHRS() throws IOException {
    return activeConnectionImpl.getCurrentNrHRS();
  }

  /**
   * @param tableNames List of table names
   * @return HTD[] table metadata
   * @throws IOException if a remote or network exception occurs
   * @deprecated Use {@link Admin#getTableDescriptor(TableName)} instead.
   */
  @Deprecated
  @Override
  public HTableDescriptor[] getHTableDescriptorsByTableName(
    List<TableName> tableNames) throws IOException {
    return activeConnectionImpl.getHTableDescriptorsByTableName(tableNames);
  }

  @Deprecated
  @Override
  public HTableDescriptor[] getHTableDescriptors(List<String> tableNames)
    throws IOException {
    return activeConnectionImpl.getHTableDescriptors(tableNames);
  }

  /**
   * @return true if this connection is closed
   */
  @Override
  public boolean isClosed() {
    return closed;
  }

  /**
   * <p>
   * Retrieve a {@link BufferedMutator} for performing client-side buffering of writes. The
   * {@link BufferedMutator} returned by this method is thread-safe. This BufferedMutator will
   * use the Connection's ExecutorService. This object can be used for long lived operations.
   * </p>
   * <p>
   * The caller is responsible for calling {@link BufferedMutator#close()} on
   * the returned {@link BufferedMutator} instance.
   * </p>
   * <p>
   * This accessor will use the connection's ExecutorService and will throw an
   * exception in the main thread when an asynchronous exception occurs.
   *
   * @param tableName the name of the table
   *
   * @return a {@link BufferedMutator} for the supplied tableName.
   */
  @Override
  public BufferedMutator getBufferedMutator(TableName tableName) {
    // Need redirect this method to
    // getBufferedMutator(BufferedMutatorParams bufferedMutatorParams)
    // to pass in the MultiClusterConnection. Otherwise, the BufferedMutator
    // will be useless after a switch happens.
    return getBufferedMutator(new BufferedMutatorParams(tableName));
  }

  /**
   * Retrieve a {@link BufferedMutator} for performing client-side buffering of writes. The
   * {@link BufferedMutator} returned by this method is thread-safe. This object can be used for
   * long lived table operations. The caller is responsible for calling
   * {@link BufferedMutator#close()} on the returned {@link BufferedMutator} instance.
   *
   * @param params details on how to instantiate the {@code BufferedMutator}.
   * @return a {@link BufferedMutator} for the supplied tableName.
   */
  @Override
  public BufferedMutator getBufferedMutator(
    BufferedMutatorParams params){
    BufferedMutator bufferedMutator = getBufferedMutatorByType(params);
    return new AliHBaseMultiBufferedMutator(this, bufferedMutator, params);
  }

  public BufferedMutator getBufferedMutatorByType(
    BufferedMutatorParams params){
    AliHBaseConstants.ClusterType clusterType = getClusterTypeFromConf(this.conf);
    if(clusterType == AliHBaseConstants.ClusterType.HBASE) {
      if (params.getTableName() == null) {
        throw new IllegalArgumentException("TableName cannot be null.");
      }
      if (params.getPool() == null) {
        params.pool(HTable.getDefaultExecutor(getConfiguration()));
      }
      if (params.getWriteBufferSize() == BufferedMutatorParams.UNSET) {
        params.writeBufferSize(getConnectionConfiguration().getWriteBufferSize());
      }
      if (params.getMaxKeyValueSize() == BufferedMutatorParams.UNSET) {
        params.maxKeyValueSize(getConnectionConfiguration().getMaxKeyValueSize());
      }
      // Pass in the MultiClusterConnection itself
      return new BufferedMutatorImpl(this, getRpcRetryingCallerFactory(),
        getRpcControllerFactory(), params);
    }else if(clusterType == AliHBaseConstants.ClusterType.HBASEUE){
      try {
        return this.activeConnectionImpl.getBufferedMutator(params);
      }catch(IOException e){
        LOG.warn("get bufferd mutator failed " + e);
        return null;
      }
    }else{
      LOG.error("getBufferedMutator ClusterType is null , return null");
      return null;
    }
  }

  @Override
  public void close(){
    try {
      if (this.closed) {
        return;
      }
      if (activeConnectionImpl != null) {
        activeConnectionImpl.close();
      }
      if (standbyConnectionImpl != null) {
        standbyConnectionImpl.close();
      }
      shutdownBatchPool(batchPool, this.cleanupPool);
      shutdownBatchPool(standbyBatchPool, this.cleanupStandbyPool);
      if (dualExecutor != null) {
        dualExecutor.close();
      }
    }catch(IOException e){
      LOG.warn("close failed " + e);
    }
  }


  @Override
  public void abort(String why, Throwable throwable) {
    // MultiClusterConnection does not support abort
    // If it is a abort triggered by ZK session expire
    // Just let the underlying connection handle it.
    this.activeConnectionImpl.abort(why, throwable);
    if(this.standbyConnectionImpl != null){
      this.standbyConnectionImpl.abort(why, throwable);
    }
    this.aborted = true;
  }

  @Override
  public boolean isAborted() {
    return aborted;
  }

  private void shutdownBatchPool(ExecutorService pool, boolean cleanupPool) {
    if (!cleanupPool || pool.isShutdown()) {
      return;
    }
    pool.shutdown();
    try {
      if (!pool.awaitTermination(10, TimeUnit.SECONDS)) {
        pool.shutdownNow();
      }
    } catch (InterruptedException e) {
      pool.shutdownNow();
    }
  }

  public Configuration getOriginalConf(){
    return this.conf;
  }

  public DualExecutor getDualExecutor(){
    return this.dualExecutor;
  }
}