/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.hbase.client;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Queue;

import com.alibaba.hbase.exception.BatchExceptions;
import com.alibaba.hbase.thrift2.generated.TAppend;
import com.alibaba.hbase.thrift2.generated.TColumnFamilyDescriptor;
import com.alibaba.hbase.thrift2.generated.TDelete;
import com.alibaba.hbase.thrift2.generated.TGet;
import com.alibaba.hbase.thrift2.generated.THBaseService;
import com.alibaba.hbase.thrift2.generated.TIncrement;
import com.alibaba.hbase.thrift2.generated.TNamespaceDescriptor;
import com.alibaba.hbase.thrift2.generated.TPut;
import com.alibaba.hbase.thrift2.generated.TResult;
import com.alibaba.hbase.thrift2.generated.TRowMutations;
import com.alibaba.hbase.thrift2.generated.TScan;
import com.alibaba.hbase.thrift2.generated.TTableDescriptor;
import com.alibaba.hbase.thrift2.generated.TTableName;
import com.alibaba.lindorm.client.core.meta.ExternalIndexConfig;
import com.alibaba.lindorm.client.core.meta.ExternalIndexField;
import com.alibaba.lindorm.client.core.widecolumnservice.WScan;
import com.alibaba.lindorm.client.dml.DmlAttributeConstants;
import com.google.protobuf.Service;
import com.google.protobuf.ServiceException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.NamespaceDescriptor;
import org.apache.hadoop.hbase.NamespaceNotFoundException;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Append;
import org.apache.hadoop.hbase.client.ConnectionUtils;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Increment;
import org.apache.hadoop.hbase.client.Mutation;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Row;
import org.apache.hadoop.hbase.client.RowMutations;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.coprocessor.Batch;
import org.apache.hadoop.hbase.client.index.AliHBaseIndexDescriptor;
import org.apache.hadoop.hbase.client.metrics.ScanMetrics;
import org.apache.hadoop.hbase.filter.CompareFilter;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.thrift.TException;
import org.apache.thrift.transport.TTransport;

public class AliHBaseAPIProxyThriftImpl implements AliHBaseAPIProxy{

  private THBaseService.Client client;
  private TTransport transport;
  private ByteBuffer tableNameInBytes;
  private int defaultScannerCaching;

  private int operationTimeout;
  private int pause;
  private int numRetries;

  private AliHBaseThriftImplFactory factory;

  private int cUInBytes;

  private static Pair<byte[][],byte[][]> fakeKeys;

  static {
    byte[][] startkeys = new byte[1][];
    byte[][] endkeys = new byte[1][];
    startkeys[0] = HConstants.EMPTY_START_ROW;
    endkeys[0] = HConstants.EMPTY_START_ROW;
    fakeKeys = new Pair<>(startkeys, endkeys);
  }



  public AliHBaseAPIProxyThriftImpl(Configuration conf, THBaseService.Client client, TTransport transport, TableName tableName, AliHBaseThriftImplFactory factory) {
    this.factory = factory;
    this.client = client;
    this.transport = transport;
    if (tableName != null) {
      tableNameInBytes = ByteBuffer.wrap(tableName.toBytes());
    }
    this.defaultScannerCaching = conf.getInt(HConstants.HBASE_CLIENT_SCANNER_CACHING,
        AliHBaseConstants.ALIHBASE_CLIENT_SCANNER_CACHING_DEFAULT);
    if (defaultScannerCaching > AliHBaseConstants.ALIHBASE_CLIENT_SCANNER_CACHING_DEFAULT) {
      this.defaultScannerCaching = AliHBaseConstants.ALIHBASE_CLIENT_SCANNER_CACHING_DEFAULT;
    }
    this.operationTimeout = conf.getInt(HConstants.HBASE_CLIENT_OPERATION_TIMEOUT, HConstants.DEFAULT_HBASE_CLIENT_OPERATION_TIMEOUT);
    this.pause = (int)conf.getLong(HConstants.HBASE_CLIENT_PAUSE, HConstants.DEFAULT_HBASE_CLIENT_PAUSE);
    this.numRetries = conf.getInt(HConstants.HBASE_CLIENT_RETRIES_NUMBER, HConstants.DEFAULT_HBASE_CLIENT_RETRIES_NUMBER);
    this.cUInBytes = conf.getInt(AliHBaseConstants.HBASE_CU_IN_BYTES, AliHBaseConstants.DEFAULT_HBASE_CU_IN_BYTES);

  }





  @Override
  public boolean tableExists(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      return client.tableExists(tTableName);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public List<HTableDescriptor> listTableDescriptors() throws IOException {
    try {
      List<TTableDescriptor> tTableDescriptors = client
          .getTableDescriptorsByPattern(null, false);
      return ThriftUtilities.tableDescriptorsFromThrift(tTableDescriptors);

    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public TableName[] listTableNames() throws IOException {
    try {
      List<TTableName> tTableNames = client.getTableNamesByPattern(null, false);
      return ThriftUtilities.tableNamesArrayFromThrift(tTableNames);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public HTableDescriptor getDescriptor(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      TTableDescriptor tTableDescriptor = client.getTableDescriptor(tTableName);
      return ThriftUtilities.tableDescriptorFromThrift(tTableDescriptor);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public TableName[] listTableNamesByNamespace(String name) throws IOException {
    try {
      List<TTableName> tTableNames = client.getTableNamesByNamespace(name);
      return ThriftUtilities.tableNamesArrayFromThrift(tTableNames);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public List<HTableDescriptor> listTableDescriptorsByNamespace(String name) throws IOException {
    try {
      List<TTableDescriptor> tTableDescriptors = client.getTableDescriptorsByNamespace(name);
      return ThriftUtilities.tableDescriptorsFromThrift(tTableDescriptors);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void createTable(HTableDescriptor desc, byte[][] splitKeys) throws IOException {
    TTableDescriptor tTableDescriptor = ThriftUtilities.tableDescriptorFromHBase(desc);
    List<ByteBuffer> splitKeyInBuffer = ThriftUtilities.splitKeyFromHBase(splitKeys);
    try {
      client.createTable(tTableDescriptor, splitKeyInBuffer);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void deleteTable(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      client.deleteTable(tTableName);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void truncateTable(TableName tableName, boolean preserveSplits) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      client.truncateTable(tTableName, preserveSplits);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void enableTable(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      client.enableTable(tTableName);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void disableTable(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      client.disableTable(tTableName);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public boolean isTableEnabled(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      return client.isTableEnabled(tTableName);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public boolean isTableDisabled(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      return client.isTableDisabled(tTableName);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public boolean isTableAvailable(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      return client.isTableAvailable(tTableName);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void addColumnFamily(TableName tableName, HColumnDescriptor columnFamily)
      throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    TColumnFamilyDescriptor tColumnFamilyDescriptor = ThriftUtilities
        .columnFamilyDescriptorFromHBase(columnFamily);
    try {
      client.addColumnFamily(tTableName, tColumnFamilyDescriptor);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void deleteColumnFamily(TableName tableName, byte[] columnFamily) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      client.deleteColumnFamily(tTableName, ByteBuffer.wrap(columnFamily));
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void modifyColumnFamily(TableName tableName, HColumnDescriptor columnFamily)
      throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    TColumnFamilyDescriptor tColumnFamilyDescriptor = ThriftUtilities
        .columnFamilyDescriptorFromHBase(columnFamily);
    try {
      client.modifyColumnFamily(tTableName, tColumnFamilyDescriptor);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void modifyTable(HTableDescriptor td) throws IOException {
    TTableDescriptor tTableDescriptor = ThriftUtilities
        .tableDescriptorFromHBase(td);
    try {
      client.modifyTable(tTableDescriptor);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void deleteNamespace(String name) throws IOException {
    try {
      client.deleteNamespace(name);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public NamespaceDescriptor getNamespaceDescriptor(String name)
      throws NamespaceNotFoundException, IOException {
    try {
      TNamespaceDescriptor tNamespaceDescriptor = client.getNamespaceDescriptor(name);
      return ThriftUtilities.namespaceDescriptorFromThrift(tNamespaceDescriptor);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public NamespaceDescriptor[] listNamespaceDescriptors() throws IOException {
    try {
      List<TNamespaceDescriptor> tNamespaceDescriptors = client.listNamespaceDescriptors();
      return ThriftUtilities.namespaceDescriptorsFromThrift(tNamespaceDescriptors);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void createNamespace(NamespaceDescriptor descriptor) throws IOException {
    TNamespaceDescriptor tNamespaceDescriptor = ThriftUtilities
        .namespaceDescriptorFromHBase(descriptor);
    try {
      client.createNamespace(tNamespaceDescriptor);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void flushRegion(String regionName) {
    throw new UnsupportedOperationException("flush is not supported");
  }

  @Override
  public void compactRegion(String regionName) {
    throw new UnsupportedOperationException("compact is not supported");
  }

  @Override
  public void majorCompactRegion(String regionName) {
    throw new UnsupportedOperationException("majorCompact is not supported");
  }

  @Override
  public void flushTable(TableName tableName) {
    throw new UnsupportedOperationException("flush is not supported");
  }

  @Override
  public void compactTable(TableName tableName) {
    throw new UnsupportedOperationException("compact is not supported");
  }

  @Override
  public void majorCompactTable(TableName tableName) {
    throw new UnsupportedOperationException("majorCompact is not supported");
  }

  @Override
  public void split(String tableNameOrRegionName, byte[] splitPoint) {
    throw new UnsupportedOperationException("split is not supported");
  }

  @Override
  public void mergeRegions(String tableNameA, String shortNameA, String shortNameB) {
    throw new UnsupportedOperationException("merge regions is not supported");
  }

  @Override
  public HRegionLocation getRegionLocation(byte[] row) throws IOException {
    throw new UnsupportedOperationException("getRegionLocation is not supported");

  }

  @Override
  public List<HRegionLocation> getAllRegionLocations() throws IOException {
    HRegionLocation loc = createFakeRegionLocation(fakeKeys.getFirst()[0], fakeKeys.getSecond()[0]);
    List<HRegionLocation> lists = new ArrayList<>();
    lists.add(loc);
    return lists;
  }

  private HRegionLocation createFakeRegionLocation(byte[] startKey, byte[] endKey) {
    HRegionInfo regionInfo = new HRegionInfo(TableName.valueOf(tableNameInBytes.array()), startKey, endKey);
    return new HRegionLocation(regionInfo, ServerName.valueOf("localhost", 0, 0));
  }

  @Override
  public Pair<byte[][], byte[][]> getStartEndKeys() throws IOException {
    return fakeKeys;
  }


  @Override
  public boolean exists(Get get) throws IOException {
    final TGet tGet = ThriftUtilities.getFromHBase(get);
    factory.acquirePermitInCU(1);
    RetryingCaller<Boolean> retryingCaller = new RetryingCaller<>();
    return retryingCaller.withRetries(new Callable<Boolean>() {
      @Override
      public Boolean call() throws Exception {
        return client.exists(tableNameInBytes, tGet);
      }
    });
  }

  @Override
  public boolean[] exists(List<Get> gets) throws IOException {
    final List<TGet> tGets = new ArrayList<>();
    for (Get get: gets) {
      tGets.add(ThriftUtilities.getFromHBase(get));
    }
    factory.acquirePermitInCU(gets.size());
    RetryingCaller<List<Boolean>> retryingCaller = new RetryingCaller<>();
    List<Boolean> results = retryingCaller.withRetries(new Callable<List<Boolean>>() {
      @Override
      public List<Boolean> call() throws Exception {
        return client.existsAll(tableNameInBytes, tGets);
      }
    });
    boolean[] booleans = new boolean[results.size()];
    for (int i = 0; i < results.size(); i++) {
      booleans[i] = results.get(i);
    }
    return booleans;
  }

  @Override
  public void batch(List<? extends Row> actions, Object[] results) throws IOException {
    List<Object> tactions = new ArrayList<Object>(actions.size());
    int writeSize = 0;
    int readSize = 0;
    boolean hasGet = false;
    for (Row action : actions) {
      if (action instanceof Get) {
        tactions.add(ThriftUtilities.getFromHBase((Get) action));
        hasGet = true;
      } else if (action instanceof Put) {
        Put put = (Put) action;
        writeSize += calculateMutationSize(put);
        tactions.add(ThriftUtilities.putFromHBase(put));
      } else if (action instanceof Delete) {
        Delete delete = (Delete) action;
        tactions.add(ThriftUtilities.deleteFromHBase(delete));
        writeSize += calculateMutationSize(delete);
      } else if (action instanceof Append) {
        Append append = (Append)action;
        tactions.add(ThriftUtilities.appendFromHBase((Append) append));
        writeSize += calculateMutationSize(append);

      } else {
        throw new UnsupportedOperationException(
            "Unsupported type " + action.getClass().getName() + " in batch operation.");
      }
    }
    factory.acquirePermitInCU(convertToCU(writeSize));
    if (hasGet) {
      // acquire 1 cu in advance for read, in case lack of CU
      factory.acquirePermitInCU(1);
    }

    Object[] hbaseResults = null;
    BatchExceptions batchExceptions = new BatchExceptions();
    try {
      hbaseResults = batch(tactions);
    } finally {
      for (int i = 0; i < hbaseResults.length; i++) {
        if (results != null) {
          if (hbaseResults[i] == null) {
            results[i] = null;
          } else if (hbaseResults[i] instanceof TResult) {
            Result result = ThriftUtilities
                .resultFromThrift((TResult) hbaseResults[i]);
            results[i] = result;
            readSize += calculateResultSize(result);
          } else if (hbaseResults[i] instanceof Result) {
            results[i] = hbaseResults[i];
          } else if (hbaseResults[i] instanceof Throwable) {
            results[i] = hbaseResults[i];
            batchExceptions.add((Throwable) hbaseResults[i], actions.get(i), null);
          } else {
            throw new IOException("Not unsupported result type " + hbaseResults[i]);
          }
        } else {
          if (hbaseResults[i] != null && hbaseResults[i] instanceof Throwable) {
            batchExceptions.add((Throwable) hbaseResults[i], actions.get(i), null);
          }
        }
      }

      if (batchExceptions.hasErrors()) {
        throw batchExceptions.makeException();
      } else {
        factory.acquirePermitInCU(convertToCU(readSize, 1));
      }
    }
  }

  private Object[] batch(List<Object> actions)
      throws IOException {
    Object[] results = new Object[actions.size()];

    final List<TGet> getActions = new ArrayList<>();
    List<Integer> getActionsIndex = new ArrayList<>();

    final List<TPut> putActions = new ArrayList<>();
    List<Integer> putActionsIndex = new ArrayList<>();

    final List<TDelete> delActions = new ArrayList<>();
    List<Integer> delActionsIndex = new ArrayList<>();

    final List<TAppend> appendActions = new ArrayList<>();
    List<Integer> appendActionsIndex = new ArrayList<>();

    for (int i = 0; i < actions.size(); i++) {
      if (actions.get(i) instanceof TGet) {
        getActions.add((TGet) actions.get(i));
        getActionsIndex.add(i);
      } else if (actions.get(i) instanceof TPut) {
        putActions.add((TPut) actions.get(i));
        putActionsIndex.add(i);
      } else if (actions.get(i) instanceof TDelete) {
        delActions.add((TDelete) actions.get(i));
        delActionsIndex.add(i);
      } else if (actions.get(i) instanceof TAppend) {
        appendActions.add((TAppend) actions.get(i));
        appendActionsIndex.add(i);
      }
      else {
        throw new UnsupportedOperationException("Not supported action " + actions.get(i).getClass().getName());
      }
    }
    if (!getActions.isEmpty()) {
      try {
        RetryingCaller<List<TResult>> retryingCaller = new RetryingCaller<>();
        List<TResult> getResult = retryingCaller.withRetries(new Callable<List<TResult>>() {
          @Override
          public List<TResult> call() throws Exception {
            return client.getMultiple(tableNameInBytes, getActions);
          }
        });
        for (int i = 0; i < getActions.size(); i++) {
          int index = getActionsIndex.get(i);
          results[index] = getResult.get(i);
        }
      } catch (Throwable e) {
        for (Integer index : getActionsIndex) {
          results[index] = e;
        }
      }
    }
    if (!putActions.isEmpty()) {
      try {
        RetryingCaller<Void> retryingCaller = new RetryingCaller<>();
        retryingCaller.withRetries(new Callable<Void>() {
          @Override
          public Void call() throws Exception {
            client.putMultiple(tableNameInBytes, putActions);
            return null;
          }
        });
        for (Integer index : putActionsIndex) {
          results[index] = new Result();
        }
      } catch (Throwable e) {
        for (Integer index : putActionsIndex) {
          results[index] = e;
        }
      }
    }
    if (!delActions.isEmpty()) {
      try {
        RetryingCaller<Void> retryingCaller = new RetryingCaller<>();
        retryingCaller.withRetries(new Callable<Void>() {
          @Override
          public Void call() throws Exception {
            client.deleteMultiple(tableNameInBytes, delActions);
            return null;
          }
        });
        for (Integer index : delActionsIndex) {
          results[index] = new Result();
        }
      } catch (Throwable e) {
        for (Integer index : delActionsIndex) {
          results[index] = e;
        }
      }
    }
    if (!appendActions.isEmpty()) {
      try {
        for (int i = 0; i < appendActions.size(); i++) {
          final TAppend append = appendActions.get(i);
          RetryingCaller<TResult> retryingCaller = new RetryingCaller<>();
          TResult tResult = retryingCaller.withRetries(new Callable<TResult>() {
            @Override
            public TResult call() throws Exception {
              return client.append(tableNameInBytes, append);
            }
          });
          Result result = ThriftUtilities.resultFromThrift(tResult);
          results[appendActionsIndex.get(i)] = result;
        }
      } catch (Throwable e) {
        for (Integer index : appendActionsIndex) {
          results[index] = e;
        }
      }
    }
    return results;
  }

  public static int calculateResultSize(Result result) {
    if (result == null || result.rawCells() == null) {
      return 0;
    }
    int size = 0;
    for (Cell cell : result.rawCells()) {
      if (cell != null) {
        size += CellUtil.estimatedSerializedSizeOf(cell);
      }
    }
    return size;
  }

  public int convertToCU(int size, int cUPaiedInadvance) {
    int cu = size / cUInBytes - cUPaiedInadvance;
    return Math.max(0, cu);

  }

  public int convertToCU(int size) {
    return convertToCU(size, 0);
  }

  public static int calculateResultSize(Result[] results) {
    if (results == null) {
      return 0;
    }
    int size = 0;
    for (Result result : results) {
      size += calculateResultSize(result);
    }
    return size;
  }

  public static int calculatePutsSize(List<Put> puts) {
    if (puts == null) {
      return 0;
    }
    int size = 0;
    for (Put put : puts) {
      size += calculateMutationSize(put);
    }
    return size;
  }

  public static int calculateDeletesSize(List<Delete> deletes) {
    if (deletes == null) {
      return 0;
    }
    int size = 0;
    for (Delete delete : deletes) {
      size += calculateMutationSize(delete);
    }
    return size;
  }

  public static int calculateMutationsSize(List<Mutation> mutations) {
    if (mutations == null) {
      return 0;
    }
    int size = 0;
    for (Mutation mutation : mutations) {
      size += calculateMutationSize(mutation);
    }
    return size;
  }

  public static int calculateMutationSize(Mutation mutation) {
    if (mutation == null) {
      return 0;
    }
    if (mutation.getFamilyCellMap() == null) {
      return 0;
    }
    int size = 0;
    for(List<Cell> cells : mutation.getFamilyCellMap().values()) {
      if (cells != null) {
        for(Cell cell : cells) {
          if (cell != null) {
            size += CellUtil.estimatedSerializedSizeOf(cell);
          }
        }
      }
    }
    return size;
  }

  @Override
  public Result get(Get get) throws IOException {
    final TGet tGet = ThriftUtilities.getFromHBase(get);
    // acquire 1 cu in advance for read, in case lack of CU
    factory.acquirePermitInCU(1);
    RetryingCaller<TResult> retryingCaller = new RetryingCaller<>();
    TResult tResult = retryingCaller.withRetries(new Callable<TResult>() {
      @Override
      public TResult call() throws Exception {
        return client.get(tableNameInBytes, tGet);
      }
    });
    Result result = ThriftUtilities.resultFromThrift(tResult);
    factory.acquirePermitInCU(convertToCU(calculateResultSize(result), 1));
    return result;
  }

  @Override
  public Result[] get(List<Get> gets) throws IOException {
    final List<TGet> tGets = ThriftUtilities.getsFromHBase(gets);
    // acquire 1 cu in advance for read, in case lack of CU
    factory.acquirePermitInCU(1);
    RetryingCaller<List<TResult>> retryingCaller = new RetryingCaller<>();
    List<TResult> results = retryingCaller.withRetries(new Callable<List<TResult>>() {
      @Override
      public List<TResult> call() throws Exception {
        return client.getMultiple(tableNameInBytes, tGets);
      }
    });
    Result[] resultArray = ThriftUtilities.resultsFromThrift(results);
    factory.acquirePermitInCU(convertToCU(calculateResultSize(resultArray), 1));
    return resultArray;
  }

  @Override
  public ResultScanner getScanner(Scan scan) throws IOException {
    return new Scanner(scan);
  }

  @Override
  public void put(Put put) throws IOException {
    final TPut tPut = ThriftUtilities.putFromHBase(put);
    factory.acquirePermitInCU(convertToCU(calculateMutationSize(put)));
    RetryingCaller<Void> retryingCaller = new RetryingCaller<>();
    retryingCaller.withRetries(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        client.put(tableNameInBytes, tPut);
        return null;
      }
    });
  }

  @Override
  public void put(List<Put> puts) throws IOException {
    final List<TPut> tPuts = ThriftUtilities.putsFromHBase(puts);
    factory.acquirePermitInCU(convertToCU(calculatePutsSize(puts)));
    RetryingCaller<Void> retryingCaller = new RetryingCaller<>();
    retryingCaller.withRetries(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        client.putMultiple(tableNameInBytes, tPuts);
        return null;
      }
    });
  }

  @Override
  public void delete(Delete delete) throws IOException {
    final TDelete tDelete = ThriftUtilities.deleteFromHBase(delete);
    factory.acquirePermitInCU(convertToCU(calculateMutationSize(delete)));
    RetryingCaller<Void> retryingCaller = new RetryingCaller<>();
    retryingCaller.withRetries(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        client.deleteSingle(tableNameInBytes, tDelete);
        return null;
      }
    });
  }

  @Override
  public void delete(List<Delete> deletes) throws IOException {
    final List<TDelete> tDeletes = ThriftUtilities.deletesFromHBase(deletes);
    factory.acquirePermitInCU(convertToCU(calculateDeletesSize(deletes)));
    RetryingCaller<Void> retryingCaller = new RetryingCaller<>();
    retryingCaller.withRetries(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        client.deleteMultiple(tableNameInBytes, tDeletes);
        return null;
      }
    });
  }

  @Override
  public boolean checkAndPut(byte[] row, byte[] family, byte[] qualifier,
      CompareFilter.CompareOp compareOp, byte[] value, Put put) throws IOException {
    RowMutations mutations = new RowMutations(put.getRow());
    mutations.add(put);
    return checkAndMutate(row, family, qualifier, compareOp, value, mutations);
  }

  @Override
  public boolean checkAndDelete(byte[] row, byte[] family, byte[] qualifier,
      CompareFilter.CompareOp compareOp, byte[] value, Delete delete) throws IOException {
    RowMutations mutations = new RowMutations(delete.getRow());
    mutations.add(delete);
    return checkAndMutate(row, family, qualifier, compareOp, value, mutations);
  }

  @Override
  public boolean checkAndMutate(final byte[] row, final byte[] family, final byte[] qualifier,
      final CompareFilter.CompareOp op, final byte[] value, final RowMutations rowMutations) throws IOException {
    final ByteBuffer valueBuffer = value == null? null : ByteBuffer.wrap(value);
    RetryingCaller<Boolean> retryingCaller = new RetryingCaller<>();
    factory.acquirePermitInCU(convertToCU(calculateMutationsSize(rowMutations.getMutations())));
    return retryingCaller.withRetries(new Callable<Boolean>() {
      @Override
      public Boolean call() throws Exception {
        return client.checkAndMutate(tableNameInBytes, ByteBuffer.wrap(row), ByteBuffer.wrap(family),
            ByteBuffer.wrap(qualifier), ThriftUtilities.compareOpFromHBase(op), valueBuffer,
            ThriftUtilities.rowMutationsFromHBase(rowMutations));
      }
    });
  }

  @Override
  public void mutateRow(RowMutations rm) throws IOException {
    final TRowMutations tRowMutations = ThriftUtilities.rowMutationsFromHBase(rm);
    factory.acquirePermitInCU(convertToCU(calculateMutationsSize(rm.getMutations())));
    RetryingCaller<Void> retryingCaller = new RetryingCaller<>();
    retryingCaller.withRetries(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        client.mutateRow(tableNameInBytes, tRowMutations);
        return null;
      }
    });
  }

  @Override
  public Result append(Append append) throws IOException {
    final TAppend tAppend = ThriftUtilities.appendFromHBase(append);
    factory.acquirePermitInCU(convertToCU(calculateMutationSize(append)));
    RetryingCaller<TResult> retryingCaller = new RetryingCaller<>();
    TResult tResult = retryingCaller.withRetries(new Callable<TResult>() {
      @Override
      public TResult call() throws Exception {
        return client.append(tableNameInBytes, tAppend);
      }
    });
    return ThriftUtilities.resultFromThrift(tResult);
  }

  @Override
  public Result increment(Increment increment) throws IOException {
    final TIncrement tIncrement = ThriftUtilities.incrementFromHBase(increment);
    factory.acquirePermitInCU(convertToCU(calculateMutationSize(increment)));
    RetryingCaller<TResult> retryingCaller = new RetryingCaller<>();
    TResult tResult = retryingCaller.withRetries(new Callable<TResult>() {
      @Override
      public TResult call() throws Exception {
        return client.increment(tableNameInBytes, tIncrement);
      }
    });
    return ThriftUtilities.resultFromThrift(tResult);
  }

  public class RetryingCaller<T> {


    protected int getRetrySleepTime(int tries) {
      return (int) ConnectionUtils.getPauseTime(pause, tries);
    }

    protected int getSleepTime(int remainingTime, int tries) {
      int sleepTime = getRetrySleepTime(tries);
      sleepTime = Math.min(sleepTime, remainingTime / 2);
      return sleepTime;
    }

    protected boolean shouldRetry(Throwable t) {
      String msg = t.getMessage();
      if (msg.contains("AllIDCFailedException") || // if there is more than one IDC, all idc will be retries before return to client
          msg.contains("OperationTimeoutException") || // Client has its own operation timeout
          msg.contains("QuotaExceededException") ||
          msg.contains("RetriesExhaustedException")) {
        return true;
      } else {
        return false;
      }
    }

    protected long getRemainingTime(int operationTimeout, long now, long startTime) {
      return Math.max(0, operationTimeout - (now - startTime));
    }

    public T withRetries(Callable<T> callable) throws IOException {
      int tries = 0;
      long startTs = System.currentTimeMillis();
      RetryContext retryContext = new RetryContext();
      AliHBaseThriftImplFactory.RETRYCONTEXTS.set(retryContext);
      try {
        for (; tries < numRetries; tries++) {
          try {
            retryContext.setRetries(tries);
            retryContext.setTimeRemaining((int)getRemainingTime(operationTimeout, System.currentTimeMillis(), startTs));
            T result = callable.call();
            return result;

          } catch (Throwable t) {
            if (!shouldRetry(t)) {
              if (t instanceof IOException) {
                throw (IOException)t;
              } else {
                throw new IOException(t);
              }
            }
            if (tries >= numRetries - 1) {
              throw new IOException("Failed after " + tries + " tries", t);
            }

            int remainingTime = (int)getRemainingTime(operationTimeout, System.currentTimeMillis(), startTs);
            if (remainingTime <= 0) {
              throw new IOException("Timeout after " + tries + " tries, operationTimeout=" + operationTimeout, t);
            } else {
              int sleepTime = getSleepTime(remainingTime, tries);
              if (sleepTime > 0) {
                Thread.sleep(sleepTime);
              }
            }
          }
        }
        throw new RuntimeException("Should not reach here");
      }  catch (InterruptedException e) {
        Thread.currentThread().interrupt();
        throw new IOException("Giving up after tries=" + tries, e);
      } finally {
        AliHBaseThriftImplFactory.RETRYCONTEXTS.set(null);
      }
    }
  }


  @Override
  public <T extends Service, R> void coprocessorService(Class<T> service, byte[] startKey,
      byte[] endKey, Batch.Call<T, R> callable, Batch.Callback<R> callback)
      throws ServiceException, Throwable {
    throw new UnsupportedOperationException("coprocessorService not supported");

  }

  @Override
  public void addExternalIndex(ExternalIndexConfig config, List<ExternalIndexField> fields)
      throws IOException {
    throw new UnsupportedOperationException("addExternalIndex not supported");

  }

  @Override
  public void alterExternalIndex(ExternalIndexConfig config) throws IOException {
    throw new UnsupportedOperationException("alterExternalIndex not supported");

  }

  @Override
  public void removeExternalIndex(List<String> fields) throws IOException {
    throw new UnsupportedOperationException("removeExternalIndex not supported");

  }

  @Override
  public void buildExternalIndex() throws IOException {
    throw new UnsupportedOperationException("buildExternalIndex not supported");

  }

  @Override
  public void cancelBuildExternalIndex() throws IOException {
    throw new UnsupportedOperationException("cancelBuildExternalIndex not supported");

  }

  @Override
  public void close() throws IOException {
    transport.close();
  }

  @Override
  public List<AliHBaseIndexDescriptor> describeIndex(TableName dataTableName) throws IOException {
    throw new UnsupportedOperationException("describeIndex not supported");
  }

  /**
   * A scanner to perform scan from thrift server
   * getScannerResults is used in this scanner
   */
  public class Scanner implements ResultScanner {
    protected TScan scan;
    protected Result lastResult = null;
    protected final Queue<Result> cache = new ArrayDeque<>();
    protected byte[] nextStartRow = null;
    int rpcNum = 0;
    private int returned = 0;

    public int getRpcNum() {
      return rpcNum;
    }

    public Scanner(Scan scan) throws IOException {
      if (scan.getCaching() <= 0) {
        scan.setCaching(defaultScannerCaching);
      } else if (scan.getCaching() == 1 && scan.isReversed()){
        // for reverse scan, we need to pass the last row to the next scanner
        // we need caching number bigger than 1
        scan.setCaching(scan.getCaching() + 1);
      }
      // tell the server that we can support empty result
      scan.setAttribute(DmlAttributeConstants.EMPTY_RESULT_ATTR, Bytes.toBytes(true));
      this.scan = ThriftUtilities.scanFromHBase(scan);
      this.scan = ThriftUtilities.scanFromHBase(scan);
    }

    @Override
    public Result[] next(int nbRows) throws IOException {
      // Collect values to be returned here
      ArrayList<Result> resultSets = new ArrayList<Result>(nbRows);
      for(int i = 0; i < nbRows; i++) {
        Result next = next();
        if (next != null) {
          resultSets.add(next);
        } else {
          break;
        }
      }
      return resultSets.toArray(new Result[resultSets.size()]);
    }

    @Override
    public Iterator<Result> iterator() {
      return new Iterator<Result>() {
        // The next RowResult, possibly pre-read
        Result next = null;

        // return true if there is another item pending, false if there isn't.
        // this method is where the actual advancing takes place, but you need
        // to call next() to consume it. hasNext() will only advance if there
        // isn't a pending next().
        @Override
        public boolean hasNext() {
          if (next == null) {
            try {
              next = Scanner.this.next();
              return next != null;
            } catch (IOException e) {
              throw new RuntimeException(e);
            }
          }
          return true;
        }

        // get the pending next item and advance the iterator. returns null if
        // there is no next item.
        @Override
        public Result next() {
          // since hasNext() does the real advancing, we call this to determine
          // if there is a next before proceeding.
          if (!hasNext()) {
            return null;
          }

          // if we get to here, then hasNext() has given us an item to return.
          // we want to return the item and then null out the next pointer, so
          // we use a temporary variable.
          Result temp = next;
          next = null;
          return temp;
        }

        @Override
        public void remove() {
          throw new UnsupportedOperationException();
        }
      };
    }

    @Override
    public Result next() throws IOException {
      while (cache.size() == 0) {
        long begin = System.currentTimeMillis();
        setupNextScanner();
        // acquire 1 cu in advance for read, in case lack of CU
        factory.acquirePermitInCU(1);
        RetryingCaller<List<TResult>> retryingCaller = new RetryingCaller<>();
        List<TResult> tResults = retryingCaller.withRetries(new Callable<List<TResult>>() {
          @Override
          public List<TResult> call() throws Exception {
            return client.getScannerResults(tableNameInBytes, scan, scan.getCaching());
          }
        });
        rpcNum++;
        TResult lastTresult = tResults.size() > 0 ? tResults.get(tResults.size() - 1) : null;
        // the last result is a empty result, we need to handle it
        if (lastTresult != null && lastTresult.isSetRow() && (
            !lastTresult.isSetColumnValues() || lastTresult.getColumnValues().isEmpty())) {
          nextStartRow = lastTresult.getRow();
          // remove the empty result
          tResults.remove(tResults.size() - 1);
          // After remove the empty result, there is no results at all, we need request again
          if (tResults.isEmpty()) {
            long timeUsed = System.currentTimeMillis() - begin;
            if (operationTimeout > 0 && timeUsed > operationTimeout) {
              throw new DoNotRetryIOException(
                  "Timeout when request table " + Bytes.toString(tableNameInBytes.array())
                      + ", operationTimeout=" + operationTimeout + " ms" + ", time used=" + timeUsed
                      + "ms, with scan=" + scan + ", executed rpc=" + rpcNum + ", returned result="
                      + returned);
            } else {
              continue;
            }
          }
        }
        Result[] results = ThriftUtilities.resultsFromThrift(tResults);
        factory.acquirePermitInCU(convertToCU(calculateResultSize(results), 1));
        boolean firstKey = true;
        for (Result result : results) {
          // If it is a reverse scan, we use the last result's key as the startkey, since there is
          // no way to construct a closet rowkey smaller than the last result
          // So when the results return, we must rule out the first result, since it has already
          // returned to user.
          if (firstKey) {
            firstKey = false;
            if (scan.isReversed() && lastResult != null) {
              if (Bytes.equals(lastResult.getRow(), result.getRow())) {
                continue;
              }
            }
          }
          cache.add(result);
          lastResult = result;
        }
        // Whether we have any results returned or not, we break here.
        // no result returned means the scan is over.
        break;

      }

      if (cache.size() > 0) {
        returned++;
        return cache.poll();
      } else {
        //scan finished
        return null;
      }
    }

    @Override
    public void close() {
    }


    public boolean renewLease() {
      throw new RuntimeException("renewLease() not supported");
    }

    public ScanMetrics getScanMetrics() {
      throw new RuntimeException("getScanMetrics() not supported");
    }

    private void setupNextScanner() {
      //if lastResult is null, it means it is not the fist scan
      if (nextStartRow != null) {
        // if the server tells us the next start key, just use it
        scan.setStartRow(nextStartRow);
        nextStartRow = null;
        lastResult = null;
      } else if (lastResult!= null) {
        byte[] lastRow = lastResult.getRow();
        if (scan.isReversed()) {
          //for reverse scan, we can't find the closet row before this row
          scan.setStartRow(lastRow);
        } else {
          scan.setStartRow(createClosestRowAfter(lastRow));
        }
      }
    }


    /**
     * Create the closest row after the specified row
     */
    protected byte[] createClosestRowAfter(byte[] row) {
      if (row == null) {
        throw new RuntimeException("The passed row is null");
      }
      return Arrays.copyOf(row, row.length + 1);
    }
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor indexDescriptor)
      throws IOException {
    throw new UnsupportedOperationException("createIndex not supported");
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor indexDescriptor,
      byte[][] splitKeys) throws IOException {
    throw new UnsupportedOperationException("createIndex not supported");
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor index,
      byte[] startKey, byte[] endKey, int numRegions) throws IOException {
    throw new UnsupportedOperationException("createIndex not supported");
  }

  @Override
  public void deleteIndex(String indexName, TableName dataTable)
      throws IOException {
    throw new UnsupportedOperationException("deleteIndex not supported");
  }

  @Override
  public void offlineIndex(String indexName, TableName dataTable)
      throws IOException {
    throw new UnsupportedOperationException("offlineIndex not supported");
  }


  @Override
  public void registerBDSCluster(String hbaseSourceName, String bdsClusterkey,
      String hbaseConnectionString, String username, String password) {
    throw new UnsupportedOperationException("registerBDSCluster not supported");
  }

  @Override
  public void registerSolrCluster(String solrSourceName, String solrConnectionStr)
      throws IOException {
    throw new UnsupportedOperationException("registerSolrCluster not supported");
  }

  @Override
  public void registerESCluster(String esSourceName, String esConnectionStr, String userName,
      String password) throws IOException {
    throw new UnsupportedOperationException("registerESCluster not supported");
  }

  @Override
  public void unregisterSolrCluster(boolean force) throws IOException {
    throw new UnsupportedOperationException("unregisterSolrCluster not supported");
  }

  @Override
  public void unregisterESCluster(boolean force) throws IOException {
    throw new UnsupportedOperationException("unregisterESCluster not supported");
  }

  @Override
  public void unregisterBDSCluster(boolean force) throws IOException {
    throw new UnsupportedOperationException("unregisterBDSCluster not supported");
  }

  @Override
  public void alterExternalIndex(String json) throws IOException {
    throw new UnsupportedOperationException("alterExternalIndex not supported");
  }

  @Override
  public String describeExternalIndex() throws IOException {
    throw new UnsupportedOperationException("describeExternalIndex not supported");
  }
}
