/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.hbase.client;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.lang.reflect.Constructor;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import javax.net.ssl.SSLException;

import com.alibaba.hbase.thrift2.generated.THBaseService;
import com.alibaba.hbase.util.RateLimiter;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.ConnectionUtils;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.http.HttpRequest;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.protocol.HttpContext;
import org.apache.thrift.protocol.TBinaryProtocol;
import org.apache.thrift.protocol.TProtocol;
import org.apache.thrift.transport.THttpClient;
import org.apache.thrift.transport.TTransport;
import org.apache.thrift.transport.TTransportException;


public class AliHBaseThriftImplFactory extends AliHBaseImplFactory {
  public static final String ALIHBASE_THRIFT_CLIENT_BUIDLER_CLASS =
      "alihbase.thrift.client.builder.class";

  public static ThreadLocal<RetryContext> RETRYCONTEXTS = new ThreadLocal<>();

  private Configuration conf;

  private ThriftClientBuilder clientBuilder;
  private String host;


  private int operationTimeout;
  private int connectTimeout;

  private RateLimiter rateLimiter;
  private int cuLimit = -1;


  public AliHBaseThriftImplFactory(Configuration conf) throws IOException {
    this.conf = conf;
    this.host = conf.get(AliHBaseConstants.ALIHBASE_SERVER_NAME);
    if (host == null) {
      throw new RuntimeException(AliHBaseConstants.ALIHBASE_SERVER_NAME + " is not set");
    }
    host = host.trim();
    if (!host.startsWith("http://") && !host.startsWith("https://")) {
      throw new RuntimeException(
          AliHBaseConstants.ALIHBASE_SERVER_NAME + " should start with http:// or https://");
    }
    String hostOnly = host.split("//")[1];
    int port = -1;
    if (!hostOnly.contains(":")) {
      this.host = this.host + ":" + AliHBaseConstants.DEFAULT_THRIFT_PORT;
      port = AliHBaseConstants.DEFAULT_THRIFT_PORT;
    } else {
      port = Integer.valueOf(hostOnly.split(":")[1]);
    }

    // instance id is not set
    if (conf.get(AliHBaseConstants.INSTANCEID) == null || conf.get(AliHBaseConstants.INSTANCEID).length() == 0) {
      String [] parts = hostOnly.split("-");
      String instanceid = parts[0].trim() + "-" + parts[1].trim();
      conf.set(AliHBaseConstants.INSTANCEID, instanceid);
    }
    // not in serverless mode
    if (port == AliHBaseConstants.DEFAULT_THRIFT_PORT && conf.get(ALIHBASE_THRIFT_CLIENT_BUIDLER_CLASS) == null) {
      conf.set(ALIHBASE_THRIFT_CLIENT_BUIDLER_CLASS, AliHBaseThriftImplFactory.HTTPThriftClientBuilder.class.getName());
    }

    this.operationTimeout = conf.getInt(HConstants.HBASE_CLIENT_OPERATION_TIMEOUT,
        HConstants.DEFAULT_HBASE_CLIENT_OPERATION_TIMEOUT);
    this.connectTimeout = conf.getInt(AliHBaseConstants.SOCKET_TIMEOUT_CONNECT, AliHBaseConstants.DEFAULT_SOCKET_TIMEOUT_CONNECT);

    String className = conf.get(ALIHBASE_THRIFT_CLIENT_BUIDLER_CLASS,
        AliHBaseThriftClientBuilder.class.getName());
    try {
      Class<?> clazz = Class.forName(className);
      Constructor<?> constructor = clazz
          .getDeclaredConstructor(AliHBaseThriftImplFactory.class);
      constructor.setAccessible(true);
      clientBuilder = (ThriftClientBuilder) constructor.newInstance(this);
    }catch (Exception e) {
      throw new IOException(e);
    }

    this.cuLimit = conf.getInt(AliHBaseConstants.HBASE_CU_LIMIT, AliHBaseConstants.DEFAULT_HBASEUE_CU_LIMIT);
    setCULimit(cuLimit);
  }

  /**
   * Set CULimit to enable qps limiter. we can limit the request
   * limit sent to HBase, in case of Quota Exceed Exception.
   * @param cuLimit
   */
  @Override
  public synchronized void setCULimit(int cuLimit) {
    this.cuLimit = cuLimit;
    if (cuLimit > 0) {
      if (rateLimiter == null) {
        rateLimiter = RateLimiter.create(cuLimit);
      } else {
        rateLimiter.setRate(cuLimit);
      }
    }
  }

  public void acquirePermitInCU(int cu) {
    if (rateLimiter == null || cuLimit < 0 || cu <= 0) {
      return;
    }
    rateLimiter.acquire(cu);
  }

  public ThriftClientBuilder getClientBuilder() {
    return clientBuilder;
  }

  public String getHost() {
    return host;
  }



  /**
   * the default thrift http client builder. ONLY FOR TEST.
   * One can extend the ThriftClientBuilder to builder custom http client, implement
   * features like authentication or 'DoAs'(hbase-examples/thrift/HttpDoAsClient)
   *
   */
  public static class HTTPThriftClientBuilder extends ThriftClientBuilder {
    Map<String,String> customHeader = new HashMap<>();

    public HTTPThriftClientBuilder(AliHBaseThriftImplFactory factory) {
      super(factory);
    }

    public void addCostumHeader(String key, String value) {
      customHeader.put(key, value);
    }

    @Override
    public Pair<THBaseService.Client, TTransport> getClient() throws IOException {
      String url = factory.getHost();
      try {
        THttpClient httpClient = new THttpClient(url, factory.getHttpClient());
        for (Map.Entry<String, String> header : customHeader.entrySet()) {
          httpClient.setCustomHeader(header.getKey(), header.getValue());
        }
        if (factory.getConf().get(AliHBaseConstants.USERNAME) != null
            && factory.getConf().get(AliHBaseConstants.PASSWORD) != null) {
          String name = factory.getConf().get(AliHBaseConstants.USERNAME);
          String password = factory.getConf().get(AliHBaseConstants.PASSWORD);
          httpClient.setCustomHeader(AliHBaseThriftClientBuilder.ACCESSKEYHEADER, name);
          httpClient.setCustomHeader(AliHBaseThriftClientBuilder.ACCESSSIGNATUREHEADER, password);
        }
        httpClient.open();
        TProtocol prot = new TBinaryProtocol(httpClient);
        THBaseService.Client client = new THBaseService.Client(prot);
        return new Pair<THBaseService.Client, TTransport>(client, httpClient);
      } catch (TTransportException e) {
        throw new IOException(e);
      }
    }
  }

  public static class DelayRetryHandler extends DefaultHttpRequestRetryHandler {
    private long pause;

    public DelayRetryHandler(int retryCount, long pause) {
      super(retryCount, true, Arrays.asList(
          InterruptedIOException.class,
          UnknownHostException.class,
          SSLException.class));
      this.pause = pause;
    }

    @Override
    public boolean retryRequest(IOException exception, int executionCount, HttpContext context) {
      // Don't sleep for retrying the first time
      if (executionCount > 1 && pause > 0) {
        try {
          long sleepTime = ConnectionUtils.getPauseTime(pause, executionCount - 1);
          Thread.sleep(sleepTime);
        } catch (InterruptedException ie) {
          //reset interrupt marker
          Thread.currentThread().interrupt();
        }
      }
      return super.retryRequest(exception, executionCount, context);
    }

    @Override
    protected boolean handleAsIdempotent(HttpRequest request) {
      return true;
    }
  }

  public synchronized HttpClient getHttpClient() {
    int retry = conf.getInt(HConstants.HBASE_CLIENT_RETRIES_NUMBER,
        HConstants.DEFAULT_HBASE_CLIENT_RETRIES_NUMBER);
    long pause = conf.getLong(HConstants.HBASE_CLIENT_PAUSE, 5);
    HttpClientBuilder builder = HttpClientBuilder.create();
    RequestConfig.Builder requestBuilder = RequestConfig.custom();
    requestBuilder = requestBuilder.setConnectTimeout(connectTimeout);
    requestBuilder = requestBuilder.setSocketTimeout(operationTimeout);
    builder.setRetryHandler(new AliHBaseThriftImplFactory.DelayRetryHandler(retry, pause));
    builder.setDefaultRequestConfig(requestBuilder.build());
    return builder.build();
  }


  @Override
  public AliHBaseAPIProxy getHBaseAPIProxyImpl(TableName tableName) throws IOException {
    Pair<THBaseService.Client, TTransport> client = clientBuilder.getClient();
    return new AliHBaseAPIProxyThriftImpl(conf, client.getFirst(), client.getSecond(), tableName, this);
  }

  @Override
  public void close() throws IOException {
  }

  @Override
  public Configuration getConf() {
    return conf;
  }
}
