/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.hbase.client;

import java.io.IOException;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;

import com.alibaba.lindorm.client.AdminService;
import com.alibaba.lindorm.client.LindormClientConfig;
import com.alibaba.lindorm.client.LindormClientConstants;
import com.alibaba.lindorm.client.LindormServiceProvider;
import com.alibaba.lindorm.client.SystemService;
import com.alibaba.lindorm.client.TableService;
import com.alibaba.lindorm.client.WideColumnService;
import com.alibaba.lindorm.client.core.utils.ClientEnvLogUtil;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.AliHBaseUEAdmin;
import org.apache.hadoop.hbase.client.AliHBaseUETable;
import org.apache.hadoop.hbase.client.BufferedMutator;
import org.apache.hadoop.hbase.client.BufferedMutatorParams;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.HTableInterface;
import org.apache.hadoop.hbase.client.NoopRegistry;
import org.apache.hadoop.hbase.client.RegionLocator;
import org.apache.hadoop.hbase.security.User;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AliHBaseUEConnection implements Connection {
  private static final Logger LOG = LoggerFactory.getLogger(AliHBaseUEConnection.class);


  private Configuration conf;


  private boolean isClosed = false;
  private boolean isAborted = false;

  private AliHBaseImplFactory factory;


  private int operationTimeout;


  // For HBase-1.x
  public AliHBaseUEConnection(Configuration conf, boolean managed, ExecutorService pool,
      final User user) throws IOException {
    this(conf, pool, user);
  }

  public AliHBaseUEConnection(Configuration conf, ExecutorService pool, final User user) throws IOException {
    this.conf = new Configuration(conf);
    // disable RegionSizeCalculator which is not supported when using MR
    this.conf.setBoolean("hbase.regionsizecalculator.enable", false);
    //disable log redirect
    this.conf.setBoolean("lindorm.client.redirectlog", false);
    // use for compatibility client, we need to stop create AliHBaseUTTable/Admin inside
    // HTable or HBaseAdmin if it is created by AliHBaseUTTable/Admin itself.
    this.conf.setBoolean(AliHBaseConstants.STOP_FAKE_HBASE_CLIENT, true);
    this.conf.set("hbase.client.registry.impl", NoopRegistry.class.getName());
    this.operationTimeout = conf.getInt(HConstants.HBASE_CLIENT_OPERATION_TIMEOUT,
        HConstants.DEFAULT_HBASE_CLIENT_OPERATION_TIMEOUT);
    this.factory = AliHBaseImplFactory.create(conf);
  }

  /**
   * Only meaningful in Thrift impl
   * @param cuLimit
   */
  public void setCULimit(int cuLimit) {
    this.factory.setCULimit(cuLimit);
  }

  public AliHBaseAPIProxy getAPIProxy(TableName tableName) throws IOException {
    return factory.getHBaseAPIProxyImpl(tableName);
  }


  @Override
  public Configuration getConfiguration() {
    return conf;
  }

  @Override
  public BufferedMutator getBufferedMutator(TableName tableName) throws IOException {
    return new AliHBaseUEBufferedMutator(AliHBaseUEConnection.this, tableName);
  }

  @Override
  public BufferedMutator getBufferedMutator(BufferedMutatorParams params)
      throws IOException {
    if (params.getTableName() == null) {
      throw new IllegalArgumentException("TableName cannot be null.");
    } else {
      return new AliHBaseUEBufferedMutator(AliHBaseUEConnection.this, params.getTableName());
    }
  }

  @Override
  public RegionLocator getRegionLocator(TableName tableName) throws IOException {
    return new AliHBaseUETable(tableName, AliHBaseUEConnection.this);
  }

  @Override
  public Admin getAdmin() throws IOException {
    return new AliHBaseUEAdmin(this);
  }

  @Override
  public synchronized void close() throws IOException {
    if (isClosed) {
      return;
    }
    this.factory.close();
    this.isClosed = true;
  }

  @Override
  public boolean isClosed() {
    return isClosed;
  }

  @Override
  public HTableInterface getTable(TableName tableName) throws IOException {
    return new AliHBaseUETable(tableName, AliHBaseUEConnection.this);
  }

  public HTableInterface getTable(String tableName) throws IOException {
    return getTable(TableName.valueOf(tableName));
  }

  @Override
  public HTableInterface getTable(TableName tableName, ExecutorService executorService) throws IOException {
    return getTable(tableName);
  }

  @Override
  public void abort(String s, Throwable throwable) {
    isAborted = true;
    try {
      close();
    } catch (IOException ioE) {
      LOG.error("Error happened when aborting", ioE);
    }
  }

  @Override
  public boolean isAborted() {
    return isAborted;
  }

  public int getOperationTimeout() {
    return operationTimeout;
  }

  public AliHBaseImplFactory getFactory() {
    return factory;
  }
}
