package com.alibaba.hbase.haclient;

import static com.alibaba.hbase.client.AliHBaseConstants.DEFAULT_DUALSERVICE_TRACE_ENABLE;
import static com.alibaba.hbase.client.AliHBaseConstants.DUALSERVICE_TRACE_ENABLE;
import static com.alibaba.hbase.haclient.ConnectInfoUtil.LINK_RETRY_COUNT;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.haclient.dualservice.AutoSwitch;
import com.alibaba.hbase.haclient.dualservice.DualConfigTracker;
import com.alibaba.hbase.haclient.dualservice.DualExecutor;
import com.alibaba.hbase.haclient.dualservice.DualTrace;
import com.alibaba.hbase.haclient.dualservice.DualUtil;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.client.AliHBaseMultiClusterConnection;
import org.apache.hadoop.hbase.security.User;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;
import org.apache.hadoop.hbase.zookeeper.ZooKeeperWatcher;
import org.apache.zookeeper.CreateMode;

public class AliHBaseMultiClusterConnectionImpl extends AliHBaseMultiClusterConnection implements Switchable {
  private static final Log LOG = LogFactory
    .getLog(AliHBaseMultiClusterConnectionImpl.class);
  private ClusterSwitchTracker masterClusterSwitchTracker;
  private List<ClusterSwitchTracker> slaveClusterSwitchTrackers = new ArrayList<>();
  private volatile SwitchCommand currentSwitchCommand = null;
  private final Configuration originalConf;
  private final String masterClusterKey;
  private volatile String currentClusterKey = null;
  private volatile ConnectInfo connectInfo = null;
  private ZooKeeperWatcher linkZooKeeper = null;
  private Lock linkZookeeperLock = new ReentrantLock();
  private DualConfigTracker dualConfigTracker;

  protected AliHBaseMultiClusterConnectionImpl(Configuration conf,
                                       ExecutorService pool, User user) throws IOException {
    this(conf, false, pool, user);
  }

  protected AliHBaseMultiClusterConnectionImpl(Configuration conf, boolean managed)
    throws IOException {
    this(conf, managed, null, null);

  }

  public AliHBaseMultiClusterConnectionImpl(Configuration conf, boolean managed,
                                            ExecutorService pool, User user) throws IOException {
    super(conf, managed, pool, user);
    originalConf = new Configuration(conf);
    //get ha endpoint
    String endpoint = originalConf.get(AliHBaseConstants.ALIHBASE_SERVER_NAME);
    if(StringUtils.isBlank(endpoint)){
      endpoint = originalConf.get(HConstants.ZOOKEEPER_QUORUM);
      if(StringUtils.isBlank(endpoint)) {
        throw new IOException("hbase ha address can not be blank");
      }
    }
    if(StringUtils.isBlank(AliHBaseConstants.getHaClusterID(originalConf))){
      throw new IOException("haclient cluster id can not be blank");
    }
    try{
      this.connectInfo = ConnectInfoUtil.getConnectInfoFromZK(endpoint, conf);
      ConnectInfoUtil.flushConnectInfo(this.connectInfo, conf);
    }catch(Exception e){
      LOG.warn("Get connect info from endpoint failed, " + e);
      this.connectInfo = ConnectInfoUtil.getConnectInfoFromXML(conf);
    }
    if(null == this.connectInfo){
      throw new IOException("Get connect info failed.");
    }
    masterClusterKey = this.connectInfo.getMasterWatchZK();
    // We only need wait for one cluster.
    CountDownLatch countDownLatch = new CountDownLatch(1);
    try {
      Configuration masterConf = ClusterSwitchUtil.createConfWithConnectKey(masterClusterKey, conf);
      masterClusterSwitchTracker = new ClusterSwitchTracker(masterConf, this, countDownLatch);
      List<String> slaveClusterKeys = this.connectInfo.getSlaveWatchZKList();
      if (!slaveClusterKeys.isEmpty()) {
        //Let master cluster to connect a little time ahead
        Thread.sleep(10);
        for (String slaveKey : slaveClusterKeys) {
          Configuration slaveConf = ClusterSwitchUtil
            .createConfWithConnectKey(slaveKey, conf);
          ClusterSwitchTracker slaveTracker = new ClusterSwitchTracker(
            slaveConf, this, countDownLatch);
          slaveClusterSwitchTrackers.add(slaveTracker);
        }
      }
      countDownLatch.await();
      if(dualServiceEnable) {
        autoSwitch = new AutoSwitch(originalConf);
        dualExecutor = new DualExecutor(this.conf, autoSwitch);
        dualConfigTracker = new DualConfigTracker(ConnectInfoUtil.getConfFromEndpoint(endpoint, originalConf),
          this);
        if(conf.getBoolean(DUALSERVICE_TRACE_ENABLE, DEFAULT_DUALSERVICE_TRACE_ENABLE)){
          DualTrace.init(endpoint, conf);
        }
      }
    } catch (InterruptedException ie) {
      LOG.error("Interrupted during init", ie);
      Thread.currentThread().interrupt();
      throw new IOException(ie);
    }
  }

  private boolean isNeedToSwitch(SwitchCommand command) {
    if (currentSwitchCommand == null) {
      return true;
    }
    return command.getTs() > currentSwitchCommand.getTs();
  }

  private void onChangeCluster(Configuration activeConf, Configuration standbyConf,
                               SwitchCommand command, String targetkey)
    throws IOException {

    this.onChangeCluster(activeConf, standbyConf);
    this.currentSwitchCommand = command;
    this.currentClusterKey = targetkey;
    LOG.info("Switch successfully to " + command);
    String targetZk = ConnectInfoUtil.getZkFromConnectInfo(this.connectInfo, targetkey);
    LOG.info("Create link node to " + targetZk);
    try {
      this.startThreadForCreateLinkNode(targetZk, this.conf.getInt(LINK_RETRY_COUNT, 10));
    }catch(Exception e){
      LOG.warn("Create link node failed : " + e);
    }
    LOG.info("Create link node to " + targetZk + " success");
  }

  private void switchToActiveWithNoCommand() throws IOException{
    String activeKey = this.connectInfo.getActiveConnectKey();
    SwitchCommand command = SwitchCommand.NOCOMMAND;
    Configuration activeConf = ClusterSwitchUtil.createConfWithConnectKey(activeKey, originalConf);
    Configuration standbyConf = null;
    if(this.dualServiceEnable){
      standbyConf =
        ClusterSwitchUtil.createConfWithConnectKey(this.connectInfo.getStandbyConnectKey(), originalConf);
    }
    onChangeCluster(activeConf, standbyConf, command, activeKey);
  }

  @Override
  public synchronized void onNodeChange(byte[] bytes) {
    try {
      SwitchCommand command = ClusterSwitchUtil.toSwitchCommand(bytes);
      LOG.info("Received " + command + ", current=" + currentSwitchCommand);
      if (!isNeedToSwitch(command)) {
        LOG.info("Ignore switch command since current command is newer");
        return;
      }
      if (command.isSwitchBackToMaster()) {
        String activeKey = this.connectInfo.getActiveConnectKey();
        if(currentClusterKey != null && currentClusterKey.equals(activeKey)){
          LOG.info("Current is connect to active, skip switch");
          return;
        }
        LOG.info("From current " + currentClusterKey + ", Switch back to master cluster: " + activeKey);
        switchToActiveWithNoCommand();
        return;
      }

      if (!ClusterSwitchUtil.isValidConnectKey(command.getClusterKey())) {
        //target cluster is not vaild
        if (currentSwitchCommand != null) {
          LOG.info("Ignore switch command since invalid cluster key");
          return;
        } else {
          // if there is no cluster connected for now(happens on startup)
          // connect master cluster instead
          String activeKey = this.connectInfo.getActiveConnectKey();
          LOG.info("Invalid cluster key; From current " + currentClusterKey + ", switch back " +
            "to master cluster: " + activeKey);
          switchToActiveWithNoCommand();
          return;
        }
      } else {
        //target cluster is vaild
        try {
          String connectKey = command.getClusterKey();
          if(!ConnectInfoUtil.isVaildTargetConnectKey(this.connectInfo, connectKey)){
            LOG.info("Invalid target cluster key " + connectKey + ", is not active "
              + this.connectInfo.getConnectConf().getActive() + " or standby " +
              this.connectInfo.getConnectConf().getStandby());
            if(currentSwitchCommand == null){
              LOG.info("Current SwitchCommand is null, switch to active");
              switchToActiveWithNoCommand();
            }
            return;
          }
          if(connectKey.equals(connectKey.equals(currentClusterKey))){
            LOG.info("Target connect key " + connectKey + " is same with current cluster key " + currentClusterKey + ", skip switch");
            return;
          }
          String lastConnectKey = currentClusterKey;
          Configuration activeConf = ClusterSwitchUtil.createConfWithConnectKey(connectKey,
            originalConf);
          Configuration standbyConf = null;
          if(this.dualServiceEnable){
            if(connectKey.equals(this.connectInfo.getActiveConnectKey())){
              standbyConf =
                ClusterSwitchUtil.createConfWithConnectKey(this.connectInfo.getStandbyConnectKey(), originalConf);
            }else{
              standbyConf =
                ClusterSwitchUtil.createConfWithConnectKey(this.connectInfo.getActiveConnectKey(), originalConf);
            }
          }
          onChangeCluster(activeConf, standbyConf, command, connectKey);
          LOG.info("From current " + lastConnectKey + ",Switch to cluster: " + command.getClusterKey());
        } catch (IOException ioE) {
          if (currentSwitchCommand == null) {
            String activeKey = this.connectInfo.getActiveConnectKey();
            LOG.error(
              "Failed to switch to " + command + ", switch back to master cluster "
                + activeKey + " since no cluster is connected ever.", ioE);
            switchToActiveWithNoCommand();
          } else {
            throw ioE;
          }
        }
      }
    } catch (Throwable t) {
      LOG.error("Error happened when switching cluster, switch not finished", t);
    }
  }

  /**
   * Start a thread to create link node to active cluster
   */
  private void startThreadForCreateLinkNode(String zkCluster, int retry) {
    final String zk = zkCluster;
    final int retryCount = retry;
    if (this.linkZooKeeper != null) {
      // close old link zookeeper connection
      try {
        this.linkZooKeeper.close();
        this.linkZooKeeper = null;
      }catch(Exception e){
        LOG.warn("old link zookeeper close failed : " + e);
      }
    }
    Thread t = new Thread() {
      public void run() {
        if (!linkZookeeperLock.tryLock()) {
          return;
        }
        try {
          int count = 0;
          while (count < retryCount) {
            count++;
            try {
              if (linkZooKeeper == null) {
                Configuration linkConf = ClusterSwitchUtil.createConfWithConnectKey(zk, conf);
                linkZooKeeper = new ZooKeeperWatcher(linkConf, "Link", null,
                  false);
                String baseNode =ClusterSwitchUtil.getBaseNode(linkConf.get(AliHBaseConstants.HACLIENT_BASE_NODE,
                  AliHBaseConstants.HACLIENT_BASE_NODE_DEFAULT), AliHBaseConstants.getHaClusterID(linkConf));
                String linkNodeName = linkConf.get(ClusterSwitchUtil.ZOOKEEPER_LINK_NODE,
                  ClusterSwitchUtil.ZOOKEEPER_LINK_NODE_DEFAULT);
                String linkNode = ZKUtil.joinZNode(baseNode, linkNodeName);

                //create if basenode not exist
                ZKUtil.createNodeIfNotExistsNoWatch(linkZooKeeper, baseNode, null,
                  CreateMode.PERSISTENT);

                //create if linkNode not exist
                ZKUtil.createNodeIfNotExistsNoWatch(linkZooKeeper, linkNode, null,
                  CreateMode.PERSISTENT);

                String node = ZKUtil.joinZNode(linkNode, StrUtil.generateRandomString(5));
                boolean result = ZKUtil.createEphemeralNodeAndWatch(linkZooKeeper, node, null);
                if(!result){
                  String message = "create link node failed with " + count;
                  LOG.warn(message);
                  throw new Exception(message);
                }
              }
              LOG.info("Successfully set link node on zk '"
                + linkZooKeeper.getQuorum() + "'");
              break;
            } catch (Throwable t) {
              LOG.warn("Failed set link node on zk '"
                + linkZooKeeper.getQuorum() + "', will retry again...", t);
              Threads.sleep(30 * 1000);
            }
          }
        } finally {
          linkZookeeperLock.unlock();
        }
      }
    };
    t.setDaemon(true);
    t.setName("LinkNode-"
      + System.currentTimeMillis());
    t.start();
  }

  /**
   * Start a thread to update dual config from zk
   */
  private void startThreadForUpdateDualConfig(final String endpoint,
                                              final Configuration clientConf) {
    Thread t = new Thread() {
      public void run() {
        List<String> dualTables = null;
        try{
          dualTables = DualUtil.getDualTablesFromZK(endpoint, clientConf);
          DualUtil.flushDualTables(dualTables, clientConf);
        }catch(Exception e){
          LOG.warn("Get dual tables from endpoint failed, " + e);
          dualTables = DualUtil.getDualTablesFromXML(conf);
        }
        updateDualConfig(dualTables);
      }
    };
    t.setDaemon(true);
    t.setName("UpdateDualTable-"
      + System.currentTimeMillis());
    t.start();
  }

  public void updateDualConfig(List<String> dualTables){
    if(dualTables != null && !dualTables.isEmpty()){
      for(String dualTable : dualTables){
        if(!dualTable.contains("#")){
          enableTableDualService(dualTable);
          continue;
        }
        int glitchtimeout = AliHBaseConstants.DEFAULT_DUALSERVICE_GLITCHTIMEOUT;
        String[] confs =  dualTable.split(DualUtil.DUAL_CONFIG_SEP);
        if(confs.length != 2){
          LOG.warn("Wrong dual table conf " + dualTable);
          continue;
        }
        try{
          glitchtimeout = Integer.valueOf(confs[1]);
        }catch(Exception e){
          LOG.warn("Wrong dual table conf " + dualTable);
          continue;
        }
        if(dualTable.startsWith(DualUtil.DUAL_CONFIG_GLITCH + DualUtil.DUAL_CONFIG_SEP)){
          conf.setInt(AliHBaseConstants.DUALSERVICE_GLITCHTIMEOUT, glitchtimeout);
          continue;
        }
        if(dualTable.startsWith(DualUtil.DUAL_CONFIG_AUTOSWITCH + DualUtil.DUAL_CONFIG_SEP)){
          if(autoSwitch != null){
            if(glitchtimeout == 1){
              autoSwitch.setAutoSwitchEnable(true);
            }else if(glitchtimeout == 0){
              autoSwitch.setAutoSwitchEnable(false);
            }
          }
          continue;
        }
        setTableGlitchTimeout(confs[0], glitchtimeout);
      }
      LOG.info("Successfully update dual table conf for " + StringUtils.join(dualTables, ","));
    }
  }

  private void enableTableDualService(String dualTable){
    if(conf != null){
      conf.setBoolean(DualExecutor.createTableConfKey(dualTable,
        AliHBaseConstants.DUALSERVICE_ENABLE), true);
    }
    if(originalConf != null){
      originalConf.setBoolean(DualExecutor.createTableConfKey(dualTable,
        AliHBaseConstants.DUALSERVICE_ENABLE), true);
    }
  }

  private void setTableGlitchTimeout(String dualTable, int glitchTimeout){
    if(conf != null){
      conf.setInt(DualExecutor.createTableConfKey(dualTable,
        AliHBaseConstants.DUALSERVICE_GLITCHTIMEOUT), glitchTimeout);
    }
    if(originalConf != null){
      originalConf.setInt(DualExecutor.createTableConfKey(dualTable,
        AliHBaseConstants.DUALSERVICE_GLITCHTIMEOUT), glitchTimeout);
    }
  }

  @Override
  public void close(){
    if (this.closed) {
      return;
    }
    super.close();
    if(masterClusterSwitchTracker != null){
      masterClusterSwitchTracker.stop();
    }
    if(slaveClusterSwitchTrackers != null){
      for(ClusterSwitchTracker slaveClusterSwitchTracker : slaveClusterSwitchTrackers){
        if(slaveClusterSwitchTracker != null){
          slaveClusterSwitchTracker.stop();
        }
      }
    }
    if(dualConfigTracker != null){
      dualConfigTracker.stop();
    }
    if(this.linkZooKeeper != null){
      try {
        this.linkZooKeeper.close();
      }catch(Exception e){
        LOG.warn("link zookeeper close failed : " + e);
      }
    }
    this.closed = true;
  }

  @Override
  public Configuration getOriginalConf(){
    return this.originalConf;
  }
}