package com.alibaba.hbase.haclient;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.regex.Pattern;

import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.client.AliHBaseUEConnection;
import com.alibaba.hbase.protobuf.generated.ClusterSwitchProto;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.client.ClusterConnection;
import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.protobuf.ProtobufUtil;

public class ClusterSwitchUtil {

  public static String ZOOKEEPER_REDIRECT_NODE = "hbase.haclient.redirect.node";

  public static String ZOOKEEPER_CONNECT_NODE = "hbase.haclient.connect.node";

  public static String ZOOKEEPER_LINK_NODE = "hbase.haclient.link.node";

  public static String ZOOKEEPER_CREATE_NODE = "hbase.haclient.create.node";

  public static String ZOOKEEPER_MODIFY_NODE = "hbase.haclient.modify.node";

  public static String ZOOKEEPER_DUAL_TABLE_NODE = "hbase.haclient.dual.table.node";

  public static String ZOOKEEPER_DUAL_TRACE_NODE = "hbase.haclient.dual.trace.node";

  // Cluster key format: zk1,zk2:port:/znode
  public static final Pattern CLUSTER_KEY_PATTERN = Pattern.compile("^[^,]+(,[^,]+)*:[0-9]+:/.+$");

  // Endpoint format: hostname:port
  public static final Pattern ENDPOINT_PATTERN = Pattern.compile("^[0-9a-zA-Z\\.\\-]+:[0-9]+");

  //Start will a dot, prevent possible same name with HBase
  public static String ZOOKEEPER_REDIRECT_NODE_DEFAULT = ".redirect";

  //Start will a dot, prevent possible same name with HBase
  public static String ZOOKEEPER_CONNECT_NODE_DEFAULT = ".connect";

  //Start will a dot, prevent possible same name with HBase
  public static String ZOOKEEPER_LINK_NODE_DEFAULT = ".link";

  //Start will a dot, prevent possible same name with HBase
  public static String ZOOKEEPER_CREATE_NODE_DEFAULT = ".create";

  //Start will a dot, prevent possible same name with HBase
  public static String ZOOKEEPER_MODIFY_NODE_DEFAULT = ".modify";

  public static String ZOOKEEPER_DUAL_TABLE_NODE_DEFAULT = ".dualtable";

  public static String ZOOKEEPER_DUAL_TRACE_NODE_DEFAULT = ".dualtrace";

  public static String CONF_SEPARATOR = ".";

  // The Quorum for the ZK cluster can have one the following format (see examples below):
  // (1). s1,s2,s3 (no client port in the list, the client port could be obtained from clientPort)
  // (2). s1:p1,s2:p2,s3:p3 (with client port, which could be same or different for each server,
  //      in this case, the clientPort would be ignored)
  // (3). s1:p1,s2,s3:p3 (mix of (1) and (2) - if port is not specified in a server, it would use
  //      the clientPort; otherwise, it would use the specified port)
  public static class ZKClusterKey {
    public String quorumString;
    public int clientPort;
    public String znodeParent;

    ZKClusterKey(String quorumString, int clientPort, String znodeParent) {
      this.quorumString = quorumString;
      this.clientPort = clientPort;
      this.znodeParent = znodeParent;
    }
  }

  /**
   * Build the ZK quorum server string with "server:clientport" list, separated by ','
   *
   * @param serverHosts a list of servers for ZK quorum
   * @param clientPort the default client port
   * @return the string for a list of "server:port" separated by ","
   */
  public static String buildQuorumServerString(String[] serverHosts, String clientPort) {
    StringBuilder quorumStringBuilder = new StringBuilder();
    String serverHost;
    for (int i = 0; i < serverHosts.length; ++i) {
      if (serverHosts[i].contains(":")) {
        serverHost = serverHosts[i]; // just use the port specified from the input
      } else {
        serverHost = serverHosts[i] + ":" + clientPort;
      }
      if (i > 0) {
        quorumStringBuilder.append(',');
      }
      quorumStringBuilder.append(serverHost);
    }
    return quorumStringBuilder.toString();
  }

  /**
   * Separate the given key into the three configurations it should contain:
   * hbase.zookeeper.quorum, hbase.zookeeper.client.port
   * and zookeeper.znode.parent
   * @param key
   * @return the three configuration in the described order
   * @throws IOException
   */
  public static ZKClusterKey transformClusterKey(String key) throws IOException {
    String[] parts = key.split(":");

    if (parts.length == 3) {
      return new ZKClusterKey(parts [0], Integer.parseInt(parts [1]), parts [2]);
    }

    if (parts.length > 3) {
      // The quorum could contain client port in server:clientport format, try to transform more.
      String zNodeParent = parts [parts.length - 1];
      String clientPort = parts [parts.length - 2];

      // The first part length is the total length minus the lengths of other parts and minus 2 ":"
      int endQuorumIndex = key.length() - zNodeParent.length() - clientPort.length() - 2;
      String quorumStringInput = key.substring(0, endQuorumIndex);
      String[] serverHosts = quorumStringInput.split(",");

      // The common case is that every server has its own client port specified - this means
      // that (total parts - the ZNodeParent part - the ClientPort part) is equal to
      // (the number of "," + 1) - "+ 1" because the last server has no ",".
      if ((parts.length - 2) == (serverHosts.length + 1)) {
        return new ZKClusterKey(quorumStringInput, Integer.parseInt(clientPort), zNodeParent);
      }

      // For the uncommon case that some servers has no port specified, we need to build the
      // server:clientport list using default client port for servers without specified port.
      return new ZKClusterKey(
        buildQuorumServerString(serverHosts, clientPort),
        Integer.parseInt(clientPort),
        zNodeParent);
    }

    throw new IOException("Cluster key passed " + key + " is invalid, the format should be:" +
      HConstants.ZOOKEEPER_QUORUM + ":" + HConstants.ZOOKEEPER_CLIENT_PORT + ":"
      + HConstants.ZOOKEEPER_ZNODE_PARENT);
  }

  /**
   * Apply the settings in the given key to the given configuration, this is
   * used to communicate with distant clusters
   * @param conf configuration object to configure
   * @param key string that contains the 3 required configuratins
   * @throws IOException
   */
  public static void applyClusterKeyToConf(Configuration conf, String key)
    throws IOException{
    ZKClusterKey zkClusterKey = transformClusterKey(key);
    conf.set(HConstants.ZOOKEEPER_QUORUM, zkClusterKey.quorumString);
    conf.setInt(HConstants.ZOOKEEPER_CLIENT_PORT, zkClusterKey.clientPort);
    conf.set(HConstants.ZOOKEEPER_ZNODE_PARENT, zkClusterKey.znodeParent);
  }


  /**
   * Get the key to the ZK ensemble for this configuration and append
   * a name at the end
   * @param conf Configuration to use to build the key
   * @return ensemble key with a name (if any)
   */
  public static String getZooKeeperClusterKey(Configuration conf) {
    String ensemble = conf.get(HConstants.ZOOKEEPER_QUORUM).trim();
    StringBuilder builder = new StringBuilder(ensemble);
    builder.append(":");
    builder.append(conf.get(HConstants.ZOOKEEPER_CLIENT_PORT));
    builder.append(":");
    builder.append(conf.get(HConstants.ZOOKEEPER_ZNODE_PARENT));
    return builder.toString();
  }


  /**
   * Get the connect key
   * @param conf Configuration to use to build the key
   * @return ensemble key with a name (if any)
   */
  public static String getConnectKey(Configuration conf) {
    AliHBaseConstants.ClusterType clusterType  =
      AliHBaseConstants.ClusterType.valueOf(conf.get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
    if(clusterType == AliHBaseConstants.ClusterType.HBASEUE){
      return conf.get(AliHBaseConstants.ALIHBASE_SERVER_NAME);
    }else{
      return getZooKeeperClusterKey(conf);
    }
  }

  /**
   * Return a map, the key is the index of the slave clusters, value is the actual
   * conf key for this slave clusters
   * @param conf
   * @return
   */
  public static Map<Integer, String> parseSlaveClustersNames(Configuration conf)
    throws IOException {
    try {
      Map<Integer, String> slaveClusters = new HashMap<>();
      Iterator<Map.Entry<String, String>> iter = conf.iterator();
      String slavePrefix = HConstants.ZOOKEEPER_QUORUM + CONF_SEPARATOR;
      Map.Entry<String, String> entry = null;
      while (iter.hasNext()) {
        entry = iter.next();
        if (entry.getKey().startsWith(slavePrefix)) {
          String confKey = entry.getKey();
          String indexStr = confKey.substring(confKey.lastIndexOf(CONF_SEPARATOR) + 1);
          Integer index = Integer.valueOf(indexStr);
          slaveClusters.put(index, confKey);
        }
      }
      return slaveClusters;
    } catch (NumberFormatException e) {
      throw new IOException(e);
    }
  }

  public static Configuration createConfForSalveCluster(int index,
                                                        Configuration conf) throws IOException {
    String quorumKey = HConstants.ZOOKEEPER_QUORUM + CONF_SEPARATOR + index;
    String quorum = conf.get(quorumKey);
    if (quorum == null) {
      throw new IOException("Cluster Conf for slave index " + index + " does not exist");
    }
    Configuration newConf = new Configuration(conf);
    newConf.set(HConstants.ZOOKEEPER_QUORUM, quorum);
    String ClientPortKey = HConstants.ZOOKEEPER_CLIENT_PORT + CONF_SEPARATOR + index;
    if (conf.get(ClientPortKey) != null) {
      newConf.set(HConstants.ZOOKEEPER_CLIENT_PORT, conf.get(ClientPortKey));
    }
    String baseNodeKey = HConstants.ZOOKEEPER_ZNODE_PARENT + CONF_SEPARATOR + index;
    if (conf.get(baseNodeKey) != null) {
      newConf.set(HConstants.ZOOKEEPER_ZNODE_PARENT, conf.get(baseNodeKey));
    }
    return newConf;

  }

  public static Configuration createConfWithConnectKey(String connectKey, Configuration conf) throws IOException {
    Configuration newConf = null;
    if(conf != null){
      newConf = new Configuration(conf);
    }else{
      newConf = new Configuration();
    }
    if(isValidClusterKey(connectKey)) {
      ZKClusterKey zkClusterKey = transformClusterKey(connectKey);
      newConf.set(HConstants.ZOOKEEPER_QUORUM, zkClusterKey.quorumString);
      newConf.set(HConstants.ZOOKEEPER_CLIENT_PORT, Integer.toString(zkClusterKey.clientPort));
      newConf.set(HConstants.ZOOKEEPER_ZNODE_PARENT, zkClusterKey.znodeParent);
      newConf.set(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE, AliHBaseConstants.ClusterType.HBASE.toString());
      newConf.set(ClusterConnection.HBASE_CLIENT_CONNECTION_IMPL,
        "org.apache.hadoop.hbase.client.ConnectionManager.HConnectionImplementation");
    }else if(isValidEndpoint(connectKey)){
      newConf.set(AliHBaseConstants.ALIHBASE_SERVER_NAME, connectKey.trim());
      newConf.set(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE, AliHBaseConstants.ClusterType.HBASEUE.toString());
      newConf.set(ClusterConnection.HBASE_CLIENT_CONNECTION_IMPL,
        AliHBaseUEConnection.class.getName());
    }
    return newConf;
  }

  public static byte[] toSwitchCommandBytes(String clusterKey, long ts) {
    ClusterSwitchProto.SwitchCommand.Builder builder = ClusterSwitchProto.SwitchCommand
      .newBuilder();
    return ProtobufUtil.prependPBMagic(builder.setTargetClusterKey(clusterKey).setTs(ts).build()
      .toByteArray());
  }

  public static SwitchCommand toSwitchCommand(byte[] data) throws IOException {
    if (data == null) {
      return SwitchCommand.NOCOMMAND;
    }
    try {
      ProtobufUtil.expectPBMagicPrefix(data);
      int prefixLen = ProtobufUtil.lengthOfPBMagic();
      ClusterSwitchProto.SwitchCommand.Builder builder =
        ClusterSwitchProto.SwitchCommand.newBuilder();
      ProtobufUtil.mergeFrom(builder, data, prefixLen, data.length - prefixLen);
      ClusterSwitchProto.SwitchCommand switchProto = builder.build();
      return new SwitchCommand(switchProto.getTargetClusterKey(), switchProto.getTs());
    } catch (DeserializationException e) {
      throw new IOException(e);
    }
  }

  /**
   * Check whether the given cluster key is a valid format: zk1,zk2:port:/znode
   * @param clusterKey
   * @return true if is a valid format of cluster key
   */
  public static boolean isValidClusterKey(String clusterKey) {
    return CLUSTER_KEY_PATTERN.matcher(clusterKey).matches();
  }

  /**
   * Check whether the given endpoint is a valid format: hostname:port
   * @param endpoint
   * @return true if is a valid format of cluster key
   */
  public static boolean isValidEndpoint(String endpoint) {
    return ENDPOINT_PATTERN.matcher(endpoint).matches();
  }


  /**
   * Check whether the given string is a valid endpoint or cluster key
   * @param key
   * @return true if is a valid format of key
   */
  public static boolean isValidConnectKey(String key) {
    return isValidClusterKey(key) || isValidEndpoint(key);
  }

  /**
   * get different base node for different hacluster
   * @param baseNode
   * @return String
   */
  public static String getBaseNode(String baseNode, String hacluster){
    return baseNode + "_" + hacluster;
  }

}