package com.alibaba.hbase.haclient.dualservice;

import java.io.IOException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.ipc.CallTimeoutException;
import org.apache.hadoop.hbase.util.Bytes;

public class DualContext<T> {

  private static final Log LOG = LogFactory.getLog(DualContext.class);

  private volatile T activeResult;
  private volatile T standbyResult;
  private Throwable activeError;
  private Throwable standbyError;
  private CountDownLatch countDown = new CountDownLatch(1);
  private final int glitchTimeout;
  private final int operationTimeout;
  private byte[] tableName;
  private long start;
  // used to identify success of put operation which return "null"
  private volatile boolean hasValue = false;
  private Boolean usePrimaryAsResult = null;
  private byte[] key;
  private HRegionLocation location;
  private DualExecutor.ActionType actionType;
  private AutoSwitch.ExecuteStrategy executeStrategy = AutoSwitch.ExecuteStrategy.DEFAULT;
  private volatile boolean activeReturn = false;
  private volatile boolean standbyReturn = false;

  public DualContext(final byte[] tableName, int glitchTimeout,
                     int operationTimeout) {
    this.glitchTimeout = glitchTimeout;
    this.operationTimeout = operationTimeout;
    this.tableName = tableName;
  }

  public DualContext(final byte[] tableName, int glitchTimeout, int operationTimeout, byte[] key,
                     HRegionLocation location, DualExecutor.ActionType actionType){
    this.glitchTimeout = glitchTimeout;
    this.operationTimeout = operationTimeout;
    this.tableName = tableName;
    this.key = key;
    this.location = location;
    this.actionType = actionType;
  }

  public void onActiveComplete(T result) {
    activeResult = result;
    hasValue = true;
    activeReturn = true;
    countDown.countDown();
  }

  public void onStandbyComplete(T result) {
    standbyResult = result;
    hasValue = true;
    standbyReturn = true;
    countDown.countDown();
  }

  public T getActiveResult(){
    return this.activeResult;
  }

  public T getStandbyResult(){
    return this.standbyResult;
  }

  public synchronized void onActiveError(Throwable error) {
    activeError = error;
    if(standbyError != null) {
      countDown.countDown();
    }
  }

  public synchronized void onStandbyError(Throwable error) {
    standbyError = error;
    if(activeError != null) {
      countDown.countDown();
    }
  }

  public T getResult() throws IOException {
    if (hasValue) {
      if (activeResult != null) {
        usePrimaryAsResult = true;
        return activeResult;
      }
      if (standbyResult != null) {
        usePrimaryAsResult = false;
        return standbyResult;
      }
      // in case result itself is null;
      return null;
    }
    if (activeError == null) {
      activeError = new CallTimeoutException("OperationTimeout");
    }
    if (standbyError == null) {
      standbyError = new CallTimeoutException("OperationTimeout");
    }
    // error handling
    long now = System.currentTimeMillis();
    String msg = "Failed exectuing dual operation for table "
      + Bytes.toString(tableName) + " with active error=" + activeError.getMessage()
      + " and with standby error=" + standbyError.getMessage() + ", timeout=" + operationTimeout + ", wait=" + (now-start);
    throw new IOException(msg, activeError);
  }

  public T getResultInGlitchTimeout() throws IOException {
    try {
      if (countDown.await(this.glitchTimeout, TimeUnit.MILLISECONDS)) {
        if(hasValue && activeReturn){
          this.usePrimaryAsResult = true;
          return this.activeResult;
        }else if(hasValue && standbyReturn){
          return this.standbyResult;
        }else{
          LOG.debug("get result in glitch timeout, return null");
          return null;
        }
      }
    } catch (InterruptedException e) {
      throw new IOException(e);
    }
    return null;
  }

  public void waitOperationTimeout() throws IOException {
    try {
      long remaining = 0;
      remaining = Math.max(0, operationTimeout + start - System.currentTimeMillis());
      remaining += 10;
      countDown.await(remaining, TimeUnit.MILLISECONDS);
    } catch (InterruptedException e) {
      throw new IOException(e);
    }
  }

  public int getOperationTimeout() {
    return this.operationTimeout;
  }

  public byte[] getTableName() {
    return this.tableName;
  }

  public Boolean usePrimaryAsResult() {
    return hasValue && activeReturn;
  }

  public void start(){
    this.start = System.currentTimeMillis();
  }

  public long getStart(){
    return this.start;
  }

  public boolean useStandbyAsResult(){
    return hasValue && !activeReturn;
  }

  public boolean activeHasError(){
    return activeError != null;
  }

  public boolean standbyHasError(){
    return standbyError != null;
  }

  public byte[] getKey() {
    return key;
  }

  public void setKey(byte[] key) {
    this.key = key;
  }

  public HRegionLocation getLocation() {
    return location;
  }

  public void setLocations(HRegionLocation location) {
    this.location = location;
  }

  public DualExecutor.ActionType getActionType() {
    return actionType;
  }

  public AutoSwitch.ExecuteStrategy getExecuteStrategy() {
    return executeStrategy;
  }

  public void setExecuteStrategy(AutoSwitch.ExecuteStrategy executeStrategy) {
    this.executeStrategy = executeStrategy;
  }

  public String toString(){
    StringBuilder sb = new StringBuilder();
    sb.append("activeReturn : " + activeReturn);
    sb.append(", standbyReturn : " + standbyReturn);
    sb.append(", hasValue : " + hasValue);
    sb.append(", usePrimaryAsResult : " + usePrimaryAsResult);
    if(activeResult != null){
      sb.append(", activeResult not null ");
    }
    if(standbyResult != null){
      sb.append(", standbyResult not null ");
    }
    return sb.toString();
  }
}