package com.alibaba.hbase.haclient.dualservice;

import java.util.concurrent.atomic.AtomicLong;

import com.google.gson.JsonObject;
import com.yammer.metrics.stats.ExponentiallyDecayingSample;
import com.yammer.metrics.stats.Sample;
import com.yammer.metrics.stats.Snapshot;

public class DualHistogram {

  public static final String NUM_OPS_METRIC_NAME = "_num_ops";
  public static final String MIN_METRIC_NAME = "_min";
  public static final String MAX_METRIC_NAME = "_max";
  public static final String SUM_METRIC_NAME = "_sum";
  public static final String MEAN_METRIC_NAME = "_mean";
  public static final String MEDIAN_METRIC_NAME = "_median";
  public static final  String SEVENTY_FIFTH_PERCENTILE_METRIC_NAME = "_75th_percentile";
  public static final String NINETY_FIFTH_PERCENTILE_METRIC_NAME = "_95th_percentile";
  public static final String NINETY_NINETH_PERCENTILE_METRIC_NAME = "_99th_percentile";
  public static final String NINETY_NINETH_NINETH_PERCENTILE_METRIC_NAME = "_999th_percentile";

  private static final int DEFAULT_SAMPLE_SIZE = 2046;
  // the bias towards sampling from more recent data.
  // Per Cormode et al. an alpha of 0.015 strongly biases to the last 5 minutes
  private static final double DEFAULT_ALPHA = 0.015;

  protected long lastUpdate = System.currentTimeMillis();

  protected final String name;
  protected final String desc;
  protected final Sample sample;
  private final AtomicLong min;
  private final AtomicLong max;
  private final AtomicLong sum;
  private final AtomicLong count;


  public DualHistogram(String name, String description) {
    this.name = name;
    this.desc = description;
    sample = new ExponentiallyDecayingSample(DEFAULT_SAMPLE_SIZE, DEFAULT_ALPHA);
    count = new AtomicLong();
    min = new AtomicLong(Long.MAX_VALUE);
    max = new AtomicLong(Long.MIN_VALUE);
    sum = new AtomicLong();
  }

  /**
   * Clears all recorded values.
   */
  public void clear() {
    this.sample.clear();
    this.count.set(0);
    this.max.set(Long.MIN_VALUE);
    this.min.set(Long.MAX_VALUE);
    this.sum.set(0);
  }

  public void clearWithOutSample() {
    this.count.set(0);
    this.max.set(Long.MIN_VALUE);
    this.min.set(Long.MAX_VALUE);
    this.sum.set(0);
  }

  public void add(final long val) {
    count.incrementAndGet();
    sample.update(val);
    setMax(val);
    setMin(val);
    sum.getAndAdd(val);
  }

  public void addCountAndSum(final int val){
    count.incrementAndGet();
    sum.getAndAdd(val);
  }


  private void setMax(final long potentialMax) {
    boolean done = false;
    while (!done) {
      final long currentMax = max.get();
      done = currentMax >= potentialMax
        || max.compareAndSet(currentMax, potentialMax);
    }
  }

  private void setMin(long potentialMin) {
    boolean done = false;
    while (!done) {
      final long currentMin = min.get();
      done = currentMin <= potentialMin
        || min.compareAndSet(currentMin, potentialMin);
    }
  }

  public long getCount() {
    return count.get();
  }

  public long getSum() {
    if (getCount() > 0) {
      return sum.get();
    }
    return 0L;
  }

  public long getMax() {
    if (count.get() > 0) {
      return max.get();
    }
    return 0L;
  }

  public long getMin() {
    if (count.get() > 0) {
      return min.get();
    }
    return 0L;
  }

  public double getMean() {
    long cCount = count.get();
    if (cCount > 0) {
      return sum.get() / (double) cCount;
    }
    return 0.0;
  }

  public JsonObject snapshot() {
    JsonObject result = new JsonObject();
    final Snapshot s = sample.getSnapshot();
    long now = System.currentTimeMillis();
    long diff = (now - lastUpdate) / 1000;
    if (diff == 0)diff = 1; // sigh this is crap.

    result.addProperty(name + NUM_OPS_METRIC_NAME, (this.getCount() * 1.0 / diff));
    result.addProperty(name + MIN_METRIC_NAME, getMin());
    result.addProperty(name + MAX_METRIC_NAME, getMax());
    result.addProperty(name + SUM_METRIC_NAME, this.getCount());
    result.addProperty(name + MEAN_METRIC_NAME, getMean());
    result.addProperty(name + MEDIAN_METRIC_NAME, s.getMedian());
    result.addProperty(name + SEVENTY_FIFTH_PERCENTILE_METRIC_NAME, s.get75thPercentile());
    result.addProperty(name + NINETY_FIFTH_PERCENTILE_METRIC_NAME, s.get95thPercentile());
    result.addProperty(name + NINETY_NINETH_PERCENTILE_METRIC_NAME, s.get99thPercentile());
    result.addProperty(name + NINETY_NINETH_NINETH_PERCENTILE_METRIC_NAME, s.get999thPercentile());
    this.clear();
    return result;
  }

  public JsonObject snapshotWithCountAndOps() {
    JsonObject result = new JsonObject();
    long now = System.currentTimeMillis();
    long diff = (now - lastUpdate) / 1000;
    if (diff == 0)diff = 1; // sigh this is crap.

    result.addProperty(name + NUM_OPS_METRIC_NAME, (this.getCount() * 1.0 / diff));
    result.addProperty(name + SUM_METRIC_NAME, getCount());
    this.clearWithOutSample();
    return result;
  }

}