package com.alibaba.hbase.haclient.dualservice;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListMap;

import com.alibaba.hbase.client.AliHBaseConstants;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.util.Bytes;

public class KeyCounter implements ExecuteCounter {

  private static final Log LOG = LogFactory.getLog(KeyCounter.class);

  private ConcurrentHashMap<String, ConcurrentSkipListMap<byte[], HRegionLocation>> keyMap;

  private ConcurrentHashMap<byte[], ResultCounter> keyCounts;

  private ConcurrentHashMap<byte[], Boolean> keySwitchStatus;

  private Configuration conf;

  private int activeFailLimit;

  private int standbySuccessLimit;

  public KeyCounter(Configuration conf){
    this.conf = conf;
    this.keyCounts = new ConcurrentHashMap<>();
    this.keyMap = new ConcurrentHashMap<>();
    this.keySwitchStatus = new ConcurrentHashMap<>();
    this.activeFailLimit = this.conf.getInt(AliHBaseConstants.KEY_ACTIVE_FAIL_LIMIT,
      AliHBaseConstants.DEFAULT_KEY_ACTIVE_FAIL_LIMIT);
    this.standbySuccessLimit = this.conf.getInt(AliHBaseConstants.KEY_STANDBY_SUCCESS_LIMIT,
      AliHBaseConstants.DEFAULT_KEY_STANDBY_SUCCESS_LIMIT);
  }

  @Override
  public synchronized <T> void update(DualContext<T> dualContext){
    boolean isSwitch = this.isSwitchStatus(dualContext);
    if(dualContext.getExecuteStrategy() == AutoSwitch.ExecuteStrategy.DEFAULT && isSwitch){
      return;
    }
    if(dualContext.getExecuteStrategy() == AutoSwitch.ExecuteStrategy.SWITCH && !isSwitch){
      return;
    }
    String tableNameAsString = TableName.valueOf(dualContext.getTableName()).getNameAsString();
    if(keyMap.containsKey(tableNameAsString)){
      ConcurrentSkipListMap<byte[], HRegionLocation> regionLocations =
        keyMap.get(tableNameAsString);
      byte[] requestKey = dualContext.getKey();
      byte[] existKey = null;
      if(requestKey.length == 0){
        existKey = regionLocations.firstKey();
      }else{
        existKey = regionLocations.floorKey(requestKey);
      }

      if(existKey == null){
        if(dualContext.getLocation() != null){
          regionLocations.put(requestKey, dualContext.getLocation());
          ResultCounter newCounter = new ResultCounter(ResultCounter.CountStrategy.SUCCESSION);
          newCounter.increment(dualContext);
          keyCounts.put(requestKey, newCounter);
          keySwitchStatus.put(requestKey, false);
        }
        return;
      }

      HRegionLocation existLocation = regionLocations.get(existKey);
      byte[] locationEndKey = existLocation.getRegionInfo().getEndKey();
      if(Bytes.BYTES_COMPARATOR.compare(requestKey, locationEndKey) < 0 || dualContext.getLocation() == null){
        keyCounts.get(existKey).increment(dualContext);
      }else{
        regionLocations.put(requestKey, dualContext.getLocation());
        ResultCounter newCounter = new ResultCounter(ResultCounter.CountStrategy.SUCCESSION);
        newCounter.increment(dualContext);
        keyCounts.put(requestKey, newCounter);
        keySwitchStatus.put(requestKey, false);
      }
    }else if(dualContext.getLocation() != null){
      ConcurrentSkipListMap<byte[], HRegionLocation> regionLocations =
        new ConcurrentSkipListMap<>(Bytes.BYTES_COMPARATOR);
      regionLocations.put(dualContext.getKey(), dualContext.getLocation());
      keyMap.put(tableNameAsString, regionLocations);

      ResultCounter newCounter = new ResultCounter(ResultCounter.CountStrategy.SUCCESSION);
      newCounter.increment(dualContext);
      keyCounts.put(dualContext.getKey(), newCounter);
      keySwitchStatus.put(dualContext.getKey(), false);
    }
  }

  @Override
  public synchronized <T> boolean isSwitchStatus(DualContext<T> dualContext) {
    String tableNameAsString = TableName.valueOf(dualContext.getTableName()).getNameAsString();
    if(keyMap.containsKey(tableNameAsString)){
      ConcurrentSkipListMap<byte[], HRegionLocation> regionLocations =
        keyMap.get(tableNameAsString);
      byte[] requestKey = dualContext.getKey();
      byte[] existKey = regionLocations.floorKey(requestKey);
      if(existKey == null){
        return false;
      }
      return keySwitchStatus.get(existKey);
    }else{
      return false;
    }
  }

  @Override
  public synchronized <T> void setSwitchStatus(DualContext<T> dualContext, boolean switchStatus) {
    String tableNameAsString = TableName.valueOf(dualContext.getTableName()).getNameAsString();
    byte[] existKey = keyMap.get(tableNameAsString).floorKey(dualContext.getKey());
    keySwitchStatus.put(existKey, switchStatus);
    this.reset(dualContext);
  }

  @Override
  public synchronized <T> boolean reachLimit(DualContext<T> dualContext){
    String tableNameAsString = TableName.valueOf(dualContext.getTableName()).getNameAsString();
    if(keyMap.containsKey(tableNameAsString)){
      byte[] existKey = keyMap.get(tableNameAsString).floorKey(dualContext.getKey());
      if(existKey == null){
        return false;
      }
      ResultCounter resultCounter = null;
      resultCounter = keyCounts.get(existKey);
      if(resultCounter == null){
        return false;
      }
      boolean result;
      if(!isSwitchStatus(dualContext)){
        result = resultCounter.getActiveFailCount() > this.activeFailLimit;
      }else{
        result = resultCounter.getStandbySuccessCount() > this.standbySuccessLimit;
      }
      if(result){
        LOG.debug("Reach limit active fail count=" + resultCounter.getActiveFailCount() + ", " +
          "standby success count=" + resultCounter.getStandbySuccessCount());
      }
      return result;
    }
    return false;
  }

  @Override
  public synchronized <T> void reset(DualContext<T> dualContext){
    String tableNameAsString = TableName.valueOf(dualContext.getTableName()).getNameAsString();
    if(keyMap.containsKey(tableNameAsString)){
      byte[] existKey = keyMap.get(tableNameAsString).floorKey(dualContext.getKey());
      if(existKey != null) {
        keyCounts.get(existKey).activeCountClear();
        keyCounts.get(existKey).standbyCountClear();
      }
    }
  }
}