/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.hbase.util;

import com.alibaba.lindorm.client.core.utils.DataTypeUtils;
import com.alibaba.lindorm.client.schema.SortOrder;
import org.apache.hadoop.hbase.client.index.AliHBaseColumn;
import org.apache.hadoop.hbase.util.Bytes;

import java.math.BigDecimal;

/**
 * Provide a set of functions to convert numeric values into/from binary-comparable form.
 * For Boolean and String, please use HBase's {@link Bytes} instead.
 *
 * For example:
 * Consider integer values 100 and -100, their binary forms are :
 *    100    ->    0x0000 0064
 *   -100    ->    0xFFFF FF9C
 * Apparently -100 is 'greater than' 100 by directly comparing to -100 in binary(similar to memcmp() in C).
 * In order to get it right we have to flip the signed bit in this scenario:
 *    100    ->    0x0000 0064    ->    0x8000 0064
 *   -100    ->    0xFFFF FF9C    ->    0x7FFF FF9C
 * By flipping the signed bit, negative values will be sorted before the positives.
 *
 * Note that this utility class has two parts:
 * 1) Basic functions to handle ordered bytes conversion, this is enough for most of the cases.
 * 2) Support user-defined sort order
 *
 * for each data type, there are 4 methods in each part, consider INT:
 * 1) convert int to byte[]
 *     {@link #toBytes(int)}
 *     {@link #putInt(byte[], int, int)}
 *
 * 2) convert byte[] to int
 *     {@link #toInt(byte[])}
 *     {@link #toInt(byte[], int)}
 */
public class OrderedBytes {
  //**************************************************************************
  // PART I: Basic functions to do ordered bytes conversion
  // Supported basic numeric types:
  //     byte
  //     short
  //     int
  //     long
  //     float
  //     double
  //     BigDecimal
  //**************************************************************************

  /**
   * Serialize a byte value into its binary-comparable byte array.
   * @param val value to be serialized.
   * @return binary-comparable byte array of the specified value
   */
  public static byte[] toBytes(byte val) {
    byte[] output = new byte[Bytes.SIZEOF_BYTE];
    DataTypeUtils.encodeByte(val, output, 0, SortOrder.getDefault());
    return output;
  }

  /**
   * Put a byte value into the specified byte array position, using its binary-comparable form.
   * @param output the output byte array
   * @param offset position in the array
   * @param val byte value to write to
   * @return incremented offset
   * @throws IllegalArgumentException if the specified byte array doesn't have enough room to hold this value
   */
  public static int putByte(byte[] output, int offset, byte val) {
    return offset + DataTypeUtils.encodeByte(val, output, offset, SortOrder.getDefault());
  }

  /**
   * Deserialize a byte value from its binary-comparable byte array.
   * @param bytes binary-comparable byte array of a byte value
   * @return deserialized value
   */
  public static byte toByte(byte[] bytes) {
    return DataTypeUtils.decodeByte(bytes, 0, SortOrder.getDefault());
  }

  /**
   * Deserialize a byte value from its binary-comparable byte array.
   * @param bytes binary-comparable byte array of a byte value
   * @param offset start position
   * @return deserialized value
   */
  public static byte toByte(byte[] bytes, int offset) {
    return DataTypeUtils.decodeByte(bytes, offset, SortOrder.getDefault());
  }


  /**
   * Serialize a short value into its binary-comparable byte array.
   * @param val value to be serialized.
   * @return binary-comparable byte array of the specified value
   */
  public static byte[] toBytes(short val) {
    byte[] output = new byte[Bytes.SIZEOF_SHORT];
    DataTypeUtils.encodeShort(val, output, 0, SortOrder.getDefault());
    return output;
  }

  /**
   * Put a short value into the specified byte array position, using its binary-comparable form.
   * @param output the output byte array
   * @param offset position in the array
   * @param val short value to write to
   * @return incremented offset
   * @throws IllegalArgumentException if the specified byte array doesn't have enough room to hold this value
   */
  public static int putShort(byte[] output, int offset, short val) {
    return offset + DataTypeUtils.encodeShort(val, output, offset, SortOrder.getDefault());
  }

  /**
   * Deserialize a short value from its binary-comparable byte array.
   * @param bytes binary-comparable byte array of a short value
   * @return deserialized value
   */
  public static short toShort(byte[] bytes) {
    return DataTypeUtils.decodeShort(bytes, 0, SortOrder.getDefault());
  }

  /**
   * Deserialize a short value from its binary-comparable byte array.
   * @param bytes binary-comparable byte array of a short value
   * @param offset start position
   * @return deserialized value
   */
  public static short toShort(byte[] bytes, int offset) {
    return DataTypeUtils.decodeShort(bytes, offset, SortOrder.getDefault());
  }


  /**
   * Serialize an integer value into its binary-comparable byte array.
   * @param val value to be serialized.
   * @return binary-comparable byte array of the specified value
   */
  public static byte[] toBytes(int val) {
    byte[] output = new byte[Bytes.SIZEOF_INT];
    DataTypeUtils.encodeInt(val, output, 0, SortOrder.getDefault());
    return output;
  }

  /**
   * Put an integer value into the specified byte array position, using its binary-comparable form.
   * @param output the output byte array
   * @param offset position in the array
   * @param val integer value to write to
   * @return incremented offset
   * @throws IllegalArgumentException if the specified byte array doesn't have enough room to hold this value
   */
  public static int putInt(byte[] output, int offset, int val) {
    return offset + DataTypeUtils.encodeInt(val, output, offset, SortOrder.getDefault());
  }

  /**
   * Deserialize an integer value from its binary-comparable byte array.
   * @param bytes binary-comparable byte array of an integer value
   * @return deserialized value
   */
  public static int toInt(byte[] bytes) {
    return DataTypeUtils.decodeInt(bytes, 0, SortOrder.getDefault());
  }

  /**
   * Deserialize an integer value from its binary-comparable byte array.
   * @param bytes binary-comparable byte array of an integer value
   * @param offset start position
   * @return deserialized value
   */
  public static int toInt(byte[] bytes, int offset) {
    return DataTypeUtils.decodeInt(bytes, offset, SortOrder.getDefault());
  }


  /**
   * Serialize a long value into its binary-comparable byte array.
   * @param val value to be serialized.
   * @return binary-comparable byte array of the specified value
   */
  public static byte[] toBytes(long val) {
    byte[] output = new byte[Bytes.SIZEOF_LONG];
    DataTypeUtils.encodeLong(val, output, 0, SortOrder.getDefault());
    return output;
  }

  /**
   * Put a long value into the specified byte array position, using its binary-comparable form.
   * @param output the output byte array
   * @param offset position in the array
   * @param val long value to write to
   * @return incremented offset
   * @throws IllegalArgumentException if the specified byte array doesn't have enough room to hold this value
   */
  public static int putLong(byte[] output, int offset, long val) {
    return offset + DataTypeUtils.encodeLong(val, output, offset, SortOrder.getDefault());
  }

  /**
   * Deserialize a long value from its binary-comparable byte array.
   * @param bytes binary-comparable byte array of a long value
   * @return deserialized value
   */
  public static long toLong(byte[] bytes) {
    return DataTypeUtils.decodeLong(bytes, 0, SortOrder.getDefault());
  }

  /**
   * Deserialize a long value from its binary-comparable byte array.
   * @param bytes binary-comparable byte array of a long value
   * @param offset start position
   * @return deserialized value
   */
  public static long toLong(byte[] bytes, int offset) {
    return DataTypeUtils.decodeLong(bytes, offset, SortOrder.getDefault());
  }


  /**
   * Serialize a float value into its binary-comparable byte array.
   * @param val value to be serialized.
   * @return binary-comparable byte array of the specified value
   */
  public static byte[] toBytes(float val) {
    byte[] output = new byte[Bytes.SIZEOF_FLOAT];
    DataTypeUtils.encodeFloat(val, output, 0, SortOrder.getDefault());
    return output;
  }

  /**
   * Put a float value into the specified byte array position, using its binary-comparable form.
   * @param output the output byte array
   * @param offset position in the array
   * @param val float value to write to
   * @return incremented offset
   * @throws IllegalArgumentException if the specified byte array doesn't have enough room to hold this value
   */
  public static int putFloat(byte[] output, int offset, float val) {
    return offset + DataTypeUtils.encodeFloat(val, output, offset, SortOrder.getDefault());
  }

  /**
   * Deserialize a float value from its binary-comparable byte array.
   * @param bytes binary-comparable byte array of a float value
   * @return deserialized value
   */
  public static float toFloat(byte[] bytes) {
    return DataTypeUtils.decodeFloat(bytes, 0, SortOrder.getDefault());
  }

  /**
   * Deserialize a float value from its binary-comparable byte array.
   * @param bytes binary-comparable byte array of a float value
   * @param offset start position
   * @return deserialized value
   */
  public static float toFloat(byte[] bytes, int offset) {
    return DataTypeUtils.decodeFloat(bytes, offset, SortOrder.getDefault());
  }


  /**
   * Serialize a double value into its binary-comparable byte array.
   * @param val value to be serialized.
   * @return binary-comparable byte array of the specified value
   */
  public static byte[] toBytes(double val) {
    byte[] output = new byte[Bytes.SIZEOF_DOUBLE];
    DataTypeUtils.encodeDouble(val, output, 0, SortOrder.getDefault());
    return output;
  }

  /**
   * Put a double value into the specified byte array position, using its binary-comparable form.
   * @param output the output byte array
   * @param offset position in the array
   * @param val double value to write to
   * @return incremented offset
   * @throws IllegalArgumentException if the specified byte array doesn't have enough room to hold this value
   */
  public static int putDouble(byte[] output, int offset, double val) {
    return offset + DataTypeUtils.encodeDouble(val, output, offset, SortOrder.getDefault());
  }

  /**
   * Deserialize a double value from its binary-comparable byte array.
   * @param bytes binary-comparable byte array of a double value
   * @return deserialized value
   */
  public static double toDouble(byte[] bytes) {
    return DataTypeUtils.decodeDouble(bytes, 0, SortOrder.getDefault());
  }

  /**
   * Deserialize a double value from its binary-comparable byte array.
   * @param bytes binary-comparable byte array of a double value
   * @param offset start position
   * @return deserialized value
   */
  public static double toDouble(byte[] bytes, int offset) {
    return DataTypeUtils.decodeDouble(bytes, offset, SortOrder.getDefault());
  }


  /**
   * Serialize a BigDecimal value into its binary-comparable byte array.
   * @param val value to be serialized.
   * @return binary-comparable byte array of the specified value
   */
  public static byte[] toBytes(BigDecimal val) {
    return DataTypeUtils.encodeDecimal(val, SortOrder.getDefault());
  }

  /**
   * Put a BigDecimal value into the specified byte array position, using its binary-comparable form.
   * Note: BigDecimal has variable length in binary format, so it's difficult to estimate the actual size and
   *       create a proper byte array in advance. So this method is rarely used.
   * @param output the output byte array
   * @param offset position in the array
   * @param val BigDecimal value to write to
   * @return incremented offset
   * @throws IllegalArgumentException if the specified byte array doesn't have enough room to hold this value
   */
  public static int putBigDecimal(byte[] output, int offset, BigDecimal val) {
    byte[] temp = DataTypeUtils.encodeDecimal(val, SortOrder.getDefault());
    System.arraycopy(temp, 0, output, offset, temp.length);
    return offset + temp.length;
  }
  /**
   * Deserialize a BigDecimal value from its binary-comparable byte array.
   * @param bytes binary-comparable byte array of a BigDecimal value
   * @return deserialized value
   */
  public static BigDecimal toBigDecimal(byte[] bytes) {
    return DataTypeUtils.decodeDecimal(bytes, 0, bytes.length, SortOrder.getDefault());
  }

  /**
   * Deserialize a BigDecimal value from its binary-comparable byte array.
   * @param bytes binary-comparable byte array of a BigDecimal value
   * @param offset start position
   * @param length number of bytes that represents this BigDecimal value
   * @return deserialized value
   */
  public static BigDecimal toBigDecimal(byte[] bytes, int offset, int length) {
    return DataTypeUtils.decodeDecimal(bytes, offset, length, SortOrder.getDefault());
  }

  //**************************************************************************
  // PART II.a: Signed values with sort order support
  // Supported basic numeric types:
  //     byte
  //     short
  //     int
  //     long
  //     float
  //     double
  //     BigDecimal
  //**************************************************************************

  /**
   * Serialize a byte value into its binary-comparable byte array with explicit sort order.
   * @param val value to be serialized.
   * @param order sort order of this value
   * @return binary-comparable byte array of the specified value
   */
  public static byte[] toBytes(byte val, AliHBaseColumn.SortOrder order) {
    byte[] output = new byte[Bytes.SIZEOF_BYTE];
    DataTypeUtils.encodeByte(val, output, 0, order.getImplSortOrder());
    return output;
  }

  /**
   * Put a byte value into the specified byte array position, using its binary-comparable form with explicit sort order.
   * @param output the output byte array
   * @param offset position in the array
   * @param val byte value to write to
   * @param order sort order of this value
   * @return incremented offset
   * @throws IllegalArgumentException if the specified byte array doesn't have enough room to hold this value
   */
  public static int putByte(byte[] output, int offset, byte val, AliHBaseColumn.SortOrder order) {
    return offset + DataTypeUtils.encodeByte(val, output, offset, order.getImplSortOrder());
  }

  /**
   * Deserialize a byte value from its binary-comparable byte array with explicit sort order.
   * @param bytes binary-comparable byte array of a byte value
   * @param order sort order of this value
   * @return deserialized value
   */
  public static byte toByte(byte[] bytes, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.decodeByte(bytes, 0, order.getImplSortOrder());
  }

  /**
   * Deserialize a byte value from its binary-comparable byte array with explicit sort order.
   * @param bytes binary-comparable byte array of a byte value
   * @param offset start position
   * @param order sort order of this value
   * @return deserialized value
   */
  public static byte toByte(byte[] bytes, int offset, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.decodeByte(bytes, offset, order.getImplSortOrder());
  }




  /**
   * Serialize a short value into its binary-comparable byte array with explicit sort order.
   * @param val value to be serialized.
   * @param order sort order of this value
   * @return binary-comparable byte array of the specified value
   */
  public static byte[] toBytes(short val, AliHBaseColumn.SortOrder order) {
    byte[] output = new byte[Bytes.SIZEOF_SHORT];
    DataTypeUtils.encodeShort(val, output, 0, order.getImplSortOrder());
    return output;
  }

  /**
   * Put a short value into the specified byte array position, using its binary-comparable form with explicit sort order.
   * @param output the output byte array
   * @param offset position in the array
   * @param val short value to write to
   * @param order sort order of this value
   * @return incremented offset
   * @throws IllegalArgumentException if the specified byte array doesn't have enough room to hold this value
   */
  public static int putShort(byte[] output, int offset, short val, AliHBaseColumn.SortOrder order) {
    return offset + DataTypeUtils.encodeShort(val, output, offset, order.getImplSortOrder());
  }

  /**
   * Deserialize a short value from its binary-comparable byte array with explicit sort order.
   * @param bytes binary-comparable byte array of a short value
   * @param order sort order of this value
   * @return deserialized value
   */
  public static short toShort(byte[] bytes, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.decodeShort(bytes, 0, order.getImplSortOrder());
  }

  /**
   * Deserialize a short value from its binary-comparable byte array with explicit sort order.
   * @param bytes binary-comparable byte array of a short value
   * @param offset start position
   * @param order sort order of this value
   * @return deserialized value
   */
  public static short toShort(byte[] bytes, int offset, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.decodeShort(bytes, offset, order.getImplSortOrder());
  }

  /**
   * Serialize an integer value into its binary-comparable byte array with explicit sort order.
   * @param val value to be serialized.
   * @param order sort order of this value
   * @return binary-comparable byte array of the specified value
   */
  public static byte[] toBytes(int val, AliHBaseColumn.SortOrder order) {
    byte[] output = new byte[Bytes.SIZEOF_INT];
    DataTypeUtils.encodeInt(val, output, 0, order.getImplSortOrder());
    return output;
  }

  /**
   * Put an integer value into the specified byte array position, using its binary-comparable form with explicit sort order.
   * @param output the output byte array
   * @param offset position in the array
   * @param val integer value to write to
   * @param order sort order of this value
   * @return incremented offset
   * @throws IllegalArgumentException if the specified byte array doesn't have enough room to hold this value
   */
  public static int putInt(byte[] output, int offset, int val, AliHBaseColumn.SortOrder order) {
    return offset + DataTypeUtils.encodeInt(val, output, offset, order.getImplSortOrder());
  }

  /**
   * Deserialize an integer value from its binary-comparable byte array with explicit sort order.
   * @param bytes binary-comparable byte array of an integer value
   * @param order sort order of this value
   * @return deserialized value
   */
  public static int toInt(byte[] bytes, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.decodeInt(bytes, 0, order.getImplSortOrder());
  }

  /**
   * Deserialize an integer value from its binary-comparable byte array with explicit sort order.
   * @param bytes binary-comparable byte array of an integer value
   * @param offset start position
   * @param order sort order of this value
   * @return deserialized value
   */
  public static int toInt(byte[] bytes, int offset, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.decodeInt(bytes, offset, order.getImplSortOrder());
  }

  /**
   * Serialize a long value into its binary-comparable byte array with explicit sort order.
   * @param val value to be serialized.
   * @param order sort order of this value
   * @return binary-comparable byte array of the specified value
   */
  public static byte[] toBytes(long val, AliHBaseColumn.SortOrder order) {
    byte[] output = new byte[Bytes.SIZEOF_LONG];
    DataTypeUtils.encodeLong(val, output, 0, order.getImplSortOrder());
    return output;
  }

  /**
   * Put a long value into the specified byte array position, using its binary-comparable form with explicit sort order.
   * @param output the output byte array
   * @param offset position in the array
   * @param val long value to write to
   * @param order sort order of this value
   * @return incremented offset
   * @throws IllegalArgumentException if the specified byte array doesn't have enough room to hold this value
   */
  public static int putLong(byte[] output, int offset, long val, AliHBaseColumn.SortOrder order) {
    return offset + DataTypeUtils.encodeLong(val, output, offset, order.getImplSortOrder());
  }

  /**
   * Deserialize a long value from its binary-comparable byte array with explicit sort order.
   * @param bytes binary-comparable byte array of a long value
   * @param order sort order of this value
   * @return deserialized value
   */
  public static long toLong(byte[] bytes, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.decodeLong(bytes, 0, order.getImplSortOrder());
  }

  /**
   * Deserialize a long value from its binary-comparable byte array with explicit sort order.
   * @param bytes binary-comparable byte array of a long value
   * @param offset start position
   * @param order sort order of this value
   * @return deserialized value
   */
  public static long toLong(byte[] bytes, int offset, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.decodeLong(bytes, offset, order.getImplSortOrder());
  }


  /**
   * Serialize a float value into its binary-comparable byte array with explicit sort order.
   * @param val value to be serialized.
   * @param order sort order of this value
   * @return binary-comparable byte array of the specified value
   */
  public static byte[] toBytes(float val, AliHBaseColumn.SortOrder order) {
    byte[] output = new byte[Bytes.SIZEOF_FLOAT];
    DataTypeUtils.encodeFloat(val, output, 0, order.getImplSortOrder());
    return output;
  }

  /**
   * Put a float value into the specified byte array position, using its binary-comparable form with explicit sort order.
   * @param output the output byte array
   * @param offset position in the array
   * @param val float value to write to
   * @param order sort order of this value
   * @return incremented offset
   * @throws IllegalArgumentException if the specified byte array doesn't have enough room to hold this value
   */
  public static int putFloat(byte[] output, int offset, float val, AliHBaseColumn.SortOrder order) {
    return offset + DataTypeUtils.encodeFloat(val, output, offset, order.getImplSortOrder());
  }

  /**
   * Deserialize a float value from its binary-comparable byte array with explicit sort order.
   * @param bytes binary-comparable byte array of a float value
   * @param order sort order of this value
   * @return deserialized value
   */
  public static float toFloat(byte[] bytes, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.decodeFloat(bytes, 0, order.getImplSortOrder());
  }

  /**
   * Deserialize a float value from its binary-comparable byte array with explicit sort order.
   * @param bytes binary-comparable byte array of a float value
   * @param offset start position
   * @param order sort order of this value
   * @return deserialized value
   */
  public static float toFloat(byte[] bytes, int offset, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.decodeFloat(bytes, offset, order.getImplSortOrder());
  }


  /**
   * Serialize a double value into its binary-comparable byte array with explicit sort order.
   * @param val value to be serialized.
   * @param order sort order of this value
   * @return binary-comparable byte array of the specified value
   */
  public static byte[] toBytes(double val, AliHBaseColumn.SortOrder order) {
    byte[] output = new byte[Bytes.SIZEOF_DOUBLE];
    DataTypeUtils.encodeDouble(val, output, 0, order.getImplSortOrder());
    return output;
  }

  /**
   * Put a double value into the specified byte array position, using its binary-comparable form with explicit sort order.
   * @param output the output byte array
   * @param offset position in the array
   * @param val double value to write to
   * @param order sort order of this value
   * @return incremented offset
   * @throws IllegalArgumentException if the specified byte array doesn't have enough room to hold this value
   */
  public static int putDouble(byte[] output, int offset, double val, AliHBaseColumn.SortOrder order) {
    return offset + DataTypeUtils.encodeDouble(val, output, offset, order.getImplSortOrder());
  }

  /**
   * Deserialize a double value from its binary-comparable byte array with explicit sort order.
   * @param bytes binary-comparable byte array of a double value
   * @param order sort order of this value
   * @return deserialized value
   */
  public static double toDouble(byte[] bytes, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.decodeDouble(bytes, 0, order.getImplSortOrder());
  }

  /**
   * Deserialize a double value from its binary-comparable byte array with explicit sort order.
   * @param bytes binary-comparable byte array of a double value
   * @param offset start position
   * @param order sort order of this value
   * @return deserialized value
   */
  public static double toDouble(byte[] bytes, int offset, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.decodeDouble(bytes, offset, order.getImplSortOrder());
  }


  /**
   * Serialize a BigDecimal value into its binary-comparable byte array with explicit sort order.
   * @param val value to be serialized.
   * @param order sort order of this value
   * @return binary-comparable byte array of the specified value
   */
  public static byte[] toBytes(BigDecimal val, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.encodeDecimal(val, order.getImplSortOrder());
  }

  /**
   * Put a BigDecimal value into the specified byte array position, using its binary-comparable form with explicit sort order.
   * Note: BigDecimal has variable length in binary format, so it's difficult to estimate the actual size and
   *       create a proper byte array in advance. So this method is rarely used.
   * @param output the output byte array
   * @param offset position in the array
   * @param val BigDecimal value to write to
   * @param order sort order of this value
   * @return incremented offset
   * @throws IllegalArgumentException if the specified byte array doesn't have enough room to hold this value
   */
  public static int putBigDecimal(byte[] output, int offset, BigDecimal val, AliHBaseColumn.SortOrder order) {
    byte[] temp = DataTypeUtils.encodeDecimal(val, order.getImplSortOrder());
    System.arraycopy(temp, 0, output, offset, temp.length);
    return offset + temp.length;
  }

  /**
   * Deserialize a BigDecimal value from its binary-comparable byte array with explicit sort order.
   * @param bytes binary-comparable byte array of a BigDecimal value
   * @param order sort order of this value
   * @return deserialized value
   */
  public static BigDecimal toBigDecimal(byte[] bytes, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.decodeDecimal(bytes, 0, bytes.length, order.getImplSortOrder());
  }

  /**
   * Deserialize a BigDecimal value from its binary-comparable byte array with explicit sort order.
   * @param bytes binary-comparable byte array of a BigDecimal value
   * @param offset start position
   * @param length number of bytes that represents this BigDecimal value
   * @param order sort order of this value
   * @return deserialized value
   */
  public static BigDecimal toBigDecimal(byte[] bytes, int offset, int length, AliHBaseColumn.SortOrder order) {
    return DataTypeUtils.decodeDecimal(bytes, offset, length, order.getImplSortOrder());
  }
}
