/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.client;

import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.index.AliHBaseIndexDescriptor;

import java.io.IOException;
import java.util.List;

/**
 * Provide some new feature interfaces
 */
public interface AliHBaseAdminInterface extends Admin {

  /**
   * Get index schema for specified data table. Return empty list if no index found, never return null.
   * @param dataTableName data table name
   * @return all index descriptors of the specified data table; if no index found for this data table, then return an empty list.
   */
  List<AliHBaseIndexDescriptor> describeIndex(TableName dataTableName) throws IOException;

  /**
   * Create a global secondary index table against a specified data table, block forever until operation success.
   * This call needs to wait for the index data to be built, may consume relatively long time, and thus may block current thread.
   * @param indexDescriptor descriptor of index table
   * @throws IOException
   */
  void createIndex(AliHBaseIndexDescriptor indexDescriptor) throws IOException ;

  /**
   * Create a global secondary index table against a specified data table with split keys, block forever until operation success.
   * This call needs to wait for the index data to be built, may consume relatively long time, and thus may block current thread.
   * @param indexDescriptor descriptor of index table
   * @param splitKeys pre-split range of index table
   * @throws IOException
   */
  void createIndex(AliHBaseIndexDescriptor indexDescriptor, byte[][] splitKeys) throws IOException;

  /**
   * Create a global secondary index table against a specified data table, block forever until operation success.
   * Index table will be split into numRegions initial regions, with startKey, endKey specified.
   * This call needs to wait for the index data to be built, may consume relatively long time, and thus may block current thread.
   * @param indexDescriptor descriptor of index table
   * @param startKey start range key of index table
   * @param endKey end range key of index table
   * @param numRegions number of initial regions
   * @throws IOException
   */
  void createIndex(AliHBaseIndexDescriptor indexDescriptor, byte[] startKey, byte[] endKey, int numRegions) throws IOException;

  /**
   * Drop index and clean all index data synchronously, block forever until operation done.
   * The #{@link AliHBaseAdminInterface#offlineIndex(String, TableName)} operation must be performed first.
   * @param indexName index name
   * @param dataTable name of main table
   * @throws IOException
   */
  void deleteIndex(String indexName, TableName dataTable) throws IOException;

  /**
   * Modify index status to DISABLED, block forever until operation done.
   * Re-online is not supported, index must be deleted and re-created.
   * @param indexName index name
   * @param dataTable name of main table
   * @throws IOException
   */
  void offlineIndex(String indexName, TableName dataTable) throws IOException;


  /**
   * Register
   * @param hbaseSourceName the Source Name of HBase Cluster, any name you want is OK, if null, Default name will be used
   * @param bdsClusterkey the BDS's cluster key. it is a zookeeper quorum key like xxx,xxx:2181:/bds
   * @param hbaseConnectionString the hbase connection string, like ld-xxxxxx-proxy-hbaseue.hbaseue.rds.aliyuncs.com:30020
   * @param username the user name of the HBase cluster
   * @param password the password of the HBase cluster
   */
  void registerBDSCluster(String hbaseSourceName, final String bdsClusterkey, final String hbaseConnectionString, final String username,
      final String password) throws IOException;

  /**
   * Register a solr cluster
   * @param solrSourceName  The Source Name of Solr, any name you want is OK, if null, Default name will be used
   * @param solrConnectionStr Solr connection string, it should be a cluster key like
   *                          xxx,xxx:2181/solr(there is no ":" before '/solr' and after client port 2181)
   * @throws IOException
   */
  void registerSolrCluster(final String solrSourceName, final String solrConnectionStr) throws IOException;

  /**
   * Register a ES cluster
   * @param esSourceName  source name of this ES cluster in BDS. If null or empty, DEFAULTESSOURCENAME
   *                  will be used.
   * @param esConnectionStr ES connection string, should be host:port,host:port, more than one host:port can be passed in
   * @param userName user name, if don't have any, left empty or nulll
   * @param password password to access this es, if don't have any, left empty or nulll
   */
  void registerESCluster(final String esSourceName, final String esConnectionStr,
      final String userName, final String password) throws IOException;

  /**
   * unregister solr cluster
   * @param force if force=true, all check will be skipped, even if there is a solr table syncing.
   * @throws IOException
   */
  void unregisterSolrCluster(final boolean force) throws IOException;

  /**
   * unregister ES cluster
   * @param force if force=true, all check will be skipped, even if there is a ES table syncing.
   * @throws IOException
   */
  void unregisterESCluster(final boolean force) throws IOException;

  /**
   * unregister BDS cluster
   * @param force if force=true, all check will be skipped, even if there is a external index table syncing.
   * @throws IOException
   */
  void unregisterBDSCluster(final boolean force) throws IOException;
}
