/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.ExecutorService;

import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.client.AliHBaseUEClientService;
import com.alibaba.hbase.client.AliHBaseUEConnection;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.MasterNotRunningException;
import org.apache.hadoop.hbase.RegionLocations;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.ZooKeeperConnectionException;
import org.apache.hadoop.hbase.client.backoff.ClientBackoffPolicy;
import org.apache.hadoop.hbase.client.coprocessor.Batch;
import org.apache.hadoop.hbase.ipc.RpcControllerFactory;
import org.apache.hadoop.hbase.protobuf.generated.AdminProtos;
import org.apache.hadoop.hbase.protobuf.generated.ClientProtos;
import org.apache.hadoop.hbase.protobuf.generated.MasterProtos;
import org.apache.hadoop.hbase.security.User;

public class AliHBaseUEClusterConnection extends
    ConnectionManager.HConnectionImplementation {

  private AliHBaseUEConnection connection;

  // For HBase-1.x
  public AliHBaseUEClusterConnection(Configuration conf, boolean managed, ExecutorService pool,
      final User user) throws IOException {
    this(conf, pool, user);
  }

  private static Configuration setupConf(Configuration conf) {
    conf.setBoolean("hbase.regionsizecalculator.enable", false);
    conf.set("hbase.client.registry.impl", NoopRegistry.class.getName());
    return conf;
  }

  @Override
  public Configuration getConfiguration() {
      return super.getConfiguration();
  }

  public AliHBaseUEClusterConnection(Configuration conf, ExecutorService pool, final User user) throws
      IOException {
    super(setupConf(conf), false);
    this.connection = new AliHBaseUEConnection(conf, pool, user);
  }

  void retrieveClusterId() {
    return;
  }

  public AliHBaseUEClusterConnection(AliHBaseUEConnection connection)
      throws IOException {
    super(connection.getConfiguration(), false);
    this.connection = connection;
  }

  /**
   * Only meaningful in Thrift impl
   * @param cuLimit
   */
  public void setCULimit(int cuLimit) {
    this.connection.setCULimit(cuLimit);
  }

  public HTableInterface getTable(TableName tableName) throws IOException {
    return connection.getTable(tableName);
  }

  public HTableInterface getTable(TableName tableName, ExecutorService pool)  throws IOException {
    return connection.getTable(tableName, pool);
  }


  public BufferedMutator getBufferedMutator(TableName tableName) {
    try {
      return connection.getBufferedMutator(tableName);
    } catch (IOException e) {
      throw new IllegalStateException("getBufferedMutator exception ",e);
    }
  }

  public BufferedMutator getBufferedMutator(BufferedMutatorParams params) {
    try {
      return connection.getBufferedMutator(params);
    } catch (IOException e) {
      throw new IllegalStateException("getBufferedMutator exception ",e);
    }
  }

  public RegionLocator getRegionLocator(TableName tableName) throws IOException {
    return connection.getRegionLocator(tableName);
  }

  /**
   * Retrieve an Admin implementation to administer an HBase cluster.
   * The returned Admin is not guaranteed to be thread-safe.  A new instance should be created for
   * each using thread.  This is a lightweight operation.  Pooling or caching of the returned
   * Admin is not recommended.
   * <br>
   * The caller is responsible for calling {@link Admin#close()} on the returned
   * Admin instance.
   *
   * @return an Admin instance for cluster administration
   */
  public Admin getAdmin() throws IOException {
    return connection.getAdmin();
  }

  @Override
  void internalClose() {
    if (isZeroReference()) {
      try {
        connection.close();
      } catch (IOException e) {
        throw new IllegalStateException("close exception ",e);
      }
    }
  }


  public boolean isClosed() {
    return connection.isClosed();
  }



  @Override
  public boolean isMasterRunning() throws MasterNotRunningException, ZooKeeperConnectionException {
    return true;
  }

  @Override
  public boolean isTableAvailable(TableName tableName, byte[][] splitKeys) throws IOException {
    try (Admin admin = getAdmin()) {
        return admin.isTableAvailable(tableName, splitKeys);
    }
  }

  @Override
  public HRegionLocation locateRegion(TableName tableName, byte[] row) throws IOException {
    try (AliHBaseUETable table = (AliHBaseUETable)getTable(tableName)) {
      return table.getRegionLocation(row);
    }
  }

  @Override
  public void clearRegionCache() {
  }


  public void cacheLocation(TableName tableName, RegionLocations location) {

  }

  @Override
  public void clearRegionCache(TableName tableName) {

  }

  @Override
  public void deleteCachedRegionLocation(HRegionLocation location) {

  }

  @Override
  public HRegionLocation relocateRegion(TableName tableName, byte[] row) throws IOException {
    return locateRegion(tableName, row);
  }

  @Override
  public RegionLocations relocateRegion(TableName tableName, byte[] row, int replicaId)
      throws IOException {
    RegionLocations regionLocations = new RegionLocations(locateRegion(tableName, row));
    return regionLocations;
  }

  @Override
  public void updateCachedLocations(TableName tableName, byte[] regionName, byte[] rowkey,
      Object exception, ServerName source) {

  }

  @Override
  public HRegionLocation locateRegion(byte[] regionName) throws IOException {
    throw new UnsupportedOperationException("locateRegion by region name is not supported");
  }

  @Override
  public List<HRegionLocation> locateRegions(TableName tableName) throws IOException {
    try (AliHBaseUETable table = (AliHBaseUETable)getTable(tableName)) {
      return table.getAllRegionLocations();
    }
  }

  @Override
  public List<HRegionLocation> locateRegions(TableName tableName, boolean useCache,
      boolean offlined) throws IOException {
    return locateRegions(tableName);
  }

  @Override
  public RegionLocations locateRegion(TableName tableName, byte[] row, boolean useCache,
      boolean retry) throws IOException {
    RegionLocations regionLocations = new RegionLocations(locateRegion(tableName, row));
    return regionLocations;
  }

  @Override
  public RegionLocations locateRegion(TableName tableName, byte[] row, boolean useCache,
      boolean retry, int replicaId) throws IOException {
    return locateRegion(tableName, row, useCache, retry);
  }

  @Override
  public MasterProtos.MasterService.BlockingInterface getMaster() {
    return null;
  }

  @Override
  public AdminProtos.AdminService.BlockingInterface getAdmin(ServerName serverName)
      throws IOException {
    return null;
  }

  @Override
  public ClientProtos.ClientService.BlockingInterface getClient(ServerName serverName)
      throws IOException {
    return new AliHBaseUEClientService(connection);
  }

  @Override
  public HRegionLocation getRegionLocation(TableName tableName, byte[] row, boolean reload)
      throws IOException {
    return locateRegion(tableName, row);
  }

  @Override
  public void clearCaches(ServerName sn) {

  }

  @Override
  public MasterKeepAliveConnection getKeepAliveMasterService() throws MasterNotRunningException {
    return null;
  }

  @Override
  public boolean isDeadServer(ServerName serverName) {
    return false;
  }

  @Override
  public NonceGenerator getNonceGenerator() {
    return null;
  }

  @Override
  public AsyncProcess getAsyncProcess() {
    return null;
  }

  @Override
  public RpcRetryingCallerFactory getNewRpcRetryingCallerFactory(Configuration conf) {
    return null;
  }


  public RpcRetryingCallerFactory getRpcRetryingCallerFactory() {
    return null;
  }


  public RpcControllerFactory getRpcControllerFactory() {
    return null;
  }


  public ConnectionConfiguration getConnectionConfiguration() {
    if(connection == null){
      return new ConnectionConfiguration(new Configuration());
    }
    return new ConnectionConfiguration(connection.getConfiguration());
  }

  @Override
  public boolean isManaged() {
    return false;
  }

  @Override
  public ServerStatisticTracker getStatisticsTracker() {
    return null;
  }

  @Override
  public ClientBackoffPolicy getBackoffPolicy() {
    return null;
  }

  @Override
  public boolean hasCellBlockSupport() {
    return false;
  }

  @Override
  public HTableInterface getTable(String tableName) throws IOException {
    return getTable(TableName.valueOf(tableName));
  }

  @Override
  public HTableInterface getTable(byte[] tableName) throws IOException {
    return getTable(TableName.valueOf(tableName));
  }

  @Override
  public HTableInterface getTable(String tableName, ExecutorService pool) throws IOException {
    return getTable(TableName.valueOf(tableName), pool);
  }

  @Override
  public HTableInterface getTable(byte[] tableName, ExecutorService pool) throws IOException {
    return getTable(TableName.valueOf(tableName), pool);
  }

  @Override
  public boolean isTableEnabled(TableName tableName) throws IOException {
    try(Admin admin = getAdmin()) {
      return admin.isTableAvailable(tableName);
    }
  }

  @Override
  public boolean isTableEnabled(byte[] tableName) throws IOException {
    return isTableEnabled(TableName.valueOf(tableName));
  }

  @Override
  public boolean isTableDisabled(TableName tableName) throws IOException {
    try(Admin admin = getAdmin()) {
      return admin.isTableDisabled(tableName);
    }
  }

  @Override
  public boolean isTableDisabled(byte[] tableName) throws IOException {
    return isTableDisabled(TableName.valueOf(tableName));
  }

  @Override
  public boolean isTableAvailable(TableName tableName) throws IOException {
    try (Admin admin = getAdmin()) {
      return admin.isTableAvailable(tableName);
    }
  }

  @Override
  public boolean isTableAvailable(byte[] tableName) throws IOException {
    return isTableAvailable(TableName.valueOf(tableName));
  }

  @Override
  public boolean isTableAvailable(byte[] tableName, byte[][] splitKeys) throws IOException {
    try(Admin admin = getAdmin()) {
      return admin.isTableAvailable(TableName.valueOf(tableName), splitKeys);
    }
  }

  @Override
  public HTableDescriptor[] listTables() throws IOException {
    try (Admin admin = getAdmin()) {
      return admin.listTables();
    }
  }

  @Override
  public String[] getTableNames() throws IOException {
    TableName[] tableNames = listTableNames();
    String[] tables = new String[tableNames.length];
    for (int i = 0; i < tableNames.length; i++) {
      tables[i] = tableNames[i].toString();
    }
    return tables;
  }

  @Override
  public TableName[] listTableNames() throws IOException {
    try (Admin admin = getAdmin()) {
      return admin.listTableNames();
    }
  }

  @Override
  public HTableDescriptor getHTableDescriptor(TableName tableName) throws IOException {
    try (Admin admin = getAdmin()) {
      return admin.getTableDescriptor(tableName);
    }
  }

  @Override
  public HTableDescriptor getHTableDescriptor(byte[] tableName) throws IOException {
    return getHTableDescriptor(TableName.valueOf(tableName));
  }

  @Override
  public HRegionLocation locateRegion(byte[] tableName, byte[] row) throws IOException {
    try(AliHBaseUETable table = (AliHBaseUETable)getTable(tableName)) {
      return table.getRegionLocation(row);
    }
  }

  @Override
  public void clearRegionCache(byte[] tableName) {

  }

  @Override
  public HRegionLocation relocateRegion(byte[] tableName, byte[] row) throws IOException {
    return locateRegion(tableName, row);
  }

  @Override
  public void updateCachedLocations(TableName tableName, byte[] rowkey, Object exception,
      HRegionLocation source) {

  }

  @Override
  public void updateCachedLocations(byte[] tableName, byte[] rowkey, Object exception,
      HRegionLocation source) {

  }

  @Override
  public List<HRegionLocation> locateRegions(byte[] tableName) throws IOException {
    try (AliHBaseUETable table = (AliHBaseUETable) getTable(tableName)) {
      return table.getAllRegionLocations();
    }
  }

  @Override
  public List<HRegionLocation> locateRegions(byte[] tableName, boolean useCache, boolean offlined)
      throws IOException {
    return locateRegions(tableName);
  }

  @Override
  public AdminProtos.AdminService.BlockingInterface getAdmin(ServerName serverName,
      boolean getMaster) throws IOException {
    return null;
  }

  @Override
  public HRegionLocation getRegionLocation(byte[] tableName, byte[] row, boolean reload)
      throws IOException {
    return locateRegion(tableName, row);
  }

  @Override
  public void processBatch(List<? extends Row> actions, TableName tableName, ExecutorService pool,
      Object[] results) throws IOException, InterruptedException {
    try(AliHBaseUETable table = (AliHBaseUETable)getTable(tableName)) {
      table.batch(actions, results);
    }
  }

  @Override
  public void processBatch(List<? extends Row> actions, byte[] tableName, ExecutorService pool,
      Object[] results) throws IOException, InterruptedException {
    processBatch(actions, TableName.valueOf(tableName), pool, results);
  }

  @Override
  public <R> void processBatchCallback(List<? extends Row> list, TableName tableName,
      ExecutorService pool, Object[] results, Batch.Callback<R> callback)
      throws IOException, InterruptedException {
    throw new UnsupportedOperationException("processBatchCallback with callback is not supported");

  }

  @Override
  public <R> void processBatchCallback(List<? extends Row> list, byte[] tableName,
      ExecutorService pool, Object[] results, Batch.Callback<R> callback)
      throws IOException, InterruptedException {
    throw new UnsupportedOperationException("processBatchCallback with callback is not supported");

  }

  @Override
  public void setRegionCachePrefetch(TableName tableName, boolean enable) {

  }

  @Override
  public void setRegionCachePrefetch(byte[] tableName, boolean enable) {

  }

  @Override
  public boolean getRegionCachePrefetch(TableName tableName) {
    return false;
  }

  @Override
  public boolean getRegionCachePrefetch(byte[] tableName) {
    return false;
  }

  @Override
  public int getCurrentNrHRS() throws IOException {
    return 0;
  }

  @Override
  public HTableDescriptor[] getHTableDescriptorsByTableName(List<TableName> tableNames)
      throws IOException {
    try (Admin admin = getAdmin()) {
      return admin.getTableDescriptorsByTableName(tableNames);
    }
  }

  @Override
  public HTableDescriptor[] getHTableDescriptors(List<String> tableNames) throws IOException {
    try (Admin admin = getAdmin()) {
      return admin.getTableDescriptors(tableNames);
    }
  }

  @Override
  public void abort(String why, Throwable e) {

  }

  @Override
  public boolean isAborted() {
    return false;
  }

  @Override
  public String toString() {
    return "AliHBaseUEClusterConnection-0x" + Integer.toHexString(hashCode());
  }

  @Override
  public void clearRegionCache(TableName tableName, byte[] row) {
  }

}
