/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.client.index;

import com.alibaba.lindorm.client.core.LindormWideColumnService;
import org.apache.hadoop.hbase.util.Bytes;

import java.util.Arrays;

/**
 * Unique object that represent a column which exists both in data table and index table.
 */
public class AliHBaseColumn {
  public static AliHBaseColumn COVER_ALL = new AliHBaseColumn();

  public enum SortOrder {
    ASC(com.alibaba.lindorm.client.schema.SortOrder.ASC),
    DESC(com.alibaba.lindorm.client.schema.SortOrder.DESC),
    ;

    // low-level sort order mapped to Lindorm, internal used only
    private com.alibaba.lindorm.client.schema.SortOrder impl;

    SortOrder(com.alibaba.lindorm.client.schema.SortOrder impl) {
      this.impl = impl;
    }

    // for internal used only, clients should never call this method.
    public com.alibaba.lindorm.client.schema.SortOrder getImplSortOrder() {
      return impl;
    }

    public static SortOrder fromImplSortOrder(com.alibaba.lindorm.client.schema.SortOrder so) {
      if (so == com.alibaba.lindorm.client.schema.SortOrder.ASC) {
        return ASC;
      } else {
        return DESC;
      }
    }

    public static SortOrder getDefault() {
      return ASC;
    }
  }

  private final byte[] family;
  private final byte[] qualifier;
  private final SortOrder sortOrder;   // only indexed column has sort order
  private int hashCode = 0;     // generate hash code if necessary

  /**
   * Create a covered column, the sort order will be set to null.
   * @param family family name bytes
   * @param qualifier qualifier bytes
   */
  public static AliHBaseColumn createCoveredColumn(byte[] family, byte[] qualifier) {
    return new AliHBaseColumn(family, qualifier, null);
  }

  public static AliHBaseColumn createIndexedColumn(byte[] family, byte[] qualifier) {
    return new AliHBaseColumn(family, qualifier, SortOrder.getDefault());
  }

  public static AliHBaseColumn createIndexedColumn(byte[] family, byte[] qualifier, SortOrder sortOrder) {
    if (sortOrder == null) {
      throw new IllegalArgumentException("SortOrder must not be null.");
    }
    return new AliHBaseColumn(family, qualifier, sortOrder);
  }


  private AliHBaseColumn(byte[] family, byte[] qualifier) {
    this(family, qualifier, null);
  }

  /**
   * default constructor, internal used only.
   */
  private AliHBaseColumn() {
    this.family = null;
    this.qualifier = null;
    this.sortOrder = null;
  }

  private AliHBaseColumn(byte[] family, byte[] qualifier, SortOrder sortOrder) {
    checkNotNull(qualifier, "qualifier name");
    if (!Bytes.equals(qualifier, Bytes.toBytes(LindormWideColumnService.UNIFIED_PK_COLUMN_NAME))) {
      checkNotNull(family, "family name");
    }
    this.family = family;
    this.qualifier = qualifier;
    this.sortOrder = sortOrder;
  }

  /**
   * @return family name bytes
   */
  public byte[] getFamily() {
    return family;
  }

  /**
   * @return qualifier name bytes
   */
  public byte[] getQualifier() {
    return qualifier;
  }

  /**
   * @return sort order if this is an indexed column, or null otherwise.
   */
  public SortOrder getSortOrder() {
    return sortOrder;
  }

  @Override
  public String toString() {
    if (this == COVER_ALL) {
      return "@@ALL@@";
    } else {
      String column = (getFamily() != null ? (Bytes.toString(getFamily()) + ".") : "") + Bytes.toString(getQualifier());
      if (sortOrder == null) {
        return column;
      } else {
        return column + " " + sortOrder;
      }
    }
  }

  @Override
  public int hashCode() {
    if (this.hashCode == 0) {
      this.hashCode = Arrays.hashCode(family) * 31 + Arrays.hashCode(qualifier);
      if (sortOrder != null) {
        this.hashCode += sortOrder.hashCode();
      }
    }
    return hashCode;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (!(obj instanceof AliHBaseColumn)) {
      return false;
    }

    AliHBaseColumn other = (AliHBaseColumn) obj;
    if (hashCode() != other.hashCode()) {
      return false;
    }
    if (!Bytes.equals(qualifier, other.qualifier)) {
      return false;
    }
    if (!Bytes.equals(family, other.family)) {
      return false;
    }
    if (sortOrder != other.sortOrder) {
      return false;
    }
    return true;
  }

  private static void checkNotNull(byte[] value, String msg) {
    if (value == null || value.length == 0) {
      throw new IllegalArgumentException(msg + " must not be null or empty");
    }
  }
}
