/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.hbase.client;

import java.io.IOException;
import java.util.List;

import com.alibaba.lindorm.client.core.meta.ExternalIndexConfig;
import com.alibaba.lindorm.client.core.meta.ExternalIndexField;
import com.google.protobuf.Service;
import com.google.protobuf.ServiceException;
import org.apache.hadoop.hbase.CompareOperator;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.NamespaceDescriptor;
import org.apache.hadoop.hbase.NamespaceNotFoundException;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Append;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptor;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Increment;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Row;
import org.apache.hadoop.hbase.client.RowMutations;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.client.coprocessor.Batch;
import org.apache.hadoop.hbase.client.index.AliHBaseIndexDescriptor;
import org.apache.hadoop.hbase.util.Pair;

public interface AliHBaseAPIProxy {

  // Admin APIs

  boolean tableExists(TableName tableName) throws IOException;

  List<TableDescriptor> listTableDescriptors() throws IOException;

  TableName[] listTableNames() throws IOException;

  HTableDescriptor getDescriptor(TableName tableName)throws IOException;

  TableName[] listTableNamesByNamespace(String name) throws IOException;

  List<TableDescriptor> listTableDescriptorsByNamespace(byte[] name) throws IOException;

  void createTable(TableDescriptor desc, byte[][] splitKeys) throws IOException;

  void deleteTable(TableName tableName) throws IOException;

  void truncateTable(TableName tableName, boolean preserveSplits) throws IOException;

  void enableTable(TableName tableName) throws IOException;

  void disableTable(TableName tableName) throws IOException;

  boolean isTableEnabled(TableName tableName) throws IOException;

  boolean isTableDisabled(TableName tableName) throws IOException;

  boolean isTableAvailable(TableName tableName) throws IOException;

  void addColumnFamily(TableName tableName, ColumnFamilyDescriptor columnFamily) throws IOException;

  void deleteColumnFamily(TableName tableName, byte[] columnFamily) throws IOException;

  void modifyColumnFamily(TableName tableName, ColumnFamilyDescriptor columnFamily) throws IOException;

  void modifyTable(TableDescriptor td) throws IOException;

  void deleteNamespace(String name) throws IOException;

  NamespaceDescriptor getNamespaceDescriptor(String name) throws NamespaceNotFoundException, IOException;

  NamespaceDescriptor[] listNamespaceDescriptors() throws IOException;

  void createNamespace(NamespaceDescriptor descriptor) throws IOException;

  void flushRegion(String regionName);

  void compactRegion(String regionName);

  void majorCompactRegion(String regionName);

  void flushTable(TableName tableName);

  void compactTable(TableName tableName);

  void majorCompactTable(TableName tableName);

  //Table API
  HRegionLocation getRegionLocation(byte[] row) throws IOException;

  List<HRegionLocation> getAllRegionLocations() throws IOException;

  Pair<byte[][], byte[][]> getStartEndKeys() throws IOException;

  boolean exists(Get get) throws IOException;

  boolean[] exists(List<Get> gets) throws IOException;

  void batch(List<? extends Row> actions, Object[] results) throws IOException;

  Result get(Get get) throws IOException;

  Result[] get(List<Get> gets) throws IOException;

  ResultScanner getScanner(Scan scan) throws IOException;

  void put(Put put) throws IOException;

  void put(List<Put> puts) throws IOException;

  void delete(Delete delete) throws IOException;

  void delete(List<Delete> deletes) throws IOException;

  Table.CheckAndMutateBuilder checkAndMutate(byte[] row, byte[] family);

  boolean checkAndPut(byte[] row, byte[] family, byte[] qualifier, CompareOperator op,
      byte[] value, Put put) throws IOException;

  boolean checkAndDelete(byte[] row, byte[] family, byte[] qualifier, CompareOperator op,
      byte[] value, Delete delete) throws IOException;

  void mutateRow(RowMutations rm) throws IOException;

  Result append(Append append) throws IOException;

  Result increment(Increment increment) throws IOException;

  <T extends Service, R> void coprocessorService(Class<T> service, byte[] startKey, byte[] endKey,
      Batch.Call<T, R> callable, Batch.Callback<R> callback) throws ServiceException, Throwable;

  void addExternalIndex(ExternalIndexConfig config, List<ExternalIndexField> fields) throws IOException;

  void removeExternalIndex(List<String> fields) throws IOException;

  void buildExternalIndex() throws IOException;

  void cancelBuildExternalIndex() throws IOException;

  void close() throws IOException;

  List<AliHBaseIndexDescriptor> describeIndex(TableName dataTableName) throws IOException;

  void createIndex(AliHBaseIndexDescriptor indexDescriptor) throws IOException;

  void createIndex(AliHBaseIndexDescriptor indexDescriptor, byte[][] splitKeys) throws IOException;

  void createIndex(AliHBaseIndexDescriptor index, byte[] startKey, byte[] endKey, int numRegions) throws IOException;

  void deleteIndex(String indexName, TableName dataTable) throws IOException;

  void offlineIndex(String indexName, TableName dataTable) throws IOException;
}
