/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.hbase.client;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Queue;

import com.alibaba.hbase.exception.BatchExceptions;
import com.alibaba.hbase.thrift2.generated.TAppend;
import com.alibaba.hbase.thrift2.generated.TColumnFamilyDescriptor;
import com.alibaba.hbase.thrift2.generated.TDelete;
import com.alibaba.hbase.thrift2.generated.TGet;
import com.alibaba.hbase.thrift2.generated.THBaseService;
import com.alibaba.hbase.thrift2.generated.TIncrement;
import com.alibaba.hbase.thrift2.generated.TNamespaceDescriptor;
import com.alibaba.hbase.thrift2.generated.TPut;
import com.alibaba.hbase.thrift2.generated.TResult;
import com.alibaba.hbase.thrift2.generated.TRowMutations;
import com.alibaba.hbase.thrift2.generated.TScan;
import com.alibaba.hbase.thrift2.generated.TTableDescriptor;
import com.alibaba.hbase.thrift2.generated.TTableName;
import com.alibaba.lindorm.client.core.meta.ExternalIndexConfig;
import com.alibaba.lindorm.client.core.meta.ExternalIndexField;
import com.google.protobuf.Service;
import com.google.protobuf.ServiceException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.CompareOperator;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.NamespaceDescriptor;
import org.apache.hadoop.hbase.NamespaceNotFoundException;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Append;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptor;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Increment;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Row;
import org.apache.hadoop.hbase.client.RowMutations;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.client.coprocessor.Batch;
import org.apache.hadoop.hbase.client.index.AliHBaseIndexDescriptor;
import org.apache.hadoop.hbase.client.metrics.ScanMetrics;
import org.apache.hadoop.hbase.io.TimeRange;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.thrift.TException;
import org.apache.thrift.transport.TTransport;

public class AliHBaseAPIProxyThriftImpl implements AliHBaseAPIProxy {

  private THBaseService.Client client;
  private TTransport transport;
  private ByteBuffer tableNameInBytes;
  private int defaultScannerCaching;

  public AliHBaseAPIProxyThriftImpl(Configuration conf, THBaseService.Client client, TTransport transport, TableName tableName) {
    this.client = client;
    this.transport = transport;
    if (tableName != null) {
      tableNameInBytes = ByteBuffer.wrap(tableName.toBytes());
    }
    this.defaultScannerCaching = conf.getInt(HConstants.HBASE_CLIENT_SCANNER_CACHING,
        AliHBaseConstants.ALIHBASE_CLIENT_SCANNER_CACHING_DEFAULT);
    if (defaultScannerCaching > AliHBaseConstants.ALIHBASE_CLIENT_SCANNER_CACHING_DEFAULT) {
      this.defaultScannerCaching = AliHBaseConstants.ALIHBASE_CLIENT_SCANNER_CACHING_DEFAULT;
    }
  }

  @Override
  public boolean tableExists(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      return client.tableExists(tTableName);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public List<TableDescriptor> listTableDescriptors() throws IOException {
    try {
      List<TTableDescriptor> tTableDescriptors = client
          .getTableDescriptorsByPattern(null, false);
      return ThriftUtilities.tableDescriptorsFromThrift(tTableDescriptors);

    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public TableName[] listTableNames() throws IOException {
    try {
      List<TTableName> tTableNames = client.getTableNamesByPattern(null, false);
      return ThriftUtilities.tableNamesArrayFromThrift(tTableNames);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public HTableDescriptor getDescriptor(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      TTableDescriptor tTableDescriptor = client.getTableDescriptor(tTableName);
      return ThriftUtilities.hTableDescriptorFromThrift(tTableDescriptor);
    } catch (TException e) {
      throw new IOException(e);
    }  }

  @Override
  public TableName[] listTableNamesByNamespace(String name) throws IOException {
    try {
      List<TTableName> tTableNames = client.getTableNamesByNamespace(name);
      return ThriftUtilities.tableNamesArrayFromThrift(tTableNames);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public List<TableDescriptor> listTableDescriptorsByNamespace(byte[] name) throws IOException {
    try {
      List<TTableDescriptor> tTableDescriptors = client
          .getTableDescriptorsByNamespace(Bytes.toString(name));
      return ThriftUtilities.tableDescriptorsFromThrift(tTableDescriptors);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void createTable(TableDescriptor desc, byte[][] splitKeys) throws IOException {
    TTableDescriptor tTableDescriptor = ThriftUtilities.tableDescriptorFromHBase(desc);
    List<ByteBuffer> splitKeyInBuffer = ThriftUtilities.splitKeyFromHBase(splitKeys);
    try {
      client.createTable(tTableDescriptor, splitKeyInBuffer);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void deleteTable(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      client.deleteTable(tTableName);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void truncateTable(TableName tableName, boolean preserveSplits) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      client.truncateTable(tTableName, preserveSplits);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void enableTable(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      client.enableTable(tTableName);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void disableTable(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      client.disableTable(tTableName);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public boolean isTableEnabled(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      return client.isTableEnabled(tTableName);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public boolean isTableDisabled(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      return client.isTableDisabled(tTableName);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public boolean isTableAvailable(TableName tableName) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      return client.isTableAvailable(tTableName);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void addColumnFamily(TableName tableName, ColumnFamilyDescriptor columnFamily)
      throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    TColumnFamilyDescriptor tColumnFamilyDescriptor = ThriftUtilities
        .columnFamilyDescriptorFromHBase(columnFamily);
    try {
      client.addColumnFamily(tTableName, tColumnFamilyDescriptor);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void deleteColumnFamily(TableName tableName, byte[] columnFamily) throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    try {
      client.deleteColumnFamily(tTableName, ByteBuffer.wrap(columnFamily));
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void modifyColumnFamily(TableName tableName, ColumnFamilyDescriptor columnFamily)
      throws IOException {
    TTableName tTableName = ThriftUtilities.tableNameFromHBase(tableName);
    TColumnFamilyDescriptor tColumnFamilyDescriptor = ThriftUtilities
        .columnFamilyDescriptorFromHBase(columnFamily);
    try {
      client.modifyColumnFamily(tTableName, tColumnFamilyDescriptor);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void modifyTable(TableDescriptor td) throws IOException {
    TTableDescriptor tTableDescriptor = ThriftUtilities
        .tableDescriptorFromHBase(td);
    try {
      client.modifyTable(tTableDescriptor);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void deleteNamespace(String name) throws IOException {
    try {
      client.deleteNamespace(name);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public NamespaceDescriptor getNamespaceDescriptor(String name)
      throws NamespaceNotFoundException, IOException {
    try {
      TNamespaceDescriptor tNamespaceDescriptor = client.getNamespaceDescriptor(name);
      return ThriftUtilities.namespaceDescriptorFromThrift(tNamespaceDescriptor);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public NamespaceDescriptor[] listNamespaceDescriptors() throws IOException {
    try {
      List<TNamespaceDescriptor> tNamespaceDescriptors = client.listNamespaceDescriptors();
      return ThriftUtilities.namespaceDescriptorsFromThrift(tNamespaceDescriptors);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void createNamespace(NamespaceDescriptor descriptor) throws IOException {
    TNamespaceDescriptor tNamespaceDescriptor = ThriftUtilities
        .namespaceDescriptorFromHBase(descriptor);
    try {
      client.createNamespace(tNamespaceDescriptor);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void flushRegion(String regionName) {
    throw new UnsupportedOperationException("flush is not supported");
  }

  @Override
  public void compactRegion(String regionName) {
    throw new UnsupportedOperationException("compact is not supported");
  }

  @Override
  public void majorCompactRegion(String regionName) {
    throw new UnsupportedOperationException("majorCompact is not supported");
  }

  @Override
  public void flushTable(TableName tableName) {
    throw new UnsupportedOperationException("flush is not supported");
  }

  @Override
  public void compactTable(TableName tableName) {
    throw new UnsupportedOperationException("compact is not supported");
  }

  @Override
  public void majorCompactTable(TableName tableName) {
    throw new UnsupportedOperationException("majorCompact is not supported");
  }

  @Override
  public HRegionLocation getRegionLocation(byte[] row) throws IOException {
    throw new UnsupportedOperationException("getRegionLocation is not supported");

  }

  @Override
  public List<HRegionLocation> getAllRegionLocations() throws IOException {
    throw new UnsupportedOperationException("getAllRegionLocations is not supported");
  }

  @Override
  public Pair<byte[][], byte[][]> getStartEndKeys() throws IOException {
    throw new UnsupportedOperationException("getStartEndKeys is not supported");

  }

  @Override
  public boolean exists(Get get) throws IOException {
    TGet tGet = ThriftUtilities.getFromHBase(get);
    try {
      return client.exists(tableNameInBytes, tGet);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public boolean[] exists(List<Get> gets) throws IOException {
    List<TGet> tGets = new ArrayList<>();
    for (Get get: gets) {
      tGets.add(ThriftUtilities.getFromHBase(get));
    }
    try {
      List<Boolean> results = client.existsAll(tableNameInBytes, tGets);
      boolean[] booleans = new boolean[results.size()];
      for (int i = 0; i < results.size(); i++) {
        booleans[i] = results.get(i);
      }
      return booleans;
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void batch(List<? extends Row> actions, Object[] results) throws IOException {
    List<Object> tactions = new ArrayList<Object>(actions.size());
    for (Row action : actions) {
      if (action instanceof Get) {
        tactions.add(ThriftUtilities.getFromHBase((Get) action));
      } else if (action instanceof Put) {
        tactions.add(ThriftUtilities.putFromHBase((Put) action));
      } else if (action instanceof Delete) {
        tactions.add(ThriftUtilities.deleteFromHBase((Delete) action));
      } else if (action instanceof Append) {
        tactions.add(ThriftUtilities.appendFromHBase((Append) action));
      } else {
        throw new UnsupportedOperationException(
            "Unsupported type " + action.getClass().getName() + " in batch operation.");
      }
    }
    Object[] hbaseResults = null;
    BatchExceptions batchExceptions = new BatchExceptions();
    try {
      hbaseResults = batch(tactions);
    } finally {
      for (int i = 0; i < hbaseResults.length; i++) {
        if (results != null) {
          if (hbaseResults[i] == null) {
            results[i] = null;
          } else if (hbaseResults[i] instanceof TResult) {
            results[i] = ThriftUtilities
                .resultFromThrift((TResult) hbaseResults[i]);
          } else if (hbaseResults[i] instanceof Result) {
            results[i] = hbaseResults[i];
          } else if (hbaseResults[i] instanceof Throwable) {
            results[i] = hbaseResults[i];
            batchExceptions.add((Throwable) hbaseResults[i], actions.get(i), null);
          } else {
            throw new IOException("Not unsupported result type " + hbaseResults[i]);
          }
        } else {
          if (hbaseResults[i] != null && hbaseResults[i] instanceof Throwable) {
            batchExceptions.add((Throwable) hbaseResults[i], actions.get(i), null);
          }
        }
      }
      if (batchExceptions.hasErrors()) {
        throw batchExceptions.makeException();
      }
    }
  }

  private Object[] batch(List<Object> actions)
      throws IOException {
    Object[] results = new Object[actions.size()];

    List<TGet> getActions = new ArrayList<>();
    List<Integer> getActionsIndex = new ArrayList<>();

    List<TPut> putActions = new ArrayList<>();
    List<Integer> putActionsIndex = new ArrayList<>();

    List<TDelete> delActions = new ArrayList<>();
    List<Integer> delActionsIndex = new ArrayList<>();

    List<TAppend> appendActions = new ArrayList<>();
    List<Integer> appendActionsIndex = new ArrayList<>();

    for (int i = 0; i < actions.size(); i++) {
      if (actions.get(i) instanceof TGet) {
        getActions.add((TGet) actions.get(i));
        getActionsIndex.add(i);
      } else if (actions.get(i) instanceof TPut) {
        putActions.add((TPut) actions.get(i));
        putActionsIndex.add(i);
      } else if (actions.get(i) instanceof TDelete) {
        delActions.add((TDelete) actions.get(i));
        delActionsIndex.add(i);
      } else if (actions.get(i) instanceof TAppend) {
        appendActions.add((TAppend) actions.get(i));
        appendActionsIndex.add(i);
      }
      else {
        throw new UnsupportedOperationException("Not supported action " + actions.get(i).getClass().getName());
      }
    }
    if (!getActions.isEmpty()) {
      try {
        List<TResult> getResult = client
            .getMultiple(tableNameInBytes, getActions);
        for (int i = 0; i < getActions.size(); i++) {
          int index = getActionsIndex.get(i);
          results[index] = getResult.get(i);
        }
      } catch (Throwable e) {
        for (Integer index : getActionsIndex) {
          results[index] = e;
        }
      }
    }
    if (!putActions.isEmpty()) {
      try {
        client.putMultiple(tableNameInBytes, putActions);
        for (Integer index : putActionsIndex) {
          results[index] = new Result();
        }
      } catch (Throwable e) {
        for (Integer index : putActionsIndex) {
          results[index] = e;
        }
      }
    }
    if (!delActions.isEmpty()) {
      try {
        client.deleteMultiple(tableNameInBytes, delActions);
        for (Integer index : delActionsIndex) {
          results[index] = new Result();
        }
      } catch (Throwable e) {
        for (Integer index : delActionsIndex) {
          results[index] = e;
        }
      }
    }
    if (!appendActions.isEmpty()) {
      try {
        for (TAppend append : appendActions) {
          client.append(tableNameInBytes, append);
        }
        for (Integer index : appendActionsIndex) {
          results[index] = new Result();
        }
      } catch (Throwable e) {
        for (Integer index : appendActionsIndex) {
          results[index] = e;
        }
      }
    }
    return results;
  }

  @Override
  public Result get(Get get) throws IOException {
    TGet tGet = ThriftUtilities.getFromHBase(get);
    try {
      TResult tResult = client.get(tableNameInBytes, tGet);
      return ThriftUtilities.resultFromThrift(tResult);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public Result[] get(List<Get> gets) throws IOException {
    List<TGet> tGets = ThriftUtilities.getsFromHBase(gets);
    try {
      List<TResult> results = client.getMultiple(tableNameInBytes, tGets);
      return ThriftUtilities.resultsFromThrift(results);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  /**
   * A scanner to perform scan from thrift server
   * getScannerResults is used in this scanner
   */
  private class Scanner implements ResultScanner {
    protected TScan scan;
    protected Result lastResult = null;
    protected final Queue<Result> cache = new ArrayDeque<>();

    private int limit = -1;
    private int returned = 0;


    public Scanner(Scan scan) throws IOException {
      if (scan.getCaching() <= 0) {
        scan.setCaching(defaultScannerCaching);
      } else if (scan.getCaching() == 1 && scan.isReversed()){
        // for reverse scan, we need to pass the last row to the next scanner
        // we need caching number bigger than 1
        scan.setCaching(scan.getCaching() + 1);
      }
      this.limit = scan.getLimit();
      this.scan = ThriftUtilities.scanFromHBase(scan);
    }

    private boolean isLimitReached() {
      if (this.limit > 0 && this.returned >= this.limit) {
        return true;
      } else {
        return false;
      }
    }


    @Override
    public Result next() throws IOException {
      if (isLimitReached()) {
        return null;
      }
      if (cache.size() == 0) {
        setupNextScanner();
        try {
          List<TResult> tResults = client
              .getScannerResults(tableNameInBytes, scan, scan.getCaching());
          Result[] results = ThriftUtilities.resultsFromThrift(tResults);
          boolean firstKey = true;
          for (Result result : results) {
            // If it is a reverse scan, we use the last result's key as the startkey, since there is
            // no way to construct a closet rowkey smaller than the last result
            // So when the results return, we must rule out the first result, since it has already
            // returned to user.
            if (firstKey) {
              firstKey = false;
              if (scan.isReversed() && lastResult != null) {
                if (Bytes.equals(lastResult.getRow(), result.getRow())) {
                  continue;
                }
              }
            }
            cache.add(result);
            lastResult = result;
          }
        } catch (TException e) {
          throw new IOException(e);
        }
      }

      if (cache.size() > 0) {
        returned++;
        return cache.poll();
      } else {
        //scan finished
        return null;
      }
    }

    @Override
    public void close() {
    }

    @Override
    public boolean renewLease() {
      throw new RuntimeException("renewLease() not supported");
    }

    @Override
    public ScanMetrics getScanMetrics() {
      throw new RuntimeException("getScanMetrics() not supported");
    }

    private void setupNextScanner() {
      //if lastResult is null null, it means it is not the fist scan
      if (lastResult!= null) {
        byte[] lastRow = lastResult.getRow();
        if (scan.isReversed()) {
          //for reverse scan, we can't find the closet row before this row
          scan.setStartRow(lastRow);
        } else {
          scan.setStartRow(createClosestRowAfter(lastRow));
        }
      }
    }


    /**
     * Create the closest row after the specified row
     */
    protected byte[] createClosestRowAfter(byte[] row) {
      if (row == null) {
        throw new RuntimeException("The passed row is null");
      }
      return Arrays.copyOf(row, row.length + 1);
    }
  }

  @Override
  public ResultScanner getScanner(Scan scan) throws IOException {
    return new Scanner(scan);
  }

  @Override
  public void put(Put put) throws IOException {
    TPut tPut = ThriftUtilities.putFromHBase(put);
    try {
      client.put(tableNameInBytes, tPut);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void put(List<Put> puts) throws IOException {
    List<TPut> tPuts = ThriftUtilities.putsFromHBase(puts);
    try {
      client.putMultiple(tableNameInBytes, tPuts);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void delete(Delete delete) throws IOException {
    TDelete tDelete = ThriftUtilities.deleteFromHBase(delete);
    try {
      client.deleteSingle(tableNameInBytes, tDelete);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void delete(List<Delete> deletes) throws IOException {
    List<TDelete> tDeletes = ThriftUtilities.deletesFromHBase(deletes);
    try {
      client.deleteMultiple(tableNameInBytes, tDeletes);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  private class CheckAndMutateBuilderImpl implements Table.CheckAndMutateBuilder {

    private final byte[] row;
    private final byte[] family;
    private byte[] qualifier;
    private CompareOperator op;
    private byte[] value;

    CheckAndMutateBuilderImpl(byte[] row, byte[] family) {
      this.row = row;
      this.family = family;
    }

    @Override
    public Table.CheckAndMutateBuilder qualifier(byte[] qualifier) {
      /**
       * If qualifier is null. Consider using an empty byte array,
       * or just do not call this method if you want a null qualifier
       */
      this.qualifier = qualifier;
      return this;
    }

    @Override
    public Table.CheckAndMutateBuilder timeRange(TimeRange timeRange) {
      throw new UnsupportedOperationException("timeRange not supported");
    }

    @Override
    public Table.CheckAndMutateBuilder ifNotExists() {
      this.op = CompareOperator.EQUAL;
      this.value = null;
      return this;
    }

    @Override
    public Table.CheckAndMutateBuilder ifMatches(CompareOperator compareOp, byte[] value) {
      this.op = compareOp;
      this.value = value;
      return this;
    }

    private void preCheck() {
      if (op == null) {
        throw new RuntimeException("condition is null. You need to specify the condition by" +
            " calling ifNotExists/ifEquals/ifMatches before executing the request");
      }
    }

    @Override
    public boolean thenPut(Put put) throws IOException {
      preCheck();
      RowMutations rowMutations = new RowMutations(put.getRow());
      rowMutations.add(put);
      return checkAndMutate(row, family, qualifier, op, value, rowMutations);
    }

    @Override
    public boolean thenDelete(Delete delete) throws IOException {
      preCheck();
      RowMutations rowMutations = new RowMutations(delete.getRow());
      rowMutations.add(delete);
      return checkAndMutate(row, family, qualifier, op, value, rowMutations);
    }

    @Override
    public boolean thenMutate(RowMutations mutation) throws IOException {
      preCheck();
      return checkAndMutate(row, family, qualifier, op, value, mutation);
    }
  }

  private boolean checkAndMutate(byte[] row, byte[] family, byte[] qualifier, CompareOperator op,
      byte[] value, RowMutations mutation) throws IOException {
    try {
      ByteBuffer valueBuffer = value == null? null : ByteBuffer.wrap(value);
      return client.checkAndMutate(tableNameInBytes, ByteBuffer.wrap(row), ByteBuffer.wrap(family),
          ByteBuffer.wrap(qualifier), ThriftUtilities.compareOpFromHBase(op), valueBuffer,
          ThriftUtilities.rowMutationsFromHBase(mutation));
    } catch (TException e) {
      throw new IOException(e);
    }
  }


  @Override
  public Table.CheckAndMutateBuilder checkAndMutate(byte[] row, byte[] family) {
    return new CheckAndMutateBuilderImpl(row, family);
  }

  @Override
  public boolean checkAndPut(byte[] row, byte[] family, byte[] qualifier, CompareOperator op,
      byte[] value, Put put) throws IOException {
    RowMutations mutations = new RowMutations(put.getRow(), 1);
    mutations.add(put);
    return checkAndMutate(row, family, qualifier, op, value, mutations);
  }

  @Override
  public boolean checkAndDelete(byte[] row, byte[] family, byte[] qualifier, CompareOperator op,
      byte[] value, Delete delete) throws IOException {
    RowMutations mutations = new RowMutations(delete.getRow(), 1);
    mutations.add(delete);

    return checkAndMutate(row, family, qualifier, op, value, mutations);
  }

  @Override
  public void mutateRow(RowMutations rm) throws IOException {
    TRowMutations tRowMutations = ThriftUtilities.rowMutationsFromHBase(rm);
    try {
      client.mutateRow(tableNameInBytes, tRowMutations);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public Result append(Append append) throws IOException {
    TAppend tAppend = ThriftUtilities.appendFromHBase(append);
    try {
      TResult tResult = client.append(tableNameInBytes, tAppend);
      return ThriftUtilities.resultFromThrift(tResult);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public Result increment(Increment increment) throws IOException {
    TIncrement tIncrement = ThriftUtilities.incrementFromHBase(increment);
    try {
      TResult tResult = client.increment(tableNameInBytes, tIncrement);
      return ThriftUtilities.resultFromThrift(tResult);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public <T extends Service, R> void coprocessorService(Class<T> service, byte[] startKey,
      byte[] endKey, Batch.Call<T, R> callable, Batch.Callback<R> callback)
      throws ServiceException, Throwable {
    throw new UnsupportedOperationException("coprocessorService not supported");
  }

  @Override
  public void addExternalIndex(ExternalIndexConfig config, List<ExternalIndexField> fields)
      throws IOException {
    throw new UnsupportedOperationException("addExternalIndex not supported");

  }

  @Override
  public void removeExternalIndex(List<String> fields) throws IOException {
    throw new UnsupportedOperationException("removeExternalIndex not supported");

  }

  @Override
  public void buildExternalIndex() throws IOException {
    throw new UnsupportedOperationException("buildExternalIndex not supported");

  }

  @Override
  public void cancelBuildExternalIndex() throws IOException {
    throw new UnsupportedOperationException("cancelBuildExternalIndex not supported");

  }

  @Override
  public void close() throws IOException {
    transport.close();

  }

  @Override
  public List<AliHBaseIndexDescriptor> describeIndex(TableName dataTableName) throws IOException {
    throw new UnsupportedOperationException("describeIndex not supported");
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor indexDescriptor)
      throws IOException {
    throw new UnsupportedOperationException("createIndex not supported");
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor indexDescriptor,
      byte[][] splitKeys) throws IOException {
    throw new UnsupportedOperationException("createIndex not supported");
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor index,
      byte[] startKey, byte[] endKey, int numRegions) throws IOException {
    throw new UnsupportedOperationException("createIndex not supported");
  }

  @Override
  public void deleteIndex(String indexName, TableName dataTable)
      throws IOException {
    throw new UnsupportedOperationException("deleteIndex not supported");
  }

  @Override
  public void offlineIndex(String indexName, TableName dataTable)
      throws IOException {
    throw new UnsupportedOperationException("offlineIndex not supported");
  }
}
