package com.alibaba.hbase.haclient;

import java.io.IOException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import com.alibaba.hbase.client.AliHBaseConstants;
import com.google.common.annotations.VisibleForTesting;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.hbase.zookeeper.RecoverableZooKeeper;
import org.apache.hadoop.hbase.zookeeper.ZKConfig;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;
import org.apache.hadoop.hbase.zookeeper.ZNodePaths;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;

public class ClusterSwitchTracker implements Watcher {
  private static final Log LOG = LogFactory.getLog(ClusterSwitchTracker.class);

  // zookeeper quorum
  private String quorum;

  private CountDownLatch syncLatch;

  // zookeeper connection
  private RecoverableZooKeeper zooKeeper;
  // base znode for this cluster
  public String redirectNode;

  private String identifier;
  private final Switchable switchable;

  private Configuration conf;

  private volatile boolean stopped = false;

  private boolean everConnected = false;

  private int expireTime;

  private Lock zkConnectionLock = new ReentrantLock();

  public ClusterSwitchTracker(Configuration conf, Switchable switchable)
    throws IOException {
    this(conf, switchable, null);
  }

  public ClusterSwitchTracker(Configuration conf, Switchable switchable,
                              CountDownLatch syncLatch) throws IOException {
    this.conf = conf;
    this.switchable = switchable;
    this.syncLatch = syncLatch;
    this.quorum = ZKConfig.getZKQuorumServersString(conf);
    this.redirectNode =
      ZNodePaths.joinZNode(ClusterSwitchUtil.getBaseNode(conf.get(AliHBaseConstants.HACLIENT_BASE_NODE,
      AliHBaseConstants.HACLIENT_BASE_NODE_DEFAULT), conf.get(AliHBaseConstants.HACLIENT_CLUSTER_ID)),
      conf.get(ClusterSwitchUtil.ZOOKEEPER_REDIRECT_NODE, ClusterSwitchUtil.ZOOKEEPER_REDIRECT_NODE_DEFAULT));
    this.expireTime = conf.getInt(HConstants.ZK_SESSION_TIMEOUT,
      HConstants.DEFAULT_ZK_SESSION_TIMEOUT);
    this.identifier = quorum + "_ClusterSwitchTracker0x0";
    this.zooKeeper = ZKUtil.connect(conf, quorum, this, identifier);
  }

  private void fetchKeyAndSwitch() {
    try {
      // this call will set a watch on the patch to watch Create/Delete event
      if (zooKeeper.exists(redirectNode, true) != null) {
        // this call will set a watch on the path to watch NodeChange event
        byte[] bytes = zooKeeper.getData(redirectNode, true, null);
        switchable.onNodeChange(bytes);
      } else {
        // The node does not exist
        switchable.onNodeChange(null);
      }
      if (!everConnected && syncLatch != null) {
        syncLatch.countDown();
      }
    } catch (KeeperException.NoNodeException noNodeEx) {
      // The switch node does not exist
      // This can happen if we check the exist, but the node is deleted
      // before we can get data from it
      switchable.onNodeChange(null);
      if (!everConnected && syncLatch != null) {
        syncLatch.countDown();
      }
    } catch (KeeperException keeperException) {
      // The switch somehow failed, the operator can schedule other switch push later
      LOG.warn("Switch failed since", keeperException);
    } catch (InterruptedException ie) {
      LOG.error(this.toString() + " is interrupted", ie);
      Thread.currentThread().interrupt();
    }
  }

  public String getRedirectNode() {
    return redirectNode;
  }

  public String getQuorum() {
    return quorum;
  }

  @Override
  public void process(WatchedEvent event) {
    LOG.trace(this.toString() + "Received ZooKeeper Event, " +
      "type=" + event.getType() + ", " +
      "state=" + event.getState() + ", " +
      "path=" + event.getPath());
    switch(event.getType()) {
      // If event type is NONE, this is a connection status change
      case None: {
        connectionEvent(event);
        break;
      }
      case NodeCreated:
      case NodeDeleted:
      case NodeDataChanged:
        fetchKeyAndSwitch();
        break;

      //We will not deal with NodeChildrenChanged, this state should not be received
      default:
        throw new IllegalStateException("Received event is not valid: " + event.getState());
    }

  }

  public void connectionEvent(WatchedEvent event) {
    switch (event.getState()) {
      case SyncConnected:
        if (zooKeeper == null) {
          long finished = System.currentTimeMillis() + this.conf
            .getLong("hbase.zookeeper.watcher.sync.connected.wait", 2000);
          while (System.currentTimeMillis() < finished) {
            try {
              if (this.zooKeeper != null) {
                break;
              }
              Thread.sleep(1);
            } catch (InterruptedException e) {
              LOG.warn("Interrupted while sleeping");
              throw new RuntimeException("Interrupted while waiting for"
                + " recoverableZooKeeper is set");
            }
          }
          // We wait for a moment, zookeeper is still null, this is not right
          if (this.zooKeeper == null) {
            LOG.error("ZK is null on connection event for " + quorum);
            throw new NullPointerException("ZK is null for " + quorum);
          }
        }
        this.identifier = identifier + "-0x" + Long.toHexString(this.zooKeeper.getSessionId());
        // Update our identifier.  Otherwise ignore.
        LOG.debug(this.identifier + " connected");
        fetchKeyAndSwitch();
        if (!everConnected) {
          everConnected = true;
        }
        break;

      // Abort the server if Disconnected or Expired
      case Disconnected:
        LOG.debug(this.toString() + "Received Disconnected from ZooKeeper, ignoring");
        break;

      case Expired:
        LOG.debug(this.toString()+ " received expired from ZooKeeper, reconnecting");
        startThreadForZKReconnection();
        break;

      case ConnectedReadOnly:
      case SaslAuthenticated:
      case AuthFailed:
        break;
      default:
        throw new IllegalStateException("Received event is not valid: " + event.getState());

    }
  }

  public synchronized void stop() {
    if (!this.stopped) {
      this.stopped = true;
      LOG.info("Stopping tracker on cluster " + quorum);
      closeZookeeper();
    }
  }

  private void closeZookeeper() {
    if (zooKeeper != null) {
      try {
        if (zooKeeper != null) {
          zooKeeper.close();
        }
      } catch (InterruptedException e) {
        Thread.currentThread().interrupt();
      }
      zooKeeper = null;
    }
  }

  /**
   * Start a thread to reconnect the zk for Cluster Switch Tracker
   */
  private void startThreadForZKReconnection() {
    if (this.zooKeeper == null) {
      // Already has started a thread to reconnect zk
      return;
    }
    Thread t = new Thread() {
      public void run() {
        if (!zkConnectionLock.tryLock()) {
          return;
        }
        try {
          closeZookeeper();
          while (!stopped) {
            try {
              if (zooKeeper == null) {
                zooKeeper = ZKUtil.connect(conf, quorum, ClusterSwitchTracker.this,
                  identifier);
              }
              LOG.info("Successfully reconnected to cluster switch tracker on zk '"
                + quorum + "'");
              break;
            } catch (Throwable t) {
              LOG.warn("Failed reconnecting to cluster switch tracker on zk '"
                + quorum + "', will retry again...", t);
              Threads.sleep(30 * 1000);
            }
          }
        } finally {
          zkConnectionLock.unlock();
        }
      }
    };
    t.setDaemon(true);
    t.setName("ClusterSwitchTracker-ZK-Reconnection-"
      + System.currentTimeMillis());
    t.start();
  }

  @VisibleForTesting
  public RecoverableZooKeeper getZooKeeper() {
    return zooKeeper;
  }

  @Override
  public String toString() {
    return this.identifier + ", trackNode=" + redirectNode + " ";
  }

}