package com.alibaba.hbase.haclient;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.haclient.dualservice.DualUtil;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;
import org.apache.hadoop.hbase.zookeeper.ZKWatcher;
import org.apache.hadoop.hbase.zookeeper.ZNodePaths;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;

public class ClusterSwitcher implements Closeable {
  static final Log LOG = LogFactory.getLog(ClusterSwitcher.class);

  private Configuration masterConf;

  private Configuration originalConf;

  private ConnectInfo connectInfo;

  private List<Configuration> slaveConfs = new ArrayList<>();

  private ExecutorService executor;

  private int totalClusters;

  private boolean skipCheck = false;

  private boolean isListAction = false;

  private boolean isSwitchAction = false;

  private boolean isSetConnectAction = false;

  private boolean isGetConnectAction = false;

  private boolean isGetLinkCountAction = false;

  private String targetClusterKey = null;

  private Options options = new Options();

  public ClusterSwitcher(ConnectInfo connectInfo, Configuration conf) throws IOException {
    this.connectInfo = connectInfo;
    this.originalConf = conf;
    if(StringUtils.isBlank(conf.get(AliHBaseConstants.HACLIENT_CLUSTER_ID))){
      throw new IOException("haclient cluster id can not be blank");
    }
    addOptions();
    init();
  }

  private void init() throws IOException {
    String masterClusterKey = this.connectInfo.getMasterWatchZK();
    masterConf = ClusterSwitchUtil.createConfWithConnectKey(masterClusterKey, this.originalConf );
    LOG.info("Master cluster: " + masterConf);

    List<String> slaveClusterKeys = this.connectInfo.getSlaveWatchZKList();

    int clusterCount = 1;
    for(String slaveClusterKey : slaveClusterKeys){
      Configuration slaveConf = ClusterSwitchUtil.createConfWithConnectKey(slaveClusterKey, this.originalConf );
      LOG.info("Slave cluster " + clusterCount + " : " + slaveClusterKey);
      slaveConfs.add(slaveConf);
      clusterCount++;
    }
    // master cluster plus all salve clusters
    totalClusters = clusterCount;
    executor = Threads
      .getBoundedCachedThreadPool(totalClusters, 60000, TimeUnit.MILLISECONDS,
        Threads.newDaemonThreadFactory("ClusterSwitcher"));
  }

  private boolean checkIfHBaseCluster(String rootNode,
                                      ZKWatcher zkw) throws KeeperException {
    if (ZKUtil.checkExists(zkw, rootNode) == -1) {
      return false;
    }
    List<String> hbaseNodes = ZKUtil.listChildrenNoWatch(zkw, rootNode);
    if (hbaseNodes.contains("master") && hbaseNodes.contains("rs"))
      return true;
    else {
      return false;
    }
  }

  protected SwitchCommand getCommandFrom(Configuration conf)
    throws IOException, KeeperException, InterruptedException {
    String redirectNode = conf.get(ClusterSwitchUtil.ZOOKEEPER_REDIRECT_NODE,
      ClusterSwitchUtil.ZOOKEEPER_REDIRECT_NODE_DEFAULT);
    byte[] data = getDataFrom(conf, redirectNode);
    if(data == null){
      return SwitchCommand.NOCOMMAND;
    }else{
      SwitchCommand ret = ClusterSwitchUtil.toSwitchCommand(data);
      LOG.debug("Get Command:" + ret + " from " +  conf.get(HConstants.ZOOKEEPER_QUORUM) + redirectNode);
      return ret;
    }
  }

  protected ConnectInfo getConnectInfoFrom(Configuration conf)
    throws IOException, KeeperException, InterruptedException {
    String connectNode = conf.get(ClusterSwitchUtil.ZOOKEEPER_CONNECT_NODE,
      ClusterSwitchUtil.ZOOKEEPER_CONNECT_NODE_DEFAULT);
    byte[] data = getDataFrom(conf, connectNode);
    if(data == null){
      return null;
    }else{
      ConnectInfo result = ConnectInfoUtil.toConnectInfo(data);
      LOG.debug("Get ConnectInfo:" + result + " from " + conf.get(HConstants.ZOOKEEPER_QUORUM) + connectNode);
      return result;
    }
  }

  protected List<String> getDualTablesFrom(Configuration conf)
    throws IOException, KeeperException, InterruptedException {
    String dualTableNode = conf.get(ClusterSwitchUtil.ZOOKEEPER_DUAL_TABLE_NODE,
      ClusterSwitchUtil.ZOOKEEPER_DUAL_TABLE_NODE_DEFAULT);
    byte[] data = getDataFrom(conf, dualTableNode);
    if(data == null){
      return null;
    }else{
      List<String> result = DualUtil.toDualTables(data);
      LOG.debug("Get dual tables:" + result + " from " + conf.get(HConstants.ZOOKEEPER_QUORUM) + dualTableNode);
      return result;
    }
  }

  protected Map<String, JSONObject> getDualTraceFrom(Configuration conf)
    throws IOException, KeeperException, InterruptedException {
    String dualTraceNode = conf.get(ClusterSwitchUtil.ZOOKEEPER_DUAL_TRACE_NODE,
      ClusterSwitchUtil.ZOOKEEPER_DUAL_TRACE_NODE_DEFAULT);
    Map<String, byte[]> data = getChildDataFrom(conf, dualTraceNode);
    if(data == null || data.isEmpty()){
      return null;
    }else{
      Map<String, JSONObject> result = new HashMap<>();
      for(String key : data.keySet()){
        result.put(key, JSONObject.parseObject(DualUtil.getDualTraceMetrics(data.get(key))));
      }
      return result;
    }
  }

  protected Integer getLinkCountFrom(Configuration conf)
    throws IOException, KeeperException, InterruptedException {
    String linkNode = conf.get(ClusterSwitchUtil.ZOOKEEPER_LINK_NODE,
      ClusterSwitchUtil.ZOOKEEPER_LINK_NODE_DEFAULT);
    ZKWatcher zk = new ZKWatcher(conf, "GetLinkCount", null, false);
    try {
      String baseNode = ClusterSwitchUtil.getBaseNode(conf.get(AliHBaseConstants.HACLIENT_BASE_NODE,
        AliHBaseConstants.HACLIENT_BASE_NODE_DEFAULT), this.originalConf.get(AliHBaseConstants.HACLIENT_CLUSTER_ID));
      if (ZKUtil.checkExists(zk, baseNode) == -1) {
        return 0;
      }
      String targetNode = ZNodePaths.joinZNode(baseNode, linkNode);
      if (ZKUtil.checkExists(zk, targetNode) == -1) {
        return 0;
      } else {
        int count = ZKUtil.getNumberOfChildren(zk, targetNode);
        LOG.debug("Get Link Count :" + count + " from " +  conf.get(HConstants.ZOOKEEPER_QUORUM) + linkNode);
        return count;
      }
    } finally {
      if (zk != null) {
        zk.close();
      }
    }
  }

  private byte[] getDataFrom(Configuration conf, String nodeName)throws IOException,
    KeeperException, InterruptedException{
    ZKWatcher zk = new ZKWatcher(conf, "GetCommand", null, false);
    try {
      String baseNode = ClusterSwitchUtil.getBaseNode(conf.get(AliHBaseConstants.HACLIENT_BASE_NODE,
        AliHBaseConstants.HACLIENT_BASE_NODE_DEFAULT), this.originalConf.get(AliHBaseConstants.HACLIENT_CLUSTER_ID));
      if (ZKUtil.checkExists(zk, baseNode) == -1) {
        return null;
      }
      String targetNode = ZNodePaths.joinZNode(baseNode, nodeName);
      if (ZKUtil.checkExists(zk, targetNode) == -1) {
        return null;
      } else {
        byte[] data = ZKUtil.getDataNoWatch(zk, targetNode,null);
        return data;
      }
    } finally {
      if (zk != null) {
        zk.close();
      }
    }
  }

  private Map<String, byte[]> getChildDataFrom(Configuration conf,
                                               String nodeName)throws IOException, KeeperException, InterruptedException{
    ZKWatcher zk = new ZKWatcher(conf, "GetCommand", null, false);
    try {
      String baseNode = ClusterSwitchUtil.getBaseNode(conf.get(AliHBaseConstants.HACLIENT_BASE_NODE,
        AliHBaseConstants.HACLIENT_BASE_NODE_DEFAULT), this.originalConf.get(AliHBaseConstants.HACLIENT_CLUSTER_ID));
      if (ZKUtil.checkExists(zk, baseNode) == -1) {
        return null;
      }
      String targetNode = ZNodePaths.joinZNode(baseNode, nodeName);
      if (ZKUtil.checkExists(zk, targetNode) == -1) {
        return null;
      }
      Map<String, byte[]> result = new HashMap<>();
      List<String> nodes = ZKUtil.listChildrenNoWatch(zk, targetNode);
      for(String node : nodes){
        byte[] data = ZKUtil.getDataNoWatch(zk, ZNodePaths.joinZNode(targetNode, node), null);
        if(data != null && data.length > 0){
          result.put(node, data);
        }
      }
      return result;
    } finally {
      if (zk != null) {
        zk.close();
      }
    }
  }

  protected void checkClusterKey(String targetClusterKey) throws IOException, KeeperException {
    targetClusterKey = targetClusterKey.trim();
    if (targetClusterKey == null) {
      throw new IOException("targetClusterKey should not b e null!");
    }
    //Empty cluster means switch back to master
    if (targetClusterKey.length() == 0) {
      return;
    }
    if(ClusterSwitchUtil.isValidEndpoint(targetClusterKey)){
      //lindorm cluster，skip check
      return;
    }
    if (!ClusterSwitchUtil.isValidClusterKey(targetClusterKey)) {
      throw new IOException(targetClusterKey
        + " is not a valid clusterkey, should be host1,host2:port:/baseNode");
    }
    Configuration configuration = new Configuration(this.masterConf);
    ClusterSwitchUtil.applyClusterKeyToConf(configuration, targetClusterKey);
    ZKWatcher zk = new ZKWatcher(configuration, "checkCluster", null,
      false);
    try {
      if (!checkIfHBaseCluster(configuration
        .get(HConstants.ZOOKEEPER_ZNODE_PARENT,
          HConstants.DEFAULT_ZOOKEEPER_ZNODE_PARENT), zk)) {
        throw new IOException(targetClusterKey + " is not a valid cluster, "
          + "a HBase cluster should have /master and /rs node under znode parent");
      }
    } finally {
      if (zk != null) {
        zk.close();
      }
    }
  }

  @Override
  public void close() {
    this.executor.shutdown();
    try {
      if (!executor.awaitTermination(10, TimeUnit.SECONDS)) {
        executor.shutdownNow();
      }
    } catch (InterruptedException e) {
      executor.shutdownNow();
    }
  }

  protected int switchTo(String targetClusterKey, long ts, boolean skipCheck)
    throws IOException, KeeperException, InterruptedException {
    if(!skipCheck) {
      checkClusterKey(targetClusterKey);
    }
    SwitchCommand command = new SwitchCommand(targetClusterKey, ts);
    CountDownLatch latch = new CountDownLatch(totalClusters);
    List<SendCommandRunner> runners = new ArrayList<>();
    SendCommandRunner runner = new SendCommandRunner(latch, masterConf, command);
    runners.add(runner);
    executor.submit(runner);
    for (Configuration slaveConf: slaveConfs) {
      runner = new SendCommandRunner(latch, slaveConf, command);
      runners.add(runner);
      executor.submit(runner);
    }
    latch.await();
    int totalSuccess = 0;
    for (SendCommandRunner sendCommandRunner : runners) {
      if (sendCommandRunner.hasError()) {
        LOG.warn("Command " + command + " hasn't been set on cluster " + ClusterSwitchUtil
            .getZooKeeperClusterKey(sendCommandRunner.getConf()) + " since ",
          sendCommandRunner.getError());
      } else {
        totalSuccess++;
      }
    }
    return totalSuccess;
  }

  protected List<Pair<String, SwitchCommand>> listCurrentSwitchCommands() throws InterruptedException {
    CountDownLatch latch = new CountDownLatch(totalClusters);
    List<GetCommandRunner> runners = new ArrayList<>();
    GetCommandRunner runner = new GetCommandRunner(latch, masterConf);
    runners.add(runner);
    executor.submit(runner);
    for (Configuration conf: slaveConfs) {
      runner = new GetCommandRunner(latch, conf);
      runners.add(runner);
      executor.submit(runner);
    }
    latch.await();
    List<Pair<String, SwitchCommand>> results = new ArrayList<>();
    for (GetCommandRunner getCommandRunner : runners) {
      String clusterKey = ClusterSwitchUtil
        .getZooKeeperClusterKey(getCommandRunner.getConf());
      if (getCommandRunner.hasError()) {
        LOG.warn("GetCommand failed on cluster " + clusterKey + " since ",
          getCommandRunner.getError());
      }
      Pair<String, SwitchCommand> result = new Pair<String, SwitchCommand>(clusterKey,
        getCommandRunner.getResult());
      results.add(result);
    }
    return results;
  }

  protected List<Pair<String, Integer>> listCurrentLinkCount() throws InterruptedException {
    CountDownLatch latch = new CountDownLatch(totalClusters);
    List<GetLinkCountRunner> runners = new ArrayList<>();
    GetLinkCountRunner runner = new GetLinkCountRunner(latch, masterConf);
    runners.add(runner);
    executor.submit(runner);
    for (Configuration conf: slaveConfs) {
      runner = new GetLinkCountRunner(latch, conf);
      runners.add(runner);
      executor.submit(runner);
    }
    latch.await();
    List<Pair<String, Integer>> results = new ArrayList<>();
    for (GetLinkCountRunner getLinkCountRunner : runners) {
      String clusterKey = ClusterSwitchUtil
        .getZooKeeperClusterKey(getLinkCountRunner.getConf());
      if (getLinkCountRunner.hasError()) {
        LOG.warn("GetCommand failed on cluster " + clusterKey + " since ",
          getLinkCountRunner.getError());
      }
      Pair<String, Integer> result = new Pair<String, Integer>(clusterKey,
        getLinkCountRunner.getResult());
      results.add(result);
    }
    return results;
  }

  public int switchTo(String targetClusterKey)
    throws IOException, KeeperException, InterruptedException {
    return switchTo(targetClusterKey, System.currentTimeMillis(), false);
  }

  public void setConnectNode() throws IOException, KeeperException{
    Configuration conf =
      ClusterSwitchUtil.createConfWithConnectKey(connectInfo.getZkClusterKey(), null);
      sendConnectInfo(conf, this.connectInfo);
  }

  public ConnectInfo getConnectNode() throws IOException, KeeperException, InterruptedException{
    Configuration conf =
      ClusterSwitchUtil.createConfWithConnectKey(connectInfo.getZkClusterKey(), null);
    return getConnectInfoFrom(conf);
  }

  public void setDualTables(List<String> dualTables) throws IOException, KeeperException{
    Configuration conf =
      ClusterSwitchUtil.createConfWithConnectKey(connectInfo.getZkClusterKey(), null);
    sendDualTable(conf, dualTables);
  }

  public List<String> getDualTables() throws IOException, KeeperException, InterruptedException{
    Configuration conf =
      ClusterSwitchUtil.createConfWithConnectKey(connectInfo.getZkClusterKey(), null);
    return getDualTablesFrom(conf);
  }

  public Map<String, JSONObject> getDualTrace() throws IOException, KeeperException,
    InterruptedException{
    Configuration conf =
      ClusterSwitchUtil.createConfWithConnectKey(connectInfo.getZkClusterKey(), null);
    return getDualTraceFrom(conf);
  }

  public String getCurrentActiveCluster() throws InterruptedException, IOException {
    List<Pair<String, SwitchCommand>> results = listCurrentSwitchCommands();
    SwitchCommand commandWithMinTs = SwitchCommand.NOCOMMAND;
    for (Pair<String, SwitchCommand> result : results) {
      SwitchCommand current = result.getSecond();
      if (current == null) {
        continue;
      }
      if (current.getTs() > commandWithMinTs.getTs()) {
        commandWithMinTs = result.getSecond();
      } else if (current.getTs() == commandWithMinTs.getTs()) {
        if (!current.equals(commandWithMinTs)) {
          throw new IOException(
            "There is two command have same timestamp but with different target cluster, one is "
              + current + ", the other one is " + commandWithMinTs
              + ", please issue a new switch command to fix this.");
        }
      }
    }
    if (commandWithMinTs.isSwitchBackToMaster()) {
      return this.connectInfo.getActiveConnectKey();
    } else {
      return commandWithMinTs.getClusterKey();
    }
  }

  public Map<String, Integer> getCurrentLinkCount() throws InterruptedException, IOException {
    List<Pair<String, Integer>> zkCountPairs = listCurrentLinkCount();
    Map<String, Integer> results = new HashMap<>();
    for(Pair<String, Integer> zkCountPair : zkCountPairs){
      String clusterKey = ConnectInfoUtil.getClusterKeyFromConnectInfo(connectInfo,
        zkCountPair.getFirst());
      results.put(clusterKey, zkCountPair.getSecond());
    }
    return results;
  }

  private void sendCommand(Configuration conf, String targetClusterKey, long ts)
    throws IOException, KeeperException {
    String redirectNode = conf.get(ClusterSwitchUtil.ZOOKEEPER_REDIRECT_NODE,
      ClusterSwitchUtil.ZOOKEEPER_REDIRECT_NODE_DEFAULT);
    byte[] data = ClusterSwitchUtil.toSwitchCommandBytes(targetClusterKey, ts);
    sendCommand(conf, data, redirectNode);
    LOG.debug("Set targetClusterKey=" + targetClusterKey + ", ts=" + ts + " on " + redirectNode);
  }

  private void sendConnectInfo(Configuration conf, ConnectInfo connectInfo) throws IOException,
    KeeperException {
    String connectNode = conf.get(ClusterSwitchUtil.ZOOKEEPER_CONNECT_NODE,
      ClusterSwitchUtil.ZOOKEEPER_CONNECT_NODE_DEFAULT);
    byte[] data = ConnectInfoUtil.toConnectInfoBytes(connectInfo);
    sendCommand(conf, data, connectNode);
  }

  private void sendDualTable(Configuration conf, List<String> dualTables) throws IOException,
    KeeperException {
    if(dualTables == null || dualTables.isEmpty()){
      return;
    }
    String dualTableNode = conf.get(ClusterSwitchUtil.ZOOKEEPER_DUAL_TABLE_NODE,
      ClusterSwitchUtil.ZOOKEEPER_DUAL_TABLE_NODE_DEFAULT);
    byte[] data = DualUtil.toDualTablesBytes(dualTables);
    sendCommand(conf, data, dualTableNode);
  }

  private void sendCommand(Configuration conf, byte[] data, String nodeName)
    throws IOException, KeeperException {
    ZKWatcher zk = new ZKWatcher(conf, "ClusterSwitcher", null,
      false);
    try {
      String baseNode = ClusterSwitchUtil.getBaseNode(conf.get(AliHBaseConstants.HACLIENT_BASE_NODE,
        AliHBaseConstants.HACLIENT_BASE_NODE_DEFAULT), this.originalConf.get(AliHBaseConstants.HACLIENT_CLUSTER_ID));
      ZKUtil.createNodeIfNotExistsNoWatch(zk, baseNode, null,
        CreateMode.PERSISTENT);
      String targetNode = ZNodePaths.joinZNode(baseNode, nodeName);
      if (ZKUtil.checkExists(zk, targetNode) == -1) {
        ZKUtil.createNodeIfNotExistsNoWatch(zk, targetNode, data,
          CreateMode.PERSISTENT);
      } else {
        ZKUtil.setData(zk, targetNode, data);
      }
    } finally {
      if (zk != null) {
        zk.close();
      }
    }
  }

  private class SendCommandRunner extends Runner<Boolean> {


    private SwitchCommand command;

    public SendCommandRunner(CountDownLatch latch, Configuration conf, SwitchCommand command) {
      super(latch, conf);
      this.command = command;
    }

    @Override
    public Boolean task() throws IOException, KeeperException, InterruptedException {
      SwitchCommand oldCommand = getCommandFrom(configuration);
      if (oldCommand.getTs() >= command.getTs()) {
        LOG.warn("Old command " + oldCommand
          + " has a ts greater or equal than current command " + command);
      }
      sendCommand(configuration, command.getClusterKey(), command.getTs());
      return true;
    }
  }

  private class GetCommandRunner extends Runner<SwitchCommand> {
    public GetCommandRunner(CountDownLatch latch, Configuration conf) {
      super(latch, conf);
    }

    @Override
    public SwitchCommand task() throws IOException, KeeperException, InterruptedException {
      return getCommandFrom(configuration);
    }
  }

  private class GetLinkCountRunner extends Runner<Integer> {
    public GetLinkCountRunner(CountDownLatch latch, Configuration conf) {
      super(latch, conf);
    }

    @Override
    public Integer task() throws IOException, KeeperException, InterruptedException {
      return getLinkCountFrom(configuration);
    }
  }

  private abstract class Runner<T> implements Runnable {
    private CountDownLatch latch;
    private T result = null;
    private Throwable throwable = null;
    protected Configuration configuration;

    public Runner(CountDownLatch latch, Configuration conf) {
      this.latch = latch;
      this.configuration = conf;
    }

    public Configuration getConf() {
      return configuration;
    }

    public boolean hasError() {
      return throwable != null;
    }

    public T getResult() {
      return result;
    }

    public Throwable getError() {
      return throwable;
    }

    public abstract T task() throws IOException, KeeperException, InterruptedException;

    @Override
    public void run() {
      try {
        result = task();
      } catch (Throwable t) {
        this.throwable = t;
      } finally {
        latch.countDown();
      }
    }
  }

  private void addOptions() {
    options.addOption("switchTo", "switchTo", true,
      "Switch to the giving cluster. Empty string means switch back to master cluster");
    options.addOption("skipCheck", "skipCheck", false,
      "Skip check when switch clusters."
        + "\nWARNING: Skip check may make client switch to a invalid cluster");
    options.addOption("getCurrentActive", "getCurrentActive",
      false, "Get current active cluster");
    options.addOption("setConnectNode", "setConnectNode",
      false, "set connent info to connect node");
    options.addOption("getConnectNode", "getConnectNode",
      false, "get connent info to connect node");
  }

  private void printUsage() {
    HelpFormatter formatter = new HelpFormatter();
    formatter.printHelp("ClusterSwitch", options, true);
  }

  private boolean parseOptions(String args[]) throws ParseException,
    IOException {
    if (args.length == 0) {
      printUsage();
      return false;
    }
    CommandLineParser parser = new PosixParser();
    CommandLine cmd = parser.parse(options, args);
    int commandCount = 0;
    if (cmd.hasOption("switchTo")) {
      isSwitchAction = true;
      targetClusterKey = cmd.getOptionValue("switchTo").trim();
      commandCount ++;
    }
    if (cmd.hasOption("skipCheck")) {
      skipCheck = true;
    }
    if (cmd.hasOption("getCurrentActive")) {
      isListAction = true;
      commandCount ++;
    }
    if(cmd.hasOption("setConnectNode")){
      isSetConnectAction = true;
      commandCount ++;
    }
    if(cmd.hasOption("getConnectNode")){
      isGetConnectAction = true;
      commandCount ++;
    }
    if(cmd.hasOption("getLinkCount")){
      isGetLinkCountAction = true;
      commandCount ++;
    }
    if (commandCount > 1) {
      System.out.println("Only one command can execute at the same time");
      printUsage();
      return false;
    }
    if (!isSwitchAction && !isListAction && !isSetConnectAction && !isGetConnectAction && !isGetLinkCountAction) {
      System.out.println("Please specify the command to execute.");
      printUsage();
      return false;
    }
    return true;
  }

  private int run(String[] args) throws Exception {
    if (!parseOptions(args)) {
      return 1;
    }
    if (isListAction) {
      String currentActive = getCurrentActiveCluster();
      System.out.println("Current Active cluster is: " + currentActive);
    } else if (isSwitchAction) {
      if (skipCheck) {
        System.out.println("WARNING: skipCheck is set, target cluster won't be validated. "
          + "Clients could be switched to a invalid cluster");
      }
      if (targetClusterKey == null || targetClusterKey.isEmpty()) {
        System.out.println(
          "Target cluster is not given, will switch to master cluster: " + this.connectInfo.getActiveConnectKey());
      }
      int success = switchTo(targetClusterKey, System.currentTimeMillis(), skipCheck);
      if (success == totalClusters) {
        System.out.println("Switch to " + targetClusterKey + " is successfully pushed to all "
          + success + " clusters");
      } else if (success == 0) {
        System.out.println("Switch to " + targetClusterKey + " is failed on all " + totalClusters
          + "clusters, please refer to the log for more details.");
      } else {
        System.out.println("Switch to " + targetClusterKey + " is successfully pushed to "
          + success + " clusters, " + (totalClusters - success)
          + " cluster(s) are failed, please refer to the log for more details.");
      }
    }else if(isSetConnectAction){
      try {
        setConnectNode();
      }catch(Exception e){
        System.out.println("Set connect info failed : " + e);
        return -1;
      }
    }else if(isGetConnectAction){
      try {
        ConnectInfo connectInfo = getConnectNode();
        System.out.println("ConnectInfo is: " + connectInfo);
      }catch(Exception e){
        System.out.println("Get connect info failed : " + e);
        return -1;
      }
    }else if(isGetLinkCountAction){
      try {
        Map<String, Integer> linkCounts = getCurrentLinkCount();
        for(Map.Entry<String, Integer> linkCount : linkCounts.entrySet()){
          System.out.println("Cluster : " + linkCount.getKey() + " with link count " + linkCount.getValue());
        }
      }catch(Exception e){
        System.out.println("Get link count failed : " + e);
        return -1;
      }
    }
    return 0;
  }

  public static void main(String[] args) throws Exception {
    ConnectInfo connectInfo = ConnectInfoUtil.getConnectInfoFromXML("hbase-site.xml");
    if(connectInfo.getZkClusterKey() == null){
      System.out.println("ConnectInfo::zkClusterKey is null");
      System.exit(-1);
    }
    Configuration conf = HBaseConfiguration.create();
    ClusterSwitcher switcher = new ClusterSwitcher(connectInfo, conf);
    System.exit(switcher.run(args));
  }
}