package com.alibaba.hbase.haclient.dualservice;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.haclient.AliHBaseMultiClusterConnectionImpl;
import com.alibaba.hbase.haclient.ClusterSwitchUtil;
import com.google.common.annotations.VisibleForTesting;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.hbase.zookeeper.RecoverableZooKeeper;
import org.apache.hadoop.hbase.zookeeper.ZKConfig;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;
import org.apache.hadoop.hbase.zookeeper.ZNodePaths;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;

public class DualConfigTracker implements Watcher {
  private static final Log LOG = LogFactory.getLog(DualConfigTracker.class);

  // zookeeper quorum
  private String quorum;

  // zookeeper connection
  private RecoverableZooKeeper zooKeeper;
  // base znode for this conf
  public String dualTableNode;

  private String identifier;
  private final AliHBaseMultiClusterConnectionImpl connection;

  private Configuration conf;

  private volatile boolean stopped = false;

  private boolean everConnected = false;

  private int expireTime;

  private final int RETRYCOUNT = 10;

  private Lock zkConnectionLock = new ReentrantLock();

  public DualConfigTracker(Configuration conf, AliHBaseMultiClusterConnectionImpl connection)
    throws IOException {
    this.conf = conf;
    this.connection = connection;
    this.quorum = ZKConfig.getZKQuorumServersString(conf);
    this.dualTableNode =
      ZNodePaths.joinZNode(ClusterSwitchUtil.getBaseNode(conf.get(AliHBaseConstants.HACLIENT_BASE_NODE,
      AliHBaseConstants.HACLIENT_BASE_NODE_DEFAULT), conf.get(AliHBaseConstants.HACLIENT_CLUSTER_ID)),
      conf.get(ClusterSwitchUtil.ZOOKEEPER_DUAL_TABLE_NODE, ClusterSwitchUtil.ZOOKEEPER_DUAL_TABLE_NODE_DEFAULT));
    this.expireTime = conf.getInt(HConstants.ZK_SESSION_TIMEOUT,
      HConstants.DEFAULT_ZK_SESSION_TIMEOUT);
    this.identifier = quorum + "_DualConfigTracker0x0";
    this.zooKeeper = ZKUtil.connect(conf, quorum, this, identifier);
  }


  public String getDualTableNode() {
    return dualTableNode;
  }

  public String getQuorum() {
    return quorum;
  }

  @Override
  public void process(WatchedEvent event) {
    LOG.trace(this.toString() + "Received ZooKeeper Event, " +
      "type=" + event.getType() + ", " +
      "state=" + event.getState() + ", " +
      "path=" + event.getPath());
    switch(event.getType()) {
      case None:
        connectionEvent(event);
        break;
      case NodeCreated:
        if(event != null && event.getPath() != null && event.getPath().startsWith((dualTableNode))){
          updateDualConfig();
        }
        break;
      case NodeDeleted:
        break;
      case NodeDataChanged:
        if(event != null && event.getPath() != null && event.getPath().startsWith((dualTableNode))){
          updateDualConfig();
        }
        break;

      //We will not deal with NodeChildrenChanged, this state should not be received
      default:
        throw new IllegalStateException("Received event is not valid: " + event.getState());
    }

  }

  public void connectionEvent(WatchedEvent event) {
    switch (event.getState()) {
      case SyncConnected:
        if (zooKeeper == null) {
          long finished = System.currentTimeMillis() + this.conf
            .getLong("hbase.zookeeper.watcher.sync.connected.wait", 2000);
          while (System.currentTimeMillis() < finished) {
            try {
              if (this.zooKeeper != null) {
                break;
              }
              Thread.sleep(1);
            } catch (InterruptedException e) {
              LOG.warn("Interrupted while sleeping");
              throw new RuntimeException("Interrupted while waiting for"
                + " recoverableZooKeeper is set");
            }
          }
          // We wait for a moment, zookeeper is still null, this is not right
          if (this.zooKeeper == null) {
            LOG.error("ZK is null on connection event for " + quorum);
            throw new NullPointerException("ZK is null for " + quorum);
          }
        }
        this.identifier = identifier + "-0x" + Long.toHexString(this.zooKeeper.getSessionId());
        // Update our identifier.  Otherwise ignore.
        LOG.debug(this.identifier + " connected");
        updateDualConfig();
        if (!everConnected) {
          everConnected = true;
        }
        break;

      // Abort the server if Disconnected or Expired
      case Disconnected:
        LOG.debug(this.toString() + "Received Disconnected from ZooKeeper, ignoring");
        break;

      case Expired:
        LOG.debug(this.toString()+ " received expired from ZooKeeper, reconnecting");
        startThreadForZKReconnection();
        break;

      case ConnectedReadOnly:
      case SaslAuthenticated:
      case AuthFailed:
        break;
      default:
        throw new IllegalStateException("Received event is not valid: " + event.getState());

    }
  }

  public void updateDualConfig(){
    try {
      if (zooKeeper.exists(dualTableNode, true) != null) {
        List<String> dualTables = null;
        try {
            byte[] data = zooKeeper.getData(dualTableNode, true, null);
            dualTables = DualUtil.toDualTables(data);
            DualUtil.flushDualTables(dualTables, this.connection.getConfiguration());
        }catch(Exception e){
          LOG.warn("Get dual tables from zk failed, " + e);
          dualTables = DualUtil.getDualTablesFromXML(conf);
        }
        this.connection.updateDualConfig(dualTables);
      }
    }catch(Exception e){
      LOG.warn("Get data from dual table node " + dualTableNode + " failed " + e);
    }
  }

  public synchronized void stop() {
    if (!this.stopped) {
      this.stopped = true;
      LOG.info("Stopping tracker on cluster " + quorum);
      closeZookeeper();
    }
  }

  private void closeZookeeper() {
    if (zooKeeper != null) {
      try {
        if (zooKeeper != null) {
          zooKeeper.close();
        }
      } catch (InterruptedException e) {
        Thread.currentThread().interrupt();
      }
      zooKeeper = null;
    }
  }

  /**
   * Start a thread to reconnect the zk for Cluster Switch Tracker
   */
  private void startThreadForZKReconnection() {

    if (this.zooKeeper == null) {
      // Already has started a thread to reconnect zk
      return;
    }
    Thread t = new Thread() {
      private int retryCount = RETRYCOUNT;
      public void run() {
        if (!zkConnectionLock.tryLock()) {
          return;
        }
        try {
          closeZookeeper();
          while (!stopped && retryCount > 0) {
            try {
              if (zooKeeper == null) {
                zooKeeper = ZKUtil.connect(conf, quorum, DualConfigTracker.this,
                  identifier);
              }
              LOG.info("Successfully reconnected to dual config tracker on zk '"
                + quorum + "'");
              break;
            } catch (Throwable t) {
              LOG.warn("Failed reconnecting to dual config tracker on zk '"
                + quorum + "', will retry again...", t);
              Threads.sleep(30 * 1000);
            }
            retryCount--;
          }
          if(zooKeeper == null && retryCount == 0){
            LOG.warn("Failed reconnecting to dual config tracker on zk '"
              + quorum + " after " + RETRYCOUNT + " times");
          }
        } finally {
          zkConnectionLock.unlock();
        }
      }
    };
    t.setDaemon(true);
    t.setName("DualConfigTracker-ZK-Reconnection-"
      + System.currentTimeMillis());
    t.start();
  }

  @VisibleForTesting
  public RecoverableZooKeeper getZooKeeper() {
    return zooKeeper;
  }

  @Override
  public String toString() {
    return this.identifier + ", trackNode=" + dualTableNode + " ";
  }

}