package com.alibaba.hbase.haclient.dualservice;

import java.io.IOException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.apache.hadoop.hbase.ipc.CallTimeoutException;
import org.apache.hadoop.hbase.util.Bytes;

public class DualContext<T> {
  private volatile T activeResult;
  private volatile T standbyResult;
  private Throwable activeError;
  private Throwable standbyError;
  private CountDownLatch countDown = new CountDownLatch(1);
  private final int glitchTimeout;
  private final int operationTimeout;
  private byte[] tableName;
  private long start;
  // used to identify success of put operation which return "null"
  private volatile boolean hasValue = false;
  private Boolean usePrimaryAsResult = null;

  public DualContext(final byte[] tableName, int glitchTimeout,
                     int operationTimeout) {
    this.glitchTimeout = glitchTimeout;
    this.operationTimeout = operationTimeout;
    this.tableName = tableName;
  }

  public void onActiveComplete(T result) {
    activeResult = result;
    hasValue = true;
    countDown.countDown();
  }

  public void onStandbyComplete(T result) {
    standbyResult = result;
    hasValue = true;
    countDown.countDown();
  }

  public synchronized void onActiveError(Throwable error) {
    activeError = error;
    if (standbyError != null) {
      countDown.countDown();
    }
  }

  public synchronized void onStandbyError(Throwable error) {
    standbyError = error;
    if (activeError != null) {
      countDown.countDown();
    }
  }

  public T getResult() throws IOException {
    if (hasValue) {
      if (activeResult != null) {
        usePrimaryAsResult = true;
        return activeResult;
      }
      if (standbyResult != null) {
        usePrimaryAsResult = false;
        return standbyResult;
      }
      // in case result itself is null;
      return null;
    }
    if (activeError == null) {
      activeError = new CallTimeoutException("OperationTimeout");
    }
    if (standbyError == null) {
      standbyError = new CallTimeoutException("OperationTimeout");
    }
    // error handling
    long now = System.currentTimeMillis();
    String msg = "Failed exectuing dual operation for table "
      + Bytes.toString(tableName) + " with active error=" + activeError.getMessage()
      + " and with standby error=" + standbyError.getMessage() + ", timeout=" + operationTimeout + ", wait=" + (now - start);
    throw new IOException(msg, activeError);
  }

  public T getResultInGlitchTimeout() throws IOException {
    try {
      if (countDown.await(this.glitchTimeout, TimeUnit.MILLISECONDS)) {
        this.usePrimaryAsResult = true;
        return this.activeResult;
      }
    } catch (InterruptedException e) {
      throw new IOException(e);
    }
    return null;
  }

  public void waitOperationTimeout() throws IOException {
    try {
      long remaining = 0;
      remaining = Math.max(0, operationTimeout + start - System.currentTimeMillis());
      remaining += 10;
      countDown.await(remaining, TimeUnit.MILLISECONDS);
    } catch (InterruptedException e) {
      throw new IOException(e);
    }
  }

  public int getOperationTimeout() {
    return this.operationTimeout;
  }

  public byte[] getTableName() {
    return this.tableName;
  }

  public Boolean usePrimaryAsResult() {
    return this.usePrimaryAsResult;
  }

  public void start() {
    this.start = System.currentTimeMillis();
  }

  public long getStart(){
    return this.start;
  }

  public boolean useStandbyAsResult(){
    return hasValue && activeResult == null;
  }
}