/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.hbase.client;

import java.io.IOException;
import java.util.List;

import com.alibaba.lindorm.client.core.meta.ExternalIndexConfig;
import com.alibaba.lindorm.client.core.meta.ExternalIndexField;
import com.alibaba.lindorm.client.core.meta.ExternalIndexRowFormatterType;
import com.alibaba.lindorm.client.core.meta.ExternalIndexType;
import org.apache.hadoop.hbase.client.Table;

/**
 * AliHBase Table interface which provide features beyond Table interface
 */
public interface AliHBaseTableInterface extends Table {

  /**
   * Add the fields to index to the TableDescriptor
   *
   * @param targetIndexName the target index name, for example, if the indexer is solr, then it will
   *                        be the collection name, if indexer is ES, then index name will be it. The
   *                        targetIndexName will be set to TableIndexConfig and stored in TableDescriptor.
   *                        TargetIndexName will be compared each time if index field changed. The fields in
   *                        one table can only index to the same target. Set targetIndexName to null to use
   *                        the stored targetIndexName directly.
   * @param indexType        the indexer type, only SOLR can supported for now.
   *                        All fields in one table can only be indexed to one indexer, so the type will be
   *                        compared with the previous stored. Set type to null to use the stored type directly.
   * @param rowFormatterType  Rowkey of HBase KeyValue will be indexed to an unique field named 'id' in the external index.
   *                        User may use this 'id' field to read back the whole row from HBase. the 'id' field is String type
   *                        and rowkey of HBase is byte[]. RowFormatterType is used to Tell the system how to index
   *                        rowkey of HBase KeyValue to a unique id in the external indexer.
   *                        If your rowkey can convert to a string, then use ExternalIndexRowFormatterType.STRING.
   *                        It will use Bytes.toString(byte[]) to convert rowkey to an unique "id" string in the external indexer.
   *                        Otherwise use HEX, it wll use org.apache.commons.codec.binary.Hex.encodeAsString(byte[]) to
   *                        convert rowkey to an unique "id" string in the external index
   * @param fields          the field to add. To define a column to index, a ExternalIndexField class need to be
   *                        constructed. For example, if we want to index f:q1 in Solr, the filed name in solr f
   *                        or this field is q1_s, and the data stored in f:q1 is string, type will be STRING.
   *                        Then the code will be like this:
   *                        ExternalIndexField field = new ExternalIndexField("f", "q1", "q1_s", ValueType.STRING)
   *                        When indexing this filed,  Bytes.toString(f:q1) will be used to convert it to a String,
   *                        when type=LONG, then Bytes.toLong(f:q1) will be used. It is important to set field type right,
   *                        otherwise, the converting will wrong and end up in data corruption.
   *                        Type supported:INT, LONG, STRING, BOOLEAN, FLOAT, DOUBLE, SHORT("short"), BIGDECIMAL
   */
  void addExternalIndex(String targetIndexName,
      ExternalIndexType indexType, ExternalIndexRowFormatterType rowFormatterType, List<ExternalIndexField> fields) throws IOException;

  /**
   * @see AliHBaseTableInterface#addExternalIndex(String, ExternalIndexType, ExternalIndexRowFormatterType, List)
   * @param targetIndexName
   * @param type
   * @param fields
   * @throws IOException
   */
  void addExternalIndex(String targetIndexName,
      ExternalIndexType type, ExternalIndexRowFormatterType rowFormatterType, ExternalIndexField... fields) throws IOException;


  /**
   * @see AliHBaseTableInterface#addExternalIndex(String, ExternalIndexType, ExternalIndexRowFormatterType, List)
   * @param config see {@link ExternalIndexConfig} for more details, config target index name, index type and rowformatter type inside config.
   * @param fields
   * @throws IOException
   */
  void addExternalIndex(ExternalIndexConfig config, List<ExternalIndexField> fields) throws IOException;

  /**
   * @see AliHBaseTableInterface#addExternalIndex(String, ExternalIndexType, ExternalIndexRowFormatterType, List)
   * @param config see {@link ExternalIndexConfig} for more details, config target index name, index type and rowformatter type inside config.
   * @param fields
   * @throws IOException
   */
  void addExternalIndex(ExternalIndexConfig config, ExternalIndexField... fields) throws IOException;


  /**
   * @see AliHBaseTableInterface#addExternalIndex(String, ExternalIndexType, ExternalIndexRowFormatterType,  List) for details
   * If index has added before, targetIndexName and type can be retrived by previous index config,
   * so, we can omit those parameters.
   * @param fields
   * @throws IOException
   */
  void addExternalIndex(List<ExternalIndexField> fields) throws IOException;

  /**
   * @see AliHBaseTableInterface#addExternalIndex(List) for detatils
   * @param fields
   * @throws IOException
   */
  void addExternalIndex(ExternalIndexField... fields) throws IOException;

  /**
   * modify external index's attributes
   * @param config the config to modify.
   * @throws IOException
   */
  void modifyExternalIndexAttr(ExternalIndexConfig config) throws IOException;


  /**
   * remove indexed field
   * @param fields the field to remove, field name should include family name and qualifier name
   *               For example, if we want to stop to index a column with family 'f' and qualifier name 'q1'
   *               then filed name will be 'f:q1'
   * @throws IOException
   */
  void removeExternalIndex(List<String> fields) throws IOException;

  void removeExternalIndex(String... fields) throws IOException;

  /**
   * build index for data in the table
   * @throws IOException
   */
  void buildExternalIndex() throws IOException;

  /**
   * Cancel the currently building external index job
   * @throws IOException
   */
  void cancelBuildExternalIndex() throws IOException;

  /**
   * Alter the externalIndex of the table, all the config is described in the json.
   * @param json
   * @throws IOException
   */
  void alterExternalIndex(String json) throws IOException;

  /**
   * describe the external index of the table
   * @return
   * @throws IOException
   */
  String describeExternalIndex() throws IOException;


}
