/*
 * Copyright Alibaba Group Holding Ltd.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.hbase.client;

import com.alibaba.lindorm.client.SystemService;
import com.alibaba.lindorm.client.core.utils.Bytes;
import com.alibaba.lindorm.client.core.utils.Pair;
import org.apache.hadoop.hbase.CellScannable;
import org.apache.hadoop.hbase.CellScanner;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.HBaseIOException;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Append;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Increment;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.Row;
import org.apache.hadoop.hbase.exceptions.UnknownProtocolException;
import org.apache.hadoop.hbase.ipc.HBaseRpcController;
import org.apache.hadoop.hbase.ipc.RpcCallContext;
import org.apache.hadoop.hbase.ipc.RpcServer;
import org.apache.hadoop.hbase.shaded.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.shaded.protobuf.ResponseConverter;
import org.apache.hadoop.hbase.shaded.protobuf.generated.ClientProtos;
import org.apache.hadoop.hbase.shaded.protobuf.generated.ClientProtos.MultiRequest;
import org.apache.hadoop.hbase.shaded.protobuf.generated.ClientProtos.RegionAction;
import org.apache.hadoop.hbase.shaded.protobuf.generated.ClientProtos.MultiResponse;
import org.apache.hadoop.hbase.shaded.protobuf.generated.ClientProtos.RegionActionResult;
import org.apache.hadoop.hbase.shaded.protobuf.generated.HBaseProtos;
import org.apache.hadoop.hbase.util.CollectionUtils;
import org.apache.hbase.thirdparty.com.google.protobuf.RpcController;
import org.apache.hbase.thirdparty.com.google.protobuf.ServiceException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.*;


public class AliHBaseUEClientService
    implements ClientProtos.ClientService.BlockingInterface {

  private static final Logger LOG = LoggerFactory.getLogger(AliHBaseUEClientService.class);

  private final int rowSizeWarnThreshold;

  public static boolean isTest;

  private SystemService systemService;

  private AliHBaseUEConnection connection;

  public AliHBaseUEClientService(AliHBaseUEConnection connection) {
    AliHBaseDirectImplFactory factory = (AliHBaseDirectImplFactory)
        connection.getFactory();
    try {
      this.connection = connection;
      this.systemService = factory.getSystemService();
      this.rowSizeWarnThreshold = connection.getConfiguration()
          .getInt("hbase.rpc.rows.warning.threshold", 5000);
    } catch (IOException e) {
      throw new UnsupportedOperationException("Create systemService exception ",
          e);
    }
  }

  @Override
  public ClientProtos.GetResponse get(RpcController rpcController,
      ClientProtos.GetRequest getRequest) throws ServiceException {
    throw new UnsupportedOperationException(
        "get(rpcController,getRequest) " + "not supported");
  }

  @Override
  public ClientProtos.MutateResponse mutate(RpcController rpcController,
      ClientProtos.MutateRequest mutateRequest) throws ServiceException {
    throw new UnsupportedOperationException(
        "mutate(rpcController,mutateRequest) " + "not supported");
  }

  @Override
  public ClientProtos.ScanResponse scan(RpcController rpcController,
      ClientProtos.ScanRequest scanRequest) throws ServiceException {
    throw new UnsupportedOperationException(
        "scan(rpcController,scanRequest) " + "not supported");
  }

  @Override
  public ClientProtos.BulkLoadHFileResponse bulkLoadHFile(
      RpcController rpcController,
      ClientProtos.BulkLoadHFileRequest bulkLoadHFileRequest)
      throws ServiceException {

    checkBulkLoadSupported(bulkLoadHFileRequest);

    List<ClientProtos.BulkLoadHFileRequest.FamilyPath> paths = bulkLoadHFileRequest
        .getFamilyPathList();

    List<Pair<byte[], String>> familyPaths = new ArrayList<>(paths.size());

    for (ClientProtos.BulkLoadHFileRequest.FamilyPath fp : paths) {
      familyPaths.add(new Pair<>(fp.getFamily().toByteArray(), fp.getPath()));
    }


    try {
      byte[][] regionInfos = HRegionInfo
          .parseRegionName(bulkLoadHFileRequest.getRegion().getValue().toByteArray());
      TableName tb = TableName.valueOf(regionInfos[0]);

      boolean bulkloadResult = systemService
          .bulkLoadLdFiles(systemService.getLConnection().getAllIDC().get(0),
              tb.getNamespaceAsString(), tb.getQualifierAsString(),
              regionInfos[1], familyPaths);
      if (!bulkloadResult) {
        throw new IOException("Bulkload region fail: " + Bytes.toStringBinary(
            bulkLoadHFileRequest.getRegion().getValue().toByteArray()));
      }

    } catch (Exception e) {
      LOG.info("BulkLoad failed, it can be retried later ", e);
      throw new ServiceException("BulkLoad exception ", e);
    }

    ClientProtos.BulkLoadHFileResponse.Builder builder = ClientProtos.BulkLoadHFileResponse
        .newBuilder();
    builder.setLoaded(true);

    return builder.build();
  }

  /**
   * Check if bulkLoad request supported
   * @param bulkLoadHFileRequest
   */
  private void checkBulkLoadSupported(
      ClientProtos.BulkLoadHFileRequest bulkLoadHFileRequest) {
    if (bulkLoadHFileRequest.getCopyFile()) {
      throw new UnsupportedOperationException(
          "BulkLoad with copyFile = true unsupported ");
    }

    List<String> allIdcs = systemService.getLConnection().getAllIDC();

    if (allIdcs.size() > 1 && !isTest) {
      throw new IllegalStateException("BulkLoad only support single idc now"
          + " , current cluster idc count : " + allIdcs.size());
    }

  }

  @Override
  public ClientProtos.PrepareBulkLoadResponse prepareBulkLoad(
      RpcController rpcController,
      ClientProtos.PrepareBulkLoadRequest prepareBulkLoadRequest)
      throws ServiceException {
    ClientProtos.PrepareBulkLoadResponse.Builder builder = ClientProtos.PrepareBulkLoadResponse
        .newBuilder();
    builder.setBulkToken("mock_token");
    return builder.build();
  }

  @Override
  public ClientProtos.CleanupBulkLoadResponse cleanupBulkLoad(
      RpcController rpcController,
      ClientProtos.CleanupBulkLoadRequest cleanupBulkLoadRequest)
      throws ServiceException {
    return ClientProtos.CleanupBulkLoadResponse.newBuilder().build();
  }

  @Override
  public ClientProtos.CoprocessorServiceResponse execService(
      RpcController rpcController,
      ClientProtos.CoprocessorServiceRequest coprocessorServiceRequest)
      throws ServiceException {
    throw new UnsupportedOperationException(
        "execService(rpcController,coprocessorServiceRequest) "
            + "not supported");
  }

  @Override
  public ClientProtos.CoprocessorServiceResponse execRegionServerService(
      RpcController rpcController,
      ClientProtos.CoprocessorServiceRequest coprocessorServiceRequest)
      throws ServiceException {
    throw new UnsupportedOperationException(
        "execRegionServerService(rpcController,coprocessorServiceRequest) "
            + "not supported");
  }


  private void checkBatchSizeAndLogLargeSize(MultiRequest request) {
    int sum = 0;
    String firstRegionName = null;
    for (RegionAction regionAction : request.getRegionActionList()) {
      if (sum == 0) {
        firstRegionName = org.apache.hadoop.hbase.util.Bytes
            .toStringBinary(regionAction.getRegion().getValue().toByteArray());
      }
      sum += regionAction.getActionCount();
    }
    if (sum > rowSizeWarnThreshold) {
      LOG.warn("Large batch operation detected (greater than " + rowSizeWarnThreshold
          + ") (HBASE-18023)." + " Requested Number of Rows: " + sum + " Client: "
          + RpcServer.getRequestUserName().orElse(null) + "/"
          + RpcServer.getRemoteAddress().orElse(null)
          + " first region in multi=" + firstRegionName);
    }
  }

  @Override
  public ClientProtos.MultiResponse multi(RpcController rpcc,
      ClientProtos.MultiRequest request) throws ServiceException {

    checkBatchSizeAndLogLargeSize(request);

    // rpc controller is how we bring in data via the back door;  it is unprotobuf'ed data.
    // It is also the conduit via which we pass back data.
    HBaseRpcController controller = (HBaseRpcController) rpcc;
    CellScanner cellScanner =
        controller != null ? controller.cellScanner() : null;
    if (controller != null) {
      controller.setCellScanner(null);
    }
    // this will contain all the cells that we need to return. It's created later, if needed.
    List<CellScannable> cellsToReturn = null;
    MultiResponse.Builder responseBuilder = MultiResponse.newBuilder();
    RegionActionResult.Builder regionActionResultBuilder = RegionActionResult
        .newBuilder();
    RpcCallContext context = RpcServer.getCurrentCall().orElse(null);

    for (RegionAction regionAction : request.getRegionActionList()) {
      if (regionAction.hasAtomic() && regionAction.getAtomic()) {
        throw new UnsupportedOperationException("Atomic action unsupported ");
      }
      regionActionResultBuilder.clear();
      HBaseProtos.RegionSpecifier regionSpecifier = regionAction.getRegion();
      try {
        byte[][] regionInfos = HRegionInfo
            .parseRegionName(regionSpecifier.getValue().toByteArray());
        TableName tb = TableName.valueOf(regionInfos[0]);
        cellsToReturn = doNonAtomicRegionMutation(tb, regionAction, cellScanner,
            regionActionResultBuilder, cellsToReturn, context);
      } catch (IOException e) {
        throw new ServiceException("Multi exception ", e);
      }
      responseBuilder.addRegionActionResult(regionActionResultBuilder.build());
    }
    // Load the controller with the Cells to return.
    if (cellsToReturn != null && !cellsToReturn.isEmpty()
        && controller != null) {
      controller.setCellScanner(CellUtil.createCellScanner(cellsToReturn));
    }

    return responseBuilder.build();
  }

  private List<CellScannable> doNonAtomicRegionMutation(final TableName tb,
      final RegionAction actions, final CellScanner cellScanner,
      final RegionActionResult.Builder builder,
      List<CellScannable> cellsToReturn, RpcCallContext context) {
    ClientProtos.ResultOrException.Builder resultOrExceptionBuilder = ClientProtos.ResultOrException
        .newBuilder();
    boolean hasResultOrException;
    List<Row> rows = new ArrayList<>();
    for (ClientProtos.Action action : actions.getActionList()) {
      hasResultOrException = false;
      resultOrExceptionBuilder.clear();
      try {
        Result r = null;
        if (action.hasGet()) {
          ClientProtos.Get pbGet = action.getGet();
          // An asynchbase client, https://github.com/OpenTSDB/asynchbase, starts by trying to do
          // a get closest before. Throwing the UnknownProtocolException signals it that it needs
          // to switch and do hbase2 protocol (HBase servers do not tell clients what versions
          // they are; its a problem for non-native clients like asynchbase. HBASE-20225.
          if (pbGet.hasClosestRowBefore() && pbGet.getClosestRowBefore()) {
            throw new UnknownProtocolException(
                "Is this a pre-hbase-1.0.0 or asynchbase client? "
                    + "Client is invoking getClosestRowBefore removed in hbase-2.0.0 replaced by "
                    + "reverse Scan.");
          }
          Get get = ProtobufUtil.toGet(pbGet);
          rows.add(get);
        } else if (action.hasServiceCall()) {
          throw new UnsupportedOperationException(
              "Service call in multi unsupported !");
        } else if (action.hasMutation()) {
          ClientProtos.MutationProto.MutationType type = action.getMutation()
              .getMutateType();
          switch (type) {
          case APPEND:
            Append append = ProtobufUtil
                .toAppend(action.getMutation(), cellScanner);
            rows.add(append);
            break;
          case INCREMENT:
            Increment increment = ProtobufUtil
                .toIncrement(action.getMutation(), cellScanner);
            rows.add(increment);
            break;
          case PUT:
            Put put = ProtobufUtil.toPut(action.getMutation(), cellScanner);
            rows.add(put);
            break;
          case DELETE:
            Delete delete = ProtobufUtil
                .toDelete(action.getMutation(), cellScanner);
            rows.add(delete);
            break;
          default:
            throw new DoNotRetryIOException(
                "Unsupported mutate type: " + type.name());
          }
        } else {
          throw new HBaseIOException("Unexpected Action type");
        }
        // Could get to here and there was no result and no exception.  Presumes we added
        // a Put or Delete to the collecting Mutations List for adding later.  In this
        // case the corresponding ResultOrException instance for the Put or Delete will be added
        // down in the doNonAtomicBatchOp method call rather than up here.
      } catch (IOException ie) {
        hasResultOrException = true;
        HBaseProtos.NameBytesPair pair = ResponseConverter.buildException(ie);
        resultOrExceptionBuilder.setException(pair);
        context.incrementResponseExceptionSize(pair.getSerializedSize());
      }
      if (hasResultOrException) {
        // Propagate index.
        resultOrExceptionBuilder.setIndex(action.getIndex());
        builder.addResultOrException(resultOrExceptionBuilder.build());
      }
    }
    // Finish up any outstanding mutations
    if (!rows.isEmpty()) {
      Object[] results = new Object[rows.size()];
      try {
        connection.getTable(tb).batch(rows, results);
        for (int i = 0; i < rows.size(); i++) {
          Object r = results[i];
          if (r instanceof Throwable) {
            Throwable throwable = (Throwable) r;
            builder.addResultOrException(getResultOrException(
                new Exception("Action exception ", throwable), i));
          } else {
            ClientProtos.Result pbResult = ProtobufUtil.toResult((Result) r);
            resultOrExceptionBuilder.clear();
            resultOrExceptionBuilder.setResult(pbResult);
            resultOrExceptionBuilder.setIndex(i);
            builder.addResultOrException(resultOrExceptionBuilder.build());
          }
        }
      } catch (Exception e) {
        for (int i = 0; i < results.length; i++) {
          Object r = results[i];
          if (r instanceof Throwable) {
            ((Throwable) r).printStackTrace();
            builder.addResultOrException(getResultOrException(
                new Exception("Action exception ", (Throwable) r), i));
          } else {
            builder.addResultOrException(getResultOrException(e, i));
          }
        }
      }
    }
    return cellsToReturn;
  }

  private static ClientProtos.ResultOrException getResultOrException(final Exception e, final int index) {
    return getResultOrException(ResponseConverter.buildActionResult(e), index);
  }

  private static ClientProtos.ResultOrException getResultOrException(
      final ClientProtos.ResultOrException.Builder builder, final int index) {
    return builder.setIndex(index).build();
  }

}
