package com.alibaba.hbase.haclient.dualservice;

import com.alibaba.hbase.client.AliHBaseConstants;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.TableName;

public class AutoSwitch {

  private static final Log LOG = LogFactory.getLog(AutoSwitch.class);

  private boolean autoSwitchEnable;

  private long autoSwitchInterval;

  private Configuration conf;

  private SwitchLevel switchLevel;

  private ExecuteCounter executeCounter;

  private ExecuteCounter tableCounter;

  private volatile boolean locateAvaliable = true;

  private long switchTime = 0l;

  public AutoSwitch(Configuration conf){
    this.conf = conf;
    this.autoSwitchEnable = conf.getBoolean(AliHBaseConstants.AUTOSWITCH_ENABLE,
      AliHBaseConstants.DEFAULT_AUTOSWITCH_ENABLE);
    this.switchLevel = SwitchLevel.valueOf(conf.get(AliHBaseConstants.AUTOSWITCH_SWITCH_LEVEL,
      AliHBaseConstants.DEFAULT_AUTOSWITCH_SWITCH_LEVEL.toString()));
    initExecuteCounter(this.switchLevel);
    this.autoSwitchInterval = conf.getLong(AliHBaseConstants.AUTOSWITCH_INTERVAL,
      AliHBaseConstants.DEFAULT_AUTOSWITCH_INTERVAL);
  }

  private void initExecuteCounter(SwitchLevel switchLevel){
    switch(this.switchLevel){
      case KEY:
        executeCounter = new KeyCounter(this.conf);
        tableCounter = new TableCounter(this.conf);
        break;
      case TABLE:
        executeCounter = new TableCounter(this.conf);
        break;
      case CONNECTION:
        executeCounter = new ConnectionCounter(this.conf);
        break;
    }
  }

  private <T> ExecuteCounter getExecuteCounter(DualContext<T> dualContext){
    if(switchLevel == SwitchLevel.KEY){
      if(dualContext.getActionType() != null){
        return executeCounter;
      }else{
        return tableCounter;
      }
    }else{
      return executeCounter;
    }
  }

  public <T> void update(DualContext<T> dualContext){
    if(!autoSwitchEnable){
      return;
    }
    ExecuteCounter counter = getExecuteCounter(dualContext);
    counter.update(dualContext);
  }

  public synchronized <T> ExecuteStrategy getExecuteStrategy(DualContext<T> dualContext){
    if(!autoSwitchEnable){
      return ExecuteStrategy.DEFAULT;
    }
    ExecuteCounter counter = getExecuteCounter(dualContext);
    if(!counter.reachLimit(dualContext)){
      if(counter.isSwitchStatus(dualContext)){
        return ExecuteStrategy.SWITCH;
      }else{
        return ExecuteStrategy.DEFAULT;
      }
    }else{
      if(!counter.isSwitchStatus(dualContext)){
        switchTime = System.currentTimeMillis();
        counter.setSwitchStatus(dualContext, true);
        LOG.debug(TableName.valueOf(dualContext.getTableName()).getNameAsString() + " active error " +
          "count reach limit, switch to standby");
        return ExecuteStrategy.SWITCH;
      }else if((System.currentTimeMillis() - switchTime) > autoSwitchInterval){
        counter.setSwitchStatus(dualContext,false);
        LOG.debug(TableName.valueOf(dualContext.getTableName()).getNameAsString() + " old active " +
          "success count reach limit, switch back to old active");
        locateAvaliable = true;
        return ExecuteStrategy.DEFAULT;
      }else{
        return ExecuteStrategy.SWITCH;
      }
    }
  }

  public enum SwitchLevel{
    CONNECTION("CONNECTION"),
    TABLE("TABLE"),
    KEY("KEY");

    private final String value;

    SwitchLevel(String value){
      this.value = value;
    }

    public String toString(){
      return this.value;
    }
  }

  public enum ExecuteStrategy{
    DEFAULT,
    SWITCH;
  }

  public boolean isAutoSwitchEnable() {
    return autoSwitchEnable;
  }

  public void setAutoSwitchEnable(boolean autoSwitchEnable) {
    this.autoSwitchEnable = autoSwitchEnable;
  }

  public SwitchLevel getSwitchLevel() {
    return switchLevel;
  }

  public void setSwitchLevel(SwitchLevel switchLevel) {
    this.switchLevel = switchLevel;
  }

  public boolean needKeyInfo(){
    return locateAvaliable && switchLevel == SwitchLevel.KEY ;
  }

  public void reset(){
    initExecuteCounter(this.switchLevel);
  }

  public void setLocateAvaliable(boolean locateAvaliable) {
    this.locateAvaliable = locateAvaliable;
  }
}