package com.alibaba.hbase.haclient.dualservice;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import com.alibaba.hbase.client.AliHBaseConstants;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;

public class ConnectionCounter implements ExecuteCounter {

  private static final Log LOG = LogFactory.getLog(ConnectionCounter.class);

  private ResultCounter resultCounter;

  private boolean switchStatus = false;

  private Configuration conf;

  private int activeFailLimit;

  private float activeFailLimitPercentage;

  private int standbySuccessLimit;

  private ScheduledExecutorService scheduledExecutorService = Executors.newScheduledThreadPool(1);

  public ConnectionCounter(Configuration conf){
    this.conf = conf;
    this.resultCounter = new ResultCounter(ResultCounter.CountStrategy.PERCENTAGE);
    this.activeFailLimit = this.conf.getInt(AliHBaseConstants.CONNECTION_ACTIVE_FAIL_LIMIT,
      AliHBaseConstants.DEFAULT_CONNECTION_ACTIVE_FAIL_LIMIT);
    this.activeFailLimitPercentage =
      this.conf.getFloat(AliHBaseConstants.CONNECTION_ACTIVE_FAIL_LIMIT_PECENTAGE,
      AliHBaseConstants.DEFAULT_CONNECTION_ACTIVE_FAIL_LIMIT_PECENTAGE);
    this.standbySuccessLimit = this.conf.getInt(AliHBaseConstants.CONNECTION_STANDBY_SUCCESS_LIMIT,
      AliHBaseConstants.DEFAULT_CONNECTION_STANDBY_SUCCESS_LIMIT);
    long period = conf.getLong(CLEAR_INTERVAL, DEFAULT_CLEAR_INTERVAL);
    scheduledExecutorService.scheduleAtFixedRate(new Thread("connection-counter") {
      @Override
      public void run() {
        resultCounter.updateActiveErrorPercentage();
      }
    }, period, period, TimeUnit.MILLISECONDS);
  }

  @Override
  public synchronized <T> void update(DualContext<T> dualContext){
    boolean isSwitch = this.isSwitchStatus(dualContext);
    if(dualContext.getExecuteStrategy() == AutoSwitch.ExecuteStrategy.DEFAULT && isSwitch){
      return;
    }
    if(dualContext.getExecuteStrategy() == AutoSwitch.ExecuteStrategy.SWITCH && !isSwitch){
      return;
    }
    this.resultCounter.increment(dualContext);
  }

  @Override
  public synchronized <T> boolean isSwitchStatus(DualContext<T> dualContext) {
    return switchStatus;
  }

  @Override
  public synchronized <T> void setSwitchStatus(DualContext<T> dualContext, boolean switchStatus) {
    this.switchStatus = switchStatus;
    this.reset(dualContext);
  }

  @Override
  public synchronized <T> boolean reachLimit(DualContext<T> dualContext){
    boolean result;
    if(!switchStatus){
      result =
        (resultCounter.getActiveFailCount() > this.activeFailLimit) || (resultCounter.getActiveFailPercentage() > this.activeFailLimitPercentage);
    }else{
      result = resultCounter.getStandbySuccessCount() > this.standbySuccessLimit;
    }
    if(result) {
      LOG.debug("Reach limit active fail count=" + this.activeFailLimit +
        ", active fail count percentage=" + resultCounter.getActiveFailPercentage() +
        ", standby success count=" + resultCounter.getStandbySuccessCount() +
        ", total fail count=" + resultCounter.getTotalActiveFailCount());
    }
    return result;
  }

  @Override
  public synchronized <T> void reset(DualContext<T> dualContext){
    this.resultCounter.totalAndActiveCountClear();
    this.resultCounter.activeCountClear();
    this.resultCounter.standbyCountClear();
  }

}