package com.alibaba.hbase.haclient.dualservice;

import java.util.concurrent.atomic.AtomicLong;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class ResultCounter {

  private static final Log LOG = LogFactory.getLog(ResultCounter.class);

  private AtomicLong activeErrorCount = new AtomicLong(0);

  private AtomicLong activeTimeoutCount = new AtomicLong(0);

  private AtomicLong standbySuccessCount = new AtomicLong(0);

  private AtomicLong totalRequestCount = new AtomicLong(0);

  private AtomicLong totalActiveErrorCount = new AtomicLong(0);

  private AtomicLong totalActiveTimeoutCount = new AtomicLong(0);

  private volatile double activeFailPecentage = 0.0D;

  private CountStrategy strategy;

  public ResultCounter(CountStrategy strategy){
    this.strategy = strategy;
  }

  public<T> void increment(DualContext<T> dualContext) {
    if(strategy == CountStrategy.PERCENTAGE){
      totalRequestCount.incrementAndGet();
    }
    if (dualContext.getExecuteStrategy() == AutoSwitch.ExecuteStrategy.DEFAULT){
      if (dualContext.usePrimaryAsResult()) {
        //active cluster is fine
        activeCountClear();
        return;
      }
      if (dualContext.activeHasError()) {
        //active result with error
        activeErrorCount.incrementAndGet();
        if(strategy == CountStrategy.PERCENTAGE){
          totalActiveErrorCount.incrementAndGet();
        }
      } else if (dualContext.useStandbyAsResult()) {
        //use standby as result, active is timeout
        activeTimeoutCount.incrementAndGet();
        if(strategy == CountStrategy.PERCENTAGE){
          totalActiveTimeoutCount.incrementAndGet();
        }
        standbySuccessCount.incrementAndGet();
      } else {
        //active and standby all timeout or standy has error
        standbySuccessCount.set(0);
      }
    }else{
      if (dualContext.useStandbyAsResult()){
        if(dualContext.activeHasError()){
          standbySuccessCount.set(0);
        }
        return;
      }
      if (dualContext.usePrimaryAsResult()) {
        standbySuccessCount.incrementAndGet();
        return;
      }
      //active and standby all timeout or standy has error
      standbySuccessCount.set(0);
    }
  }

  public long getActiveFailCount(){
    return activeErrorCount.get() + activeTimeoutCount.get();
  }

  public long getTotalActiveFailCount(){
    return totalActiveErrorCount.get() + totalActiveTimeoutCount.get();
  }

  public long getStandbySuccessCount(){
    return standbySuccessCount.get();
  }

  public long totalRequestCount(){
    return totalRequestCount.get();
  }

  public void activeCountClear(){
    activeErrorCount.set(0);
    activeTimeoutCount.set(0);
  }

  public void totalActiveCountClear(){
    totalActiveErrorCount.set(0);
    totalActiveTimeoutCount.set(0);
  }

  public void totalAndActiveCountClear(){
    totalRequestCount.set(0);
    totalActiveCountClear();
  }

  public void standbyCountClear(){
    standbySuccessCount.set(0);
  }

  public void updateActiveErrorPercentage(){
    long errorCount = getTotalActiveFailCount();
    long total = totalRequestCount.get();
    if(total == 0 || total <= 100){
      activeFailPecentage = 0.0D;
    }
    activeFailPecentage = errorCount/(double)total;
    totalAndActiveCountClear();
  }

  public double getActiveFailPercentage(){
    return activeFailPecentage;
  }

  public enum CountStrategy{
    SUCCESSION,
    PERCENTAGE;
  }
}