package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.haclient.AdminUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.CacheEvictionStats;
import org.apache.hadoop.hbase.ClusterMetrics;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.NamespaceDescriptor;
import org.apache.hadoop.hbase.NamespaceNotFoundException;
import org.apache.hadoop.hbase.RegionMetrics;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.TableNotFoundException;
import org.apache.hadoop.hbase.client.index.AliHBaseIndexDescriptor;
import org.apache.hadoop.hbase.client.replication.TableCFs;
import org.apache.hadoop.hbase.client.security.SecurityCapability;
import org.apache.hadoop.hbase.ipc.CoprocessorRpcChannel;
import org.apache.hadoop.hbase.quotas.QuotaFilter;
import org.apache.hadoop.hbase.quotas.QuotaRetriever;
import org.apache.hadoop.hbase.quotas.QuotaSettings;
import org.apache.hadoop.hbase.replication.ReplicationException;
import org.apache.hadoop.hbase.replication.ReplicationPeerConfig;
import org.apache.hadoop.hbase.replication.ReplicationPeerDescription;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;

public class AliHBaseMultiAdmin extends HBaseAdmin implements AliHBaseAdminInterface {
  private static final Log LOG = LogFactory.getLog(AliHBaseMultiAdmin.class);

  private volatile HBaseAdmin currentHBaseAdmin = null;
  //switch count decide switch or not
  private long switchCount = 0;
  private AliHBaseMultiClusterConnection connection;

  public AliHBaseMultiAdmin(AliHBaseMultiClusterConnection connection,
                            HBaseAdmin hbaseAdmin) throws IOException{
    super(connection);
    this.currentHBaseAdmin = hbaseAdmin;
    this.connection = connection;
    this.switchCount = connection.getSwitchCount();
  }

  private HBaseAdmin getCurrentHBaseAdmin() throws IOException {
    if(this.switchCount < connection.getSwitchCount()){
      AliHBaseConstants.ClusterType clusterType =
        AliHBaseConstants.ClusterType.valueOf(this.connection.getConfiguration().get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
      //only HBase to HBase no need update BufferedMutator
      if(!(this.currentHBaseAdmin instanceof HBaseAdmin && clusterType == AliHBaseConstants.ClusterType.HBASE)) {
        HBaseAdmin newHBaseAdmin = connection.getHBaseAdmin();
        HBaseAdmin lastHBaseAdmin = this.currentHBaseAdmin;
        this.currentHBaseAdmin = newHBaseAdmin;
        if (lastHBaseAdmin != null) {
          try {
            lastHBaseAdmin.close();
          } catch (IOException e) {
            LOG.warn("last hbaseAdmin close failed" + e);
          }
        }
      }
      this.switchCount = connection.getSwitchCount();
    }
    return this.currentHBaseAdmin;
  }

  @Override
  public int getOperationTimeout() {
    return this.currentHBaseAdmin.getOperationTimeout();
  }

  @Override
  public void abort(String why, Throwable e) {
    this.currentHBaseAdmin.abort(why, e);
  }

  @Override
  public boolean isAborted() {
    return this.currentHBaseAdmin.isAborted();
  }

  @Override
  public void close() throws IOException {
    this.currentHBaseAdmin.close();
  }

  @Override
  public Configuration getConfiguration() {
    return this.currentHBaseAdmin.getConfiguration();
  }

  @Override
  public boolean tableExists(TableName tableName) throws IOException {
    return this.getCurrentHBaseAdmin().tableExists(tableName);
  }

  @Override
  public Connection getConnection() {
    return connection;
  }

  @Override
  public HTableDescriptor[] listTables() throws IOException {
    return this.getCurrentHBaseAdmin().listTables();
  }

  @Override
  public List<TableDescriptor> listTableDescriptors() throws IOException {
    return this.getCurrentHBaseAdmin().listTableDescriptors();
  }

  @Override
  public HTableDescriptor[] listTables(Pattern pattern) throws IOException {
    return listTables();
  }

  @Override
  public List<TableDescriptor> listTableDescriptors(Pattern pattern) throws IOException {
    return listTableDescriptors();
  }

  @Override
  public HTableDescriptor[] listTables(String regex) throws IOException {
    return listTables();
  }

  @Override
  public HTableDescriptor[] listTables(Pattern pattern, boolean includeSysTables)
    throws IOException {
    return listTables();
  }

  @Override
  public List<TableDescriptor> listTableDescriptors(Pattern pattern, boolean includeSysTables)
    throws IOException {
    return listTableDescriptors();
  }

  @Override
  public HTableDescriptor[] listTables(String regex, boolean includeSysTables) throws IOException {
    return listTables();
  }

  @Override
  public TableName[] listTableNames() throws IOException {
    return this.getCurrentHBaseAdmin().listTableNames();
  }

  @Override
  public TableName[] listTableNames(Pattern pattern) throws IOException {
    return listTableNames();
  }

  @Override
  public TableName[] listTableNames(String regex) throws IOException {
    return listTableNames();
  }

  @Override
  public TableName[] listTableNames(Pattern pattern, boolean includeSysTables) throws IOException {
    return listTableNames();
  }

  @Override
  public TableName[] listTableNames(String regex, boolean includeSysTables) throws IOException {
    return listTableNames();
  }

  @Override
  public HTableDescriptor getTableDescriptor(TableName tableName)
    throws TableNotFoundException, IOException {
    TableDescriptor tableDescriptor = getDescriptor(tableName);
    return new HTableDescriptor(tableDescriptor);
  }

  @Override
  public TableDescriptor getDescriptor(TableName tableName)
    throws TableNotFoundException, IOException {
    return this.getCurrentHBaseAdmin().getDescriptor(tableName);
  }

  @Override
  public HTableDescriptor[] listTableDescriptorsByNamespace(String name) throws IOException {
    return this.getCurrentHBaseAdmin().listTableDescriptorsByNamespace(name);
  }

  @Override
  public List<TableDescriptor> listTableDescriptorsByNamespace(byte[] name) throws IOException {
    return this.getCurrentHBaseAdmin().listTableDescriptorsByNamespace(name);
  }

  @Override
  public TableName[] listTableNamesByNamespace(String name) throws IOException {
    return this.getCurrentHBaseAdmin().listTableNamesByNamespace(name);
  }

  @Override
  public HTableDescriptor[] getTableDescriptorsByTableName(List<TableName> tableNames)
    throws IOException {
    return this.getCurrentHBaseAdmin().getTableDescriptorsByTableName(tableNames);
  }

  @Override
  public List<TableDescriptor> listTableDescriptors(List<TableName> tableNames) throws IOException {
    return this.getCurrentHBaseAdmin().listTableDescriptors(tableNames);
  }

  @Override
  public HTableDescriptor[] getTableDescriptors(List<String> names) throws IOException {
    List<TableName> tableNames = names.stream().map(name -> TableName.valueOf(name)).collect(
      Collectors.toList());
    return getTableDescriptorsByTableName(tableNames);
  }

  @Override
  public void createTable(TableDescriptor desc) throws IOException {
    createTable(desc, null);
  }

  @Override
  public void createTable(TableDescriptor desc, byte[] startKey, byte[] endKey, int numRegions)
    throws IOException {
    if(numRegions < 3) {
      throw new IllegalArgumentException("Must create at least three regions");
    } else if(Bytes.compareTo(startKey, endKey) >= 0) {
      throw new IllegalArgumentException("Start key must be smaller than end key");
    }
    if (numRegions == 3) {
      createTable(desc, new byte[][]{startKey, endKey});
      return;
    }
    byte [][] splitKeys = Bytes.split(startKey, endKey, numRegions - 3);
    if(splitKeys == null || splitKeys.length != numRegions - 1) {
      throw new IllegalArgumentException("Unable to split key range into enough regions");
    }
    createTable(desc, splitKeys);
  }

  @Override
  public void createTable(TableDescriptor desc, byte[][] splitKeys) throws IOException {
    this.getCurrentHBaseAdmin().createTable(desc, splitKeys);
    try{
      AdminUtil.setCreateInfoNode(this.connection.getOriginalConf(), desc, splitKeys);
    }catch(Exception e){
      throw new IOException(e);
    }
  }

  @Override
  public void deleteTable(TableName tableName) throws IOException {
    this.getCurrentHBaseAdmin().deleteTable(tableName);
  }

  @Override
  public void truncateTable(TableName tableName, boolean preserveSplits) throws IOException {
    this.getCurrentHBaseAdmin().truncateTable(tableName, preserveSplits);
  }

  @Override
  public void enableTable(TableName tableName) throws IOException {
    this.getCurrentHBaseAdmin().enableTable(tableName);
  }

  @Override
  public void disableTable(TableName tableName) throws IOException {
    this.getCurrentHBaseAdmin().disableTable(tableName);
  }

  @Override
  public boolean isTableEnabled(TableName tableName) throws IOException {
    return this.getCurrentHBaseAdmin().isTableEnabled(tableName);
  }

  @Override
  public boolean isTableDisabled(TableName tableName) throws IOException {
    return this.getCurrentHBaseAdmin().isTableDisabled(tableName);
  }

  @Override
  public boolean isTableAvailable(TableName tableName) throws IOException {
    return isTableEnabled(tableName);
  }

  @Override
  public boolean isTableAvailable(TableName tableName, byte[][] splitKeys) throws IOException {
    return this.getCurrentHBaseAdmin().isTableAvailable(tableName, splitKeys);
  }

  @Override
  public void addColumnFamily(TableName tableName, ColumnFamilyDescriptor columnFamily)
    throws IOException {
    this.getCurrentHBaseAdmin().addColumnFamily(tableName, columnFamily);
  }

  @Override
  public void deleteColumn(TableName tableName, byte[] columnFamily) throws IOException {
    deleteColumnFamily(tableName, columnFamily);
  }

  @Override
  public void deleteColumnFamily(TableName tableName, byte[] columnFamily) throws IOException {
    this.getCurrentHBaseAdmin().deleteColumnFamily(tableName, columnFamily);
  }

  @Override
  public void modifyColumnFamily(TableName tableName, ColumnFamilyDescriptor columnFamily)
    throws IOException {
    this.getCurrentHBaseAdmin().modifyColumnFamily(tableName, columnFamily);
  }

  @Override
  public void modifyTable(TableName tableName, TableDescriptor td) throws IOException {
    modifyTable(td);
  }

  @Override
  public void modifyTable(TableDescriptor td) throws IOException {
    this.getCurrentHBaseAdmin().modifyTable(td);
    try{
      AdminUtil.setModifyInfoNode(this.connection.getOriginalConf(), td);
    }catch(Exception e){
      throw new IOException(e);
    }
  }

  @Override
  public void modifyNamespace(NamespaceDescriptor descriptor) throws IOException {
    this.getCurrentHBaseAdmin().modifyNamespace(descriptor);
  }

  @Override
  public void deleteNamespace(String name) throws IOException {
    this.getCurrentHBaseAdmin().deleteNamespace(name);
  }

  @Override
  public NamespaceDescriptor getNamespaceDescriptor(String name)
    throws NamespaceNotFoundException, IOException {
    return this.getCurrentHBaseAdmin().getNamespaceDescriptor(name);
  }

  @Override
  public NamespaceDescriptor[] listNamespaceDescriptors() throws IOException {
    return this.getCurrentHBaseAdmin().listNamespaceDescriptors();
  }

  @Override
  public void createNamespace(NamespaceDescriptor descriptor) throws IOException {
    this.getCurrentHBaseAdmin().createNamespace(descriptor);
  }

  @Override
  public void flush(TableName tableName) throws IOException{
    this.getCurrentHBaseAdmin().flush(tableName);
  }

  @Override
  public void flushRegion(byte[] regionName) throws IOException{
    this.getCurrentHBaseAdmin().flushRegion(regionName);
  }

  @Override
  public void compact(TableName tableName) throws IOException{
      this.getCurrentHBaseAdmin().compact(tableName);
  }

  @Override
  public void compactRegion(byte[] regionName) throws IOException {
    this.getCurrentHBaseAdmin().compactRegion(regionName);
  }

  @Override
  public void compact(TableName tableName, byte[] columnFamily) throws IOException{
    this.getCurrentHBaseAdmin().compact(tableName, columnFamily);
  }

  @Override
  public void compactRegion(byte[] regionName, byte[] columnFamily) throws IOException{
    this.getCurrentHBaseAdmin().compactRegion(regionName, columnFamily);
  }

  @Override
  public void compact(TableName tableName, CompactType compactType) throws IOException, InterruptedException{
    this.getCurrentHBaseAdmin().compact(tableName, compactType);
  }

  @Override
  public void compact(TableName tableName, byte[] columnFamily, CompactType compactType) throws IOException, InterruptedException{
    this.getCurrentHBaseAdmin().compact(tableName, columnFamily, compactType);
  }

  @Override
  public void majorCompact(TableName tableName) throws IOException{
    this.getCurrentHBaseAdmin().majorCompact(tableName);
  }

  @Override
  public void majorCompactRegion(byte[] regionName) throws IOException{
    this.getCurrentHBaseAdmin().majorCompactRegion(regionName);
  }

  @Override
  public void majorCompact(TableName tableName, byte[] columnFamily)throws IOException {
    this.getCurrentHBaseAdmin().majorCompact(tableName, columnFamily);
  }

  @Override
  public void majorCompactRegion(byte[] regionName, byte[] columnFamily) throws IOException{
    this.getCurrentHBaseAdmin().majorCompactRegion(regionName, columnFamily);
  }

  @Override
  public void majorCompact(TableName tableName, CompactType compactType) throws IOException, InterruptedException{
    this.getCurrentHBaseAdmin().majorCompact(tableName, compactType);
  }

  @Override
  public void majorCompact(TableName tableName, byte[] columnFamily, CompactType compactType)throws IOException, InterruptedException{
    this.getCurrentHBaseAdmin().majorCompact(tableName, columnFamily, compactType);
  }

  @Override
  public HTableDescriptor[] disableTables(String regex) throws IOException {
    return this.getCurrentHBaseAdmin().disableTables(regex);
  }

  @Override
  public HTableDescriptor[] disableTables(Pattern pattern) throws IOException {
    return this.getCurrentHBaseAdmin().disableTables(pattern);
  }

  @Override
  public HTableDescriptor[] enableTables(String regex) throws IOException {
    return this.getCurrentHBaseAdmin().enableTables(regex);
  }

  @Override
  public HTableDescriptor[] enableTables(Pattern pattern) throws IOException {
    return this.getCurrentHBaseAdmin().enableTables(pattern);
  }

  @Override
  public HTableDescriptor[] deleteTables(String regex) throws IOException {
    return this.getCurrentHBaseAdmin().deleteTables(regex);
  }

  @Override
  public HTableDescriptor[] deleteTables(Pattern pattern) throws IOException {
    return this.getCurrentHBaseAdmin().deleteTables(pattern);
  }


  @Override
  public void closeRegion(String regionname, String serverName) throws IOException {
    this.getCurrentHBaseAdmin().closeRegion(regionname, serverName);
  }

  @Override
  public void closeRegion(byte[] regionname, String serverName) throws IOException{
    this.getCurrentHBaseAdmin().closeRegion(regionname, serverName);
  }

  @Override
  public boolean closeRegionWithEncodedRegionName(String encodedRegionName, String serverName) throws IOException{
    return this.getCurrentHBaseAdmin().closeRegionWithEncodedRegionName(encodedRegionName, serverName);
}

  @Override
  public void closeRegion(ServerName sn, HRegionInfo hri) throws IOException{
    this.getCurrentHBaseAdmin().closeRegion(sn, hri);
  }

  @Override
  public List<HRegionInfo> getOnlineRegions(ServerName sn) throws IOException{
    return this.getCurrentHBaseAdmin().getOnlineRegions(sn);
  }

  @Override
  public List<RegionInfo> getRegions(ServerName serverName) throws IOException{
    return this.getCurrentHBaseAdmin().getRegions(serverName);
  }

  @Override
  public void flushRegionServer(ServerName serverName) throws IOException{
    this.getCurrentHBaseAdmin().flushRegionServer(serverName);
  }

  @Override
  public void compactRegionServer(ServerName serverName) throws IOException {
    this.getCurrentHBaseAdmin().compactRegionServer(serverName);
  }

  @Override
  public void majorCompactRegionServer(ServerName serverName) throws IOException {
    this.getCurrentHBaseAdmin().majorCompactRegionServer(serverName);
  }

  @Override
  public void assign(byte[] regionName) throws IOException {
    this.getCurrentHBaseAdmin().assign(regionName);
  }

  @Override
  public void unassign(byte[] regionName, boolean force) throws IOException {
    this.getCurrentHBaseAdmin().unassign(regionName, force);
  }

  @Override
  public void offline(byte[] regionName) throws IOException {
    this.getCurrentHBaseAdmin().offline(regionName);
  }

  @Override
  public boolean balancerSwitch(boolean onOrOff, boolean synchronous) throws IOException {
    return this.getCurrentHBaseAdmin().balancerSwitch(onOrOff, synchronous);
  }

  @Override
  public boolean balance() throws IOException {
    return this.getCurrentHBaseAdmin().balance();
  }

  @Override
  public boolean balance(boolean force) throws IOException {
    return this.getCurrentHBaseAdmin().balance(force);
  }

  @Override
  public boolean isBalancerEnabled() throws IOException {
    return this.getCurrentHBaseAdmin().isBalancerEnabled();
  }

  @Override
  public CacheEvictionStats clearBlockCache(TableName tableName) throws IOException {
    return this.getCurrentHBaseAdmin().clearBlockCache(tableName);
  }

  @Override
  public boolean normalize() throws IOException {
    return this.getCurrentHBaseAdmin().normalize();
  }

  @Override
  public boolean isNormalizerEnabled() throws IOException {
    return this.getCurrentHBaseAdmin().isNormalizerEnabled();
  }

  @Override
  public boolean normalizerSwitch(boolean on) throws IOException {
    return this.getCurrentHBaseAdmin().normalizerSwitch(on);
  }

  @Override
  public boolean catalogJanitorSwitch(boolean onOrOff) throws IOException {
    return this.getCurrentHBaseAdmin().catalogJanitorSwitch(onOrOff);
  }

  @Override
  public int runCatalogJanitor() throws IOException {
    return this.getCurrentHBaseAdmin().runCatalogJanitor();
  }

  @Override
  public boolean isCatalogJanitorEnabled() throws IOException {
    return this.getCurrentHBaseAdmin().isCatalogJanitorEnabled();
  }

  @Override
  public boolean cleanerChoreSwitch(boolean onOrOff) throws IOException {
    return this.getCurrentHBaseAdmin().cleanerChoreSwitch(onOrOff);
  }

  @Override
  public boolean runCleanerChore() throws IOException {
    return this.getCurrentHBaseAdmin().runCleanerChore();
  }

  @Override
  public boolean isCleanerChoreEnabled() throws IOException {
    return this.getCurrentHBaseAdmin().isCleanerChoreEnabled();
  }

  @Override
  public void mergeRegions(byte[] nameOfRegionA, byte[] nameOfRegionB, boolean forcible) throws IOException {
    this.getCurrentHBaseAdmin().mergeRegions(nameOfRegionA, nameOfRegionB, forcible);

  }

  @Override
  public Future<Void> mergeRegionsAsync(byte[] nameOfRegionA, byte[] nameOfRegionB,
                                        boolean forcible) throws IOException {
    return this.getCurrentHBaseAdmin().mergeRegionsAsync(nameOfRegionA, nameOfRegionB, forcible);
  }

  @Override
  public Future<Void> mergeRegionsAsync(byte[][] nameofRegionsToMerge, boolean forcible) throws IOException {
    return this.getCurrentHBaseAdmin().mergeRegionsAsync(nameofRegionsToMerge, forcible);
  }

  @Override
  public void split(TableName tableName) throws IOException {
    this.getCurrentHBaseAdmin().split(tableName);
  }

  @Override
  public void splitRegion(byte[] regionName) throws IOException {
    this.getCurrentHBaseAdmin().splitRegion(regionName);
  }

  @Override
  public void split(TableName tableName, byte[] splitPoint) throws IOException {
    this.getCurrentHBaseAdmin().split(tableName, splitPoint);
  }

  @Override
  public void splitRegion(byte[] regionName, byte[] splitPoint) throws IOException {
    this.getCurrentHBaseAdmin().splitRegion(regionName, splitPoint);
  }

  @Override
  public Future<Void> splitRegionAsync(byte[] regionName, byte[] splitPoint) throws IOException {
    return this.getCurrentHBaseAdmin().splitRegionAsync(regionName, splitPoint);
  }

  @Override
  public Future<Void> modifyTableAsync(TableName tableName, TableDescriptor td) throws IOException {
    return this.getCurrentHBaseAdmin().modifyTableAsync(tableName, td);
  }

  @Override
  public Future<Void> modifyTableAsync(TableDescriptor td) throws IOException {
    return this.getCurrentHBaseAdmin().modifyTableAsync(td);
  }

  @Override
  public void shutdown() throws IOException {
    this.getCurrentHBaseAdmin().shutdown();
  }

  @Override
  public void stopMaster() throws IOException{
    this.getCurrentHBaseAdmin().stopMaster();
  }

  @Override
  public boolean isMasterInMaintenanceMode() throws IOException {
    return this.getCurrentHBaseAdmin().isMasterInMaintenanceMode();
  }

  @Override
  public void stopRegionServer(String hostnamePort) throws IOException {
    this.getCurrentHBaseAdmin().stopRegionServer(hostnamePort);
  }

  @Override
  public ClusterMetrics getClusterMetrics(EnumSet<ClusterMetrics.Option> options) throws IOException {
    return this.getCurrentHBaseAdmin().getClusterMetrics(options);
  }

  @Override
  public List<RegionMetrics> getRegionMetrics(ServerName serverName, TableName tableName) throws IOException {
    return this.getCurrentHBaseAdmin().getRegionMetrics(serverName, tableName);
  }

  @Override
  public Future<Void> createNamespaceAsync(NamespaceDescriptor descriptor) throws IOException {
    return this.getCurrentHBaseAdmin().createNamespaceAsync(descriptor);
  }

  @Override
  public Future<Void> modifyNamespaceAsync(NamespaceDescriptor descriptor) throws IOException {
    return this.getCurrentHBaseAdmin().modifyNamespaceAsync(descriptor);
  }

  @Override
  public List<HRegionInfo> getTableRegions(TableName tableName) throws IOException {
    return this.getCurrentHBaseAdmin().getTableRegions(tableName);
  }

  @Override
  public List<RegionInfo> getRegions(TableName tableName) throws IOException {
    return this.getCurrentHBaseAdmin().getRegions(tableName);
  }

  @Override
  public boolean abortProcedure(long procId, boolean mayInterruptIfRunning) throws IOException {
    return this.getCurrentHBaseAdmin().abortProcedure(procId, mayInterruptIfRunning);
  }

  @Override
  public Future<Boolean> abortProcedureAsync(long procId, boolean mayInterruptIfRunning) throws IOException {
    return this.getCurrentHBaseAdmin().abortProcedureAsync(procId, mayInterruptIfRunning);
  }

  @Override
  public String getProcedures() throws IOException {
    return this.getCurrentHBaseAdmin().getProcedures();
  }

  @Override
  public String getLocks() throws IOException {
    return this.getCurrentHBaseAdmin().getLocks();
  }

  @Override
  public void rollWALWriter(ServerName serverName) throws IOException {
    this.getCurrentHBaseAdmin().rollWALWriter(serverName);
  }

  @Override
  public CompactionState getCompactionState(TableName tableName) throws IOException {
    return this.getCurrentHBaseAdmin().getCompactionState(tableName);
  }

  @Override
  public CompactionState getCompactionState(TableName tableName, CompactType compactType) throws IOException {
    return this.getCurrentHBaseAdmin().getCompactionState(tableName, compactType);
  }

  @Override
  public CompactionState getCompactionStateForRegion(byte[] regionName) throws IOException {
    return this.getCurrentHBaseAdmin().getCompactionStateForRegion(regionName);
  }

  @Override
  public long getLastMajorCompactionTimestamp(TableName tableName) throws IOException {
    return this.getCurrentHBaseAdmin().getLastMajorCompactionTimestamp(tableName);
  }

  @Override
  public long getLastMajorCompactionTimestampForRegion(byte[] regionName) throws IOException {
    return this.getCurrentHBaseAdmin().getLastMajorCompactionTimestampForRegion(regionName);
  }

  @Override
  public void snapshot(String snapshotName, TableName tableName) throws IOException {
    this.getCurrentHBaseAdmin().snapshot(snapshotName, tableName);
  }

  @Override
  public void snapshot(byte[] snapshotName, TableName tableName) throws IOException {
    this.getCurrentHBaseAdmin().snapshot(snapshotName, tableName);
  }

  @Override
  public void snapshot(String snapshotName, TableName tableName, SnapshotType type) throws IOException {
    this.getCurrentHBaseAdmin().snapshot(snapshotName, tableName, type);
  }

  @Override
  public void snapshot(SnapshotDescription snapshot) throws IOException {
    this.getCurrentHBaseAdmin().snapshot(snapshot);
  }

  @Override
  public void snapshotAsync(SnapshotDescription snapshot) throws IOException {
    this.getCurrentHBaseAdmin().snapshotAsync(snapshot);
  }

  @Override
  public boolean isSnapshotFinished(SnapshotDescription snapshot) throws IOException {
    return this.getCurrentHBaseAdmin().isSnapshotFinished(snapshot);
  }

  @Override
  public void restoreSnapshot(byte[] snapshotName) throws IOException {
    this.getCurrentHBaseAdmin().restoreSnapshot(snapshotName);
  }

  @Override
  public void restoreSnapshot(String snapshotName) throws IOException {
    this.getCurrentHBaseAdmin().restoreSnapshot(snapshotName);
  }

  @Override
  public Future<Void> restoreSnapshotAsync(String snapshotName) throws IOException {
    return this.getCurrentHBaseAdmin().restoreSnapshotAsync(snapshotName);
  }

  @Override
  public void restoreSnapshot(byte[] snapshotName, boolean takeFailSafeSnapshot) throws IOException {
    this.getCurrentHBaseAdmin().restoreSnapshot(snapshotName, takeFailSafeSnapshot);
  }

  @Override
  public void restoreSnapshot(String snapshotName, boolean takeFailSafeSnapshot) throws IOException {
    this.getCurrentHBaseAdmin().restoreSnapshot(snapshotName, takeFailSafeSnapshot);
  }

  @Override
  public void restoreSnapshot(String snapshotName, boolean takeFailSafeSnapshot,
                              boolean restoreAcl) throws IOException {
    this.getCurrentHBaseAdmin().restoreSnapshot(snapshotName, takeFailSafeSnapshot, restoreAcl);
  }

  @Override
  public void cloneSnapshot(byte[] snapshotName, TableName tableName) throws IOException {
    this.getCurrentHBaseAdmin().cloneSnapshot(snapshotName, tableName);
  }

  @Override
  public void cloneSnapshot(String snapshotName, TableName tableName, boolean restoreAcl) throws IOException {
    this.getCurrentHBaseAdmin().cloneSnapshot(snapshotName, tableName, restoreAcl);
  }

  @Override
  public void cloneSnapshot(String snapshotName, TableName tableName) throws IOException {
    this.getCurrentHBaseAdmin().cloneSnapshot(snapshotName, tableName);
  }

  @Override
  public Future<Void> cloneSnapshotAsync(String snapshotName, TableName tableName) throws IOException {
    return this.getCurrentHBaseAdmin().cloneSnapshotAsync(snapshotName, tableName);
  }

  @Override
  public void execProcedure(String signature, String instance, Map<String, String> props) throws IOException {
    this.getCurrentHBaseAdmin().execProcedure(signature, instance, props);
  }

  @Override
  public byte[] execProcedureWithReturn(String signature, String instance, Map<String, String> props) throws IOException {
    return this.getCurrentHBaseAdmin().execProcedureWithReturn(signature, instance, props);
  }

  @Override
  public boolean isProcedureFinished(String signature, String instance, Map<String, String> props) throws IOException {
    return this.getCurrentHBaseAdmin().isProcedureFinished(signature, instance, props);
  }

  @Override
  public List<SnapshotDescription> listSnapshots() throws IOException {
    return this.getCurrentHBaseAdmin().listSnapshots();
  }

  @Override
  public List<SnapshotDescription> listSnapshots(String regex) throws IOException {
    return this.getCurrentHBaseAdmin().listSnapshots(regex);
  }

  @Override
  public List<SnapshotDescription> listSnapshots(Pattern pattern) throws IOException {
    return this.getCurrentHBaseAdmin().listSnapshots(pattern);
  }

  @Override
  public List<SnapshotDescription> listTableSnapshots(String tableNameRegex,
                                                      String snapshotNameRegex) throws IOException {
    return this.getCurrentHBaseAdmin().listTableSnapshots(tableNameRegex, snapshotNameRegex);
  }

  @Override
  public List<SnapshotDescription> listTableSnapshots(Pattern tableNamePattern,
                                                      Pattern snapshotNamePattern) throws IOException {
    return this.getCurrentHBaseAdmin().listTableSnapshots(tableNamePattern, snapshotNamePattern);
  }

  @Override
  public void deleteSnapshot(byte[] snapshotName) throws IOException {
    this.getCurrentHBaseAdmin().deleteSnapshot(snapshotName);
  }

  @Override
  public void deleteSnapshot(String snapshotName) throws IOException {
    this.getCurrentHBaseAdmin().deleteSnapshot(snapshotName);
  }

  @Override
  public void deleteSnapshots(String regex) throws IOException {
    this.getCurrentHBaseAdmin().deleteSnapshots(regex);
  }

  @Override
  public void deleteSnapshots(Pattern pattern) throws IOException {
    this.getCurrentHBaseAdmin().deleteSnapshots(pattern);
  }

  @Override
  public void deleteTableSnapshots(String tableNameRegex, String snapshotNameRegex) throws IOException {
    this.getCurrentHBaseAdmin().deleteTableSnapshots(tableNameRegex, snapshotNameRegex);
  }

  @Override
  public void deleteTableSnapshots(Pattern tableNamePattern, Pattern snapshotNamePattern) throws IOException {
    this.getCurrentHBaseAdmin().deleteTableSnapshots(tableNamePattern, snapshotNamePattern);
  }

  @Override
  public void setQuota(QuotaSettings quota) throws IOException {
    this.getCurrentHBaseAdmin().setQuota(quota);
  }

  @Override
  public QuotaRetriever getQuotaRetriever(QuotaFilter filter) throws IOException {
    return this.getCurrentHBaseAdmin().getQuotaRetriever(filter);
  }

  @Override
  public List<QuotaSettings> getQuota(QuotaFilter filter) throws IOException {
    return this.getCurrentHBaseAdmin().getQuota(filter);
  }

  @Override
  public CoprocessorRpcChannel coprocessorService() {
    try {
      return this.getCurrentHBaseAdmin().coprocessorService();
    }catch(IOException e){
      LOG.error("coprocessorService failed : " + e);
      return null;
    }
  }

  @Override
  public CoprocessorRpcChannel coprocessorService(ServerName serverName)  {
    try {
      return this.getCurrentHBaseAdmin().coprocessorService(serverName);
    } catch(IOException e){
        LOG.error("coprocessorService failed : " + e);
        return null;
    }
  }

  @Override
  public void updateConfiguration(ServerName server) throws IOException {
    this.getCurrentHBaseAdmin().updateConfiguration(server);
  }

  @Override
  public void updateConfiguration() throws IOException {
    this.getCurrentHBaseAdmin().updateConfiguration();
  }

  @Override
  public List<SecurityCapability> getSecurityCapabilities() throws IOException {
    return this.getCurrentHBaseAdmin().getSecurityCapabilities();
  }

  @Override
  public boolean splitSwitch(boolean enabled, boolean synchronous) throws IOException {
    return this.getCurrentHBaseAdmin().splitSwitch(enabled, synchronous);
  }

  @Override
  public boolean mergeSwitch(boolean enabled, boolean synchronous) throws IOException {
    return this.getCurrentHBaseAdmin().mergeSwitch(enabled, synchronous);
  }

  @Override
  public boolean isSplitEnabled() throws IOException {
    return this.getCurrentHBaseAdmin().isSplitEnabled();
  }

  @Override
  public boolean isMergeEnabled() throws IOException {
    return this.getCurrentHBaseAdmin().isMergeEnabled();
  }

  @Override
  public void addReplicationPeer(String peerId, ReplicationPeerConfig peerConfig, boolean enabled) throws IOException {
    this.getCurrentHBaseAdmin().addReplicationPeer(peerId, peerConfig, enabled);
  }

  @Override
  public void removeReplicationPeer(String peerId)  throws IOException {
    this.getCurrentHBaseAdmin().removeReplicationPeer(peerId);
  }

  @Override
  public void enableReplicationPeer(String peerId) throws IOException {
    this.getCurrentHBaseAdmin().enableReplicationPeer(peerId);
  }


  @Override
  public void disableReplicationPeer(String peerId) throws IOException {
    this.getCurrentHBaseAdmin().disableReplicationPeer(peerId);
  }

  @Override
  public ReplicationPeerConfig getReplicationPeerConfig(String peerId) throws IOException {
    return this.getCurrentHBaseAdmin().getReplicationPeerConfig(peerId);
  }

  @Override
  public void updateReplicationPeerConfig(String peerId, ReplicationPeerConfig peerConfig) throws IOException {
    this.getCurrentHBaseAdmin().updateReplicationPeerConfig(peerId, peerConfig);
  }

  @Override
  public void appendReplicationPeerTableCFs(String id, Map<TableName, List<String>> tableCfs)
    throws ReplicationException, IOException {
    this.getCurrentHBaseAdmin().appendReplicationPeerTableCFs(id, tableCfs);
  }

  @Override
  public void removeReplicationPeerTableCFs(String id, Map<TableName, List<String>> tableCfs)
    throws ReplicationException, IOException {
    this.getCurrentHBaseAdmin().removeReplicationPeerTableCFs(id, tableCfs);
  }

  @Override
  public List<ReplicationPeerDescription> listReplicationPeers() throws IOException {
    return this.getCurrentHBaseAdmin().listReplicationPeers();
  }

  @Override
  public List<ReplicationPeerDescription> listReplicationPeers(Pattern pattern) throws IOException {
    return this.getCurrentHBaseAdmin().listReplicationPeers(pattern);
  }

  @Override
  public void decommissionRegionServers(List<ServerName> servers, boolean offload) throws IOException {
    this.getCurrentHBaseAdmin().decommissionRegionServers(servers, offload);
  }

  @Override
  public List<ServerName> listDecommissionedRegionServers() throws IOException {
    return this.getCurrentHBaseAdmin().listDecommissionedRegionServers();
  }

  @Override
  public void recommissionRegionServer(ServerName server, List<byte[]> encodedRegionNames) throws IOException {
    this.getCurrentHBaseAdmin().recommissionRegionServer(server, encodedRegionNames);
  }

  @Override
  public List<TableCFs> listReplicatedTableCFs() throws IOException {
    return this.getCurrentHBaseAdmin().listReplicatedTableCFs();
  }

  @Override
  public void enableTableReplication(TableName tableName) throws IOException {
    this.getCurrentHBaseAdmin().enableTableReplication(tableName);
  }

  @Override
  public void disableTableReplication(TableName tableName) throws IOException {
    this.getCurrentHBaseAdmin().disableTableReplication(tableName);
  }

  @Override
  public void clearCompactionQueues(ServerName serverName, Set<String> queues) throws IOException
    , InterruptedException {
    this.getCurrentHBaseAdmin().clearCompactionQueues(serverName, queues);
  }

  @Override
  public List<ServerName> clearDeadServers(List<ServerName> servers) throws IOException {
    return this.getCurrentHBaseAdmin().clearDeadServers(servers);
  }

  @Override
  public Future<Void> createTableAsync(TableDescriptor desc, byte[][] splitKeys) throws IOException {
    return this.getCurrentHBaseAdmin().createTableAsync(desc, splitKeys);
  }

  @Override
  public Future<Void> deleteTableAsync(TableName tableName) throws IOException {
    return this.getCurrentHBaseAdmin().deleteTableAsync(tableName);
  }

  @Override
  public Future<Void> truncateTableAsync(TableName tableName, boolean preserveSplits) throws IOException {
    return this.getCurrentHBaseAdmin().truncateTableAsync(tableName, preserveSplits);
  }

  @Override
  public Future<Void> enableTableAsync(TableName tableName) throws IOException {
    return this.getCurrentHBaseAdmin().enableTableAsync(tableName);
  }

  @Override
  public Future<Void> disableTableAsync(TableName tableName) throws IOException {
    return this.getCurrentHBaseAdmin().disableTableAsync(tableName);
  }

  @Override
  public Pair<Integer, Integer> getAlterStatus(TableName tableName) throws IOException {
    // return a fake pair
    return this.getCurrentHBaseAdmin().getAlterStatus(tableName);
  }

  @Override
  public Pair<Integer, Integer> getAlterStatus(byte[] tableName) throws IOException {
    // return a fake pair
    return this.getCurrentHBaseAdmin().getAlterStatus(tableName);
  }

  @Override
  public Future<Void> deleteColumnFamilyAsync(TableName tableName, byte[] columnFamily) throws IOException {
    return this.getCurrentHBaseAdmin().deleteColumnFamilyAsync(tableName, columnFamily);
  }

  @Override
  public Future<Void> addColumnFamilyAsync(TableName tableName,
                                           ColumnFamilyDescriptor columnFamily) throws IOException {
    return this.getCurrentHBaseAdmin().addColumnFamilyAsync(tableName, columnFamily);
  }

  @Override
  public Future<Void> modifyColumnFamilyAsync(TableName tableName,
                                              ColumnFamilyDescriptor columnFamily) throws IOException {
    return this.getCurrentHBaseAdmin().modifyColumnFamilyAsync(tableName, columnFamily);
  }

  @Override
  public Future<Void> deleteNamespaceAsync(String name) throws IOException {
    return this.getCurrentHBaseAdmin().deleteNamespaceAsync(name);
  }

  @Override
  public void move(byte[] bytes, byte[] bytes1) throws IOException {
    this.getCurrentHBaseAdmin().move(bytes, bytes1);
  }


  @Override
  public void splitRegionSync(byte[] regionName, byte[] splitPoint) throws IOException {
    this.getCurrentHBaseAdmin().splitRegionSync(regionName, splitPoint);
  }


  @Override
  public void splitRegionSync(byte[] regionName, byte[] splitPoint,
                              final long timeout, final TimeUnit units) throws IOException {
    this.getCurrentHBaseAdmin().splitRegionSync(regionName, splitPoint, timeout, units);
  }

  @Override
  public byte[][] rollHLogWriter(String serverName) throws IOException {
    return this.getCurrentHBaseAdmin().rollHLogWriter(serverName);
  }

  @Override
  public void mergeRegionsSync(byte[] nameOfRegionA, byte[] nameOfRegionB, boolean forcible)
    throws IOException {
    this.getCurrentHBaseAdmin().mergeRegionsSync(nameOfRegionA, nameOfRegionB, forcible);
  }

  @Override
  Future<Void> splitRegionAsync(RegionInfo hri, byte[] splitPoint) throws IOException {
    return this.getCurrentHBaseAdmin().splitRegionAsync(hri, splitPoint);
  }

  @Override
  Pair<RegionInfo, ServerName> getRegion(byte[] regionName) throws IOException {
    return this.getCurrentHBaseAdmin().getRegion(regionName);
  }

  @Override
  public List<AliHBaseIndexDescriptor> describeIndex(TableName dataTableName) throws IOException {
    if (isHBaseUE()) {
      return ((AliHBaseUEAdmin) this.currentHBaseAdmin).describeIndex(dataTableName);
    } else {
      throw new UnsupportedOperationException("describeIndex not supported by HBase, please use HBaseUE instead.");
    }
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor indexDescriptor) throws IOException {
    if(isHBaseUE()) {
      ((AliHBaseUEAdmin) this.currentHBaseAdmin).createIndex(indexDescriptor);
    } else {
      throw new UnsupportedOperationException("createIndex not supported by HBase, please use HBaseUE instead.");
    }
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor indexDescriptor, byte[][] splitKeys) throws IOException {
    if(isHBaseUE()) {
      ((AliHBaseUEAdmin) this.currentHBaseAdmin).createIndex(indexDescriptor, splitKeys);
    } else {
      throw new UnsupportedOperationException("createIndex not supported by HBase, please use HBaseUE instead.");
    }
  }

  @Override
  public void createIndex(AliHBaseIndexDescriptor indexDescriptor, byte[] startKey, byte[] endKey, int numRegions)
          throws IOException {
    if(isHBaseUE()) {
      ((AliHBaseUEAdmin) this.currentHBaseAdmin).createIndex(indexDescriptor, startKey, endKey, numRegions);
    } else {
      throw new UnsupportedOperationException("createIndex not supported by HBase, please use HBaseUE instead.");
    }
  }

  @Override
  public void deleteIndex(String indexName, TableName dataTable) throws IOException {
    if(isHBaseUE()) {
      ((AliHBaseUEAdmin) this.currentHBaseAdmin).deleteIndex(indexName, dataTable);
    } else {
      throw new UnsupportedOperationException("deleteIndex not supported by HBase, please use HBaseUE instead.");
    }
    throw new UnsupportedOperationException("deleteIndex not supported");
  }

  @Override
  public void offlineIndex(String indexName, TableName dataTable) throws IOException {
    if(isHBaseUE()) {
      ((AliHBaseUEAdmin) this.currentHBaseAdmin).offlineIndex(indexName, dataTable);
    } else {
      throw new UnsupportedOperationException("offlineIndex not supported by HBase, please use HBaseUE instead.");
    }
  }

  /**
   * Whether current connection is connected to an HBaseUE cluster.
   * @return true if connected to HBaseUE, false otherwise
   */
  private boolean isHBaseUE() {
    AliHBaseConstants.ClusterType clusterType = AliHBaseConstants.ClusterType.valueOf(
            this.connection.getConfiguration().get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
    return (this.currentHBaseAdmin instanceof AliHBaseUEAdmin && clusterType == AliHBaseConstants.ClusterType.HBASEUE);
  }

  @Override
  public void registerBDSCluster(String hbaseSourceName, String bdsClusterkey,
      String hbaseConnectionString, String username, String password) throws IOException {
    if (isHBaseUE()) {
      ((AliHBaseUEAdmin) this.currentHBaseAdmin).registerBDSCluster(hbaseSourceName, bdsClusterkey, hbaseConnectionString, username, password);
    } else {
      throw new UnsupportedOperationException("registerBDSCluster not supported by HBase, please use HBaseUE instead.");
    }
  }

  @Override
  public void registerSolrCluster(String solrSourceName, String solrConnectionStr)
      throws IOException {
    if (isHBaseUE()) {
      ((AliHBaseUEAdmin) this.currentHBaseAdmin).registerSolrCluster(solrSourceName, solrConnectionStr);
    } else {
      throw new UnsupportedOperationException("registerSolrCluster not supported by HBase, please use HBaseUE instead.");
    }
  }

  @Override
  public void registerESCluster(String esSourceName, String esConnectionStr, String userName,
      String password) throws IOException {
    if (isHBaseUE()) {
      ((AliHBaseUEAdmin) this.currentHBaseAdmin).registerESCluster(esSourceName, esConnectionStr, userName, password);
    } else {
      throw new UnsupportedOperationException("registerESCluster not supported by HBase, please use HBaseUE instead.");
    }

  }

  @Override
  public void unregisterSolrCluster(boolean force) throws IOException {
    if (isHBaseUE()) {
      ((AliHBaseUEAdmin) this.currentHBaseAdmin).unregisterSolrCluster(force);
    } else {
      throw new UnsupportedOperationException("unregisterSolrCluster not supported by HBase, please use HBaseUE instead.");
    }
  }

  @Override
  public void unregisterESCluster(boolean force) throws IOException {
    if (isHBaseUE()) {
      ((AliHBaseUEAdmin) this.currentHBaseAdmin).unregisterESCluster(force);
    } else {
      throw new UnsupportedOperationException("unregisterESCluster not supported by HBase, please use HBaseUE instead.");
    }
  }

  @Override
  public void unregisterBDSCluster(boolean force) throws IOException {
    if (isHBaseUE()) {
      ((AliHBaseUEAdmin) this.currentHBaseAdmin).unregisterBDSCluster(force);
    } else {
      throw new UnsupportedOperationException("unregisterBDSCluster not supported by HBase, please use HBaseUE instead.");
    }
  }
}