package org.apache.hadoop.hbase.client;

import java.io.Closeable;
import java.io.IOException;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.haclient.ClusterSwitchUtil;
import com.alibaba.hbase.haclient.dualservice.AutoSwitch;
import com.alibaba.hbase.haclient.dualservice.DualExecutor;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.MasterNotRunningException;
import org.apache.hadoop.hbase.RegionLocations;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.ZooKeeperConnectionException;
import org.apache.hadoop.hbase.client.backoff.ClientBackoffPolicy;
import org.apache.hadoop.hbase.ipc.RpcControllerFactory;
import org.apache.hadoop.hbase.security.User;
import org.apache.hadoop.hbase.util.Threads;

import org.apache.hadoop.hbase.shaded.protobuf.generated.AdminProtos;
import org.apache.hadoop.hbase.shaded.protobuf.generated.ClientProtos;

public class AliHBaseMultiClusterConnection implements ClusterConnection, Closeable {
  private static final Log LOG = LogFactory.getLog(AliHBaseMultiClusterConnection.class);

  private volatile ClusterConnection activeConnectionImpl;
  //standby connection use for dual service
  private volatile ClusterConnection standbyConnectionImpl;

  protected volatile long switchCount = 0;
  protected volatile Configuration conf;
  protected volatile Configuration standbyConf;
  protected User user;
  protected volatile boolean closed = false;
  protected volatile boolean aborted = false;
  protected volatile boolean cleanupPool = false;
  protected volatile boolean cleanupStandbyPool = false;

  // thread executor shared by all HTableInterface instances created
  // by this connection
  private ExecutorService batchPool = null;
  private ExecutorService standbyBatchPool = null;
  private AsyncProcess asyncProcess = null;

  private AliHBaseConstants.ClusterType activeClusterType = null;
  private AliHBaseConstants.ClusterType standbyClusterType = null;

  protected boolean dualServiceEnable;
  protected DualExecutor dualExecutor;
  protected AutoSwitch autoSwitch;

  protected AliHBaseMultiClusterConnection(Configuration conf,
                                   ExecutorService pool, User user) throws IOException{
    this.conf = conf;
    this.user = user;
    this.batchPool = pool;
    //get dual service enable or not
    this.dualServiceEnable = conf.getBoolean(AliHBaseConstants.DUALSERVICE_ENABLE,
      AliHBaseConstants.DUALSERVICE_ENABLE_DEFAULT);
  }

  protected ClusterConnection createConnection(Configuration conf, User user, ExecutorService pool) throws IOException {
    ClusterConnection newClusterConnection = null;
    AliHBaseConstants.ClusterType clusterType = getClusterTypeFromConf(conf);
    if(clusterType == AliHBaseConstants.ClusterType.HBASE) {
      newClusterConnection = new ConnectionImplementation(
        conf, pool, user);
    }else if(clusterType == AliHBaseConstants.ClusterType.HBASEUE){
      newClusterConnection = new AliHBaseUEClusterConnection(
        conf, pool, user);
    }else{
      throw new IOException("Cluster is null, can not create connection");
    }
    return newClusterConnection;
  }

  // For tests to override.
  protected AsyncProcess createAsyncProcess() {
    // No default pool available.
    AliHBaseConstants.ClusterType clusterType =
      AliHBaseConstants.ClusterType.valueOf(this.conf.get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
    if(clusterType == AliHBaseConstants.ClusterType.HBASE) {
      return new AsyncProcess(this, conf,
        RpcRetryingCallerFactory.instantiate(conf, this.getStatisticsTracker()),
        RpcControllerFactory.instantiate(conf));
    }else {
      return null;
    }
  }

  protected ExecutorService getThreadPool(int maxThreads, int coreThreads, String nameHint,
                                          BlockingQueue<Runnable> passedWorkQueue) {
    // shared HTable thread executor not yet initialized
    if (maxThreads == 0) {
      maxThreads = Runtime.getRuntime().availableProcessors() * 8;
    }
    if (coreThreads == 0) {
      coreThreads = Runtime.getRuntime().availableProcessors() * 8;
    }
    long keepAliveTime = conf.getLong("hbase.hconnection.threads.keepalivetime", 60);
    BlockingQueue<Runnable> workQueue = passedWorkQueue;
    if (workQueue == null) {
      workQueue =
        new LinkedBlockingQueue<Runnable>(maxThreads *
          conf.getInt(HConstants.HBASE_CLIENT_MAX_TOTAL_TASKS,
            HConstants.DEFAULT_HBASE_CLIENT_MAX_TOTAL_TASKS));
    }
    ThreadPoolExecutor tpe = new ThreadPoolExecutor(
      coreThreads,
      maxThreads,
      keepAliveTime,
      TimeUnit.SECONDS,
      workQueue,
      Threads.newDaemonThreadFactory(toString() + nameHint));
    tpe.allowCoreThreadTimeOut(true);
    return tpe;
  }

  private ExecutorService getBatchPool() {
    if (batchPool == null) {
      synchronized (this) {
        if (batchPool == null) {
          this.batchPool = getThreadPool(conf.getInt("hbase.hconnection.threads.max", 256),
            conf.getInt("hbase.hconnection.threads.core", 256), "-shared-", null);
          cleanupPool = true;
        }
      }
    }
    return this.batchPool;
  }

  private ExecutorService getStandbyBatchPool() {
    if (standbyBatchPool == null) {
      synchronized (this) {
        if (standbyBatchPool == null) {
          this.standbyBatchPool = getThreadPool(standbyConf.getInt("hbase.hconnection.threads.max", 256),
            standbyConf.getInt("hbase.hconnection.threads.core", 256), "-shared-", null);
          cleanupStandbyPool = true;
        }
      }
    }
    return this.standbyBatchPool;
  }

  protected synchronized void onChangeCluster(Configuration activeConf, Configuration standbyConf)
    throws IOException {
    // Copy the configs in case change outside
    this.conf = new Configuration(activeConf);
    String activeConnectKey = ClusterSwitchUtil.getConnectKey(activeConf);
    AliHBaseConstants.ClusterType localActiveType = getClusterTypeFromConf(this.conf);

    if(standbyConf == null) {
      AliHBaseConstants.ClusterType lastClusterType = this.activeClusterType;
      if(lastClusterType == null){
        lastClusterType = localActiveType;
      }
      Connection lastConnection = activeConnectionImpl;
      LOG.info("Creating new connection to " + activeConnectKey);
      this.activeConnectionImpl = createConnection(activeConf, user, getBatchPool());
      this.activeClusterType = localActiveType;
      if (lastConnection != null && !lastConnection.isClosed() && lastClusterType == AliHBaseConstants.ClusterType.HBASE) {
        LOG.info("Closing old connection on "
          + ClusterSwitchUtil.getConnectKey(lastConnection.getConfiguration())
          + " since destination cluster is changed");
        lastConnection.close();
      }
    }else{
      // Copy the configs in case change outside
      this.standbyConf = new Configuration(standbyConf);
      String standbyConnectKey = ClusterSwitchUtil.getConnectKey(standbyConf);
      AliHBaseConstants.ClusterType localStandbyType = getClusterTypeFromConf(standbyConf);

      //switch active and standby
      ClusterConnection tempConnection = this.activeConnectionImpl;
      this.activeConnectionImpl = this.standbyConnectionImpl;
      this.standbyConnectionImpl = tempConnection;

      //if active connection not exist or closed, create new connection to active
      if(activeConnectionImpl == null || activeConnectionImpl.isClosed()){
        LOG.info("Creating new active connection to " + activeConnectKey);
        try {
          this.activeConnectionImpl = createConnection(activeConf, user, getBatchPool());
        }catch(Throwable t){
          LOG.warn("Create active connection failed with error : " + t);
          LOG.warn("Disable dual service, try connect standby cluster");
          standbyConf.setBoolean(AliHBaseConstants.DUALSERVICE_ENABLE, false);
          this.onChangeCluster(standbyConf, null);
        }
      }
      //if standby connection not exist or closed, create new connect to standby
      if(standbyConnectionImpl == null || standbyConnectionImpl.isClosed()){
        LOG.info("Creating new standby connection to " + standbyConnectKey);
        try {
          this.standbyConnectionImpl = createConnection(standbyConf, user, getStandbyBatchPool());
        }catch(Throwable t){
          LOG.warn("Create standby connection failed with error : " + t);
        }
      }

      this.activeClusterType = localActiveType;
      this.standbyClusterType = localStandbyType;
      if(autoSwitch != null){
        autoSwitch.reset();
      }
    }
    this.switchCount++;
  }

  private AliHBaseConstants.ClusterType getClusterTypeFromConf(Configuration conf){
    return AliHBaseConstants.ClusterType.valueOf(conf.get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
  }

  public long getSwitchCount() {
    return switchCount;
  }

  /** @return - true if the master server is running
   * @deprecated this has been deprecated without a replacement */
  @Deprecated
  @Override
  public boolean isMasterRunning()
    throws MasterNotRunningException, ZooKeeperConnectionException {
    return activeConnectionImpl.isMasterRunning();
  }

  /**
   * Use this api to check if the table has been created with the specified number of
   * splitkeys which was used while creating the given table.
   * Note : If this api is used after a table's region gets splitted, the api may return
   * false.
   * @param tableName
   *          tableName
   * @param splitKeys
   *          splitKeys used while creating table
   * @throws IOException
   *           if a remote or network exception occurs
   */
  @Override
  public boolean isTableAvailable(TableName tableName, byte[][] splitKeys)
    throws IOException {
    return activeConnectionImpl.isTableAvailable(tableName, splitKeys);
  }

  /**
   * Find the location of the region of <i>tableName</i> that <i>row</i>
   * lives in.
   * @param tableName name of the table <i>row</i> is in
   * @param row row key you're trying to find the region of
   * @return HRegionLocation that describes where to find the region in
   * question
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public HRegionLocation locateRegion(TableName tableName, byte[] row)
    throws IOException {
    return activeConnectionImpl.locateRegion(tableName, row);
  }

  /**
   * Allows flushing the region cache.
   */
  @Override
  public void clearRegionCache() {
    activeConnectionImpl.clearRegionCache();
  }

  public void cacheLocation(TableName tableName,
                            RegionLocations regionLocations) {
    activeConnectionImpl.cacheLocation(tableName, regionLocations);
  }

  /**
   * Allows flushing the region cache of all locations that pertain to
   * <code>tableName</code>
   * @param tableName Name of the table whose regions we are to remove from
   * cache.
   */
  @Override
  public void clearRegionCache(TableName tableName) {
    activeConnectionImpl.clearRegionCache(tableName);
  }

  /**
   * Deletes cached locations for the specific region.
   * @param location The location object for the region, to be purged from cache.
   */
  @Override
  public void deleteCachedRegionLocation(HRegionLocation location) {
    activeConnectionImpl.deleteCachedRegionLocation(location);
  }

  /**
   * Find the location of the region of <i>tableName</i> that <i>row</i>
   * lives in, ignoring any value that might be in the cache.
   * @param tableName name of the table <i>row</i> is in
   * @param row row key you're trying to find the region of
   * @return HRegionLocation that describes where to find the region in
   * question
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public HRegionLocation relocateRegion(TableName tableName, byte[] row)
    throws IOException {
    return activeConnectionImpl.relocateRegion(tableName, row);
  }

  /**
   * Find the location of the region of <i>tableName</i> that <i>row</i>
   * lives in, ignoring any value that might be in the cache.
   * @param tableName name of the table <i>row</i> is in
   * @param row row key you're trying to find the region of
   * @param replicaId the replicaId of the region
   * @return RegionLocations that describe where to find the region in
   * question
   * @throws IOException if a remote or network exception occurs
   */
  public RegionLocations relocateRegion(TableName tableName, byte[] row,
                                        int replicaId) throws IOException {
    return activeConnectionImpl.relocateRegion(tableName, row, replicaId);
  }

  /**
   * Update the location cache. This is used internally by HBase, in most cases it should not be
   *  used by the client application.
   * @param tableName the table name
   * @param regionName the region name
   * @param rowkey the row
   * @param exception the exception if any. Can be null.
   * @param source the previous location
   */
  @Override
  public void updateCachedLocations(TableName tableName, byte[] regionName,
                                    byte[] rowkey, Object exception, ServerName source) {
    activeConnectionImpl.updateCachedLocations(tableName, regionName, rowkey,
      exception, source);


  }

  /**
   * Gets the location of the region of <i>regionName</i>.
   * @param regionName name of the region to locate
   * @return HRegionLocation that describes where to find the region in
   * question
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public HRegionLocation locateRegion(byte[] regionName) throws IOException {
    return activeConnectionImpl.locateRegion(regionName);
  }

  /**
   * Gets the locations of all regions in the specified table, <i>tableName</i>.
   * @param tableName table to get regions of
   * @return list of region locations for all regions of table
   * @throws IOException
   */
  @Override
  public List<HRegionLocation> locateRegions(TableName tableName)
    throws IOException {
    return activeConnectionImpl.locateRegions(tableName);
  }

  /**
   * Gets the locations of all regions in the specified table, <i>tableName</i>.
   * @param tableName table to get regions of
   * @param useCache Should we use the cache to retrieve the region information.
   * @param offlined True if we are to include offlined regions, false and we'll leave out offlined
   *          regions from returned list.
   * @return list of region locations for all regions of table
   * @throws IOException
   */
  @Override
  public List<HRegionLocation> locateRegions(final TableName tableName,
                                             final boolean useCache, final boolean offlined) throws IOException {
    return activeConnectionImpl.locateRegions(tableName, useCache, offlined);
  }

  /**
   *
   * @param tableName table to get regions of
   * @param row the row
   * @param useCache Should we use the cache to retrieve the region information.
   * @param retry do we retry
   * @return region locations for this row.
   * @throws IOException
   */
  public RegionLocations locateRegion(TableName tableName,
                                      byte[] row, boolean useCache, boolean retry) throws IOException {
    return locateRegion(tableName, row, useCache, retry, RegionReplicaUtil.DEFAULT_REPLICA_ID);
  }

  /**
   *
   * @param tableName table to get regions of
   * @param row the row
   * @param useCache Should we use the cache to retrieve the region information.
   * @param retry do we retry
   * @param replicaId the replicaId for the region
   * @return region locations for this row.
   * @throws IOException
   */
  public RegionLocations locateRegion(TableName tableName,
                                      byte[] row, boolean useCache, boolean retry, int replicaId) throws IOException {
    return activeConnectionImpl.locateRegion(tableName, row, useCache, retry, replicaId);
  }

  /**
   * Returns a {@link MasterKeepAliveConnection} to the active master
   */
  @Override
  public MasterKeepAliveConnection getMaster()
    throws IOException {
    return activeConnectionImpl.getMaster();
  }

  /**
   * Establishes a connection to the region server at the specified address.
   * @param serverName
   * @return proxy for HRegionServer
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public AdminProtos.AdminService.BlockingInterface getAdmin(
    ServerName serverName) throws IOException {
    return activeConnectionImpl.getAdmin(serverName);
  }

  /**
   * Establishes a connection to the region server at the specified address, and returns
   * a region client protocol.
   *
   * @param serverName
   * @return ClientProtocol proxy for RegionServer
   * @throws IOException if a remote or network exception occurs
   *
   */
  @Override
  public ClientProtos.ClientService.BlockingInterface getClient(
    ServerName serverName) throws IOException {
    return activeConnectionImpl.getClient(serverName);
  }

  /**
   * Find region location hosting passed row
   * @param tableName table name
   * @param row Row to find.
   * @param reload If true do not use cache, otherwise bypass.
   * @return Location of row.
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public HRegionLocation getRegionLocation(TableName tableName, byte [] row,
                                           boolean reload) throws IOException {
    return activeConnectionImpl.getRegionLocation(tableName, row, reload);
  }

  /**
   * Clear any caches that pertain to server name <code>sn</code>.
   * @param sn A server name
   */
  @Override
  public void clearCaches(ServerName sn) {
    activeConnectionImpl.clearCaches(sn);
  }


  /**
   * @return Nonce generator for this HConnection; may be null if disabled in configuration.
   */
  @Override
  public NonceGenerator getNonceGenerator() {
    return activeConnectionImpl.getNonceGenerator();
  }

  /**
   * @return Default AsyncProcess associated with this connection.
   */
  @Override
  public AsyncProcess getAsyncProcess() {
    // return the asyncProcess created. Not the one in the currentConnectionImpl
    // Since when a switch happens, the pool in the currentConnectionImpl will be close
    // The user will always receive a RejectedExecutionException unless they recreate
    // the Table instance from the connection
    if (asyncProcess != null) {
      return asyncProcess;
    }
    synchronized (this) {
      if (asyncProcess != null) {
        return asyncProcess;
      }
      asyncProcess = createAsyncProcess();
      return asyncProcess;
    }
  }

  /**
   * Returns a new RpcRetryingCallerFactory from the given {@link Configuration}.
   * This RpcRetryingCallerFactory lets the users create {@link RpcRetryingCaller}s which can be
   * intercepted with the configured {@link RetryingCallerInterceptor}
   * @param conf
   * @return RpcRetryingCallerFactory
   */
  @Override
  public RpcRetryingCallerFactory getNewRpcRetryingCallerFactory(
    Configuration conf) {
    return activeConnectionImpl.getNewRpcRetryingCallerFactory(conf);
  }

  /**
   * @return Connection's RpcRetryingCallerFactory instance
   */
  public RpcRetryingCallerFactory getRpcRetryingCallerFactory() {
    return activeConnectionImpl.getRpcRetryingCallerFactory();
  }

  /**
   * @return Connection's RpcControllerFactory instance
   */
  public RpcControllerFactory getRpcControllerFactory() {
    return activeConnectionImpl.getRpcControllerFactory();
  }

  /**
   * @return a ConnectionConfiguration object holding parsed configuration values
   */
  public ConnectionConfiguration getConnectionConfiguration() {
    return activeConnectionImpl.getConnectionConfiguration();
  }

  /**
   * @return the current statistics tracker associated with this connection
   */
  @Override
  public ServerStatisticTracker getStatisticsTracker() {
    return activeConnectionImpl.getStatisticsTracker();
  }

  /**
   * @return the configured client backoff policy
   */
  @Override
  public ClientBackoffPolicy getBackoffPolicy() {
    return activeConnectionImpl.getBackoffPolicy();
  }

  /**
   * @return the MetricsConnection instance associated with this connection.
   */
  public MetricsConnection getConnectionMetrics() {
    return activeConnectionImpl.getConnectionMetrics();
  }

  /**
   * @return true when this connection uses a {@link org.apache.hadoop.hbase.codec.Codec} and so
   *         supports cell blocks.
   */
  public boolean hasCellBlockSupport() {
    return activeConnectionImpl.hasCellBlockSupport();
  }

  @Override
  public Configuration getConfiguration() {
    return conf;
  }

  @Override
  public Table getTable(TableName tableName) throws IOException {
    return getTableBuilder(tableName, getBatchPool()).build();
  }

  @Override
  public TableState getTableState(TableName tableName) throws IOException {
    return activeConnectionImpl.getTableState(tableName);
  }

  @Override
  public AdminProtos.AdminService.BlockingInterface getAdminForMaster() throws IOException {
    return activeConnectionImpl.getAdminForMaster();
  }

  public HTable getHTable(TableName tableName){
    return getHTable(tableName, getBatchPool(), this.activeClusterType, this, false);
  }

  public HTable getHTableWithStandbyConnection(TableName tableName){
    return getHTable(tableName, getStandbyBatchPool(), this.standbyClusterType,
      this.standbyConnectionImpl, true);
  }

  //use for AliHbaseMultiTable update HTable
  public HTable getHTable(TableName tableName, final ExecutorService pool, AliHBaseConstants.ClusterType clusterType,
                          ClusterConnection connection, boolean isStandby){
    if(clusterType == AliHBaseConstants.ClusterType.HBASE) {
     return new HTable(connection, new TableBuilderBase(tableName,
       connection.getConnectionConfiguration()) {
       @Override
       public Table build() {
         return null;
       }
     }, getRpcRetryingCallerFactory(), getRpcControllerFactory(), pool);
    }else if(clusterType == AliHBaseConstants.ClusterType.HBASEUE){
      if(!isStandby) {
        return (HTable)this.activeConnectionImpl.getTableBuilder(tableName, pool).build();
      }else{
        if(this.standbyConnectionImpl != null && !this.standbyConnectionImpl.isClosed()){
          return (HTable) this.standbyConnectionImpl.getTableBuilder(tableName, pool).build();
        }else{
          return null;
        }
      }
    }else{
      LOG.error("getHTable can not get cluster type, return null");
      return null;
    }
  }

  @Override
  public TableBuilder getTableBuilder(TableName tableName, final ExecutorService pool) {
    AliHBaseConstants.ClusterType clusterType =
      AliHBaseConstants.ClusterType.valueOf(this.conf.get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
    if(clusterType == AliHBaseConstants.ClusterType.HBASE) {
      return new TableBuilderBase(tableName, getConnectionConfiguration()) {
        @Override
        public Table build() {
          HTable htable = new HTable(AliHBaseMultiClusterConnection.this, this, getRpcRetryingCallerFactory(),
            getRpcControllerFactory(), pool);
          return new AliHBaseMultiTable(this.tableName, AliHBaseMultiClusterConnection.this, htable);
        }
      };
    }else if(clusterType == AliHBaseConstants.ClusterType.HBASEUE){
      return new TableBuilderBase(tableName, getConnectionConfiguration()) {
        @Override
        public Table build() {
          HTable htable = (HTable)AliHBaseMultiClusterConnection.this.activeConnectionImpl.getTableBuilder(tableName, pool).build();
          return new AliHBaseMultiTable(this.tableName, AliHBaseMultiClusterConnection.this, htable);
        }
      };
    }else{
      LOG.error("getTableBuilder can not get cluster type, return null");
      return null;
    }
  }

  /**
   * Retrieve a Table implementation for accessing a table.
   * The returned Table is not thread safe, a new instance should be created for each using thread.
   * This is a lightweight operation, pooling or caching of the returned Table
   * is neither required nor desired.
   * <p>
   * The caller is responsible for calling {@link Table#close()} on the returned
   * table instance.
   * <p>
   * Since 0.98.1 this method no longer checks table existence. An exception
   * will be thrown if the table does not exist only when the first operation is
   * attempted.
   *
   * @param tableName the name of the table
   * @param pool The thread pool to use for batch operations, null to use a default pool.
   * @return a Table to use for interactions with this table
   */
  @Override
  public Table getTable(TableName tableName,
                        ExecutorService pool) throws IOException {
    return getTableBuilder(tableName, pool).build();
  }

  /**
   * Retrieve a RegionLocator implementation to inspect region information on a table. The returned
   * RegionLocator is not thread-safe, so a new instance should be created for each using thread.
   *
   * This is a lightweight operation.  Pooling or caching of the returned RegionLocator is neither
   * required nor desired.
   *
   * RegionLocator needs to be unmanaged
   *
   * @param tableName Name of the table who's region is to be examined
   * @return A RegionLocator instance
   */
  @Override
  public RegionLocator getRegionLocator(TableName tableName)
    throws IOException {
    AliHBaseConstants.ClusterType clusterType =
      AliHBaseConstants.ClusterType.valueOf(this.conf.get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
    if(clusterType == AliHBaseConstants.ClusterType.HBASE) {
      // Need pass in the MultiClusterConnection itself
      return new HRegionLocator(tableName, this);
    }else if(clusterType == AliHBaseConstants.ClusterType.HBASEUE){
      return this.activeConnectionImpl.getRegionLocator(tableName);
    }else{
      LOG.error("getRegionLocator ClusterType is null , return null");
      return null;
    }

  }

  /**
   * Retrieve an Admin implementation to administer an HBase cluster.
   * The returned Admin is not guaranteed to be thread-safe.  A new instance should be created for
   * each using thread.  This is a lightweight operation.  Pooling or caching of the returned
   * Admin is not recommended.  Note that HConnection needs to be unmanaged
   *
   * @return an Admin instance for cluster administration
   */
  @Override
  public Admin getAdmin() throws IOException {
    HBaseAdmin hbaseAdmin = getHBaseAdmin();
    return new AliHBaseMultiAdmin(this, hbaseAdmin);
  }

  public HBaseAdmin getHBaseAdmin() throws IOException {
    AliHBaseConstants.ClusterType clusterType =
      AliHBaseConstants.ClusterType.valueOf(this.conf.get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
    if(clusterType == AliHBaseConstants.ClusterType.HBASE) {
      // Need pass in the MultiClusterConnection itself
      return new HBaseAdmin(this);
    }else if(clusterType == AliHBaseConstants.ClusterType.HBASEUE){
      return (AliHBaseUEAdmin)this.activeConnectionImpl.getAdmin();
    }else{
      LOG.error("getHBaseAdmin ClusterType is null , return null");
      return null;
    }
  }

  /**
   * A table that isTableEnabled == false and isTableDisabled == false
   * is possible. This happens when a table has a lot of regions
   * that must be processed.
   * @param tableName table name
   * @return true if the table is enabled, false otherwise
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public boolean isTableEnabled(TableName tableName) throws IOException {
    return activeConnectionImpl.isTableEnabled(tableName);
  }


  /**
   * @param tableName table name
   * @return true if the table is disabled, false otherwise
   * @throws IOException if a remote or network exception occurs
   */
  @Override
  public boolean isTableDisabled(TableName tableName) throws IOException {
    return activeConnectionImpl.isTableDisabled(tableName);
  }

  /**
   * @return the number of region servers that are currently running
   * @throws IOException if a remote or network exception occurs
   */

  public int getCurrentNrHRS() throws IOException {
    return 1;
  }

  /**
   * <p>
   * Retrieve a {@link BufferedMutator} for performing client-side buffering of writes. The
   * {@link BufferedMutator} returned by this method is thread-safe. This BufferedMutator will
   * use the Connection's ExecutorService. This object can be used for long lived operations.
   * </p>
   * <p>
   * The caller is responsible for calling {@link BufferedMutator#close()} on
   * the returned {@link BufferedMutator} instance.
   * </p>
   * <p>
   * This accessor will use the connection's ExecutorService and will throw an
   * exception in the main thread when an asynchronous exception occurs.
   *
   * @param tableName the name of the table
   *
   * @return a {@link BufferedMutator} for the supplied tableName.
   */
  @Override
  public BufferedMutator getBufferedMutator(TableName tableName)
    throws IOException {
    // Need redirect this method to
    // getBufferedMutator(BufferedMutatorParams bufferedMutatorParams)
    // to pass in the MultiClusterConnection. Otherwise, the BufferedMutator
    // will be useless after a switch happens.
    return getBufferedMutator(new BufferedMutatorParams(tableName));
  }

  /**
   * Retrieve a {@link BufferedMutator} for performing client-side buffering of writes. The
   * {@link BufferedMutator} returned by this method is thread-safe. This object can be used for
   * long lived table operations. The caller is responsible for calling
   * {@link BufferedMutator#close()} on the returned {@link BufferedMutator} instance.
   *
   * @param params details on how to instantiate the {@code BufferedMutator}.
   * @return a {@link BufferedMutator} for the supplied tableName.
   */
  @Override
  public BufferedMutator getBufferedMutator(
    BufferedMutatorParams params) throws IOException {
    BufferedMutator bufferedMutator = getBufferedMutatorByType(params);
    return new AliHBaseMultiBufferedMutator(this, bufferedMutator, params);
  }

  public BufferedMutator getBufferedMutatorByType(
    BufferedMutatorParams params) throws IOException {
    AliHBaseConstants.ClusterType clusterType =
      AliHBaseConstants.ClusterType.valueOf(this.conf.get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
    if(clusterType == AliHBaseConstants.ClusterType.HBASE) {
      if (params.getTableName() == null) {
        throw new IllegalArgumentException("TableName cannot be null.");
      }
      if (params.getPool() == null) {
        params.pool(HTable.getDefaultExecutor(getConfiguration()));
      }
      if (params.getWriteBufferSize() == BufferedMutatorParams.UNSET) {
        params.writeBufferSize(getConnectionConfiguration().getWriteBufferSize());
      }
      if (params.getMaxKeyValueSize() == BufferedMutatorParams.UNSET) {
        params.maxKeyValueSize(getConnectionConfiguration().getMaxKeyValueSize());
      }
      // Pass in the MultiClusterConnection itself
      return new BufferedMutatorImpl(this, getRpcRetryingCallerFactory(),
        getRpcControllerFactory(), params);
    }else if(clusterType == AliHBaseConstants.ClusterType.HBASEUE){
      return this.activeConnectionImpl.getBufferedMutator(params);
    }else{
      LOG.error("getBufferedMutator ClusterType is null , return null");
      return null;
    }
  }

  @Override
  public void close() throws IOException {
    if (this.closed) {
      return;
    }

    if (activeConnectionImpl != null) {
      activeConnectionImpl.close();
    }
    if(standbyConnectionImpl != null){
      standbyConnectionImpl.close();
    }
    shutdownBatchPool(batchPool, this.cleanupPool);
    shutdownBatchPool(standbyBatchPool, this.cleanupStandbyPool);
    if(dualExecutor != null){
      dualExecutor.close();
    }
    this.closed = true;
  }

  @Override
  public boolean isClosed() {
    return closed;
  }

  @Override
  public void abort(String why, Throwable throwable) {
    // MultiClusterConnection does not support abort
    // If it is a abort triggered by ZK session expire
    // Just let the underlying connection handle it.
    this.activeConnectionImpl.abort(why, throwable);
    if(this.standbyConnectionImpl != null){
      this.standbyConnectionImpl.abort(why, throwable);
    }
    aborted = true;
  }

  @Override
  public boolean isAborted() {
    return this.activeConnectionImpl.isAborted();
  }

  private void shutdownBatchPool(ExecutorService pool, boolean cleanupPool) {
    if (!cleanupPool || pool.isShutdown()) {
      return;
    }
    pool.shutdown();
    try {
      if (!pool.awaitTermination(10, TimeUnit.SECONDS)) {
        pool.shutdownNow();
      }
    } catch (InterruptedException e) {
      pool.shutdownNow();
    }
  }

  public Configuration getOriginalConf(){
    return this.conf;
  }

  public DualExecutor getDualExecutor(){
    return this.dualExecutor;
  }

  @Override
  public Hbck getHbck() throws IOException{
    return null;
  }

  @Override
  public Hbck getHbck(ServerName var1) throws IOException{
    return null;
  }

  public void clearRegionLocationCache(){
    return;
  }
}