package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.haclient.dualservice.DualExecutor;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;
import com.google.protobuf.Service;
import com.google.protobuf.ServiceException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.CompareOperator;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.coprocessor.Batch;
import org.apache.hadoop.hbase.filter.CompareFilter;
import org.apache.hadoop.hbase.ipc.CoprocessorRpcChannel;

public class AliHBaseMultiTable extends HTable implements Table {
  private static final Log LOG = LogFactory.getLog(AliHBaseMultiTable.class);

  private volatile HTable activeHTable = null;
  private volatile HTable standbyHTable = null;
  //switch count decide switch or not
  private long switchCount = 0;
  private AliHBaseMultiClusterConnection connection;
  private Configuration conf;
  private TableName tableName;
  private AliHBaseConstants.ClusterType currentType;
  private boolean tableDualEnabled = false;
  private int glitchTimeout;
  private int scannerTimeout;

  public AliHBaseMultiTable(TableName tableName, AliHBaseMultiClusterConnection connection, HTable table){
    super(connection, new AliHBaseMultiTable.HTableBuilder(tableName), null, null,
      new AliHBaseMultiTable.NoopExecutorService());
    this.activeHTable = table;
    this.connection = connection;
    this.tableName = tableName;
    this.currentType =
      AliHBaseConstants.ClusterType.valueOf(connection.getConfiguration().get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
    this.switchCount = connection.getSwitchCount();
    this.conf = connection.getConfiguration();
    this.scannerTimeout = this.conf.getInt(HConstants.HBASE_CLIENT_SCANNER_TIMEOUT_PERIOD,
      HConstants.DEFAULT_HBASE_CLIENT_SCANNER_TIMEOUT_PERIOD);
    if(this.conf.getBoolean(AliHBaseConstants.DUALSERVICE_ENABLE,
      AliHBaseConstants.DUALSERVICE_ENABLE_DEFAULT)) {
      if (this.conf.getBoolean(AliHBaseConstants.DUALSERVICE_TABLE_ENABLE,
        AliHBaseConstants.DEFAULT_DUALSERVICE_TABLE_ENABLE)) {
        tableDualEnabled = true;
      } else {
        tableDualEnabled = this.conf.getBoolean(DualExecutor.createTableConfKey(tableName.getNameAsString(),
          AliHBaseConstants.DUALSERVICE_ENABLE), false);
      }
      this.glitchTimeout = this.conf.getInt(
        DualExecutor.createTableConfKey(tableName.getNameAsString(),
          AliHBaseConstants.DUALSERVICE_GLITCHTIMEOUT), this.conf.getInt(
          AliHBaseConstants.DUALSERVICE_GLITCHTIMEOUT, AliHBaseConstants.DEFAULT_DUALSERVICE_GLITCHTIMEOUT));
      if (tableDualEnabled) {
        this.standbyHTable = this.connection.getHTableWithStandbyConnection(tableName);
      }
    }
  }

  private HTable getCurrentHTable(){
    if(this.switchCount < connection.getSwitchCount()){
      Configuration currentConf = this.connection.getConfiguration();
      AliHBaseConstants.ClusterType clusterType =
        AliHBaseConstants.ClusterType.valueOf(currentConf.get(AliHBaseConstants.ALIHBASE_CLUSTER_TYPE));
      //only HBase to HBase no need update HTable
      if(!(this.currentType == AliHBaseConstants.ClusterType.HBASE && clusterType == AliHBaseConstants.ClusterType.HBASE)) {
        LOG.debug("Update HTable, from " + this.currentType + ", to " + clusterType);
        HTable newTable = connection.getHTable(tableName);
        HTable lastHTable = this.activeHTable;
        this.activeHTable = newTable;
        if (lastHTable != null) {
          try {
            lastHTable.close();
          } catch (IOException e) {
            LOG.warn("last htable close failed" + e);
          }
        }
      }
      this.currentType = clusterType;
      this.switchCount = connection.getSwitchCount();
      this.conf = currentConf;
      if(tableDualEnabled){
        this.standbyHTable = this.connection.getHTableWithStandbyConnection(tableName);
      }
    }
    return this.activeHTable;
  }

  //visiable for test
  public boolean getTableDualEnable(){
    return this.tableDualEnabled;
  }

  private static class HTableBuilder extends TableBuilderBase {
    public HTableBuilder(TableName tableName) {
      super(tableName, new ConnectionConfiguration());
    }

    @Override
    public Table build() {
      return null;
    }
  }

  private static class NoopExecutorService implements ExecutorService {
    @Override
    public void shutdown() {

    }

    @Override
    public List<Runnable> shutdownNow() {
      return null;
    }

    @Override
    public boolean isShutdown() {
      return false;
    }

    @Override
    public boolean isTerminated() {
      return false;
    }

    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
      return false;
    }

    @Override
    public <T> Future<T> submit(Callable<T> task) {
      return null;
    }

    @Override
    public <T> Future<T> submit(Runnable task, T result) {
      return null;
    }

    @Override
    public Future<?> submit(Runnable task) {
      return null;
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks)
      throws InterruptedException {
      return null;
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks, long timeout,
                                         TimeUnit unit) throws InterruptedException {
      return null;
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks)
      throws InterruptedException, ExecutionException {
      return null;
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
      throws InterruptedException, ExecutionException, TimeoutException {
      return null;
    }

    @Override
    public void execute(Runnable command) {

    }
  }

  @Override
  public Configuration getConfiguration() {
    return this.getCurrentHTable().getConfiguration();
  }

  @Override
  public TableName getName() {
    return this.getCurrentHTable().getName();
  }

  @Override
  public HTableDescriptor getTableDescriptor() throws IOException {
    return this.getCurrentHTable().getTableDescriptor();
  }

  @Override
  public TableDescriptor getDescriptor() throws IOException {
    return this.getCurrentHTable().getDescriptor();
  }

  @Override
  public ResultScanner getScanner(Scan scan) throws IOException {
    HTable activeTable = this.getCurrentHTable();
    if(tableDualEnabled && this.standbyHTable != null){
      DualExecutor executor = this.connection.getDualExecutor();
      if(executor != null && scan.getLimit() > 0 && scan.getLimit() < 500){
        return new AliHBaseResultScanner(activeTable, this.standbyHTable, executor, scan,
          glitchTimeout, scannerTimeout);
      }
    }
   return activeTable.getScanner(scan);
  }

  @Override
  public ResultScanner getScanner(byte[] family) throws IOException {
    Scan scan = new Scan();
    scan.addFamily(family);
    return this.getScanner(scan);
  }

  @Override
  public ResultScanner getScanner(byte[] family, byte[] qualifier) throws IOException {
    Scan scan = new Scan();
    scan.addColumn(family, qualifier);
    return this.getScanner(scan);
  }

  @Override
  public Result get(Get get) throws IOException {
    HTable activeTable = this.getCurrentHTable();
    if(tableDualEnabled && this.standbyHTable != null){
      DualExecutor executor = this.connection.getDualExecutor();
      if(executor != null) {
        return executor.dualGet(activeTable, this.standbyHTable, tableName.getName(), get,
          this.glitchTimeout, this.getOperationTimeout());
      }
    }
    return activeTable.get(get);
  }

  @Override
  public Result[] get(List<Get> gets) throws IOException {
    HTable activeTable = this.getCurrentHTable();
    if(tableDualEnabled  && this.standbyHTable != null){
      DualExecutor executor = this.connection.getDualExecutor();
      if(executor != null) {
        return executor.dualBatchGet(activeTable, this.standbyHTable, tableName.getName(), gets,
          this.glitchTimeout, this.getOperationTimeout());
      }
    }
    return activeTable.get(gets);
  }

  @Override
  public void batch(List<? extends Row> actions, Object[] results) throws InterruptedException, IOException {
    this.getCurrentHTable().batch(actions, results);
  }

  @Override
  public void batch(List<? extends Row> actions, Object[] results, int rpcTimeout) throws InterruptedException, IOException {
    this.getCurrentHTable().batch(actions, results, rpcTimeout);
  }

  @Override
  public <R> void batchCallback(List<? extends Row> actions, Object[] results, Batch.Callback<R> callback) throws IOException, InterruptedException {
    this.getCurrentHTable().batchCallback(actions, results, callback);
  }

  @Override
  public void delete(final Delete delete) throws IOException {
    HTable activeTable = this.getCurrentHTable();
    if(tableDualEnabled && this.standbyHTable != null){
      DualExecutor executor = this.connection.getDualExecutor();
      if(executor != null) {
        executor.dualDelete(activeTable, this.standbyHTable, tableName.getName(), delete,
          this.glitchTimeout, this.getOperationTimeout());
        return;
      }
    }
    activeTable.delete(delete);
  }

  @Override
  public void delete(List<Delete> deletes) throws IOException {
    HTable activeTable = this.getCurrentHTable();
    if(tableDualEnabled && this.standbyHTable != null){
      DualExecutor executor = this.connection.getDualExecutor();
      if(executor != null) {
        executor.dualBatchDelete(activeTable, this.standbyHTable, tableName.getName(), deletes,
          this.glitchTimeout, this.getOperationTimeout());
        return;
      }
    }
    activeTable.delete(deletes);
  }

  @Override
  public void put(final Put put) throws IOException {
    HTable activeTable = this.getCurrentHTable();
    if(tableDualEnabled && this.standbyHTable != null){
      DualExecutor executor = this.connection.getDualExecutor();
      if(executor != null) {
        executor.dualPut(activeTable, this.standbyHTable, tableName.getName(), put,
          this.glitchTimeout, this.getOperationTimeout());
        return;
      }
    }
    activeTable.put(put);
  }

  @Override
  public void put(List<Put> puts) throws IOException {
    HTable activeTable = this.getCurrentHTable();
    if(tableDualEnabled && this.standbyHTable != null){
      DualExecutor executor = this.connection.getDualExecutor();
      if(executor != null) {
        executor.dualBatchPut(activeTable, this.standbyHTable, tableName.getName(), puts,
          this.glitchTimeout, this.getOperationTimeout());
        return;
      }
    }
    activeTable.put(puts);
  }
  @Override
  public void mutateRow(final RowMutations rm) throws IOException {
    this.getCurrentHTable().mutateRow(rm);
  }

  @Override
  public Result append(final Append append) throws IOException {
    return this.getCurrentHTable().append(append);
  }

  @Override
  public Result increment(final Increment increment) throws IOException {
    return this.getCurrentHTable().increment(increment);
  }

  @Override
  public long incrementColumnValue(byte[] row, byte[] family, byte[] qualifier, long amount) throws IOException {
    return this.getCurrentHTable().incrementColumnValue(row, family, qualifier, amount);
  }

  @Override
  public long incrementColumnValue(final byte[] row, final byte[] family, final byte[] qualifier, final long amount, final Durability durability) throws IOException {
    return this.getCurrentHTable().incrementColumnValue(row, family, qualifier, amount, durability);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public boolean checkAndPut(byte[] row, byte[] family, byte[] qualifier, byte[] value, Put put) throws IOException {
    return this.getCurrentHTable().checkAndPut(row, family, qualifier, value, put);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public boolean checkAndPut(byte[] row, byte[] family, byte[] qualifier, CompareFilter.CompareOp compareOp, byte[] value, Put put) throws IOException {
    return this.getCurrentHTable().checkAndPut(row, family, qualifier, compareOp, value, put);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public boolean checkAndPut(byte[] row, byte[] family, byte[] qualifier, CompareOperator op, byte[] value, Put put) throws IOException {
    return this.getCurrentHTable().checkAndPut(row, family, qualifier, op, value, put);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public boolean checkAndDelete(byte[] row, byte[] family, byte[] qualifier, byte[] value, Delete delete) throws IOException {
    return this.getCurrentHTable().checkAndDelete(row, family, qualifier, value, delete);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public boolean checkAndDelete(byte[] row, byte[] family, byte[] qualifier, CompareFilter.CompareOp compareOp, byte[] value, Delete delete) throws IOException {
    return this.getCurrentHTable().checkAndDelete(row, family, qualifier, compareOp, value, delete);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public boolean checkAndDelete(byte[] row, byte[] family, byte[] qualifier, CompareOperator op, byte[] value, Delete delete) throws IOException {
    return this.getCurrentHTable().checkAndDelete(row, family, qualifier, op, value, delete);
  }

  @Override
  public CheckAndMutateBuilder checkAndMutate(byte[] row, byte[] family) {
    return this.getCurrentHTable().checkAndMutate(row, family);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public boolean checkAndMutate(byte[] row, byte[] family, byte[] qualifier, CompareFilter.CompareOp compareOp, byte[] value, RowMutations rm) throws IOException {
    return this.getCurrentHTable().checkAndMutate(row, family, qualifier, compareOp, value, rm);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public boolean checkAndMutate(byte[] row, byte[] family, byte[] qualifier, CompareOperator op, byte[] value, RowMutations rm) throws IOException {
    return this.getCurrentHTable().checkAndMutate(row, family, qualifier, op, value, rm);
  }

  @Override
  public boolean exists(Get get) throws IOException {
    return this.getCurrentHTable().exists(get);
  }

  @Override
  public boolean[] exists(List<Get> gets) throws IOException {
    return this.getCurrentHTable().exists(gets);
  }

  @Override
  public <R> void processBatchCallback(List<? extends Row> list, Object[] results, Batch.Callback<R> callback) throws IOException, InterruptedException {
    this.getCurrentHTable().processBatchCallback(list, results, callback);
  }

  @Override
  public void close() throws IOException {
    super.close();
    if(this.activeHTable != null){
      this.activeHTable.close();
    }
    if(this.standbyHTable != null){
      this.standbyHTable.close();
    }
  }

  public void validatePut(Put put) throws IllegalArgumentException {
    return;
  }

  @Override
  public void clearRegionCache() {
    this.getCurrentHTable().clearRegionCache();
  }

  @Override
  public CoprocessorRpcChannel coprocessorService(byte[] row) {
    return this.getCurrentHTable().coprocessorService(row);
  }

  @Override
  public <T extends Service, R> Map<byte[], R> coprocessorService(Class<T> service, byte[] startKey, byte[] endKey, Batch.Call<T, R> callable) throws ServiceException, Throwable {
    return this.getCurrentHTable().coprocessorService(service, startKey, endKey, callable);
  }

  @Override
  public <T extends Service, R> void coprocessorService(final Class<T> service, byte[] startKey, byte[] endKey, final Batch.Call<T, R> callable, final Batch.Callback<R> callback) throws ServiceException, Throwable {
    this.getCurrentHTable().coprocessorService(service, startKey, endKey, callable, callback);
  }

  @Override
  public long getRpcTimeout(TimeUnit unit) {
    return this.getCurrentHTable().getRpcTimeout(unit);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public int getRpcTimeout() {
    return this.getCurrentHTable().getRpcTimeout();
  }

  /** @deprecated */
  @Deprecated
  @Override
  public void setRpcTimeout(int rpcTimeout) {
    this.getCurrentHTable().setRpcTimeout(rpcTimeout);
  }

  @Override
  public long getReadRpcTimeout(TimeUnit unit) {
    return this.getCurrentHTable().getReadRpcTimeout(unit);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public int getReadRpcTimeout() {
    return this.getCurrentHTable().getReadRpcTimeout();
  }

  /** @deprecated */
  @Deprecated
  @Override
  public void setReadRpcTimeout(int readRpcTimeout) {
    this.getCurrentHTable().setReadRpcTimeout(readRpcTimeout);
  }

  @Override
  public long getWriteRpcTimeout(TimeUnit unit) {
    return this.getCurrentHTable().getWriteRpcTimeout(unit);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public int getWriteRpcTimeout() {
    return this.getCurrentHTable().getWriteRpcTimeout();
  }

  /** @deprecated */
  @Deprecated
  @Override
  public void setWriteRpcTimeout(int writeRpcTimeout) {
    this.getCurrentHTable().setWriteRpcTimeout(writeRpcTimeout);
  }

  @Override
  public long getOperationTimeout(TimeUnit unit) {
    return this.getCurrentHTable().getOperationTimeout(unit);
  }

  /** @deprecated */
  @Deprecated
  @Override
  public int getOperationTimeout() {
    return this.getCurrentHTable().getOperationTimeout();
  }

  /** @deprecated */
  @Deprecated
  @Override
  public void setOperationTimeout(int operationTimeout) {
   this.getCurrentHTable().setOperationTimeout(operationTimeout);
  }

  @Override
  public String toString() {
    return this.getCurrentHTable().toString();
  }

  @Override
  public <R extends Message> Map<byte[], R> batchCoprocessorService(Descriptors.MethodDescriptor methodDescriptor, Message request, byte[] startKey, byte[] endKey, R responsePrototype) throws ServiceException, Throwable {
    return this.getCurrentHTable().batchCoprocessorService(methodDescriptor, request, startKey, endKey
      , responsePrototype);
  }

  @Override
  public <R extends Message> void batchCoprocessorService(Descriptors.MethodDescriptor methodDescriptor, Message request, byte[] startKey, byte[] endKey, R responsePrototype, Batch.Callback<R> callback) throws ServiceException, Throwable {
    this.getCurrentHTable().batchCoprocessorService(methodDescriptor, request, startKey, endKey,
      responsePrototype, callback);
  }

  @Override
  public RegionLocator getRegionLocator() {
    return this.getCurrentHTable().getRegionLocator();
  }

}