/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import com.alibaba.hbase.client.AliHBaseAPIProxy;
import com.alibaba.hbase.client.AliHBaseTableInterface;
import com.alibaba.hbase.client.AliHBaseUEAggregateService;
import com.alibaba.hbase.client.AliHBaseUEConnection;
import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.client.ElementConvertor;
import com.alibaba.hbase.exception.BatchExceptions;
import com.alibaba.lindorm.client.TableService;
import com.alibaba.lindorm.client.WideColumnService;
import com.alibaba.lindorm.client.core.meta.ExternalIndexConfig;
import com.alibaba.lindorm.client.core.meta.ExternalIndexField;
import com.alibaba.lindorm.client.core.meta.ExternalIndexRowFormatterType;
import com.alibaba.lindorm.client.core.meta.ExternalIndexType;
import com.alibaba.lindorm.client.core.widecolumnservice.WAppend;
import com.alibaba.lindorm.client.core.widecolumnservice.WDelete;
import com.alibaba.lindorm.client.core.widecolumnservice.WGet;
import com.alibaba.lindorm.client.core.widecolumnservice.WIncrement;
import com.alibaba.lindorm.client.core.widecolumnservice.WPut;
import com.alibaba.lindorm.client.core.widecolumnservice.WResult;
import com.alibaba.lindorm.client.core.widecolumnservice.WRowMutations;
import com.alibaba.lindorm.client.core.widecolumnservice.WScan;
import com.alibaba.lindorm.client.core.widecolumnservice.WScanner;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WCompareFilter;
import com.alibaba.lindorm.client.schema.LindormTableDescriptor;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;
import com.google.protobuf.Service;
import com.google.protobuf.ServiceException;
import org.apache.commons.lang3.NotImplementedException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.CompareOperator;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.coprocessor.Batch;
import org.apache.hadoop.hbase.client.metrics.ScanMetrics;
import org.apache.hadoop.hbase.filter.CompareFilter;
import org.apache.hadoop.hbase.io.TimeRange;
import org.apache.hadoop.hbase.ipc.CoprocessorRpcChannel;
import org.apache.hadoop.hbase.protobuf.generated.AggregateProtos;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;

public class AliHBaseUETable extends HTable implements AliHBaseTableInterface, RegionLocator {


  private TableName tableName;
  private AliHBaseUEConnection connection;
  private Configuration conf;
  private int operationTimeout;
  private AliHBaseAPIProxy proxy;

  private static AliHBaseUEClusterConnection superConnection = null;

  private static AliHBaseUEClusterConnection getSuperConnection(AliHBaseUEConnection connection) throws IOException{
    if(superConnection == null || superConnection.isClosed()){
        superConnection = new AliHBaseUEClusterConnection(connection);
    }
    return superConnection;
  }

  public AliHBaseUETable(TableName tableName, AliHBaseUEConnection connection) throws IOException {
    super(getSuperConnection(connection), new HTableBuilder(tableName), null, null,
        new NoopExecutorService());
    this.tableName = tableName;
    this.connection = connection;
    this.conf = connection.getConfiguration();
    this.operationTimeout = connection.getOperationTimeout();
    this.proxy = connection.getAPIProxy(tableName);
  }

  private static class HTableBuilder extends TableBuilderBase {
    public HTableBuilder(TableName tableName) {
      super(tableName, new ConnectionConfiguration());
    }

    @Override
    public Table build() {
      return null;
    }
  }

  private static class NoopExecutorService implements ExecutorService {
    @Override
    public void shutdown() {

    }

    @Override
    public List<Runnable> shutdownNow() {
      return null;
    }

    @Override
    public boolean isShutdown() {
      return false;
    }

    @Override
    public boolean isTerminated() {
      return false;
    }

    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
      return false;
    }

    @Override
    public <T> Future<T> submit(Callable<T> task) {
      return null;
    }

    @Override
    public <T> Future<T> submit(Runnable task, T result) {
      return null;
    }

    @Override
    public Future<?> submit(Runnable task) {
      return null;
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks)
        throws InterruptedException {
      return null;
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks, long timeout,
        TimeUnit unit) throws InterruptedException {
      return null;
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks)
        throws InterruptedException, ExecutionException {
      return null;
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
        throws InterruptedException, ExecutionException, TimeoutException {
      return null;
    }

    @Override
    public void execute(Runnable command) {

    }
  }



  @Override
  public HRegionLocation getRegionLocation(byte[] row) throws IOException {
    return getRegionLocation(row, false);
  }

  @Override
  public HRegionLocation getRegionLocation(byte[] row, boolean b) throws IOException {
    return proxy.getRegionLocation(row);
  }

  public HRegionLocation getRegionLocation(byte[] row, int replicaId, boolean reload) throws IOException{
    return proxy.getRegionLocation(row);
  }

  public List<HRegionLocation> getRegionLocations(byte[] row) throws IOException {
    return getRegionLocations(row, false);
  }

  public List<HRegionLocation> getRegionLocations(byte[] row, boolean reload) throws IOException{
    return Arrays.asList(proxy.getRegionLocation(row));
  }

  public void clearRegionLocationCache(){
    return;
  }

  @Override
  public List<HRegionLocation> getAllRegionLocations() throws IOException {
    return proxy.getAllRegionLocations();
  }

  @Override
  public byte[][] getStartKeys() throws IOException {
    return getStartEndKeys().getFirst();
  }

  @Override
  public byte[][] getEndKeys() throws IOException {
    return getStartEndKeys().getSecond();
  }

  @Override
  public Pair<byte[][], byte[][]> getStartEndKeys() throws IOException {
    return proxy.getStartEndKeys();
  }

  @Override
  public TableName getName() {
    return tableName;
  }

  @Override
  public Configuration getConfiguration() {
    return conf;
  }

  @Override
  public TableDescriptor getDescriptor() throws IOException {
    return proxy.getDescriptor(tableName);
  }



  @Override
  public boolean exists(Get get) throws IOException {
    return proxy.exists(get);
  }

  @Override
  public boolean[] exists(List<Get> gets) throws IOException {
    Result[] results = get(gets);
    boolean[] existResult = new boolean[results.length];
    for (int i = 0; i < results.length; i++) {
      existResult[i] = (results[i].getRow() != null);
    }
    return existResult;
  }

  @Override
  public void batch(List<? extends Row> actions, Object[] results)
      throws IOException {
    proxy.batch(actions, results);
  }

  @Override
  public <R> void batchCallback(List<? extends Row> actions, Object[] results,
      Batch.Callback<R> callback) throws IOException {
    throw new IOException("BatchCallback not supported, use put(List<Put> puts), "
        + "get(List<Get> gets) or delete(List<Delete> deletes) respectively");
  }

  @Override
  public Result get(Get get) throws IOException {
    return proxy.get(get);
  }

  @Override
  public Result[] get(List<Get> gets) throws IOException {
    return proxy.get(gets);
  }

  @Override
  public ResultScanner getScanner(Scan scan) throws IOException {
    return proxy.getScanner(scan);
  }

  @Override
  public ResultScanner getScanner(byte[] family) throws IOException {
    Scan scan = new Scan();
    scan.addFamily(family);
    return getScanner(scan);
  }

  @Override
  public ResultScanner getScanner(byte[] family, byte[] qualifier) throws IOException {
    Scan scan = new Scan();
    scan.addColumn(family, qualifier);
    return getScanner(scan);
  }

  @Override
  public void put(Put put) throws IOException {
    proxy.put(put);
  }

  @Override
  public void put(List<Put> puts) throws IOException {
    proxy.put(puts);
  }

  @Override
  public void delete(Delete delete) throws IOException {
    proxy.delete(delete);
  }

  @Override
  public void delete(List<Delete> deletes) throws IOException {
    proxy.delete(deletes);
  }

  @Override
  public CheckAndMutateBuilder checkAndMutate(byte[] row, byte[] family) {
    return proxy.checkAndMutate(row, family);
  }

  @Override
  public boolean checkAndPut(byte[] row, byte[] family, byte[] qualifier, byte[] value, Put put)
      throws IOException {
    return checkAndPut(row, family, qualifier, CompareOperator.EQUAL, value, put);
  }

  @Override
  public boolean checkAndPut(byte[] row, byte[] family, byte[] qualifier,
      CompareFilter.CompareOp compareOp, byte[] value, Put put) throws IOException {
    return checkAndPut(row, family, qualifier, CompareOperator.valueOf(compareOp.name()), value, put);
  }

  @Override
  public boolean checkAndPut(byte[] row, byte[] family, byte[] qualifier, CompareOperator op,
      byte[] value, Put put) throws IOException {
    return proxy.checkAndPut(row, family, qualifier, op, value, put);
  }

  @Override
  public boolean checkAndDelete(byte[] row, byte[] family, byte[] qualifier, byte[] value,
      Delete delete) throws IOException {
    return checkAndDelete(row, family, qualifier, CompareOperator.EQUAL, value, delete);
  }

  @Override
  public boolean checkAndDelete(byte[] row, byte[] family, byte[] qualifier,
      CompareFilter.CompareOp compareOp, byte[] value, Delete delete) throws IOException {
    return checkAndDelete(row, family, qualifier, CompareOperator.valueOf(compareOp.name()), value, delete);

  }

  @Override
  public boolean checkAndDelete(byte[] row, byte[] family, byte[] qualifier, CompareOperator op,
      byte[] value, Delete delete) throws IOException {
    return proxy.checkAndDelete(row, family, qualifier, op, value, delete);

  }

  @Override
  public boolean checkAndMutate(byte[] row, byte[] family, byte[] qualifier,
      CompareFilter.CompareOp compareOp, byte[] value, RowMutations mutation) throws IOException {
    return checkAndMutate(row, family, qualifier, CompareOperator.valueOf(compareOp.name()), value,
        mutation);
  }

  @Override
  public boolean checkAndMutate(byte[] row, byte[] family, byte[] qualifier, CompareOperator op,
      byte[] value, RowMutations mutation) throws IOException {
    return checkAndMutate(row, family).qualifier(qualifier).ifMatches(op, value).thenMutate(mutation);
  }

  @Override
  public void mutateRow(RowMutations rm) throws IOException {
    proxy.mutateRow(rm);
  }

  @Override
  public Result append(Append append) throws IOException {
    return proxy.append(append);
  }

  @Override
  public Result increment(Increment increment) throws IOException {
    return proxy.increment(increment);
  }

  @Override
  public void close() throws IOException {
    proxy.close();
  }

  @Override
  public long getRpcTimeout(TimeUnit unit) {
    return unit.convert(operationTimeout, TimeUnit.MILLISECONDS);
  }

  @Override
  public long getReadRpcTimeout(TimeUnit unit) {
    return unit.convert(operationTimeout, TimeUnit.MILLISECONDS);
  }

  @Override
  public long getWriteRpcTimeout(TimeUnit unit) {
    return unit.convert(operationTimeout, TimeUnit.MILLISECONDS);
  }

  @Override
  public long getOperationTimeout(TimeUnit unit) {
    return unit.convert(operationTimeout, TimeUnit.MILLISECONDS);
  }

  @Override
  public CoprocessorRpcChannel coprocessorService(byte[] row) {
    return proxy.coprocessorService(row);

  }

  @Override
  public <T extends Service, R> Map<byte[], R> coprocessorService(Class<T> service, byte[] startKey, byte[] endKey,
      Batch.Call<T, R> callable) throws ServiceException, Throwable {
    return proxy.coprocessorService(service, startKey, endKey, callable);
  }

  @Override
  public <T extends Service, R> void coprocessorService(Class<T> service, byte[] startKey, byte[] endKey,
      Batch.Call<T, R> callable, Batch.Callback<R> callback) throws ServiceException, Throwable {
    proxy.coprocessorService(service, startKey, endKey, callable, callback);
  }

  public void batch(final List<? extends Row> actions, final Object[] results, int rpcTimeout)
      throws InterruptedException, IOException {
    batch(actions, results);
  }

  public <R> void processBatchCallback(
      final List<? extends Row> list, final Object[] results, final Batch.Callback<R> callback)
      throws IOException, InterruptedException {
    this.batchCallback(list, results, callback);
  }

  public RegionLocator getRegionLocator() {
    return this;
  }



  @Override
  protected Connection getConnection() {
    return connection;
  }

  @Override
  public HTableDescriptor getTableDescriptor() throws IOException {
    return new HTableDescriptor(getDescriptor());
  }

  @Override
  public long incrementColumnValue(byte[] row, byte[] family, byte[] qualifier, long amount)
      throws IOException {
    Increment increment = new Increment(row).addColumn(family, qualifier, amount);
    Cell cell = increment(increment).getColumnLatestCell(family, qualifier);
    return Bytes.toLong(cell.getValueArray(), cell.getValueOffset(), cell.getValueLength());
  }

  @Override
  public long incrementColumnValue(byte[] row, byte[] family, byte[] qualifier, long amount,
      Durability durability) throws IOException {
    Increment increment = new Increment(row)
        .addColumn(family, qualifier, amount)
        .setDurability(durability);
    Cell cell = increment(increment).getColumnLatestCell(family, qualifier);
    return Bytes.toLong(cell.getValueArray(), cell.getValueOffset(), cell.getValueLength());
  }

  @Override
  public void validatePut(Put put) throws IllegalArgumentException {
    super.validatePut(put);
  }

  @Override
  ExecutorService getPool() {
    return super.getPool();
  }

  @Override
  public void clearRegionCache() {
    super.clearRegionCache();
  }

  @Override
  public int getRpcTimeout() {
    return super.getRpcTimeout();
  }

  @Override
  public void setRpcTimeout(int rpcTimeout) {
    super.setRpcTimeout(rpcTimeout);
  }

  @Override
  public int getReadRpcTimeout() {
    return super.getReadRpcTimeout();
  }

  @Override
  public void setReadRpcTimeout(int readRpcTimeout) {
    super.setReadRpcTimeout(readRpcTimeout);
  }

  @Override
  public int getWriteRpcTimeout() {
    return super.getWriteRpcTimeout();
  }

  @Override
  public void setWriteRpcTimeout(int writeRpcTimeout) {
    super.setWriteRpcTimeout(writeRpcTimeout);
  }

  @Override
  public int getOperationTimeout() {
    return super.getOperationTimeout();
  }

  @Override
  public void setOperationTimeout(int operationTimeout) {
    super.setOperationTimeout(operationTimeout);
  }

  @Override
  public <R extends Message> Map<byte[], R> batchCoprocessorService(
      Descriptors.MethodDescriptor methodDescriptor, Message request, byte[] startKey,
      byte[] endKey, R responsePrototype) throws ServiceException, Throwable {
    throw new NotImplementedException("batchCoprocessorService is not implemented!");
  }

  @Override
  public <R extends Message> void batchCoprocessorService(
      Descriptors.MethodDescriptor methodDescriptor, Message request, byte[] startKey,
      byte[] endKey, R responsePrototype, Batch.Callback<R> callback)
      throws ServiceException, Throwable {
    throw new NotImplementedException("batchCoprocessorService is not implemented!");
  }

  @Override
  public boolean[] existsAll(List<Get> gets) throws IOException {
    return exists(gets);
  }

  @Override
  public void addExternalIndex(String targetIndexName, ExternalIndexType type,
      ExternalIndexRowFormatterType rowFormatterType, List<ExternalIndexField> fields)
      throws IOException {
    ExternalIndexConfig config = new ExternalIndexConfig(targetIndexName, type, rowFormatterType);
    addExternalIndex(config, fields);
  }

  @Override
  public void addExternalIndex(String targetIndexName, ExternalIndexType type,
      ExternalIndexRowFormatterType rowFormatterType, ExternalIndexField... fields)
      throws IOException {
    List<ExternalIndexField> list = new ArrayList<ExternalIndexField>();
    for (ExternalIndexField field : fields) {
      list.add(field);
    }
    addExternalIndex(targetIndexName, type, rowFormatterType, list);

  }

  @Override
  public void addExternalIndex(ExternalIndexConfig config, List<ExternalIndexField> fields)
      throws IOException {
    proxy.addExternalIndex(config, fields);

  }

  @Override
  public void addExternalIndex(ExternalIndexConfig config, ExternalIndexField... fields)
      throws IOException {
    List<ExternalIndexField> list = new ArrayList<ExternalIndexField>();
    for (ExternalIndexField field : fields) {
      list.add(field);
    }
    addExternalIndex(config, list);
  }

  @Override
  public void addExternalIndex(List<ExternalIndexField> fields) throws IOException {
    addExternalIndex(null, fields);
  }

  @Override
  public void addExternalIndex(ExternalIndexField... fields) throws IOException {
    addExternalIndex(null, fields);
  }

  @Override
  public void modifyExternalIndexAttr(ExternalIndexConfig config) throws IOException {
    proxy.alterExternalIndex(config);
  }

  @Override
  public void removeExternalIndex(List<String> fields) throws IOException {
    proxy.removeExternalIndex(fields);
  }

  @Override
  public void removeExternalIndex(String... fields) throws IOException {
    List<String> list = new ArrayList<String>();
    for (String field : fields) {
      list.add(field);
    }
    removeExternalIndex(list);
  }

  @Override
  public void buildExternalIndex() throws IOException {
    proxy.buildExternalIndex();
  }

  @Override
  public void cancelBuildExternalIndex() throws IOException {
    proxy.cancelBuildExternalIndex();
  }

  @Override
  public void alterExternalIndex(String json) throws IOException {
    proxy.alterExternalIndex(json);
  }

  @Override
  public String describeExternalIndex() throws IOException {
    return proxy.describeExternalIndex();
  }
}
