package com.alibaba.hbase.haclient.dualservice;

import static com.alibaba.hbase.client.AliHBaseConstants.DEFAULT_DUALSERVICE_ACTIVE_EXECUTOR_QUEUE;
import static com.alibaba.hbase.client.AliHBaseConstants.DEFAULT_DUALSERVICE_ACTIVE_EXECUTOR_THREAD;
import static com.alibaba.hbase.client.AliHBaseConstants.DEFAULT_DUALSERVICE_STANDBY_EXECUTOR_QUEUE;
import static com.alibaba.hbase.client.AliHBaseConstants.DEFAULT_DUALSERVICE_STANDBY_EXECUTOR_THREAD;
import static com.alibaba.hbase.client.AliHBaseConstants.DEFAULT_DUALSERVICE_TRACE_ENABLE;
import static com.alibaba.hbase.client.AliHBaseConstants.DUALSERVICE_ACTIVE_EXECUTOR_QUEUE;
import static com.alibaba.hbase.client.AliHBaseConstants.DUALSERVICE_ACTIVE_EXECUTOR_THREAD;
import static com.alibaba.hbase.client.AliHBaseConstants.DUALSERVICE_STANDBY_EXECUTOR_QUEUE;
import static com.alibaba.hbase.client.AliHBaseConstants.DUALSERVICE_STANDBY_EXECUTOR_THREAD;
import static com.alibaba.hbase.client.AliHBaseConstants.DUALSERVICE_TRACE_ENABLE;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;


import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.util.Threads;

/**
 * A executor for dual service. executor will send request to active cluster, if
 * active request glitch timeout (defined by user) or throw exception, then send
 * request to standby cluster, excutor will merge response to client (first response
 * first return).
 *
 * request process dual executor
 *
 * 1. get execute strategy (switch or not) from autoswitch
 * 2. send request to acitve cluster
 * 3. if glitch timeout or throw exception, concurrent send request to standby cluster
 * 4. merge response return to client (first response first return).
 *
 */
public class DualExecutor {

  private static final Log LOG = LogFactory.getLog(DualExecutor.class);

  private Configuration conf;

  private ThreadPoolExecutor dualActiveThreadPool = null;

  private ThreadPoolExecutor dualStandbyThreadPool = null;

  private int activeDualThreads;

  private int standbyDualThreads;

  private int activeQueueSize;

  private int standbyQueueSize;

  private float queueRate = 0.6f;

  private boolean traceEnable;

  private AutoSwitch autoSwitch;

  public DualExecutor(Configuration conf, AutoSwitch autoSwitch){
    this.conf = conf;
    activeDualThreads = conf.getInt(DUALSERVICE_ACTIVE_EXECUTOR_THREAD,
      DEFAULT_DUALSERVICE_ACTIVE_EXECUTOR_THREAD);
    activeQueueSize = conf.getInt(DUALSERVICE_ACTIVE_EXECUTOR_QUEUE,
      DEFAULT_DUALSERVICE_ACTIVE_EXECUTOR_QUEUE);
    standbyDualThreads = conf.getInt(DUALSERVICE_STANDBY_EXECUTOR_THREAD,
      DEFAULT_DUALSERVICE_STANDBY_EXECUTOR_THREAD);
    standbyQueueSize = conf.getInt(DUALSERVICE_STANDBY_EXECUTOR_QUEUE,
      DEFAULT_DUALSERVICE_STANDBY_EXECUTOR_QUEUE);
    //init active thread pool
    dualActiveThreadPool = new ThreadPoolExecutor(activeDualThreads, activeDualThreads, 60,
      TimeUnit.SECONDS, new ArrayBlockingQueue<Runnable>(activeQueueSize), Threads.newDaemonThreadFactory("DualService-Active-Executor"),
      new ThreadPoolExecutor.AbortPolicy());
    //init standby thread pool
    dualStandbyThreadPool = new ThreadPoolExecutor(standbyDualThreads, standbyDualThreads, 60,
      TimeUnit.SECONDS, new ArrayBlockingQueue<Runnable>(standbyQueueSize),
      Threads.newDaemonThreadFactory("DualService-Standby-Executor"),
      new ThreadPoolExecutor.AbortPolicy());
    traceEnable = conf.getBoolean(DUALSERVICE_TRACE_ENABLE, DEFAULT_DUALSERVICE_TRACE_ENABLE);
    this.autoSwitch = autoSwitch;
  }

  public static String createTableConfKey(String tableName, String confKey){
    return tableName + "." + confKey;
  }


  /**
   * generate dual context for dual executor.
   * dual context include tablename, glitchTimeout, operationTimesout...
   *
   * @param activeTable
   * @param tableName
   * @param glitchTimeout
   * @param operationTimeout
   * @param actionType
   * @param row
   */
  private <T> DualContext<T> generateDualContext(HTable activeTable, final byte[] tableName,
                                                 final int glitchTimeout, final int operationTimeout,
                                                 ActionType actionType, byte[] row){
    DualContext<T> dualContext = null;
    //auto switch is enable and key level switch is enable
    if(autoSwitch.isAutoSwitchEnable() && autoSwitch.needKeyInfo()){
      HRegionLocation location = null;
      try {
        location = activeTable.getRegionLocator().getRegionLocation(row);
      }catch(IOException e){
        autoSwitch.setLocateAvaliable(false);
        return new DualContext<T>(tableName, glitchTimeout, operationTimeout, new byte[0],
          null, null);
      }
      byte[] key = null;
      if(location != null){
        key = location.getRegionInfo().getStartKey();
      }else{
        key = new byte[0];
      }
      dualContext = new DualContext<T>(tableName, glitchTimeout, operationTimeout, key,
        location, actionType);
    }else{
      dualContext = new DualContext<T>(tableName, glitchTimeout, operationTimeout);
    }
    return dualContext;
  }

  /**
   * dual get for hbase client get
   *
   * @param activeTable
   * @param standbyTable
   * @param tableName
   * @param glitchTimeout
   * @param operationTimeout
   * @throws IOException
   */
  public Result dualGet(HTable activeTable, HTable standbyTable, final byte[] tableName,
                        final Get get, final int glitchTimeout, final int operationTimeout) throws IOException{

    DualContext<Result> dualContext = this.generateDualContext(activeTable, tableName,
      glitchTimeout, operationTimeout, ActionType.GET, get.getRow());
    //get execute strategy
    AutoSwitch.ExecuteStrategy strategy = autoSwitch.getExecuteStrategy(dualContext);
    dualContext.setExecuteStrategy(strategy);
    //change request to callable
    Callable<Result> activeCallable = new DualCallable<Result, Get>(dualContext, activeTable, get
      , ActionType.GET, Role.ACTIVE);
    Callable<Result> standbyCallable = new DualCallable<Result, Get>(dualContext, standbyTable,
      get, ActionType.GET, Role.STANDBY);

    DualMetrics getMetrics = null;
    if(traceEnable){
      getMetrics = DualTrace.getInstance().getMetrics(DualTrace.GET);
    }
    return doDualOperation(activeCallable, standbyCallable, dualContext, getMetrics);
  }

  /**
   * dual put for hbase client put
   *
   * @param activeTable
   * @param standbyTable
   * @param tableName
   * @param glitchTimeout
   * @param operationTimeout
   * @throws IOException
   */
  public Void dualPut(HTable activeTable, HTable standbyTable, final byte[] tableName,
                      final Put put, final int glitchTimeout, final int operationTimeout) throws IOException{
    DualContext<Void> dualContext = this.generateDualContext(activeTable, tableName,
      glitchTimeout, operationTimeout, ActionType.PUT, put.getRow());
    //get execute strategy
    AutoSwitch.ExecuteStrategy strategy = autoSwitch.getExecuteStrategy(dualContext);
    dualContext.setExecuteStrategy(strategy);
    //change request to callable
    Callable<Void> activeCallable = new DualCallable<Void, Put>(dualContext, activeTable, put
      , ActionType.PUT, Role.ACTIVE);
    Callable<Void> standbyCallable = new DualCallable<Void, Put>(dualContext, standbyTable,
      put, ActionType.PUT, Role.STANDBY);

    DualMetrics putMetrics = null;
    if(traceEnable){
      putMetrics = DualTrace.getInstance().getMetrics(DualTrace.PUT);
    }
    return doDualOperation(activeCallable, standbyCallable, dualContext, putMetrics);
  }

  /**
   * dual delete for hbase client delete
   *
   * @param activeTable
   * @param standbyTable
   * @param tableName
   * @param glitchTimeout
   * @param operationTimeout
   * @throws IOException
   */
  public Void dualDelete(HTable activeTable, HTable standbyTable, final byte[] tableName,
                         final Delete delete, final int glitchTimeout, final int operationTimeout) throws IOException{
    DualContext<Void> dualContext = this.generateDualContext(activeTable, tableName,
      glitchTimeout, operationTimeout, ActionType.DELETE, delete.getRow());
    //get execute strategy
    AutoSwitch.ExecuteStrategy strategy = autoSwitch.getExecuteStrategy(dualContext);
    dualContext.setExecuteStrategy(strategy);
    //change request to callable
    Callable<Void> activeCallable = new DualCallable<Void, Delete>(dualContext, activeTable, delete
      , ActionType.DELETE, Role.ACTIVE);
    Callable<Void> standbyCallable = new DualCallable<Void, Delete>(dualContext, standbyTable,
      delete, ActionType.DELETE, Role.STANDBY);

    DualMetrics deleteMetrics = null;
    if(traceEnable){
      deleteMetrics = DualTrace.getInstance().getMetrics(DualTrace.DELETE);
    }
    return doDualOperation(activeCallable, standbyCallable, dualContext, deleteMetrics);
  }

  /**
   * dual batch get for hbase client batch get
   *
   * @param activeTable
   * @param standbyTable
   * @param tableName
   * @param glitchTimeout
   * @param operationTimeout
   * @throws IOException
   */
  public Result[] dualBatchGet(HTable activeTable, HTable standbyTable, final byte[] tableName,
                               final List<Get> gets, final int glitchTimeout, final int operationTimeout) throws IOException{
    DualContext<Result[]> dualContext = new DualContext<Result[]>(tableName, glitchTimeout,
      operationTimeout);
    //get execute strategy
    AutoSwitch.ExecuteStrategy strategy = autoSwitch.getExecuteStrategy(dualContext);
    dualContext.setExecuteStrategy(strategy);
    //change request to callable
    Callable<Result[]> activeCallable = new DualCallable<Result[], List<Get> >(dualContext,
      activeTable, gets, ActionType.BATCHGET, Role.ACTIVE);
    Callable<Result[]> standbyCallable = new DualCallable<Result[], List<Get> >(dualContext,
      standbyTable, gets, ActionType.BATCHGET, Role.STANDBY);

    DualMetrics batchGetMetrics = null;
    if(traceEnable){
      batchGetMetrics = DualTrace.getInstance().getMetrics(DualTrace.BATCHGET);
    }
    return doDualOperation(activeCallable, standbyCallable, dualContext, batchGetMetrics);
  }

  /**
   * dual batch put for hbase client batch put
   *
   * @param activeTable
   * @param standbyTable
   * @param tableName
   * @param glitchTimeout
   * @param operationTimeout
   * @throws IOException
   */
  public Void dualBatchPut(HTable activeTable, HTable standbyTable, final byte[] tableName,
                           final List<Put> puts, final int glitchTimeout,
                           final int operationTimeout) throws IOException{
    DualContext<Void> dualContext = new DualContext<Void>(tableName, glitchTimeout,
      operationTimeout);
    //get execute strategy
    AutoSwitch.ExecuteStrategy strategy = autoSwitch.getExecuteStrategy(dualContext);
    dualContext.setExecuteStrategy(strategy);
    //change request to callable
    Callable<Void> activeCallable = new DualCallable<Void, List<Put> >(dualContext,
      activeTable, puts, ActionType.BATCHPUT, Role.ACTIVE);
    Callable<Void> standbyCallable = new DualCallable<Void, List<Put> >(dualContext,
      standbyTable, puts, ActionType.BATCHPUT, Role.STANDBY);

    DualMetrics batchPutMetrics = null;
    if(traceEnable){
      batchPutMetrics = DualTrace.getInstance().getMetrics(DualTrace.BATCHPUT);
    }
    return doDualOperation(activeCallable, standbyCallable, dualContext, batchPutMetrics);
  }

  /**
   * dual batch delete for hbase client batch delete
   *
   * @param activeTable
   * @param standbyTable
   * @param tableName
   * @param glitchTimeout
   * @param operationTimeout
   * @throws IOException
   */
  public Void dualBatchDelete(HTable activeTable, HTable standbyTable, final byte[] tableName,
                              final List<Delete> deletes, final int glitchTimeout,
                              final int operationTimeout) throws IOException{
    DualContext<Void> dualContext = new DualContext<Void>(tableName, glitchTimeout,
      operationTimeout);
    //get execute strategy
    AutoSwitch.ExecuteStrategy strategy = autoSwitch.getExecuteStrategy(dualContext);
    dualContext.setExecuteStrategy(strategy);
    //change request to callable
    Callable<Void> activeCallable = new DualCallable<Void, List<Delete> >(dualContext,
      activeTable, deletes, ActionType.BATCHDELETE, Role.ACTIVE);
    Callable<Void> standbyCallable = new DualCallable<Void, List<Delete> >(dualContext,
      standbyTable, deletes, ActionType.BATCHDELETE, Role.STANDBY);

    DualMetrics batchDeleteMetrics = null;
    if(traceEnable){
      batchDeleteMetrics = DualTrace.getInstance().getMetrics(DualTrace.BATCHDELETE);
    }
    return doDualOperation(activeCallable, standbyCallable, dualContext, batchDeleteMetrics);
  }

  /**
   * dual next for scanner next
   *
   * @param activeScanner
   * @param standbyScanner
   * @param tableName
   * @param glitchTimeout
   * @param operationTimeout
   * @throws IOException
   */
  public DualScannerResult dualNext(ResultScanner activeScanner, ResultScanner standbyScanner,
                                    final byte[] tableName, final int glitchTimeout,
                                    final int operationTimeout) throws IOException{
    DualContext<DualScannerResult> dualContext = new DualContext<DualScannerResult>(tableName, glitchTimeout,
      operationTimeout);
    //get execute strategy
    AutoSwitch.ExecuteStrategy strategy = autoSwitch.getExecuteStrategy(dualContext);
    dualContext.setExecuteStrategy(strategy);
    //change request to callable
    Callable<DualScannerResult> activeCallable = new DualScannerCallable<DualScannerResult>(dualContext,
      activeScanner, Role.ACTIVE);
    Callable<DualScannerResult> standbyCallable = new DualScannerCallable<DualScannerResult>(dualContext,
      standbyScanner, Role.STANDBY);

    DualMetrics scanMetrics = null;
    if(traceEnable){
      scanMetrics = DualTrace.getInstance().getMetrics(DualTrace.SCAN);
    }
    return doDualOperation(activeCallable, standbyCallable, dualContext, scanMetrics);
  }

  /**
   * do dual operation
   *
   * 1. send request to acitve cluster
   * 2. if glitch timeout or throw exception, concurrent send request to standby cluster
   * 3. merge response return to client (first response first return).
   *
   * @param activeCallable
   * @param standbyCallable
   * @param dualContext
   * @param metrics
   * @throws IOException
   */
  private<T> T doDualOperation(Callable<T> activeCallable, Callable<T> standbyCallable,
                               DualContext<T> dualContext, DualMetrics metrics)throws IOException{
    Future<T> activeFuture = null;
    Future<T> standbyFuture = null;
    AutoSwitch.ExecuteStrategy executeStrategy = dualContext.getExecuteStrategy();
    try{
      if((dualActiveThreadPool != null && dualActiveThreadPool.getQueue().size() < queueRate * activeQueueSize)
        || (dualStandbyThreadPool != null && dualStandbyThreadPool.getQueue().size() < queueRate * standbyQueueSize)){
        dualContext.start();
        T result = null;
        //switch or not
        if(executeStrategy == AutoSwitch.ExecuteStrategy.DEFAULT) {
          try {
            activeFuture = dualActiveThreadPool.submit(activeCallable);
            result = dualContext.getResultInGlitchTimeout();
          } catch (Throwable t) {
            dualContext.onActiveError(t);
          }
          // result is comed back from active
          if (dualContext.usePrimaryAsResult()) {
            autoSwitch.update(dualContext);
            if(result == null){
              result = dualContext.getActiveResult();
            }
            if(result == null){
              LOG.debug("result is null , context is " + dualContext.toString());
            }
            return result;
          }
        }else{
          try {
            standbyFuture = dualStandbyThreadPool.submit(standbyCallable);
            result = dualContext.getResultInGlitchTimeout();
          } catch (Throwable t) {
            dualContext.onStandbyError(t);
          }
          // result is comed back from standby
          if (dualContext.useStandbyAsResult()) {
            autoSwitch.update(dualContext);
            if(result == null){
              result = dualContext.getStandbyResult();
            }
            if(result == null){
              LOG.debug("result is null , context is " + dualContext.toString());
            }
            return result;
          }
        }
        boolean dualsubmit = false;
        //glitch timeout or throw exception
        if(executeStrategy == AutoSwitch.ExecuteStrategy.DEFAULT) {
          try {
            standbyFuture = dualStandbyThreadPool.submit(standbyCallable);
            dualsubmit = true;
          } catch (Throwable t) {
            dualContext.onStandbyError(t);
          }
        }else{
          try {
            activeFuture = dualActiveThreadPool.submit(activeCallable);
            dualsubmit = true;
          } catch (Throwable t) {
            dualContext.onActiveError(t);
          }
        }

        if(traceEnable && dualsubmit){
          metrics.addDualExecuteHistogram();
        }
        dualContext.waitOperationTimeout();
        if (traceEnable && dualContext.useStandbyAsResult()) {
          metrics.addDualRealHistogram(System.currentTimeMillis() - dualContext.getStart());
        }
        autoSwitch.update(dualContext);
        return dualContext.getResult();
      }else{
        if(executeStrategy == AutoSwitch.ExecuteStrategy.DEFAULT) {
          LOG.debug("dual active and standby thread pool is full, disable dual service use active" +
            " call");
          try {
            return activeCallable.call();
          } catch (Exception e) {
            throw new IOException(e);
          }
        }else{
          LOG.debug("dual active and standby thread pool is full, disable dual service use " +
            "standby call");
          try {
            return standbyCallable.call();
          } catch (Exception e) {
            throw new IOException(e);
          }
        }
      }
    }finally {
      if(activeFuture != null && !activeFuture.isDone()){
        activeFuture.cancel(true);
      }
      if(standbyFuture != null && !standbyFuture.isDone()){
        standbyFuture.cancel(true);
      }
    }
  }

  public void close(){
    if(this.dualActiveThreadPool != null){
      this.dualActiveThreadPool.shutdown();
    }
  }

  //all action type dual executor support
  public enum ActionType{
    GET,
    PUT,
    DELETE,
    BATCHGET,
    BATCHPUT,
    BATCHDELETE;
  }

  public enum Role{
    ACTIVE,
    STANDBY;
  }
}