/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.hbase.client;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.BufferedMutator;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Mutation;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.RetriesExhaustedWithDetailsException;
import org.apache.hadoop.hbase.client.Table;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;

public class AliHBaseUEBufferedMutator implements BufferedMutator {

  private final TableName tableName;

  private AliHBaseUEConnection connection;

  private volatile long writeBufferSize;

  private long currentWriteBufferSize;

  private volatile boolean clearBufferOnFail;

  private final ConcurrentLinkedQueue<Mutation> writeBuffer;

  private Table table;


  public AliHBaseUEBufferedMutator(AliHBaseUEConnection connection, TableName tableName) throws IOException {
    this.tableName = tableName;
    this.connection = connection;
    this.writeBuffer = new ConcurrentLinkedQueue<Mutation>();
    this.writeBufferSize = this.connection.getConfiguration().getLong("hbase.client.write.buffer", 2097152);
    this.currentWriteBufferSize = 0;
    this.clearBufferOnFail = this.connection.getConfiguration()
        .getBoolean("hbase.client.bufferedmutator.clearbufferonfail", true);
    this.table = connection.getTable(tableName);
  }

  @Override
  public TableName getName() {
    return tableName;
  }

  @Override
  public Configuration getConfiguration() {
    return connection.getConfiguration();
  }

  @Override
  public void mutate(Mutation mutation) throws InterruptedIOException, RetriesExhaustedWithDetailsException {
    mutate(Collections.singletonList(mutation));
  }

  @Override
  public void mutate(List<? extends Mutation> list)
      throws InterruptedIOException, RetriesExhaustedWithDetailsException {
    List<Put> flushPuts = new ArrayList<Put>();
    List<Delete> flushDeletes = new ArrayList<Delete>();

    for (Mutation mutation : list) {
      writeBuffer.add(mutation);
      currentWriteBufferSize += mutation.heapSize();
    }
    try {
      if (currentWriteBufferSize >= writeBufferSize) {
        extractOMutation(flushPuts, flushDeletes);
      }
      flush(flushPuts, flushDeletes);
    } catch (IOException e) {
      // TODO: 06/09/2018
      throw new RetriesExhaustedWithDetailsException("Mutate exception", e);
    }
  }

  private void extractOMutation(List<Put> flushPuts, List<Delete> flushDeletes) throws IOException {
    for (Mutation mutation : writeBuffer) {
      if (mutation instanceof Put) {
        flushPuts.add((Put)mutation);
      } else if (mutation instanceof Delete) {
        flushDeletes.add(((Delete) mutation));
      } else {
        throw new UnsupportedOperationException("Unsupported mutation type " + mutation.getClass());
      }
    }
    writeBuffer.clear();
    currentWriteBufferSize = 0;
  }

  private void flush(List<Put> flushPuts, List<Delete> flushDeletes) throws IOException {
    if (!flushPuts.isEmpty()) {
      commitPuts(flushPuts);
    }
    if (!flushDeletes.isEmpty()) {
      commitDeletes(flushDeletes);
    }
  }

  @Override
  public void close() throws IOException {
    flush();
  }

  @Override
  public void flush() throws InterruptedIOException, RetriesExhaustedWithDetailsException{
    List<Put> flushPuts = new ArrayList<Put>();
    List<Delete> flushDeletes = new ArrayList<Delete>();
    try {
      synchronized (writeBuffer) {
        extractOMutation(flushPuts, flushDeletes);
      }
      flush(flushPuts, flushDeletes);
    } catch (IOException e) {
      // TODO: 06/09/2018
      throw new RetriesExhaustedWithDetailsException("Flush exception ", e);
    }
  }

  @Override
  public long getWriteBufferSize() {
    return writeBufferSize;
  }

  @Override
  public void setRpcTimeout(int i) {

  }

  @Override
  public void setOperationTimeout(int i) {

  }

  public void setClearBufferOnFail(boolean clearBufferOnFail) {
    this.clearBufferOnFail = clearBufferOnFail;
  }

  private void commitPuts(final List<Put> puts) throws IOException {
    boolean flushSuccessfully = false;
    try {
      this.table.put(puts);
      flushSuccessfully = true;
    } finally {
      if (!flushSuccessfully && !clearBufferOnFail) {
        synchronized (writeBuffer) {
          for (Put put : puts) {
            writeBuffer.add(put);
            currentWriteBufferSize += put.heapSize();
          }
        }
      }
    }
  }

  private void commitDeletes(final List<Delete> deletes) throws IOException {
    boolean flushSuccessfully = false;
    try {
      this.table.delete(deletes);
      flushSuccessfully = true;
    } finally {
      if (!flushSuccessfully && !clearBufferOnFail) {
        synchronized (writeBuffer) {
          for (Delete delete : deletes) {
            writeBuffer.add(delete);
            currentWriteBufferSize += delete.heapSize();
          }
        }
      }
    }
  }

}
