/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.hbase.client;


import com.alibaba.lindorm.client.core.ipc.Attributes;

import static com.alibaba.hbase.client.AliHBaseConstants.ALLOW_FULL_TALBE_SCAN;
import static com.alibaba.hbase.client.AliHBaseConstants.CHS_L;
import static com.alibaba.hbase.client.AliHBaseConstants.CHS_MAX_LAYER;
import static com.alibaba.hbase.client.AliHBaseConstants.CHS_PROMOTE_ON_MAJOR;
import static com.alibaba.hbase.client.AliHBaseConstants.COLD_BOUNDARY;
import static com.alibaba.hbase.client.AliHBaseConstants.FAMILY_ATTRIBUTES_KEY;
import static com.alibaba.hbase.client.AliHBaseConstants.FAMILY_ATTRIBUTES_KEY_BYTES;
import static com.alibaba.hbase.client.AliHBaseConstants.INDEX_ID_KEY_BYTES;
import static com.alibaba.hbase.client.AliHBaseConstants.INDEX_NAMES_KEY;
import static com.alibaba.hbase.client.AliHBaseConstants.INDEX_NAMES_KEY_BYTES;
import static com.alibaba.hbase.client.AliHBaseConstants.MUTABILITY;
import static com.alibaba.hbase.client.AliHBaseConstants.NONPK_ATTRIBUTES_KEY_BYTES;
import static com.alibaba.hbase.client.AliHBaseConstants.PK_ATTRIBUTES_KEY_BYTES;
import static com.alibaba.hbase.client.AliHBaseConstants.SILENCE_INDEXES_KEY_BYTES;
import static com.alibaba.hbase.client.AliHBaseConstants.STORAGETYPE_COLD;
import static com.alibaba.hbase.client.AliHBaseConstants.STORAGETYPE_DEFAULT;
import static com.alibaba.hbase.client.AliHBaseConstants.TABLESERVICE_ATTRIBUTES_KEY_BYTES;
import static com.alibaba.hbase.client.AliHBaseConstants.TABLE_ATTRIBUTES_KEY;
import static com.alibaba.hbase.client.AliHBaseConstants.TABLE_ATTRIBUTES_KEY_BYTES;
import static org.apache.hadoop.hbase.client.ColumnFamilyDescriptorBuilder.COMPRESSION;
import static org.apache.hadoop.hbase.client.ColumnFamilyDescriptorBuilder.DATA_BLOCK_ENCODING;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import com.alibaba.lindorm.client.core.LindormWideColumnService;
import com.alibaba.lindorm.client.core.meta.FamilyAttributes;
import com.alibaba.lindorm.client.core.meta.SilenceIndex;
import com.alibaba.lindorm.client.core.meta.TableAttributes;
import com.alibaba.lindorm.client.core.utils.BytesKeyAttributes;
import com.alibaba.lindorm.client.core.utils.DataInputBuffer;
import com.alibaba.lindorm.client.core.utils.ImmutableBytesPtr;
import com.alibaba.lindorm.client.core.utils.LindormObjectUtils;
import com.alibaba.lindorm.client.core.utils.Pair;
import com.alibaba.lindorm.client.core.utils.SchemaUtils;
import com.alibaba.lindorm.client.core.utils.StringUtils;
import com.alibaba.lindorm.client.core.utils.WritableUtils;
import com.alibaba.lindorm.client.core.widecolumnservice.WAppend;
import com.alibaba.lindorm.client.core.widecolumnservice.WColumn;
import com.alibaba.lindorm.client.core.widecolumnservice.WDelete;
import com.alibaba.lindorm.client.core.widecolumnservice.WGet;
import com.alibaba.lindorm.client.core.widecolumnservice.WIncrement;
import com.alibaba.lindorm.client.core.widecolumnservice.WPut;
import com.alibaba.lindorm.client.core.widecolumnservice.WResult;
import com.alibaba.lindorm.client.core.widecolumnservice.WRowMutations;
import com.alibaba.lindorm.client.core.widecolumnservice.WScan;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WBinaryComparator;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WBinaryPrefixComparator;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WBitComparator;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WByteArrayComparable;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WColumnCountGetFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WColumnPaginationFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WColumnPrefixFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WColumnRangeFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WColumnValueFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WCompareFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WDependentColumnFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WFamilyFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WFilterList;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WFirstKeyOnlyFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WFirstKeyValueMatchingQualifiersFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WFuzzyRowFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WInclusiveStopFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WKeyOnlyFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WMultiRowRangeFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WMultipleColumnPrefixFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WNullComparator;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WPageFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WPrefixFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WQualifierFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WRandomRowFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WRegexStringComparator;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WRowFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WSingleColumnValueExcludeFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WSingleColumnValueFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WSkipFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WSubstringComparator;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WTimestampsFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WValueFilter;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WWhileMatchFilter;
import com.alibaba.lindorm.client.dml.ColumnKey;
import com.alibaba.lindorm.client.dml.ColumnValue;
import com.alibaba.lindorm.client.exception.AccessDeniedException;
import com.alibaba.lindorm.client.exception.DoNotRetryIOException;
import com.alibaba.lindorm.client.exception.LeaseException;
import com.alibaba.lindorm.client.exception.LindormException;
import com.alibaba.lindorm.client.exception.NameSpaceNotFoundException;
import com.alibaba.lindorm.client.exception.NoSuchColumnFamilyException;
import com.alibaba.lindorm.client.exception.NotAllMetaRegionsOnlineException;
import com.alibaba.lindorm.client.exception.QuotaExceededException;
import com.alibaba.lindorm.client.exception.TableExistsException;
import com.alibaba.lindorm.client.exception.TableNotDisabledException;
import com.alibaba.lindorm.client.exception.TableNotEnabledException;
import com.alibaba.lindorm.client.exception.TableNotFoundException;
import com.alibaba.lindorm.client.exception.UnknownProtocolException;
import com.alibaba.lindorm.client.exception.UnknownScannerException;
import com.alibaba.lindorm.client.schema.ColumnFamilyDescriptor;
import com.alibaba.lindorm.client.schema.ColumnSchema;
import com.alibaba.lindorm.client.schema.DataType;
import com.alibaba.lindorm.client.schema.IndexState;
import com.alibaba.lindorm.client.schema.IndexedColumnSchema;
import com.alibaba.lindorm.client.schema.LindormAttribute;
import com.alibaba.lindorm.client.schema.LindormFamilyAttributeConstants;
import com.alibaba.lindorm.client.schema.LindormFamilyAttributes;
import com.alibaba.lindorm.client.schema.LindormIndexDescriptor;
import com.alibaba.lindorm.client.schema.LindormTableDescriptor;
import com.alibaba.lindorm.client.schema.Mutability;
import com.alibaba.lindorm.client.schema.PrimaryKeySchema;
import com.alibaba.lindorm.client.schema.SortOrder;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.CellComparator;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.CompareOperator;
import org.apache.hadoop.hbase.HBaseIOException;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.NamespaceNotFoundException;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Append;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Durability;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Increment;
import org.apache.hadoop.hbase.client.IsolationLevel;
import org.apache.hadoop.hbase.client.Mutation;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Query;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.RowMutations;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.client.coprocessor.BigDecimalColumnInterpreter;
import org.apache.hadoop.hbase.client.coprocessor.DoubleColumnInterpreter;
import org.apache.hadoop.hbase.client.coprocessor.LongColumnInterpreter;
import org.apache.hadoop.hbase.client.index.AliHBaseColumn;
import org.apache.hadoop.hbase.client.index.AliHBaseIndexDescriptor;
import org.apache.hadoop.hbase.coprocessor.ColumnInterpreter;
import org.apache.hadoop.hbase.filter.BinaryComparator;
import org.apache.hadoop.hbase.filter.BinaryPrefixComparator;
import org.apache.hadoop.hbase.filter.BitComparator;
import org.apache.hadoop.hbase.filter.ByteArrayComparable;
import org.apache.hadoop.hbase.filter.ColumnCountGetFilter;
import org.apache.hadoop.hbase.filter.ColumnPaginationFilter;
import org.apache.hadoop.hbase.filter.ColumnPrefixFilter;
import org.apache.hadoop.hbase.filter.ColumnRangeFilter;
import org.apache.hadoop.hbase.filter.ColumnValueFilter;
import org.apache.hadoop.hbase.filter.DependentColumnFilter;
import org.apache.hadoop.hbase.filter.FamilyFilter;
import org.apache.hadoop.hbase.filter.Filter;
import org.apache.hadoop.hbase.filter.FilterList;
import org.apache.hadoop.hbase.filter.FirstKeyOnlyFilter;
import org.apache.hadoop.hbase.filter.FirstKeyValueMatchingQualifiersFilter;
import org.apache.hadoop.hbase.filter.FuzzyRowFilter;
import org.apache.hadoop.hbase.filter.InclusiveStopFilter;
import org.apache.hadoop.hbase.filter.KeyOnlyFilter;
import org.apache.hadoop.hbase.filter.MultiRowRangeFilter;
import org.apache.hadoop.hbase.filter.MultipleColumnPrefixFilter;
import org.apache.hadoop.hbase.filter.NullComparator;
import org.apache.hadoop.hbase.filter.PageFilter;
import org.apache.hadoop.hbase.filter.PrefixFilter;
import org.apache.hadoop.hbase.filter.QualifierFilter;
import org.apache.hadoop.hbase.filter.RandomRowFilter;
import org.apache.hadoop.hbase.filter.RegexStringComparator;
import org.apache.hadoop.hbase.filter.RowFilter;
import org.apache.hadoop.hbase.filter.SingleColumnValueExcludeFilter;
import org.apache.hadoop.hbase.filter.SingleColumnValueFilter;
import org.apache.hadoop.hbase.filter.SkipFilter;
import org.apache.hadoop.hbase.filter.SubstringComparator;
import org.apache.hadoop.hbase.filter.TimestampsFilter;
import org.apache.hadoop.hbase.filter.ValueFilter;
import org.apache.hadoop.hbase.filter.WhileMatchFilter;
import org.apache.hadoop.hbase.io.TimeRange;
import org.apache.hadoop.hbase.regionserver.BloomType;
import org.apache.hadoop.hbase.shaded.protobuf.generated.ComparatorProtos;
import org.apache.hadoop.hbase.util.Bytes;

import org.apache.hbase.thirdparty.com.google.protobuf.InvalidProtocolBufferException;

import org.apache.hadoop.hbase.shaded.protobuf.generated.FilterProtos;
import org.apache.hadoop.hbase.shaded.protobuf.generated.HBaseProtos;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static com.alibaba.hbase.client.AliHBaseConstants.CHS_L;
import static com.alibaba.hbase.client.AliHBaseConstants.CHS_MAX_LAYER;
import static com.alibaba.hbase.client.AliHBaseConstants.CHS_PROMOTE_ON_MAJOR;
import static com.alibaba.hbase.client.AliHBaseConstants.COLD_BOUNDARY;
import static com.alibaba.hbase.client.AliHBaseConstants.STORAGETYPE_COLD;
import static com.alibaba.hbase.client.AliHBaseConstants.STORAGETYPE_DEFAULT;
import static com.alibaba.hbase.client.AliHBaseConstants.STORAGE_POLICY;
import static com.alibaba.lindorm.client.schema.LindormFamilyAttributeConstants.*;

public class ElementConvertor {
  private static final long DEFAULT_MAX_RESULT_SIZE = -1L;

  private static final String TABLEMETAVERSION = "TABLEMETAVERSION";
  private static final byte[] TABLEMETAVERSIONBYTES = Bytes.toBytes(TABLEMETAVERSION);

  private static final String COLD_HOT_COLD_STORAGETYPE_CONFIG = "storagetype=" + STORAGETYPE_COLD;
  private static final byte[] ROWKEY_NAME = Bytes.toBytes(LindormWideColumnService.UNIFIED_PK_COLUMN_NAME);

  private static final Map<String, DataType> INTERPRETER_DATATYPE_MAP = new HashMap<>();

  // used for external application to reset EXTENDED_CONVERTTOR, thus support more convert
  public static ExtendedElementConvertor EXTENDED_CONVERTTOR = new ExtendedElementConvertor();

  static {
    INTERPRETER_DATATYPE_MAP.put(LongColumnInterpreter.class.getName(), DataType.LONG);
    INTERPRETER_DATATYPE_MAP.put(DoubleColumnInterpreter.class.getName(), DataType.DOUBLE);
    INTERPRETER_DATATYPE_MAP.put(BigDecimalColumnInterpreter.class.getName(), DataType.DECIMAL);
  }


  private static WColumn.Type convertTypeToWType(byte type) {
    switch (KeyValue.Type.codeToType(type)) {
      case Minimum:
        return WColumn.Type.Minimum;
      case Put:
        return WColumn.Type.Put;
      case Delete:
        return WColumn.Type.Delete;
      case DeleteColumn:
        return WColumn.Type.DeleteColumn;
      case DeleteFamily:
        return WColumn.Type.DeleteFamily;
      case Maximum:
        return WColumn.Type.Maximum;
    }
    throw new UnsupportedOperationException("Unsupport op type " + KeyValue.Type.codeToType(type));
  }

  private static KeyValue.Type convertWTypeToHType(WColumn.Type type) {
    switch (type) {
      case Minimum:
        return KeyValue.Type.Minimum;
      case Put:
        return KeyValue.Type.Put;
      case Delete:
        return KeyValue.Type.Delete;
      case DeleteColumn:
        return KeyValue.Type.DeleteColumn;
      case DeleteFamily:
        return KeyValue.Type.DeleteFamily;
      case Maximum:
        return KeyValue.Type.Maximum;
    }
    throw new UnsupportedOperationException("Unsupport op type " + type);
  }

  /**
   * Creates a {@link WDelete} (Tablestore) from a {@link Delete} (HBase).
   *
   * @param delete the <code>Delete</code> to convert
   * @return converted <code>WDelete</code>
   */
  public static WDelete toLindormDelete(Delete delete) throws IOException {
    checkDeleteSupport(delete);
    // TODO: 06/09/2018 lock id ?
    WDelete wdel = new WDelete(delete.getRow(), delete.getTimestamp());

//    wdel.setWriteToWAL(!Durability.SKIP_WAL.equals(delete.getDurability()));

    for (Map.Entry<byte[], List<Cell>> entry : delete.getFamilyCellMap().entrySet()) {
      for (Cell kv : entry.getValue()) {
        wdel.addDeleteMarker(new WColumn(CellUtil.cloneFamily(kv), CellUtil.cloneQualifier(kv),
            CellUtil.cloneValue(kv), kv.getTimestamp(),
            convertTypeToWType(kv.getType().getCode())));
      }
    }
    copyAttributes(wdel, delete.getAttributesMap());
    return wdel;
  }

  public static List<Delete> toHBaseDeletes(List<WDelete> in) throws IOException {
    List<Delete> out = new ArrayList<Delete>(in.size());
    for (WDelete delete : in) {
      out.add(toHBaseDelete(delete));
    }
    return out;
  }

  public static Delete toHBaseDelete(WDelete in) {
    Delete out = new Delete(in.getRowKey());

    Map<byte[], List<WColumn>> familyMap = in.getFamilyMap();
    for (byte[] family : familyMap.keySet()) {
      for (WColumn columnValue : familyMap.get(family)) {
        byte[] qualifier = columnValue.getQualifier();
        long timestamp = columnValue.getTs();
        if (columnValue.getType() == WColumn.Type.Delete) {
          out.addColumn(family, qualifier, timestamp);
        } else if (columnValue.getType() == WColumn.Type.DeleteColumn) {
          out.addColumns(family, qualifier, timestamp);
        } else if (columnValue.getType() == WColumn.Type.DeleteFamily) {
          out.addFamily(family, timestamp);
        }
      }
    }
    return out;
  }

  private static void checkPutSupport(Put in) throws IOException {
    if (in.getACL() != null) {
      throw new IOException("Put#setACL() is not supported");
    }
  }

  private static void checkIncrementSupport(Increment in) throws IOException {
    if (in.getACL() != null) {
      throw new IOException("Increment#setACL() is not supported");
    }
    if (in.getTTL() != Long.MAX_VALUE) {
      throw new IOException("Increment#setTTL() is not supported");
    }
  }

  private static void checkAppendSupport(Append in) throws IOException {
    if (in.getACL() != null) {
      throw new IOException("Append#setACL() is not supported");
    }
    if (in.getTTL() != Long.MAX_VALUE) {
      throw new IOException("Append#setTTL() is not supported");
    }
  }

  private static void checkDeleteSupport(Delete in) throws IOException {
    if (in.getACL() != null) {
      throw new IOException("Delete#setACL() is not supported");
    }
    if (in.getTTL() != Long.MAX_VALUE) {
      throw new IOException("Delete#setTTL() is not supported");
    }
  }

  /**
   * Converts multiple {@link Delete}s (HBase) into a list of {@link WDelete}s
   * (Lindorm).
   *
   * @param in list of <code>Delete</code>s to convert
   * @return list of converted <code>WDelete</code>s
   */
  public static List<WDelete> toLindormDeleteList(List<Delete> in) throws IOException {
    List<WDelete> out = new ArrayList<WDelete>(in.size());
    for (Delete delete : in) {
      if (delete != null) {
        out.add(toLindormDelete(delete));
      }
    }
    return out;
  }

  private static boolean isQueryHotOnly(Query query) {
    byte[] value = query.getAttribute(AliHBaseConstants.HOT_ONLY);
    return value != null && Bytes.toBoolean(value);
  }

  private static boolean isHotColdAutoMerge(Query query) {
    byte[] value = query.getAttribute(AliHBaseConstants.COLD_HOT_MERGE);
    return value != null && Bytes.toBoolean(value);
  }

  /**
   * Creates a {@link WGet} (Lindorm) from a {@link Get} (HBase).
   *
   * @param get the <code>Get</code> to convert
   * @return <code>WGet</code> object
   * @throws IOException
   */
  public static WGet toLindormGet(Get get) throws IOException {
    checkGetRowSupport(get);
    WGet wget = new WGet(get.getRow());
    wget.setCacheBlocks(get.getCacheBlocks());
    wget.setMaxVersions(get.getMaxVersions());
    TimeRange tr = get.getTimeRange();
    if (!tr.isAllTime()) {
      wget.setTimeRange(tr.getMin(), tr.getMax());
    }
    wget.setFamilyMap(get.getFamilyMap());
    wget.setFilter(toLindormFilter(get.getFilter()));
    if (isQueryHotOnly(get)) {
      wget.setQueryHotOnly(true);
    } else if (isHotColdAutoMerge(get)) {
      wget.setHotColdAutoMerge(true);
    }
    copyAttributes(wget, get.getAttributesMap());
    return wget;
  }

  private static void checkGetRowSupport(Get in) throws IOException{
    if (in.isClosestRowBefore()) {
      throw new IOException("Get#setClosestRowBefore(true) is not supported");
    }
    if (in.getACL() != null) {
      throw new IOException("Get#setACL() is not supported");
    }
    //if (in.getMaxResultsPerColumnFamily() != -1) {
    //  throw new IOException("Get#setMaxResultsPerColumnFamily() is not supported");
    //}
    //if (in.getConsistency() == Consistency.TIMELINE) {
    //  throw new IOException(
    //      "Get#setConsistency(Consistency.TIMELINE) is not supported");
    //}
    //if (!in.getAttributesMap().isEmpty()) {
    //  throw new IOException("Get#setAttribute() is not supported");
    //}
    //if (in.getIsolationLevel() == IsolationLevel.READ_UNCOMMITTED) {
    //  throw new IOException(
    //      "Get#setIsolationLevel(READ_UNCOMMITTED) is not supported");
    //}
    //if (in.getReplicaId() > 0) {
    //  throw new IOException("Get#setReplicaId() is not supported");
    //}
    //if (in.isCheckExistenceOnly()) {
    //  throw new IOException("Get#setCheckExistenceOnly() is not supported");
    //}
    //if (in.getLoadColumnFamiliesOnDemandValue() != null &&
    //    in.getLoadColumnFamiliesOnDemandValue()) {
    //  throw new IOException("Get#setLoadColumnFamiliesOnDemand() is not supported");
    //}

    //if (in.getPriority() != -1) {
    //  throw new IOException("Get#setPriority() is not supported");
    //}
  }

  static WFilter toLindormFilter(Filter filter) throws IOException {
    if (filter == null) {
      return null;
    }
    if (filter instanceof FilterList) {
      FilterList filterList = ((FilterList) filter);
      WFilterList wlist = null;
      if (filterList.getOperator() == FilterList.Operator.MUST_PASS_ALL) {
        wlist = new WFilterList(WFilterList.Operator.MUST_PASS_ALL);
      } else {
        wlist = new WFilterList(WFilterList.Operator.MUST_PASS_ONE);
      }
      for (Filter childFilter : filterList.getFilters()) {
        wlist.addFilter(toLindormFilter(childFilter));
      }
      return wlist;
    } else if (filter instanceof SingleColumnValueExcludeFilter) {
      SingleColumnValueExcludeFilter sFilter = (SingleColumnValueExcludeFilter) filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(sFilter.getOperator());
      WByteArrayComparable wComparator = toLindormComparator(sFilter.getComparator());
      return new WSingleColumnValueExcludeFilter(sFilter.getFamily(), sFilter.getQualifier(), op,
          wComparator);
    } else if (filter instanceof SingleColumnValueFilter) {
      SingleColumnValueFilter sFilter = (SingleColumnValueFilter) filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(sFilter.getOperator());
      WByteArrayComparable wComparator = toLindormComparator(sFilter.getComparator());
      WSingleColumnValueFilter wsFilter = new WSingleColumnValueFilter(sFilter.getFamily(),
          sFilter.getQualifier(), op, wComparator);
      wsFilter.setLatestVersionOnly(sFilter.getLatestVersionOnly());
      wsFilter.setFilterIfMissing(sFilter.getFilterIfMissing());
      return wsFilter;
    } else if (filter instanceof ColumnCountGetFilter) {
      ColumnCountGetFilter sFilter = (ColumnCountGetFilter) filter;
      return new WColumnCountGetFilter(sFilter.getLimit());
    } else if (filter instanceof ColumnPrefixFilter) {
      ColumnPrefixFilter sFilter = (ColumnPrefixFilter) filter;
      return new WColumnPrefixFilter(sFilter.getPrefix());
    } else if (filter instanceof ColumnRangeFilter) {
      ColumnRangeFilter sFilter = (ColumnRangeFilter) filter;
      return new WColumnRangeFilter(sFilter.getMinColumn(), sFilter.getMinColumnInclusive(),
          sFilter.getMaxColumn(), sFilter.getMaxColumnInclusive());
    } else if (filter instanceof DependentColumnFilter) {
      DependentColumnFilter sFilter = (DependentColumnFilter) filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(sFilter.getOperator());
      WByteArrayComparable wComparator = null;
      if(null != sFilter.getComparator()){
        wComparator = toLindormComparator(sFilter.getComparator());
      }
      return new WDependentColumnFilter(sFilter.getFamily(), sFilter.getQualifier(),
          sFilter.getDropDependentColumn(), op, wComparator);
    } else if (filter instanceof FamilyFilter) {
      FamilyFilter sFilter = (FamilyFilter) filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(sFilter.getOperator());
      WByteArrayComparable wComparator = toLindormComparator(sFilter.getComparator());
      return new WFamilyFilter(op, wComparator);
    } else if (filter instanceof QualifierFilter) {
      QualifierFilter sFilter = (QualifierFilter) filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(sFilter.getOperator());
      WByteArrayComparable wComparator = toLindormComparator(sFilter.getComparator());
      return new WQualifierFilter(op, wComparator);
    } else if (filter instanceof RowFilter) {
      RowFilter sFilter = (RowFilter) filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(sFilter.getOperator());
      WByteArrayComparable wComparator = toLindormComparator(sFilter.getComparator());
      return new WRowFilter(op, wComparator);
    } else if (filter instanceof ValueFilter) {
      ValueFilter sFilter = (ValueFilter) filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(sFilter.getOperator());
      WByteArrayComparable wComparator = toLindormComparator(sFilter.getComparator());
      return new WValueFilter(op, wComparator);
    } else if(filter instanceof FirstKeyValueMatchingQualifiersFilter){
      FirstKeyValueMatchingQualifiersFilter firstKeyValueMatchingQualifiersFilter = (FirstKeyValueMatchingQualifiersFilter)filter;
      return new WFirstKeyValueMatchingQualifiersFilter(
          getQualifierFromFirstKeyValueMatchingQualifiersFilter(
              firstKeyValueMatchingQualifiersFilter));
    } else if (filter instanceof FirstKeyOnlyFilter) {
      return new WFirstKeyOnlyFilter();
    } else if (filter instanceof InclusiveStopFilter) {
      InclusiveStopFilter sFilter = (InclusiveStopFilter) filter;
      return new WInclusiveStopFilter(sFilter.getStopRowKey());
    } else if (filter instanceof MultipleColumnPrefixFilter) {
      MultipleColumnPrefixFilter sFilter = (MultipleColumnPrefixFilter) filter;
      return new WMultipleColumnPrefixFilter(sFilter.getPrefix());
      //    } else if (filter instanceof PageFilter) {
      //      PageFilter sFilter = (PageFilter) filter;
      //      return new WPageFilter(sFilter.getPageSize());
    } else if (filter instanceof PrefixFilter) {
      PrefixFilter sFilter = (PrefixFilter) filter;
      return new WPrefixFilter(sFilter.getPrefix());
    } else if (filter instanceof RandomRowFilter) {
      RandomRowFilter sFilter = (RandomRowFilter) filter;
      return new WRandomRowFilter(sFilter.getChance());
    } else if (filter instanceof SkipFilter) {
      SkipFilter sFilter = (SkipFilter) filter;
      return new WSkipFilter(toLindormFilter(sFilter.getFilter()));
    } else if (filter instanceof TimestampsFilter) {
      TimestampsFilter sFilter = (TimestampsFilter) filter;
      return new WTimestampsFilter(sFilter.getTimestamps());
    } else if (filter instanceof WhileMatchFilter) {
      WhileMatchFilter sFilter = (WhileMatchFilter) filter;
      return new WWhileMatchFilter(toLindormFilter(sFilter.getFilter()));
    } else if (filter instanceof FuzzyRowFilter) {
      FuzzyRowFilter fuzzyRowFilter = (FuzzyRowFilter)filter;
      byte[] bytes = fuzzyRowFilter.toByteArray();
      FilterProtos.FuzzyRowFilter proto;
      try {
        proto = FilterProtos.FuzzyRowFilter.parseFrom(bytes);
      } catch (InvalidProtocolBufferException e) {
        throw new IOException(e);
      }
      int count = proto.getFuzzyKeysDataCount();
      ArrayList<Pair<byte[], byte[]>> fuzzyKeysData =
          new ArrayList<>(count);
      for (int i = 0; i < count; ++i) {
        HBaseProtos.BytesBytesPair current = proto.getFuzzyKeysData(i);
        byte[] keyBytes = current.getFirst().toByteArray();
        byte[] keyMeta = current.getSecond().toByteArray();
        // mask adapter ( -1(0xff) -> 0, 2 -> 1), FuzzyRowFilter is fast version in HBase 1.x+, add adapter
        keyMeta = maskAdapter(keyMeta);
        fuzzyKeysData.add(new Pair<>(keyBytes, keyMeta));
      }
      return new WFuzzyRowFilter(fuzzyKeysData);
    } else if (filter instanceof KeyOnlyFilter) {
      KeyOnlyFilter keyOnlyFilter = (KeyOnlyFilter)filter;
      byte[] bytes = keyOnlyFilter.toByteArray();
      FilterProtos.KeyOnlyFilter proto;
      try {
        proto = FilterProtos.KeyOnlyFilter.parseFrom(bytes);
      } catch (InvalidProtocolBufferException e) {
        throw new IOException(e);
      }
      return new WKeyOnlyFilter(proto.getLenAsVal());
    } else if (filter instanceof PageFilter) {
      PageFilter pageFilter = (PageFilter)filter;
      return new WPageFilter(pageFilter.getPageSize());
    } else if (filter instanceof ColumnPaginationFilter) {
      ColumnPaginationFilter columnPaginationFilter = (ColumnPaginationFilter)filter;
      return new WColumnPaginationFilter(columnPaginationFilter.getLimit(), columnPaginationFilter.getOffset());
    } else if (filter instanceof MultiRowRangeFilter) {
      MultiRowRangeFilter multiRowRangeFilter = (MultiRowRangeFilter)filter;
      return new WMultiRowRangeFilter(convertToWRowRangeList(multiRowRangeFilter.getRowRanges()));
    } else if (filter instanceof ColumnValueFilter) {
      ColumnValueFilter columnValueFilter = (ColumnValueFilter)filter;
      WCompareFilter.CompareOp op = toLindormCompareOp(columnValueFilter.getCompareOperator());
      WByteArrayComparable wComparator = toLindormComparator(columnValueFilter.getComparator());
      WColumnValueFilter wColumnValueFilter = new WColumnValueFilter(columnValueFilter.getFamily(),
          columnValueFilter.getQualifier(), op, wComparator);
      return wColumnValueFilter;
    } else if (EXTENDED_CONVERTTOR != null) {
      return EXTENDED_CONVERTTOR.toLindormFilter(filter);
    } else {
      throw new IOException("Unsupport filter " + filter.getClass().getName());
    }
  }

  public static List<WMultiRowRangeFilter.WRowRange> convertToWRowRangeList(List<MultiRowRangeFilter.RowRange> rowRanges) {
    List<WMultiRowRangeFilter.WRowRange> ranges = new ArrayList<>();
    for (MultiRowRangeFilter.RowRange rowRange : rowRanges) {
      ranges.add(convertToWRowRange(rowRange));
    }
    return ranges;
  }

  public static WMultiRowRangeFilter.WRowRange convertToWRowRange(MultiRowRangeFilter.RowRange rowRange) {
    return new WMultiRowRangeFilter.WRowRange(rowRange.getStartRow(),
        rowRange.isStartRowInclusive(), rowRange.getStopRow(), rowRange.isStopRowInclusive());
  }

  private static Set<byte[]> getQualifierFromFirstKeyValueMatchingQualifiersFilter(FirstKeyValueMatchingQualifiersFilter filter) throws IOException {
    try {
      Field field = FirstKeyValueMatchingQualifiersFilter.class.getDeclaredField("qualifiers");
      field.setAccessible(true);
      return (Set<byte[]>)field.get(filter);
    } catch (Throwable t) {
      throw new IOException(t);
    }
  }

  private static byte[] maskAdapter(byte[] mask) {
    if (!isPreprocessedMask(mask)) return mask;
    for (int i = 0; i < mask.length; i++) {
      if (mask[i] == -1) {
        mask[i] = 0; // -1 -> 0
      } else if (mask[i] == 2) {
        mask[i] = 1;// 2 -> 1
      }
    }
    return mask;
  }

  private static boolean isPreprocessedMask(byte[] mask) {
    for (int i = 0; i < mask.length; i++) {
      if (mask[i] != -1 && mask[i] != 2) {
        return false;
      }
    }
    return true;
  }


  public static WCompareFilter.CompareOp toLindormCompareOp(
      org.apache.hadoop.hbase.filter.CompareFilter.CompareOp operator) throws IOException {
    switch (operator) {
      case LESS:
        return WCompareFilter.CompareOp.LESS;
      case LESS_OR_EQUAL:
        return WCompareFilter.CompareOp.LESS_OR_EQUAL;
      case EQUAL:
        return WCompareFilter.CompareOp.EQUAL;
      case NOT_EQUAL:
        return WCompareFilter.CompareOp.NOT_EQUAL;
      case GREATER_OR_EQUAL:
        return WCompareFilter.CompareOp.GREATER_OR_EQUAL;
      case GREATER:
        return WCompareFilter.CompareOp.GREATER;
      case NO_OP:
        return WCompareFilter.CompareOp.NO_OP;
    }
    throw new IOException("Unkown operator " + operator);
  }

  public static WCompareFilter.CompareOp toLindormCompareOp(
      CompareOperator operator) throws IOException {
    switch (operator) {
      case LESS:
        return WCompareFilter.CompareOp.LESS;
      case LESS_OR_EQUAL:
        return WCompareFilter.CompareOp.LESS_OR_EQUAL;
      case EQUAL:
        return WCompareFilter.CompareOp.EQUAL;
      case NOT_EQUAL:
        return WCompareFilter.CompareOp.NOT_EQUAL;
      case GREATER_OR_EQUAL:
        return WCompareFilter.CompareOp.GREATER_OR_EQUAL;
      case GREATER:
        return WCompareFilter.CompareOp.GREATER;
      case NO_OP:
        return WCompareFilter.CompareOp.NO_OP;
    }
    throw new IOException("Unkown operator " + operator);
  }


  private static WBitComparator.BitwiseOp toLindormBitwiseOp(
      BitComparator.BitwiseOp bitwiseOp) throws IOException {
    switch(bitwiseOp){
      case AND:
        return WBitComparator.BitwiseOp.AND;
      case OR:
        return WBitComparator.BitwiseOp.OR;
      case XOR:
        return WBitComparator.BitwiseOp.XOR;
    }
    throw new IOException("Unkown BitwiseOp " + bitwiseOp);
  }

  private static WByteArrayComparable toLindormComparator(
      ByteArrayComparable comparator) throws IOException {
    if(comparator instanceof BinaryComparator){
      return new WBinaryComparator(comparator.getValue());
    }else if(comparator instanceof BinaryPrefixComparator){
      return new WBinaryPrefixComparator(comparator.getValue());
    }else if(comparator instanceof BitComparator){
      WBitComparator.BitwiseOp wbitWistOp = toLindormBitwiseOp(((BitComparator) comparator).getOperator());
      return new WBitComparator(comparator.getValue(), wbitWistOp);
    }else if(comparator instanceof NullComparator){
      return new WNullComparator();
    }else if(comparator instanceof RegexStringComparator){
      ComparatorProtos.RegexStringComparator proto;
      try {
        proto = ComparatorProtos.RegexStringComparator.parseFrom(comparator.toByteArray());
      } catch (InvalidProtocolBufferException e) {
        throw new IOException(e.getMessage());
      }
      return new WRegexStringComparator(proto.getPattern());
    }else if(comparator instanceof SubstringComparator){
      return new WSubstringComparator(Bytes.toString(comparator.getValue()));
    }
    throw new IOException("Unknow comparator " + comparator);
  }

  /**
   * Converts multiple {@link WPut}s (Lindorm) into a list of {@link Put}s (HBase).
   *
   * @param in list of <code>WPut</code>s to convert
   * @return list of converted <code>Put</code>s
   */
  public static List<Put> toHBasePuts(List<WPut> in) {
    List<Put> out = new ArrayList<Put>(in.size());
    for (WPut put : in) {
      out.add(toHBasePut(put));
    }
    return out;
  }

  /**
   * Creates a {@link Put} (HBase) from a {@link WPut} (Lindorm)
   *
   * @param in the <code>WPut</code> to convert
   * @return converted <code>Put</code>
   */
  public static Put toHBasePut(WPut in) {
    if (in.isEmpty()) {
      throw new IllegalArgumentException("No columns to insert");
    }
    Put out = new Put(in.getRowKey());
    Map<byte[], List<WColumn>> familyMap = in.getFamilyMap();
    for (byte[] family : familyMap.keySet()) {
      for (WColumn col : familyMap.get(family)) {
        out.addColumn(col.getFamily(), col.getQualifier(), col.getTs(), col.getValue());
      }
    }
    return out;
  }

  /**
   * Converts multiple {@link Put}s (HBase) into a list of {@link WPut}s (Lindorm).
   *
   * @param in list of <code>Put</code>s to convert
   * @return list of converted <code>WPut</code>s
   */
  public static List<WPut> toLindormPuts(List<Put> in) throws IOException {
    List<WPut> out = new ArrayList<WPut>(in.size());
    for (Put put : in) {
      out.add(toLindormPut(put));
    }
    return out;
  }

  /**
   * Creates a {@link WPut} (Lindorm) from a {@link Put} (HBase)
   *
   * @param put the <code>Put</code> to convert
   * @return converted <code>WPut</code>
   */
  public static WPut toLindormPut(Put put)throws IOException {
    if (put.isEmpty()) {
      throw new IllegalArgumentException("No columns to insert");
    }
    checkPutSupport(put);
    WPut wput = new WPut(put.getRow());
//    wput.setWriteToWAL(!Durability.SKIP_WAL.equals(put.getDurability()));
    for (List<Cell> list : put.getFamilyCellMap().values()) {
      for (Cell kv : list) {
        wput.add(new WColumn(CellUtil.cloneFamily(kv), CellUtil.cloneQualifier(kv), CellUtil.cloneValue(kv),
            kv.getTimestamp(), convertTypeToWType(kv.getType().getCode())));
      }
    }
    copyAttributes(wput, put.getAttributesMap());
    return wput;
  }

  public static WIncrement toLindormIncrement(Increment inc) throws IOException {
    if (inc.isEmpty()) {
      throw new IllegalArgumentException("No incremented values found.");
    }
    checkIncrementSupport(inc);
    WIncrement winc = new WIncrement(inc.getRow(), inc.getTimeStamp());
//    winc.setWriteToWAL(!Durability.SKIP_WAL.equals(inc.getDurability()));
    for (List<Cell> list : inc.getFamilyCellMap().values()) {
      for (Cell kv : list) {
        winc.addColumn(CellUtil.cloneFamily(kv), CellUtil.cloneQualifier(kv),
            Bytes.toLong(CellUtil.cloneValue(kv)));
      }
    }
    winc.setTimeRange(inc.getTimeRange().getMin(), inc.getTimeRange().getMax());
    copyAttributes(winc, inc.getAttributesMap());

    return winc;
  }

  public static WAppend toLindormAppend(Append append) throws IOException {
    if (append.isEmpty()) {
      throw new IllegalArgumentException("No append values found.");
    }
    checkAppendSupport(append);
    WAppend wAppend = new WAppend(append.getRow(), append.getTimeStamp());
//    wAppend.setWriteToWAL(!Durability.SKIP_WAL.equals(append.getDurability()));
    for (List<Cell> list : append.getFamilyCellMap().values()) {
      for (Cell kv : list) {
        wAppend.add(CellUtil.cloneFamily(kv), CellUtil.cloneQualifier(kv),
            CellUtil.cloneValue(kv));
      }
    }
    wAppend.setReturnResults(append.isReturnResults());
    copyAttributes(wAppend, append.getAttributesMap());

    return wAppend;
  }

  /**
   * Create the closest row after the specified row
   */
  static byte[] createClosestRowAfter(byte[] row) {
    return Arrays.copyOf(row, row.length + 1);
  }

  public static WScan toLindormScan(Scan scan) throws IOException {
    checkScanSupport(scan);
    byte[] startRow = scan.getStartRow();
    byte[] stopRow = scan.getStopRow();
    if (!Bytes.equals(HConstants.EMPTY_START_ROW, startRow) && !scan.includeStartRow()) {
      startRow = createClosestRowAfter(startRow);
    }
    if (!Bytes.equals(HConstants.EMPTY_END_ROW, stopRow) && scan.includeStopRow()) {
      stopRow = createClosestRowAfter(stopRow);
    }
    WScan wScan = new WScan(startRow, stopRow);
    wScan.setFamilyMap(scan.getFamilyMap());
    wScan.setCacheBlocks(scan.getCacheBlocks());
    wScan.setCaching(scan.getCaching());
    if (scan.getMaxResultSize() > 0) {
      wScan.setMaxResultSize(scan.getMaxResultSize());
    }
    if (scan.getRowOffsetPerColumnFamily() > 0) {
      wScan.setOffset(scan.getRowOffsetPerColumnFamily());
    }
    if (scan.getMaxResultsPerColumnFamily() > 0) {
      wScan.setLimit(scan.getMaxResultsPerColumnFamily());
    }
    if (scan.getLimit() >= 0) {
      wScan.setLimit(scan.getLimit());
    }
    wScan.setReversed(scan.isReversed());
    if (scan.getMaxVersions() > 1) {
      wScan.setMaxVersions(scan.getMaxVersions());
    }
    TimeRange timeRange = scan.getTimeRange();
    if (!timeRange.isAllTime()) {
      wScan.setTimeRange(timeRange.getMin(), timeRange.getMax());
    }
    wScan.setRaw(scan.isRaw());
    wScan.setIsolationLevel(toLindormIsolationLevel(scan.getIsolationLevel()));
    if (scan.getFilter() != null) {
      wScan.setFilter(toLindormFilter(scan.getFilter()));
    }
    if (scan.getAttribute(ALLOW_FULL_TALBE_SCAN) != null) {
      boolean allowFilter = Bytes.toBoolean(scan.getAttribute(ALLOW_FULL_TALBE_SCAN));
      wScan.setAllowFiltering(allowFilter);
    }
    if (isQueryHotOnly(scan)) {
      wScan.setQueryHotOnly(true);
    } else if (isHotColdAutoMerge(scan)) {
      wScan.setHotColdAutoMerge(true);
    }
    copyAttributes(wScan, scan.getAttributesMap());
    return wScan;
  }

  private static WScan.IsolationLevel toLindormIsolationLevel(IsolationLevel isolationLevel)
      throws IOException {
    switch (isolationLevel) {
      case READ_COMMITTED:
        return WScan.IsolationLevel.READ_COMMITTED;
      case READ_UNCOMMITTED:
        return WScan.IsolationLevel.READ_UNCOMMITTED;
    }
    throw new IOException("Unkown isolationLevel " + isolationLevel);
  }

  private static void checkScanSupport(Scan in) throws IOException {
    /**
    if (in.getBatch() != -1) {
      throw new IOException("Scan#SetBatch() is not supported");
    }
    if (in.getMaxResultSize() != -1) {
      throw new IOException("Scan#SetMaxResultSize() is not supported");
    }
    if (in.getLimit() != -1) {
      throw new IOException("Scan#SetLimit() is not supported");
    }
    if (in.getLimit() != -1) {
      throw new IOException("Scan#SetLimit() is not supported");
    }
    if (!in.getIsolationLevel().equals(IsolationLevel.READ_COMMITTED)) {
      throw new IOException(
          "Scan#setIsolationLevel(READ_UNCOMMITTED) is not supported");
    }
    if (in.getReplicaId() != -1) {
      throw new IOException("Scan#setReplicaId() is not supported");
    }
    if (in.getACL() != null) {
      throw new IOException("Scan#setACL() is not supported");
    }
    if (in.getConsistency().equals(Consistency.TIMELINE)) {
      throw new IOException("Scan#setConsistency(TIMELINE) is not supported");
    }
    if (!in.getAttributesMap().isEmpty()) {
      if(in.getAttributesMap().size() > 1 || !in.getAttributesMap().containsKey("_raw_")) {
        throw new IOException("Scan#setAttribute() is not supported");
      }
    }
    if (in.getAllowPartialResults()) {
      throw new IOException("Scan#setAllowPartialResults() is not supported");
    }
    if (in.getLoadColumnFamiliesOnDemandValue() != null) {
      throw new IOException(
          "Scan#setLoadColumnFamiliesOnDemandValue() is not supported");
    }
    if (!in.getColumnFamilyTimeRange().isEmpty()) {
      throw new IOException("Scan#setColumnFamilyTimeRange() is not supported");
    }
    if (in.getRowOffsetPerColumnFamily() > 0) {
      throw new IOException("Scan#setRowOffsetPerColumnFamily() is not supported");
    }
    if (in.getMaxResultsPerColumnFamily() > 0) {
      throw new IOException("Scan#setMaxResultsPerColumnFamily() is not supported");
    }
     */
  }

  /**
   * Converts multiple {@link Get}s (HBase) into a list of {@link WGet}s (Lindorm).
   *
   * @param in list of <code>Get</code>s to convert
   * @return list of <code>WGet</code> objects
   * @throws IOException
   */
  public static List<WGet> toLindormGets(List<Get> in) throws IOException {
    List<WGet> out = new ArrayList<WGet>(in.size());
    for (Get get : in) {
      out.add(toLindormGet(get));
    }
    return out;
  }

  /**
   * Converts multiple {@link WResult}s (Lindorm) into a list of {@link Result}s
   * (HBase).
   *
   * @param in array of <code>WResult</code>s to convert
   * @return array of converted <code>Result</code>s
   */
  public static Result[] toHBaseResults(List<WResult> in) {
    List<Result> out = new ArrayList<Result>(in.size());
    for (WResult result : in) {
      out.add(toHBaseResult(result));
    }
    return out.toArray(new Result[0]);
  }

  public static Result[] toHBaseResults(WResult[] in) {
    List<Result> out = new ArrayList<Result>(in.length);
    for (WResult result : in) {
      out.add(toHBaseResult(result));
    }
    return out.toArray(new Result[0]);
  }

  public static Result[] toHBaseResults(WResult[] in, List<Get> gets) {
    List<Result> out = new ArrayList<Result>(in.length);
    for (int i = 0; i < in.length; i++) {
      out.add(toHBaseResult(in[i], gets.get(i).isCheckExistenceOnly()));
    }
    return out.toArray(new Result[0]);
  }

  public static Result toHBaseResult(WResult in) {
      return toHBaseResult(in, false);
  }

  /**
   * Create a {@link Result} (HBase) from a {@link WResult} (Lindorm)
   *
   * @param in
   * @return converted result
   */
  public static Result toHBaseResult(WResult in, boolean isCheckExistenceOnly) {
    Cell[] cells = new Cell[in.size()];
    int i = 0;
    for (WColumn c : in.raw()) {
      Cell cell = CellUtil
              .createCell(in.getRowKey(), c.getFamily(), c.getQualifier(), c.getTs(),
                      convertWTypeToHType(c.getType()).getCode(), c.getValue());
      cells[i] = cell;
      i++;
    }
    // sort in case lindorm may return unordered KVs.
    // this is necessary because Result#getColumnLatestCell use binary search to get specific column.
    Arrays.sort(cells, CellComparator.getInstance());

    return Result.create(cells, isCheckExistenceOnly ? !(cells.length == 0) : null, false);
  }

  public static HColumnDescriptor toHbasecolumnFamilyDescriptor(
      LindormTableDescriptor tableDescriptor, ColumnFamilyDescriptor in) {
    HColumnDescriptor builder = new HColumnDescriptor(in.getName());

    //Do not pass compress, data block encoding, dfsreplication
    builder.setValue(COMPRESSION, tableDescriptor.getCompression());
    builder.setValue(DATA_BLOCK_ENCODING, tableDescriptor.getDataBlockEncoding());
    //builder.setCompressionType(Compression.Algorithm.valueOf(tableDescriptor.getCompression()));
    //builder.setDataBlockEncoding(DataBlockEncoding.valueOf(tableDescriptor.getDataBlockEncoding()));
    if (tableDescriptor.getFamilyAttributes().getDfsReplicationNoDefault() != null) {
      builder.setDFSReplication(tableDescriptor.getDfsReplication());
    }
    builder.setMaxVersions(tableDescriptor.getMaxVersions());
    builder.setMinVersions(tableDescriptor.getMinVersions());
    builder.setTimeToLive(tableDescriptor.getTimeToLive());
    builder.setStoragePolicy(tableDescriptor.getStorageType());
    builder.setValue(COLD_BOUNDARY, tableDescriptor.getColdHotSeparateBoundary());
    for (int i = 1; i <= CHS_MAX_LAYER; i++) {
      String key = CHS_L + i;
      String value = tableDescriptor.getColdHotLayerConfig(i);
      builder.setValue(key, value);
    }
    builder.setValue(CHS_PROMOTE_ON_MAJOR,
        Boolean.toString(tableDescriptor.isColdHotPromoteOnMajor()));
    //column family attributes will override table level attributes
    builder.setBloomFilterType(BloomType.valueOf(in.getBloomFilterType()));
    builder.setValue(COMPRESSION, in.getCompression());
    builder.setValue(DATA_BLOCK_ENCODING, in.getDataBlockEncoding());
    if (in.getFamilyAttributes().getDfsReplicationNoDefault() != null) {
      builder.setDFSReplication(in.getDfsReplication());
    }
    builder.setMaxVersions(in.getMaxVersions());
    builder.setMinVersions(in.getMinVersions());
    builder.setTimeToLive(in.getTimeToLive());
    builder.setStoragePolicy(in.getStorageType());
    builder.setValue(COLD_BOUNDARY, in.getColdHotSeparateBoundary());
    builder.setBlockCacheEnabled(in.isBlockCacheEnabled());
    builder.setDFSReplication(tableDescriptor.getDfsReplication());
    // set major peirod;
    try {
      Long period = in.getCompactionMajorPeriodNoDefault();
      if (period != null) {
        builder.setConfiguration("hbase.hregion.majorcompaction", String.valueOf(period));
      }
      Long hotCompactionPeriod = in.getHotCompactionPeriod();
      if(hotCompactionPeriod != null){
        builder.setValue(HOT_COMPACTION_PERIOD, Long.toString(hotCompactionPeriod));
      }
    } catch (Throwable t) {
      // error setting major compaction period
    }
    for (int i = 1; i <= CHS_MAX_LAYER; i++) {
      String key = CHS_L + i;
      String value = in.getColdHotLayerConfig(i);
      builder.setValue(key, value);
    }
    builder.setValue(CHS_PROMOTE_ON_MAJOR,
        Boolean.toString(in.isColdHotPromoteOnMajor()));
    return builder;
  }

  public static ColumnFamilyDescriptor toLindormcolumnFamilyDescriptor(
      org.apache.hadoop.hbase.client.ColumnFamilyDescriptor in, boolean limitedDescriptorEnable) {
    ColumnFamilyDescriptor out = new ColumnFamilyDescriptor(in.getName());
    if(limitedDescriptorEnable){
      out.setTimeToLive(in.getTimeToLive(), TimeUnit.SECONDS);
      out.setMaxVersions(in.getMaxVersions());
      out.setMinVersions(in.getMinVersions());
    }else{
      out.setCompression(Bytes.toString(in.getValue(Bytes.toBytes(COMPRESSION))));
      out.setDataBlockEncoding(Bytes.toString(in.getValue(Bytes.toBytes(DATA_BLOCK_ENCODING))));
      out.setBlockCacheEnabled(in.isBlockCacheEnabled());
      // set major peirod;
      try {
        String majorPeriod = in.getConfigurationValue("hbase.hregion.majorcompaction");
        if (majorPeriod != null && majorPeriod.length() > 0) {
          long period = Long.valueOf(majorPeriod);
          out.setCompactionMajorPeriod(period);
        }
      } catch (Throwable t) {
        // error setting major compaction period
      }
      if (in.getDFSReplication() > 0 && in.getDFSReplication() <= 3) {
        out.setDfsReplication(in.getDFSReplication());
      }
      out.setMaxVersions(in.getMaxVersions());
      out.setMinVersions(in.getMinVersions());
      out.setTimeToLive(in.getTimeToLive(), TimeUnit.SECONDS);
      if (in.getStoragePolicy() != null) {
        String storageType = in.getStoragePolicy().toUpperCase();
        if (storageType.equals(STORAGETYPE_DEFAULT) || storageType.equals(STORAGETYPE_COLD)) {
          out.setStorageType(in.getStoragePolicy());
        } else {
          throw new IllegalArgumentException("only accept storage type:" +
              STORAGETYPE_DEFAULT + " or " + STORAGETYPE_COLD);
        }
      }
      byte[] key = Bytes.toBytes(COLD_BOUNDARY);
      String boundary = null;
      if (in.getValue(key) != null) {
        if (in.getStoragePolicy() != null &&
            in.getStoragePolicy().toUpperCase().equals(STORAGETYPE_COLD)) {
          throw new IllegalArgumentException("can not both set cold hot separate and cold storage type on family:"
              + in.getNameAsString());
        }
        boundary = Bytes.toString(in.getValue(key));
        if (StringUtils.isNullOrEmpty(boundary)) {
          boundary = null;
        } else {
          out.setColdHotSeparateBoundary(boundary);
        }
      }
      for (int i = 1; i <= CHS_MAX_LAYER && !StringUtils.isNullOrEmpty(boundary); i++) {
        String layer = CHS_L + i;
        key = Bytes.toBytes(layer);
        if (in.getValue(key) != null) {
          out.setColdHotLayerConfig(i, Bytes.toString(in.getValue(key)));
        }
      }
      // set default cold storage type to OSS
      if (!StringUtils.isNullOrEmpty(boundary)) {
        int oldestIndex = boundary.split(",").length + 1;
        if (oldestIndex <= CHS_MAX_LAYER) {
          key = Bytes.toBytes(CHS_L + oldestIndex);
          if (in.getValue(key) == null) {
            out.setColdHotLayerConfig(oldestIndex, COLD_HOT_COLD_STORAGETYPE_CONFIG);
          }
        }
      }
      key = Bytes.toBytes(CHS_PROMOTE_ON_MAJOR);
      if (in.getValue(key) != null) {
        String value = Bytes.toString(in.getValue(key));
        out.setColdHotPromoteOnMajor(Boolean.valueOf(value));
      }
      key = Bytes.toBytes(HOT_COMPACTION_PERIOD);
      if (in.getValue(key) != null) {
        String value = Bytes.toString(in.getValue(key));
        out.setHotCompactionPeriod(Long.valueOf(value));
      }
    }
    return out;
  }

  public static List<LindormFamilyAttributes> toLindormFamilyAttributes (
      org.apache.hadoop.hbase.client.ColumnFamilyDescriptor in, boolean limitedDescriptorEnable) throws IOException {
    List<LindormAttribute> lindormAttributes = new ArrayList<>();
    if(limitedDescriptorEnable){
      if (in.getTimeToLive() > 0) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.TTL,
            Integer.toString(in.getTimeToLive())));
      }
      if (in.getMaxVersions() > 0) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.MAX_VERSIONS,
            Integer.toString(in.getMaxVersions())));
      }
      if (in.getMinVersions() > 0) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.MIN_VERSIONS,
            Integer.toString(in.getMinVersions())));
      }
    }else {
      if (in.getValue(Bytes.toBytes(COMPRESSION)) != null) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.COMPRESSION,
            Bytes.toString(in.getValue(Bytes.toBytes(COMPRESSION)))));
      }
      if (in.getValue(Bytes.toBytes(DATA_BLOCK_ENCODING)) != null) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.DATA_BLOCK_ENCODING,
            Bytes.toString(in.getValue(Bytes.toBytes(DATA_BLOCK_ENCODING)))));
      }
      if (in.getMaxVersions() > 0) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.MAX_VERSIONS,
            Integer.toString(in.getMaxVersions())));
      }
      if (in.getMinVersions() > 0) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.MIN_VERSIONS,
            Integer.toString(in.getMinVersions())));
      }
      if (in.getTimeToLive() > 0) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.TTL,
            Integer.toString(in.getTimeToLive())));
      }
      if (in.getStoragePolicy() != null) {
        lindormAttributes.add(new LindormAttribute(LindormFamilyAttributeConstants.STORAGE_TYPE,
            in.getStoragePolicy()));
      }
    }
    LindormFamilyAttributes lindormFamilyAttributes = new LindormFamilyAttributes(Bytes.toString(in.getName()),
        lindormAttributes);
    return Arrays.asList(lindormFamilyAttributes);
  }

  public static HTableDescriptor toHbaseTableDescriptor(String namespace, LindormTableDescriptor in) {
    HTableDescriptor builder = new HTableDescriptor(TableName.valueOf(namespace, in.getName()));
    for (ColumnFamilyDescriptor column : in.getFamilies()) {
      if((column.getFamilyAttributes() == null || column.getFamilyAttributes().getAttributesMap().size() == 0)&& in.getFamilyAttributes()!= null){
        column.setFamilyAttributes(in.getFamilyAttributes());
      }
      builder.addFamily(toHbasecolumnFamilyDescriptor(in, column));
    }
    if(in.getTableAttributes() != null) {
      try {
        byte[] tableAttrs = LindormObjectUtils.getBytes(in.getTableAttributes());
        builder.setValue(Bytes.toBytes(TABLE_ATTRIBUTES_KEY), tableAttrs);
      } catch (IOException ioE) {
        throw new RuntimeException(
            "Error converting LindormTableDescriptor to HTableDescriptor" + in, ioE);
      }
    }
    if(in.getFamilyAttributes() != null) {
      try {
        byte[] familyAttrs = LindormObjectUtils.getBytes(in.getFamilyAttributes());
        builder.setValue(FAMILY_ATTRIBUTES_KEY_BYTES, familyAttrs);
      } catch (IOException ioE) {
        throw new RuntimeException(
          "Error converting LindormTableDescriptor to HTableDescriptor" + in, ioE);
      }
    }
    boolean isTableSerivce = false;
    if(in.getPkColumns() != null && in.getPkColumns().size() > 0){
      try {
        byte[] pkAttrs = getBytesFromPkColumn(in.getPkColumns());
        builder.setValue(PK_ATTRIBUTES_KEY_BYTES, pkAttrs);
        isTableSerivce = possibleTableServiceWithPk(in.getPkColumns());
      } catch (IOException ioE) {
        throw new RuntimeException(
          "Error converting LindormTableDescriptor to HTableDescriptor" + in, ioE);
      }
    }
    if(in.getNonPkColumns() != null && in.getNonPkColumns().size() > 0){
      try {
        byte[] nonPkAttrs = getBytesFromNonPkColumn(in.getNonPkColumns());
        builder.setValue(NONPK_ATTRIBUTES_KEY_BYTES, nonPkAttrs);
        isTableSerivce = possibleTableServiceWithNonPk(in.getNonPkColumns());
      } catch (IOException ioE) {
        throw new RuntimeException(
          "Error converting LindormTableDescriptor to HTableDescriptor" + in, ioE);
      }
    }
    builder.setValue(TABLESERVICE_ATTRIBUTES_KEY_BYTES, Bytes.toBytes(isTableSerivce));
    if (in.getOtherAttributes() != null && in.getOtherAttributes().getAttributes() != null) {
      for (Map.Entry<ImmutableBytesPtr, ImmutableBytesPtr> entry : in.getOtherAttributes().getAttributes().entrySet()) {
        builder.setValue(entry.getKey().get(), entry.getValue().get());
      }
    }
    builder.setValue(TABLEMETAVERSIONBYTES, Bytes.toBytes(in.getMetaVersion()));
    List<String> indexNames = in.getIndexNames();
    if (indexNames != null && !indexNames.isEmpty()) {
      builder.setValue(INDEX_NAMES_KEY_BYTES, SchemaUtils.indexNamesToBytes(indexNames));
    }
    Map<Byte, String> indexIds = in.getIndexIds();
    if (indexIds != null && !indexIds.isEmpty()) {
      try {
        builder.setValue(INDEX_ID_KEY_BYTES, SchemaUtils.indexIdToBytes(indexIds));
      } catch (IOException ioE) {
        throw new RuntimeException("Error converting LindormTableDescriptor to HTableDescriptor, index ids, " + in, ioE);
      }
    }
    Map<String, SilenceIndex> silenceIndexMap = in.getSilenceIndexs();
    if (silenceIndexMap != null && !silenceIndexMap.isEmpty()) {
      try {
        builder.setValue(SILENCE_INDEXES_KEY_BYTES, SchemaUtils.silenceIndexToBytes(silenceIndexMap));
      } catch (IOException ioE) {
        throw new RuntimeException("Error converting LindormTableDescriptor to HTableDescriptor, silence index, " + in, ioE);
      }
    }

    if (in.isDeferredLogFlush()) {
      builder.setDurability(Durability.ASYNC_WAL);
    } else {
      builder.setDurability(Durability.USE_DEFAULT);
    }
    if (in.getTableAttributes().getMemStoreFlushSizeNoDefault() != null) {
      builder.setMemStoreFlushSize(in.getTableAttributes().getMemStoreFlushSize());
    }
    //builder.setValue(MUTABILITY, in.getTableAttributes().getMutability().toString());
    return builder;
  }

  public static LindormTableDescriptor toLindormTableDescripter(TableDescriptor htd, boolean limitedDescriptorEnable) {
    LindormTableDescriptor lindormTableDesc = new LindormTableDescriptor(
        htd.getTableName().getQualifierAsString());
    for (org.apache.hadoop.hbase.client.ColumnFamilyDescriptor hbaseFamily : htd
        .getColumnFamilies()) {
      ColumnFamilyDescriptor column = toLindormcolumnFamilyDescriptor(hbaseFamily, limitedDescriptorEnable);
      lindormTableDesc.addFamily(column);
    }

    if(htd.getValues() != null) {
      BytesKeyAttributes otherAttributes = new BytesKeyAttributes();
      if(!limitedDescriptorEnable) {
        for (Map.Entry<Bytes, Bytes> entry : htd.getValues().entrySet()) {
          if (entry.getKey().compareTo(TABLEMETAVERSIONBYTES) == 0) {
            continue;
          }
          if (entry.getKey().compareTo(TABLESERVICE_ATTRIBUTES_KEY_BYTES) == 0) {
            continue;
          }
          // todo : use table-driven pattern to replace if-elseif pattern
          if (entry.getKey().compareTo(TABLE_ATTRIBUTES_KEY_BYTES) == 0) {
            TableAttributes tableAttributes = new TableAttributes();
            try {
              LindormObjectUtils.getWritable(entry.getValue().get(), tableAttributes);
              lindormTableDesc.setTableAttributes(tableAttributes);
            } catch (IOException ioE) {
              throw new RuntimeException(
                  "Error converting HTableDescriptor to LindormTableDescriptor" + htd, ioE);
            }
          } else if (entry.getKey().compareTo(INDEX_NAMES_KEY_BYTES) == 0) {
            List<String> indexNames = SchemaUtils.indexNamesFromBytes(entry.getValue().copyBytes());
            lindormTableDesc.setIndexNames(indexNames);
          } else if (entry.getKey().compareTo(INDEX_ID_KEY_BYTES) == 0) {
            try {
              Map<Byte, String> indexIds = SchemaUtils.indexIdFromBytes(entry.getValue().copyBytes());
              lindormTableDesc.setIndexIds(indexIds);
            } catch (IOException ioE) {
              throw new RuntimeException("Error converting HTableDescriptor to LindormTableDescriptor, index names, "
                    + "htd = " + htd, ioE);
            }
          } else if (entry.getKey().compareTo(SILENCE_INDEXES_KEY_BYTES) == 0) {
            try {
              Map<String, SilenceIndex> silenceIndexMap = SchemaUtils.silenceIndexFromBytes(entry.getValue().copyBytes());
              lindormTableDesc.setSilenceIndex(silenceIndexMap);
            } catch (IOException ioE) {
              throw new RuntimeException("Error converting HTableDescriptor to LindormTableDescriptor, silence index map, "
                      + "htd = " + htd, ioE);
            }
          } else if(entry.getKey().compareTo(FAMILY_ATTRIBUTES_KEY_BYTES) == 0) {
            FamilyAttributes familyAttributes = new FamilyAttributes();
            try {
              LindormObjectUtils.getWritable(entry.getValue().get(), familyAttributes);
              lindormTableDesc.setFamilyAttributes(familyAttributes);
            } catch (IOException ioE) {
              throw new RuntimeException(
                "Error converting HTableDescriptor to LindormTableDescriptor" + htd, ioE);
            }
          } else if(entry.getKey().compareTo(PK_ATTRIBUTES_KEY_BYTES) == 0) {
            try {
              List<PrimaryKeySchema> pks = getPkColumnFromBytes(entry.getValue().get());
              lindormTableDesc.setPkColumns(pks);
            } catch (IOException ioE) {
              throw new RuntimeException(
                "Error converting HTableDescriptor to LindormTableDescriptor" + htd, ioE);
            }
          } else if(entry.getKey().compareTo(NONPK_ATTRIBUTES_KEY_BYTES) == 0) {
            try {
              List<ColumnSchema> nonPks = getNonPkColumnFromBytes(entry.getValue().get());
              lindormTableDesc.setNonPkColumns(nonPks);
            } catch (IOException ioE) {
              throw new RuntimeException(
                "Error converting HTableDescriptor to LindormTableDescriptor" + htd, ioE);
            }
          } else {
            // if this is not a known tableAttributes, then set a as an 'other attribute'
            if (!TableAttributes.isTableAttributes(Bytes.toString(entry.getKey().get()))) {
              otherAttributes.set(new ImmutableBytesPtr(entry.getKey().get()), new ImmutableBytesPtr(entry.getValue().get()));
            }
            // no else branch here, because tableAttributes are covered in TABLE_ATTRIBUTES_KEY_BYTES attribute in HTD.
          }
        } //end for loop
      }
      lindormTableDesc.setOtherAttributes(otherAttributes);
    }

    byte[] metaVersion = htd.getValue(TABLEMETAVERSIONBYTES);
    if (metaVersion != null && metaVersion.length != 0) {
      lindormTableDesc.setMetaVersion(Bytes.toInt(metaVersion));
    }
    String mutabilityStr = htd.getValue(MUTABILITY);
    if (mutabilityStr != null) {
      Mutability mutability = Mutability.valueOf(mutabilityStr);
      lindormTableDesc.getTableAttributes().setMutability(mutability);
      // remove from HTD
    }
    lindormTableDesc.setDynamicColumnsEnabled(true);
    if (htd.getDurability().equals(Durability.ASYNC_WAL)) {
      lindormTableDesc.setDeferredLogFlush(true);
    } else {
      lindormTableDesc.setDeferredLogFlush(false);
    }
    if (htd.getMemStoreFlushSize() > 0) {
      lindormTableDesc.getTableAttributes().setMemStoreFlushSize(htd.getMemStoreFlushSize());
    }
    return lindormTableDesc;
  }

  /**
   * There are 3 types of table names in lindorm
   * a) tt
   * b) ns.tt
   * c) ns.tt.idx
   * We should support all of them.
   * @param table lindorm table name
   * @return hbase table name
   */
  public static TableName toHBaseTableName(String table) {
    int pos = table.indexOf(".");
    if (pos < 0) {
      // no '.' found, scenario a
      return TableName.valueOf(table);
    } else {
      // scenario b or c
      String namespace = table.substring(0, pos);
      String name = table.substring(pos + 1);
      return TableName.valueOf(namespace, name);
    }
  }

  public static WRowMutations toLindormWRowMutation(RowMutations rowMutations) throws IOException {
    WRowMutations wRowMutations = new WRowMutations(rowMutations.getRow());
    for (Mutation mutation : rowMutations.getMutations()) {
      if (mutation instanceof Put) {
        WPut wPut = toLindormPut((Put)mutation);
        wRowMutations.add(wPut);
      } else if (mutation instanceof Delete) {
        WDelete wDelete = toLindormDelete((Delete)mutation);
        wRowMutations.add(wDelete);
      } else  {
        throw new IOException("Only Put or Delete is supported in RowMutation: " + mutation);
      }
    }
    return wRowMutations;
  }


  public static DataType toInterpreterDataType(String interpreterClassName) {
    DataType dataType = INTERPRETER_DATATYPE_MAP.get(interpreterClassName);
    if (dataType == null) {
      throw new UnsupportedOperationException("Unsupported interpreterClass " + interpreterClassName);
    }
    return dataType;
  }

  public static byte[] toValueBytes(ColumnValue cv, ColumnInterpreter interpreterClassName) {
    BigDecimal bigDecimal = cv.getDecimal();
    if (interpreterClassName instanceof LongColumnInterpreter) {
      return Bytes.toBytes(bigDecimal.longValue());
    }
    if (interpreterClassName instanceof DoubleColumnInterpreter) {
      return Bytes.toBytes(bigDecimal.doubleValue());
    }
    if (interpreterClassName instanceof BigDecimalColumnInterpreter) {
      return Bytes.toBytes(bigDecimal);
    }
    throw new UnsupportedOperationException("Unsupported interpreterClass " + interpreterClassName.getClass());
  }

  public static String toLindormTableFullName(TableName tableName) {
    return tableName.getNamespaceAsString() + "." + tableName.getQualifierAsString();
  }

  private static void initLindormIndexFamilyDescriptor(LindormIndexDescriptor out, HColumnDescriptor in) {
    out.setCompression(in.getCompressionType().toString());
    out.setDataBlockEncoding(in.getDataBlockEncoding().name());
    out.setBlockCacheEnabled(in.isBlockCacheEnabled());
    if (in.getStoragePolicy() != null) {
      String storageType = in.getStoragePolicy().toUpperCase();
      if (storageType.equals(STORAGETYPE_DEFAULT) || storageType.equals(STORAGETYPE_COLD)) {
        out.setStorageType(in.getStoragePolicy());
      } else {
        throw new IllegalArgumentException("only accept storage type:" +
            STORAGETYPE_DEFAULT + " or " + STORAGETYPE_COLD);
      }
    }

    byte[] key = Bytes.toBytes(COLD_BOUNDARY);
    String boundary = null;
    if (in.getValue(key) != null) {
      if (in.getStoragePolicy() != null &&
          in.getStoragePolicy().toUpperCase().equals(STORAGETYPE_COLD)) {
        throw new IllegalArgumentException("can not both set cold hot separate and cold storage type on index:"
            + out.getIndexName());
      }
      boundary = Bytes.toString(in.getValue(key));
      if (StringUtils.isNullOrEmpty(boundary)) {
        boundary = null;
      } else {
        out.setColdHotSeparateBoundary(boundary);
      }
    }
    for (int i = 1; i <= CHS_MAX_LAYER && !StringUtils.isNullOrEmpty(boundary); i++) {
      String layer = CHS_L + i;
      key = Bytes.toBytes(layer);
      if (in.getValue(key) != null) {
        out.setColdHotLayerConfig(i, Bytes.toString(in.getValue(key)));
      }
    }
    // set default cold storage type to OSS
    if (!StringUtils.isNullOrEmpty(boundary)) {
      int oldestIndex = boundary.split(",").length + 1;
      if (oldestIndex <= CHS_MAX_LAYER) {
        key = Bytes.toBytes(CHS_L + oldestIndex);
        if (in.getValue(key) == null) {
          out.setColdHotLayerConfig(oldestIndex, COLD_HOT_COLD_STORAGETYPE_CONFIG);
        }
      }
    }
    key = Bytes.toBytes(CHS_PROMOTE_ON_MAJOR);
    if (in.getValue(key) != null) {
      String value = Bytes.toString(in.getValue(key));
      out.setColdHotPromoteOnMajor(Boolean.getBoolean(value));
    }
  }

  public static LindormIndexDescriptor toLindormIndexDescriptor(AliHBaseIndexDescriptor idxDesc) {
    LindormIndexDescriptor lid = new LindormIndexDescriptor(
            idxDesc.getIndexName(), idxDesc.getDataTable().getQualifierAsString());
    initLindormIndexFamilyDescriptor(lid, idxDesc.getFamilyDescriptor());

    for (AliHBaseColumn ic: idxDesc.getIndexedColumns()) {
      SortOrder order = SortOrder.getDefault();
      if (ic.getSortOrder() != null) {
        order = ic.getSortOrder() == AliHBaseColumn.SortOrder.ASC ? SortOrder.ASC : SortOrder.DESC;
      }
      IndexedColumnSchema ics = new IndexedColumnSchema(ic.getFamily(), ic.getQualifier(), order);
      lid.addIndexedColumns(ics);
    }
    for (AliHBaseColumn cc: idxDesc.getCoveredColumns()) {
      if (cc == AliHBaseColumn.COVER_ALL) {
        lid.addCoveredColumn(LindormIndexDescriptor.COVERED_DYNAMIC_COLUMNS);
        // ignore other covered columns, because it is covering all columns now.
        break;
      }
      lid.addCoveredColumn(new ColumnKey(cc.getFamily(), cc.getQualifier()));
    }

    // set index state
    lid.setIndexState(IndexState.BUILDING);
    return lid;
  }

  public static AliHBaseIndexDescriptor toAliHBaseIndexDescriptor(String namespace, LindormIndexDescriptor desc) {
    String indexName = desc.getIndexName();
    TableName dataTableName = TableName.valueOf(namespace, desc.getDataTableName());
    AliHBaseIndexDescriptor ret = new AliHBaseIndexDescriptor(indexName, dataTableName);

    for (IndexedColumnSchema col : desc.getIndexedColumns()) {
      if (isRowkeyField(col.getColumnKey())) {
        if (col.getSortOrder().equals(SortOrder.DESC)) {
          ret.addIndexedColumn(null, col.getColumnKey().getQualifier(),
              AliHBaseColumn.SortOrder.fromImplSortOrder(col.getSortOrder()));
        }
      } else {
        ret.addIndexedColumn(getFamilyName(col.getColumnKey()), col.getColumnKey().getQualifier(),
            AliHBaseColumn.SortOrder.fromImplSortOrder(col.getSortOrder()));
      }
    }

    for (ColumnKey ck : desc.getCoveredColumns()) {
      if (ck == LindormIndexDescriptor.COVERED_DYNAMIC_COLUMNS) {
        ret.setCoveredAllColumns();
        break;
      } else {
        ret.addCoveredColumn(getFamilyName(ck), ck.getQualifier());
      }
    }

    return ret;
  }

  private static boolean isRowkeyField(ColumnKey ck) {
    byte[] f = ck.getFamily();
    return (f == null || f.length == 0) && Arrays.equals(ck.getQualifier(), ROWKEY_NAME);
  }

  private static byte[] getFamilyName(ColumnKey ck) {
    byte[] family = ck.getFamily();
    if (family == null || family.length == 0) {
      family = SchemaUtils.DEFAULT_FAMILY_NAME_BYTES;
    }
    return family;
  }

  public static String indexSchemaToString(List<AliHBaseIndexDescriptor> indexes, String dataTableName) {
    if (indexes == null || indexes.isEmpty()) {
      return "No index found for table " + dataTableName;
    } else {
      StringBuilder str = new StringBuilder();
      for (AliHBaseIndexDescriptor idx : indexes) {
        str.append(idx.getSchema());
        str.append("\n");
      }
      str.append("Total " + indexes.size() + " indexes found for table " + dataTableName);
      return str.toString();
    }
  }

  private static void copyAttributes(Attributes attributes, Map<String, byte[]> attributesMap) {
    for(Map.Entry<String, byte[]> entry : attributesMap.entrySet()) {
      attributes.setAttribute(entry.getKey(), entry.getValue());
    }
  }

  /**
   * TODO PhoenixOnLindorm : ugly, find a proper way to do this conversion
   */
  public static TableName handlePhoenixSystemTable(TableName name) {
    String tableName = name.getQualifierAsString();
    if (tableName.startsWith("SYSTEM.")) {
      tableName = tableName.substring(7);
      return TableName.valueOf("SYSTEM", tableName);
    } else {
      return name;
    }
  }

  public static IOException toHBaseIOException(Throwable t){
    if(t == null){
      return null;
    }
    if(t instanceof HBaseIOException || t instanceof org.apache.hadoop.hbase.DoNotRetryIOException){
      return (IOException) t;
    }
    Throwable cause = t;
    if (t instanceof LindormException && t.getCause() != null){
      cause = t.getCause();
    }
    if (cause instanceof DoNotRetryIOException) {
      if (cause instanceof TableExistsException ||
        cause.getMessage().contains(TableExistsException.class.getName())) {
        return new org.apache.hadoop.hbase.TableExistsException(cause.getMessage());
      }
      if (cause instanceof TableNotFoundException ||
        cause.getMessage().contains(TableNotFoundException.class.getName())) {
        return new org.apache.hadoop.hbase.TableNotFoundException(cause.getMessage());
      }
      if (cause instanceof TableNotEnabledException ||
        cause.getMessage().contains(TableNotEnabledException.class.getName())) {
        return new org.apache.hadoop.hbase.TableNotEnabledException(cause.getMessage());
      }
      if (cause instanceof NameSpaceNotFoundException ||
        cause.getMessage().contains(NameSpaceNotFoundException.class.getName())) {
        return new NamespaceNotFoundException(cause.getMessage());
      }
      if (cause instanceof NoSuchColumnFamilyException ||
        cause.getMessage().contains(NoSuchColumnFamilyException.class.getName())) {
        return new org.apache.hadoop.hbase.regionserver.NoSuchColumnFamilyException(cause.getMessage());
      }
      if (cause instanceof NotAllMetaRegionsOnlineException ||
        cause.getMessage().contains(NotAllMetaRegionsOnlineException.class.getName())) {
        return new org.apache.hadoop.hbase.NotAllMetaRegionsOnlineException(cause.getMessage());
      }
      if (cause instanceof TableNotDisabledException ||
        cause.getMessage().contains(TableNotDisabledException.class.getName())) {
        return new org.apache.hadoop.hbase.TableNotDisabledException(cause.getMessage());
      }
      if (cause instanceof LeaseException ||
        cause.getMessage().contains(LeaseException.class.getName())) {
        return new org.apache.hadoop.hbase.regionserver.LeaseException(cause.getMessage());
      }
      if (cause instanceof UnknownScannerException ||
        cause.getMessage().contains(UnknownScannerException.class.getName())) {
        return new org.apache.hadoop.hbase.UnknownScannerException(cause.getMessage());
      }
      if (cause instanceof AccessDeniedException ||
        cause.getMessage().contains(AccessDeniedException.class.getName())) {
        return new org.apache.hadoop.hbase.security.AccessDeniedException(cause.getMessage());
      }
      if (cause instanceof QuotaExceededException ||
        cause.getMessage().contains(QuotaExceededException.class.getName())) {
        return new org.apache.hadoop.hbase.quotas.QuotaExceededException(cause.getMessage(),
          cause.getCause());
      }
      if (cause instanceof UnknownProtocolException ||
        cause.getMessage().contains(UnknownProtocolException.class.getName())) {
        return new org.apache.hadoop.hbase.exceptions.UnknownProtocolException(cause.getMessage());
      }
      return new org.apache.hadoop.hbase.DoNotRetryIOException(cause.getMessage(), cause.getCause());
    }else{
      return new HBaseIOException(cause.getMessage(), cause.getCause());
    }
  }

  public static byte[] getBytesFromPkColumn(List<PrimaryKeySchema> pks) throws IOException{
      ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
      DataOutputStream out = new DataOutputStream(byteStream);
      byte[] result;
      try {
        WritableUtils.writeVInt(out, pks.size());
        for(PrimaryKeySchema pk : pks){
          pkColumnWriteTo(out, pk);
        }
        out.close();
        out = null;
        result = byteStream.toByteArray();
      } finally {
        if (out != null) {
          out.close();
        }
      }
      return result;
  }

  public static byte[] getBytesFromNonPkColumn(List<ColumnSchema> nonPks) throws IOException{
    ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
    DataOutputStream out = new DataOutputStream(byteStream);
    byte[] result;
    try {
      WritableUtils.writeVInt(out, nonPks.size());
      for(ColumnSchema  nonPk : nonPks){
        nonPkColumnWriteTo(out, nonPk);
      }
      out.close();
      out = null;
      result = byteStream.toByteArray();
    } finally {
      if (out != null) {
        out.close();
      }
    }
    return result;
  }

  public static List<PrimaryKeySchema> getPkColumnFromBytes(byte[] bytes) throws IOException{
    DataInputBuffer in = new DataInputBuffer();
    in.reset(bytes, bytes.length);
    int count = WritableUtils.readVInt(in);
    List<PrimaryKeySchema> pks = new ArrayList<>();
    for (int i = 0; i < count; i++) {
      PrimaryKeySchema pk = pkColumnReadFrom(in);
      pks.add(pk);
    }
    return pks;
  }

  public static List<ColumnSchema> getNonPkColumnFromBytes(byte[] bytes) throws IOException{
    DataInputBuffer in = new DataInputBuffer();
    in.reset(bytes, bytes.length);
    int count = WritableUtils.readVInt(in);
    List<ColumnSchema> nonPks = new ArrayList<>();
    for (int i = 0; i < count; i++) {
      ColumnSchema nonPk = nonPkColumnReadFrom(in);
      nonPks.add(nonPk);
    }
    return nonPks;
  }

  public static void pkColumnWriteTo(DataOutputStream out, PrimaryKeySchema pk) throws IOException{
    if(pk == null){
      return;
    }
    WritableUtils.writeString(out, "name");
    Bytes.writeByteArray(out, pk.getName());
    WritableUtils.writeString(out, "dataType");
    Bytes.writeByteArray(out, Bytes.toBytes(pk.getDataType().toString()));
    WritableUtils.writeString(out, "maxLength");
    WritableUtils.writeVInt(out, pk.getMaxLength() != null ? pk.getMaxLength() : -1);
    WritableUtils.writeString(out, "scale");
    WritableUtils.writeVInt(out, pk.getScale() != null ? pk.getMaxLength() : -1);
    WritableUtils.writeString(out, "sortOrder");
    Bytes.writeByteArray(out, Bytes.toBytes(pk.getSortOrder().toString()));
    WritableUtils.writeString(out, "hashed");
    WritableUtils.writeVInt(out, pk.isHashed()?1:0);
  }

  public static PrimaryKeySchema pkColumnReadFrom(DataInputBuffer in) throws IOException{
    PrimaryKeySchema pk = new PrimaryKeySchema();
    setPkAttributes(in, pk,  WritableUtils.readString(in));
    setPkAttributes(in, pk,  WritableUtils.readString(in));
    setPkAttributes(in, pk,  WritableUtils.readString(in));
    setPkAttributes(in, pk,  WritableUtils.readString(in));
    return pk;
  }

  public static void setPkAttributes(DataInputBuffer in, PrimaryKeySchema pk, String name) throws IOException {
    switch (name){
      case "name":
        pk.setName(Bytes.readByteArray(in));
        break;
      case "dataType":
        DataType dataType = DataType.valueOf(Bytes.toString(Bytes.readByteArray(in)));
        WritableUtils.readString(in);
        int maxLength =  WritableUtils.readVInt(in);
        WritableUtils.readString(in);
        int scale =  WritableUtils.readVInt(in);
        if(maxLength == -1){
          pk.setType(dataType);
        }else if(scale == -1){
          pk.setType(dataType, maxLength);
        }else{
          pk.setType(dataType, maxLength, scale);
        }
        break;
      case "sortOrder":
        pk.setSortOrder(SortOrder.valueOf(Bytes.toString(Bytes.readByteArray(in))));
        break;
      case "hashed":
        pk.setHashed(WritableUtils.readVInt(in) > 0? true : false);
        break;
      default:
        throw new IOException("set pk attributes failed, unsupported attribute " + name);
    }
  }

  public static void nonPkColumnWriteTo(DataOutputStream out, ColumnSchema nonPk) throws IOException{
    if(nonPk == null){
      return;
    }
    WritableUtils.writeString(out, "familyName");
    Bytes.writeByteArray(out, nonPk.getFamilyName());
    WritableUtils.writeString(out, "columnName");
    Bytes.writeByteArray(out, nonPk.getColumnName());
    WritableUtils.writeString(out, "dataType");
    Bytes.writeByteArray(out, Bytes.toBytes(nonPk.getDataType().toString()));
    WritableUtils.writeString(out, "maxLength");
    WritableUtils.writeVInt(out, nonPk.getMaxLength() != null ? nonPk.getMaxLength() : -1);
    WritableUtils.writeString(out, "scale");
    WritableUtils.writeVInt(out, nonPk.getScale() != null ? nonPk.getScale() : -1);
  }

  public static ColumnSchema nonPkColumnReadFrom(DataInputBuffer in) throws IOException{
    ColumnSchema nonPk = new ColumnSchema();
    setNonPkAttributes(in, nonPk,  WritableUtils.readString(in));
    setNonPkAttributes(in, nonPk,  WritableUtils.readString(in));
    setNonPkAttributes(in, nonPk,  WritableUtils.readString(in));
    return nonPk;
  }

  public static void setNonPkAttributes(DataInputBuffer in, ColumnSchema nonPk, String name) throws IOException {
    switch (name){
      case "familyName":
        nonPk.setFamilyName(Bytes.readByteArray(in));
        break;
      case "columnName":
        nonPk.setColumnName(Bytes.readByteArray(in));
        break;
      case "dataType":
        DataType dataType = DataType.valueOf(Bytes.toString(Bytes.readByteArray(in)));
        WritableUtils.readString(in);
        int maxLength =  WritableUtils.readVInt(in);
        WritableUtils.readString(in);
        int scale =  WritableUtils.readVInt(in);
        if(maxLength == -1){
          nonPk.setDataType(dataType);
        }else if(scale == -1){
          nonPk.setDataType(dataType, maxLength);
        }else{
          nonPk.setDataType(dataType, maxLength, scale);
        }
        break;
      default:
        throw new IOException("set nonpk attributes failed, unsupported attribute " + name);
    }
  }

  public static boolean possibleTableServiceWithPk(List<PrimaryKeySchema> pks) {
    if (pks.size() != 1) {
      return true;
    }
    PrimaryKeySchema pk = pks.get(0);
    if (pk != null && "ROW".equals(Bytes.toString(pk.getName())) && pk.getDataType() == DataType.VARBINARY) {
      return false;
    }
    return true;
  }

  public static boolean possibleTableServiceWithNonPk(List<ColumnSchema> nonPks) {
    if (nonPks.size() != 1) {
      return true;
    }
    ColumnSchema nonPk = nonPks.get(0);
    if (nonPk != null && "COL".equals(Bytes.toString(nonPk.getColumnName())) && nonPk.getDataType() == DataType.VARBINARY) {
      return false;
    }
    return true;
  }
}

