package com.alibaba.hbase.haclient;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.alibaba.hbase.client.AliHBaseConstants;
import com.alibaba.hbase.protobuf.generated.ClusterSwitchProto;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;
import org.apache.hadoop.hbase.zookeeper.ZKWatcher;
import org.apache.hadoop.hbase.zookeeper.ZNodePaths;
import org.apache.zookeeper.KeeperException;

public class ConnectInfoUtil {
  private static Log LOG = LogFactory.getLog(ConnectInfoUtil.class);

  public static final String CONNECTINFO_DIR = "hbase.haclient.connectinfo.dir";
  public static final String CONNECTINFO_DIR_DEFATULT = "/alihbase/connectinfo/";
  public static final String CONNECTINFO_FILE = "hbase.haclient.connectinfo.filename";
  public static final String CONNECTINFO_FILE_DEFAULT = "connectinfo.xml";
  public static final String WATCH_ZK_PREFIX = "hbase.haclient.watch.zk.";
  public static final String CONNECT_CONF_ACTIVE = "hbase.haclient.connect.active";
  public static final String CONNECT_CONF_STANDBY = "hbase.haclient.connect.standby";
  public static final String CONNECTINFO_ZK_CLUSTER = "hbase.connect.info.zk";
  public static final String LINK_RETRY_COUNT = "hbase.link.retry.count";

  private static void checkConnectInfo(List<String> watchZkList, String active, String standby,
                                       String zkClusterKey) throws IOException{
    if(watchZkList == null || watchZkList.isEmpty()){
      throw new IOException("watch zk list is empty");
    }
    for(String zk : watchZkList){
      if(!ClusterSwitchUtil.isValidClusterKey(zk)){
        throw new IOException("ZK " + zk + " in watch zk list is invaild");
      }
    }
    if(!ClusterSwitchUtil.isValidConnectKey(active)){
      throw new IOException("ConnnectConf:active " + active + " is invaild");
    }
    if(!ClusterSwitchUtil.isValidConnectKey(standby)){
      throw new IOException("ConnnectConf:standby " + standby + " is invaild");
    }
    if(zkClusterKey != null && !ClusterSwitchUtil.isValidClusterKey(zkClusterKey)){
      throw new IOException("ConnnectConf:zkClusterKey " + zkClusterKey + " is invaild");
    }
  }

  public static ConnectInfo toConnectInfo(byte[] data) throws IOException {
    if (data == null) {
      return  null;
    }
    try {
      ProtobufUtil.expectPBMagicPrefix(data);
      int prefixLen = ProtobufUtil.lengthOfPBMagic();
      ClusterSwitchProto.ConnectInfo.Builder builder =
        ClusterSwitchProto.ConnectInfo.newBuilder();
      ProtobufUtil.mergeFrom(builder, data, prefixLen, data.length - prefixLen);
      ClusterSwitchProto.ConnectInfo connectInfo = builder.build();
      List<String> watchZkList = connectInfo.getWatchZkListList();
      ClusterSwitchProto.ConnectInfo.ConnectConf connectConf = connectInfo.getConnectConf();
      String active = null;
      String standby = null;
      if(null == connectConf){
        throw new IOException("ConnectConf is null");
      }else{
        active = connectConf.getActive();
        standby = connectConf.getStandby();
      }
      checkConnectInfo(watchZkList, active, standby, null);
      return new ConnectInfo(watchZkList, active, standby);
    } catch (DeserializationException e) {
      throw new IOException(e);
    }
  }


  /**
   * Get ConnectInfo from zk
   * @param endpoint
   * @return ConnectInfo
   * @throws IOException
   * @throws KeeperException
   */
  public static ConnectInfo getConnectInfoFromZK(String endpoint, Configuration clientConf)
    throws IOException, KeeperException {
    Configuration conf = new Configuration();
    if (!ClusterSwitchUtil.isValidEndpoint(endpoint)) {
      throw new IOException("Endpoint " + endpoint + " has invalid format, vaild format is " +
        "hostname:port");
    }
    conf.set(HConstants.ZOOKEEPER_QUORUM, endpoint.split(":")[0]);
    conf.set(HConstants.ZOOKEEPER_CLIENT_PORT, endpoint.split(":")[1]);
    ZKWatcher zk = new ZKWatcher(conf, "GetCommand", null, false);
    try {
      String baseNode = ClusterSwitchUtil.getBaseNode(clientConf.get(AliHBaseConstants.HACLIENT_BASE_NODE,
        AliHBaseConstants.HACLIENT_BASE_NODE_DEFAULT), AliHBaseConstants.getHaClusterID(clientConf));
      if (ZKUtil.checkExists(zk, baseNode) == -1) {
        throw new IOException("baseNode " + baseNode + " is not exist");
      }
      String connectNode = ZNodePaths.joinZNode(baseNode,
        clientConf.get(ClusterSwitchUtil.ZOOKEEPER_CONNECT_NODE,
          ClusterSwitchUtil.ZOOKEEPER_CONNECT_NODE_DEFAULT));
      if (ZKUtil.checkExists(zk, connectNode) == -1) {
        throw new IOException("connectNode " + connectNode + " is not exist");
      } else {
        byte[] data = ZKUtil.getDataNoWatch(zk, connectNode, null);
        ConnectInfo connectInfo = toConnectInfo(data);
        return connectInfo;
      }
    } finally {
      if (zk != null) {
        zk.close();
      }
    }
  }

  /**
   * if can not connect to endpoint get connectInfo from xml
   * @return ConnectInfo
   */
  public static ConnectInfo getConnectInfoFromXML(Configuration conf){
    return getConnectInfoFromXML(System.getProperty("user.home") + conf.get(CONNECTINFO_DIR,
      CONNECTINFO_DIR_DEFATULT) + conf.get(CONNECTINFO_FILE, CONNECTINFO_FILE_DEFAULT));
  }


  /**
   * get connectInfo from xml
   * @param file
   * @return ConnectInfo
   */
  public static ConnectInfo getConnectInfoFromXML(String file){
    try{
      Configuration conf  = new Configuration();
      conf.addResource(new Path(file));
      String active = conf.get(CONNECT_CONF_ACTIVE, null);
      String standby = conf.get(CONNECT_CONF_STANDBY, null);
      List<String> watchZkList = new ArrayList<>();
      Iterator<Map.Entry<String, String>> iterator = conf.iterator();
      while(iterator.hasNext()){
        Map.Entry<String, String> entry = iterator.next();
        if(entry.getKey().startsWith(WATCH_ZK_PREFIX)){
          watchZkList.add(entry.getValue());
        }
      }
      String zkClusterKey = conf.get(CONNECTINFO_ZK_CLUSTER, null);
      checkConnectInfo(watchZkList, active ,standby, zkClusterKey);
      return new ConnectInfo(watchZkList, active ,standby, zkClusterKey);
    } catch(Exception e){
      LOG.error("Get connnect info from xml failed, " + e);
      return null;
    }
  }

  /**
   * flush connnect info to user home dir, if can not connect to endpoint
   * get connectInfo from disk
   * @param connectInfo
   */
  public static void flushConnectInfo(ConnectInfo connectInfo, Configuration clientConf){
    Configuration conf  = new Configuration();
    conf.clear();
    List<String> zkWatchList = connectInfo.getWatchZkList();
    for (int i = 0; i < zkWatchList.size(); i++) {
      conf.set(WATCH_ZK_PREFIX + (i + 1), zkWatchList.get(i));
    }
    conf.set(CONNECT_CONF_ACTIVE, connectInfo.getConnectConf().getActive());
    conf.set(CONNECT_CONF_STANDBY, connectInfo.getConnectConf().getStandby());
    if(connectInfo.getZkClusterKey() != null){
      conf.set(CONNECTINFO_ZK_CLUSTER, connectInfo.getZkClusterKey());
    }
    String dirName = System.getProperty("user.home") + clientConf.get(CONNECTINFO_DIR, CONNECTINFO_DIR_DEFATULT);
    String fileName = dirName + clientConf.get(CONNECTINFO_FILE, CONNECTINFO_FILE_DEFAULT);
    File dir = new File(dirName);
    if(!dir.exists()){
      dir.mkdirs();
    }

    FileOutputStream out = null;
    try{
      File file = new File(fileName);
      if(!file.exists()){
        file.createNewFile();
      }
      out =
        new FileOutputStream(file);
      conf.writeXml(out);
      out.flush();
    } catch(Exception e){
      LOG.warn("flush connect info failed : " + e);
    } finally {
      try {
        if (out != null) {
          out.close();
        }
      }catch(IOException ioe){
        LOG.debug("flushConnectInfo FileOutputStream close failed");
      }
    }
  }

  /**
   * flush connnect info to user home dir, if can not connect to endpoint
   * get connectInfo from disk
   * @param connectInfo
   */
  public static boolean isVaildTargetConnectKey(ConnectInfo connectInfo, String connectKey){
    return connectInfo.getConnectConf().getActive().equals(connectKey.trim()) ||
      connectInfo.getConnectConf().getStandby().equals(connectKey.trim());
  }

  public static byte[] toConnectInfoBytes(ConnectInfo connectInfo) {
    ClusterSwitchProto.ConnectInfo.Builder builder = ClusterSwitchProto.ConnectInfo.newBuilder();
    List<String> watchZkList = connectInfo.getWatchZkList();
    for (String zk : watchZkList) {
      builder.addWatchZkList(zk);
    }
    ClusterSwitchProto.ConnectInfo.ConnectConf.Builder connectConfBuilder =
      ClusterSwitchProto.ConnectInfo.ConnectConf.newBuilder();
    connectConfBuilder.setActive(connectInfo.getActiveConnectKey());
    connectConfBuilder.setStandby(connectInfo.getStandbyConnectKey());
    builder.setConnectConf(connectConfBuilder.build());

    return ProtobufUtil.prependPBMagic(builder.build().toByteArray());
  }

  public static String getZkFromConnectInfo(ConnectInfo connectInfo, String clusterKey){
    if(ClusterSwitchUtil.isValidClusterKey(clusterKey)){
      return clusterKey;
    }
    List<String> watchZkList = connectInfo.getWatchZkList();
    if(clusterKey.equals(connectInfo.getActiveConnectKey())){
      return watchZkList.get(0);
    }else{
      if(watchZkList.size() > 1){
        return watchZkList.get(1);
      }
      return watchZkList.get(0);
    }
  }

  public static String getClusterKeyFromConnectInfo(ConnectInfo connectInfo, String zkCluster){
    List<String> watchZkList = connectInfo.getWatchZkList();
    if(watchZkList.size() > 1){
      if(zkCluster.equals(watchZkList.get(0))){
        return connectInfo.getActiveConnectKey();
      }else if(zkCluster.equals(watchZkList.get(1))){
        return connectInfo.getStandbyConnectKey();
      }else{
        LOG.info("ZkCluster " + zkCluster + " is not match zk in watch list, so return active " +
          "cluster key");
        return connectInfo.getActiveConnectKey();
      }
    }else{
      return connectInfo.getActiveConnectKey();
    }

  }

  public static Configuration getConfFromEndpoint(String endpoint, Configuration clientConf) {
    Configuration conf = new Configuration(clientConf);
    conf.set(HConstants.ZOOKEEPER_QUORUM, endpoint.split(":")[0]);
    conf.set(HConstants.ZOOKEEPER_CLIENT_PORT, endpoint.split(":")[1]);
    return conf;
  }
}